// ****************************************************************************
// * Piece.java                                                               *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;

class Piece implements ResultProcessor
{
  static final int kExplosionDelay=1;
  static final int kExplosionBatchSize=10;
  static final int kExplosionFillNumber=5000;
  static final int kExplosionEmptyNumber=8000;

  // key to piece types
  static final int kTriangle=0, kSquare=1, kDiagMirror=2;
  static final int kStraightMirror=3, kBeamSplitter=4, kHypercube=5;
  static final int kLaser=6, kKing=7;
  static final double kPieceBorderRatio = 0.12;

  // Key to orientations.  Possible orientations are listed starting from zero.
  //    - triangle:        mirror faces NW, NE, SE, SW
  //    - square:          mirror faces N, E, S, W
  //    - diag mirror:     mirror faces NW/SE, NE/SW
  //    - straight mirror: mirror faces N/S, E/W
  //    - beam splitter:   tip of mirror faces N, E, S, W
  //    - hypercube:       orientation is meaningless
  //    - laser:           laser faces N, E, S, W
  //    - king:            orientation is meaningless

  int team;          // which team to I belong to
  int type;          // see key above
  int ori;           // current orientation (see key, above)
  int startori;      // orientation before current pending move
  Board board;
  int col, row;
  boolean held;      // is this piece being held "in hand" by the cursor?
  boolean doomed;    // is this piece about to be destroyed?
  boolean spent;     // has this aggressor been used yet this turn?
  int left, right, top, bottom;      // inside (drawable) square
  int left2, right2, top2, bottom2;  // outside (entire) square
  int cx, cy;                        // center coordinates
  int w, h;                          // width and height
  boolean killOwnPiece;    // set by confirmation dialog

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  Piece(int type, int team, int ori, int col, int row, Board board)
  {
    this.team = team;
    this.type = type;
    this.ori = ori;
    this.col = col;
    this.row = row;
    this.board = board;
    held = spent = doomed = false;
    killOwnPiece = false;
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void ComputeBoundaries()
  {
    int pieceBorder;

    left2 = board.GetX(col)+3;
    top2 = board.GetY(row)+3;
    right2 = board.GetX(col+1)-3;
    bottom2 = board.GetY(row+1)-3;
    pieceBorder = (int) ((right2-left2)*kPieceBorderRatio);
    left = left2 + pieceBorder;
    right = right2 - pieceBorder;
    pieceBorder = (int) ((bottom2-top2)*kPieceBorderRatio);
    top = top2 + pieceBorder;
    bottom = bottom2 - pieceBorder;
    cx = (left+right)/2;
    cy = (top+bottom)/2;
    w = right - left;
    h = bottom - top;
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Erase() { Erase(board.getGraphics()); }
  void Erase(Graphics g)
  {
    if (g == null)
      return;
    ComputeBoundaries();

    g.clearRect(left2, top2, (right2-left2), (bottom2-top2));
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Draw() { Draw(board.getGraphics()); }
  void Draw(Graphics g)
  {
    PieceColor color;

    if (g == null)
      return;
    ComputeBoundaries();

    if (doomed)
      color = ColorManager.DoomedPiece();
    else if (held)
      color = ColorManager.HeldPiece();
    else
      color = ColorManager.PlayerColor(team);

    switch (type) {
    case kTriangle:
      switch (ori) {
      case 0: DoTriangle(g, color, right-1, bottom-1, left, top); break;
      case 1: DoTriangle(g, color, left, bottom-1, right-1, top); break;
      case 2: DoTriangle(g, color, left, top, right-1, bottom-1); break;
      case 3: DoTriangle(g, color, right-1, top, left, bottom-1); break;
      }
      break;
      
    case kHypercube:
      g.setColor(color.r);
      MyUtils.DrawDottedRect(g, left, top, right-left, bottom-top, 1, 0);
      MyUtils.DrawDottedRect(g, left+1, top+1, right-left-2, bottom-top-2, 1, 1);
      MyUtils.DrawDottedRect(g, left+2, top+2, right-left-4, bottom-top-4, 1, 0);
      MyUtils.DrawDottedRect(g, left+3, top+3, right-left-6, bottom-top-6, 1, 1);
      break;

    case kSquare:
      g.setColor(color.r);
      g.fillRect(left, top, right-left, bottom-top);
      g.setColor(color.h);
      switch (ori) {
      case 0:
	g.drawLine(left, top, right, top); 
	g.drawLine(left, top+1, right, top+1); 
	g.drawLine(left, top+2, right, top+2); 
	break;
      case 1:
	g.drawLine(right, top, right, bottom);
	g.drawLine(right-1, top, right-1, bottom);
	g.drawLine(right-2, top, right-2, bottom);
	break;
      case 2:
	g.drawLine(left, bottom, right, bottom);
	g.drawLine(left, bottom-1, right, bottom-1);
	g.drawLine(left, bottom-2, right, bottom-2);
	break;
      case 3:
	g.drawLine(left, top, left, bottom);
	g.drawLine(left+1, top, left+1, bottom);
	g.drawLine(left+2, top, left+2, bottom);
	break;
      }
      break;
      
    case kDiagMirror:
      g.setColor(color.h);
      switch (ori) {
      case 0:
	g.drawLine(left, top, right, bottom);
	g.drawLine(left+1, top, right, bottom-1);
	g.drawLine(left+2, top, right, bottom-2);
	g.drawLine(left, top+1, right-1, bottom);
	g.drawLine(left, top+2, right-2, bottom);
	break;
      case 1:
	g.drawLine(right, top, left, bottom);
	g.drawLine(right-1, top, left, bottom-1);
	g.drawLine(right-2, top, left, bottom-2);
	g.drawLine(right, top+1, left+1, bottom);
	g.drawLine(right, top+2, left+2, bottom);
	break;
      }
      break;
      
    case kStraightMirror:
      g.setColor(color.h);
      switch (ori) {
      case 0:
	g.drawLine(cx-2, top, cx-2, bottom);
	g.drawLine(cx-1, top, cx-1, bottom);
	g.drawLine(cx, top, cx, bottom);
	g.drawLine(cx+1, top, cx+1, bottom);
	g.drawLine(cx+2, top, cx+2, bottom);
	break;
      case 1:
	g.drawLine(right, cy-2, left, cy-2);
	g.drawLine(right, cy-1, left, cy-1);
	g.drawLine(right, cy, left, cy);
	g.drawLine(right, cy+1, left, cy+1);
	g.drawLine(right, cy+2, left, cy+2);
	break;
      }
      break;
      
    case kBeamSplitter:
      switch (ori) {
	case 0: DoBeamSplitter(g,color,left,bottom-1,right-1,bottom-1);  break;
	case 1: DoBeamSplitter(g,color,left,top,left,bottom-1);          break;
	case 2: DoBeamSplitter(g,color,left,top,right-1,top);            break;
	case 3: DoBeamSplitter(g,color,right-1,top,right-1,bottom-1);    break;
      }
      break;
      
    case kLaser:
      int bcx = (int)(w/5);
      int bcy = (int)(h/5);
      g.setColor(color.r);
      switch (ori) {
      case 0:
	g.fillArc(left, bottom-bcy*2, w, bcy*2, 180, 180);
	MyUtils.FillTriangle(g, cx-bcx, bottom-bcy, cx, top, cx+bcx, bottom-bcy);
	break;
      case 1:
	g.fillArc(left, top, bcx*2,h, 90, 180);
	MyUtils.FillTriangle(g, left+bcx, cy-bcy, right, cy, left+bcx, cy+bcy);
	break;
      case 2:
	g.fillArc(left, top, w, bcy*2, 0, 180);
	MyUtils.FillTriangle(g, cx-bcx, top+bcy, cx, bottom, cx+bcx, top+bcy);
	break;
      case 3:
	g.fillArc(right-bcx*2, top, bcx*2, h, 270, 180);
	MyUtils.FillTriangle(g, right-bcx, cy-bcy, left, cy, right-bcx, cy+bcy);
	break;
      }
      break;
      
    case kKing:
      g.setColor(color.r);

      g.drawLine(cx, top, right, cy);
      g.drawLine(cx, top+1, right-1, cy);
      g.drawLine(cx, top-1, right+1, cy);
      g.drawLine(right, cy, cx, bottom);
      g.drawLine(right-1, cy, cx, bottom-1);
      g.drawLine(right+1, cy, cx, bottom+1);
      g.drawLine(cx, bottom, left, cy); 
      g.drawLine(cx, bottom-1, left+1, cy);
      g.drawLine(cx, bottom+1, left-1, cy);
      g.drawLine(left, cy, cx, top);
      g.drawLine(left+1, cy, cx, top+1);
      g.drawLine(left-1, cy, cx, top-1);

      g.drawLine(cx, top, cx, bottom);
      g.drawLine(cx+1, top, cx+1, bottom);
      g.drawLine(cx-1, top, cx-1, bottom);
      g.drawLine(left, cy, right, cy);
      g.drawLine(left, cy+1, right, cy+1);
      g.drawLine(left, cy-1, right, cy-1);

      break;
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Rotate(boolean redraw, boolean fromTeleport)
  {
    int period;

    if (!fromTeleport)
      board.game.sounds.Click();

    switch (type) {
    case kTriangle: case kSquare: case kBeamSplitter: case kLaser:
      period = 4; break;
    case kDiagMirror: case kStraightMirror:
      period = 2; break;
    default: // hypercube, king
      period = 1; break;
    }

    ori = (ori+1) % period;

    if (redraw) {
      Erase();
      Draw();
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void ResetRotation()
  {
    ori = startori;
    Erase();
    Draw();
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void PickedUp()
  {
    held = true;
    startori = ori;
    Draw();
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  int MovesUsed(Point newsq)
  {
    int rotation=0, translation=0, laser=0;

    if (startori != ori)
      rotation = 1;

    translation = Math.abs(newsq.x-col) + Math.abs(newsq.y-row);

    if (type == kLaser && board.beam != null)
      laser = 1;

    return rotation + translation + laser;
  }

  // --------------------------------------------------------------------------
  // Seven possible scenarios for dropping a piece:
  //    1. Drop it back where it was
  //    2. Drop it in a square you can't get to, either because it's too
  //       far away, or because the path is blocked (illegal)
  //    3. Drop it in an empty, in-range square
  //    4. Drop it onto a neutral hypercube
  //    5. Drop a non-aggressor (or a used one) onto another piece (illegal)
  //    6. Drop a square or king onto another piece
  //    7. Drop a hypercube onto another piece
  //
  // Returns -1 if illegal move, or otherwise the number of turns used
  // (zero in scenario 1).  Possible side effects are that the destination
  // piece is either moved or captured.  Also takes care of redrawing all
  // involved pieces.
  // --------------------------------------------------------------------------
  int AttemptDrop(Point newsq, Piece target, int movesLeft,
		  boolean afterLaserFire, boolean local)
  {
    int retval, rotation=0;
    Graphics g = board.getGraphics();
    MessageBox confirm;
    String[] text;

    if (startori != ori)
      rotation = 1;
    retval = Math.abs(newsq.x-col) + Math.abs(newsq.y-row) + rotation;
        
    // case 1
    if (newsq.x == col && newsq.y == row) {
      if (!afterLaserFire)
	board.game.sounds.Click();
    }

    // case 2
    else if (!board.CanGetThere(col, row, newsq.x, newsq.y,
				movesLeft-rotation, this)) {
      retval = -1;
      board.game.sounds.Beep(local);
    }

    // case 3
    else if (target == null) {
      Move(g, newsq);
      board.game.sounds.Click();
    }

    // case 4
    else if (target.type == kHypercube && target.team == LaserChess.kNoTeam) {
      Erase(g);
      board.game.sounds.Teleport();
      board.TeleportPiece(this);
      Draw(g);
    }

    // case 5
    else if ((type!=kHypercube && type!=kKing && type!=kSquare) || spent) {
      retval = -1;
      board.game.sounds.Beep(local);
    }

    // case 6
    else if (type == kSquare || type == kKing) {
      // If they are about to take their own piece, warn them and
      // give them a chance to back out.
      if (!board.netplay && target.team == team) {
	board.game.sounds.Beep(true);
	board.GrabFocusNext();
	
	text = new String[1];
	text[0] = "Are you sure you want to capture your own piece?";
	board.GrabFocusNext();
	confirm = new MessageBox(board.game.frame, "Confirm Capture",
				 text, Label.LEFT, MessageBox.kOKCancel,
				 this);
      }

      if (board.netplay || killOwnPiece || target.team != team) {
	board.game.sounds.Capture();
	target.Die(false);
	Move(g, newsq);
	if (target.type == kKing)
	  board.EndGame(1-target.team, false);
	spent = true;
	killOwnPiece = false;
      }
      else
	retval = -1;
    }

    // case 7
    else {
      target.Erase(g);
      board.game.sounds.Teleport();
      board.TeleportPiece(target);
      target.Draw(g);
      Move(g, newsq);
      spent = true;
    }

    if (retval >= 0) {
      held = false;
      Draw(board.getGraphics());
    }

    return retval;
  }

  
  // --------------------------------------------------------------------------
  // Called by confirmation dialog if you try to capture your own piece.
  // --------------------------------------------------------------------------
  public void ProcessResult(Window source, Object result)
  {
    MessageBoxResult mbResult;

    mbResult = (MessageBoxResult) result;

    if (mbResult.buttonPressed == MessageBoxResult.kOK)
      killOwnPiece = true;
    else if (mbResult.buttonPressed == MessageBoxResult.kCancel)
      killOwnPiece = false;
  }
  

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Die(boolean explode)
  {
    if (explode) {
      board.ExplosionInProgress(true);
      board.game.sounds.Explode();
      ComputeBoundaries();
      Graphics g = board.getGraphics();
      int x, y;
      g.setColor(ColorManager.DoomedPiece().h);

      for (int i=0; i<kExplosionFillNumber/kExplosionBatchSize; ++i) {
	for (int j=0; j<kExplosionBatchSize; ++j) {
	  x = (int)(left2 + (right2-left2) * Math.random());
	  y = (int)(top2 + (bottom2-top2) * Math.random());
	  MyUtils.PlotPoint(g, x, y);
	}
	try { Thread.sleep(kExplosionDelay); }
	catch(InterruptedException e) { ; }
      }
      for (int i=0; i<kExplosionEmptyNumber/kExplosionBatchSize; ++i) {
	for (int j=0; j<kExplosionBatchSize; ++j) {
	  x = (int)(left2 + (right2-left2) * Math.random());
	  y = (int)(top2 + (bottom2-top2) * Math.random());
	  g.clearRect(x,y,1,1);
	}
	try { Thread.sleep(kExplosionDelay); }
	catch(InterruptedException e) { ; }
      }
    }
    Erase();
    board.pieces[col][row] = null;
    board.ExplosionInProgress(false);
  }
  
  
  // --------------------------------------------------------------------------
  // Draws a laser beam coming out of this piece in the specified direction.
  // This can be used when a beam is initially fired from a laser, but it also
  // used when a beam has bounced off a mirror and needs to continue on its
  // way.  DrawOutgoingBeam, our helper function, just needs to be told which
  // radius to start drawing outwards from: the small radius for "slender"
  // pieces (2), or the big radius for "fat" pieces (0).
  // --------------------------------------------------------------------------
  void SpitOutBeam(Direction out, Graphics g)
  {
    switch (type) {
    case kTriangle: case kStraightMirror: case kDiagMirror:
    case kBeamSplitter: case kHypercube:
      DrawOutgoingBeam(out, 2, g); break;
    case kSquare: case kLaser: case kKing:
      DrawOutgoingBeam(out, 0, g); break;
    }
  }

  // --------------------------------------------------------------------------
  // Draws a laser beam coming into this piece from the direction specified
  // by "indir".
  //
  // "outdir" and "outsplit" are outgoing parameters; their incoming values
  // are irrelevant, and they are set by this method, if appropriate.
  // "outdir" always holds the direction that the beam will continue going in
  // after hitting this piece (which is irrelevant if the piece is destroyed).
  // If the beam splits, then outsplit has the direction that the second
  // goes; otherwise, it is not set.
  // 
  // The return value is an integer that tells what kind of interaction the 
  // piece and the laser beam had.  The constants for interpreting this
  // integer are declared in the LaserBeam class.
  // --------------------------------------------------------------------------
  int AcceptBeam(Direction in, Graphics g, Direction out,
		 Direction out_split)
  {
    int diff;
    int result = LaserBeam.kNoEffect;

    out.Set(in.dir);  // prime it for kNoEffect and kReflect
    switch (type) {

    case kTriangle:
      if (in.dir == ori || ((in.dir+1)%4) == ori) {
	DrawIncomingBeam(in, 0, g);
	result = LaserBeam.kVulnerable;
      }
      else {
	DrawIncomingBeam(in, 2, g);
	result = LaserBeam.kDeflect;
	if ((ori&1) == 0)
	  out.Set(-in.dy, -in.dx);
	else
	  out.Set(in.dy, in.dx);
      }
      break;

    case kHypercube:
      DrawIncomingBeam(in, 2, g);
      if (team == LaserChess.kNoTeam)
	result = LaserBeam.kAbsorb;
      else {
	result = LaserBeam.kNoEffect;
	SpitOutBeam(out, g);
      }
      break;
      
    case kSquare:
      DrawIncomingBeam(in, 0, g);
      if (Math.abs(ori-in.dir) == 2) {
	result = LaserBeam.kReflect;
	out.Reverse();
      }
      else
	result = LaserBeam.kVulnerable;
      break;
      
    case kDiagMirror:
      DrawIncomingBeam(in, 2, g);
      result = LaserBeam.kDeflect;
      if (ori == 0)
	out.Set(in.dy, in.dx);
      else
	out.Set(-in.dy, -in.dx);
      break;
      
    case kStraightMirror:
      DrawIncomingBeam(in, 2, g);
      result = ((in.dir&1)==ori) ? LaserBeam.kNoEffect : LaserBeam.kReflect;
      if (result == LaserBeam.kReflect)
	out.Reverse();
      break;
      
    case kBeamSplitter:
      int side1=(ori+3)%4, side2=(ori+1)%4;
      diff=Math.abs(ori-in.dir);
      switch (diff) {
      case 0:
	DrawIncomingBeam(in, 0, g);
	result = LaserBeam.kVulnerable;
	break;
      case 1:
      case 3:
	DrawIncomingBeam(in, 2, g);
	result = LaserBeam.kDeflect;
	out.Set(ori);
	break;
      case 2:
	DrawIncomingBeam(in, 2, g);
	result = LaserBeam.kSplit;
	out.Set(side1);
	out_split.Set(side2);
	break;
      }
      break;
    
    case kLaser:
      diff = Math.abs(ori-in.dir);
      if (diff == 1 || diff == 3)
	DrawIncomingBeam(in, 1, g);
      else
	DrawIncomingBeam(in, 0, g);
      result = LaserBeam.kVulnerable;
      break;

    case kKing:
      DrawIncomingBeam(in, 0, g);
      result = LaserBeam.kVulnerable;
      break;
    }

    return result;
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void EnableAppropriateActions(LaserChess game)
  {
    if (board.turn == board.localTeam || !board.netplay) {
      switch (type) {
      case kTriangle: case kStraightMirror: case kDiagMirror:
      case kBeamSplitter: case kSquare:
	game.buttons.EnableActionButtons(true, true, false, true, true);
	game.menu.EnableActionItems(true, true, false, true, true);
	break;
      case kHypercube: case kKing:
	game.buttons.EnableActionButtons(true, false, false, true, true);
	game.menu.EnableActionItems(true, false, false, true, true);
	break;
      case kLaser:
	game.buttons.EnableActionButtons(true, true, true, true, true);
	game.menu.EnableActionItems(true, true, true, true, true);
	break;
      }
    }
    game.buttons.ChangeToggleMessage(true);
    game.menu.ChangeToggleMessage(true);
  }

  // --------------------------------------------------------------------------
  // Here we disable only the action buttons and menu items, leaving
  // "pick up" still available.  If this is a networked game and the local
  // team has just finished its turn, we may want those disabled... but that
  // is taken care of by Board.SetTurn.
  // --------------------------------------------------------------------------
  void DisableInappropriateActions(LaserChess game)
  {
    game.buttons.DisableActionButtons();
    game.menu.DisableActionItems();
    game.buttons.ChangeToggleMessage(false);
    game.menu.ChangeToggleMessage(false);
  }

  // --------------------------------------------------------------------------
  // Called at the end of every turn to reset the "spent" flag (whose job it
  // is to ensure that an aggressor can only attack once per round)
  // --------------------------------------------------------------------------
  void Refresh()
  { spent = false; }

  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void DoTriangle(Graphics g, PieceColor color, int tipX, int tipY,
			   int faceX, int faceY)
  {
    g.setColor(color.r);
    MyUtils.FillTriangle(g, tipX, tipY, tipX, faceY, faceX, tipY);

    g.setColor(color.h);
    g.drawLine(tipX, faceY, faceX, tipY);
    g.drawLine(tipX, faceY+1, faceX, tipY+1);
    g.drawLine(tipX, faceY-1, faceX, tipY-1);
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void DoBeamSplitter(Graphics g, PieceColor color, int p1X, int p1Y, 
			      int p2X, int p2Y)
  {
    g.setColor(color.r);
    MyUtils.FillTriangle(g, p1X, p1Y, cx, cy, p2X, p2Y);

    g.setColor(color.h);
    g.drawLine(p1X, p1Y, cx, cy);
    g.drawLine(p1X, p1Y+1, cx, cy+1);
    g.drawLine(p1X, p1Y-1, cx, cy-1);
    g.drawLine(p2X, p2Y, cx, cy);
    g.drawLine(p2X, p2Y+1, cx, cy+1);
    g.drawLine(p2X, p2Y-1, cx, cy-1);
  }

  // --------------------------------------------------------------------------
  // Draws the laser beam as it hits this piece.  "in" is the direction
  // the laser beam was heading in.  "depth" specifies how far in the beam
  // should be drawn: 0 = edge, 2 = center, and 1 is special for hitting
  // a laser from the side.
  // --------------------------------------------------------------------------
  private void DrawIncomingBeam(Direction in, int depth, Graphics g)
  {
    in.Reverse();
    DrawOutgoingBeam(in, depth, g);
    in.Reverse();
  }

  // --------------------------------------------------------------------------
  // Draws the laser beam as it leaves this piece.  Parameters are the same
  // as for DrawIncomingBeam.
  // --------------------------------------------------------------------------
  private void DrawOutgoingBeam(Direction out, int depth, Graphics g)
  {
    ComputeBoundaries();
    int hs=(cx-left2)+2, vs=(cy-top2)+2;
    switch (depth) {
    case 0:
      g.drawLine(cx + out.dx*(cx-left), cy + out.dy*(cy-top),
		 cx + out.dx*hs, cy + out.dy*vs);
      break;
    case 1:
      g.drawLine(cx + out.dx*hs/5, cy + out.dy*vs/5,
		 cx + out.dx*hs, cy + out.dy*vs);
      break;
    case 2:
      g.drawLine(cx, cy, cx + out.dx*hs, cy + out.dy*vs);
      break;
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void Move(Graphics g, Point newsq)
  {
    Erase(g);
    board.PieceMoved(col, row, newsq.x, newsq.y);
    col = newsq.x;
    row = newsq.y;
    ComputeBoundaries();
    Draw(g);
  }

}
