// ****************************************************************************
// * Network.java                                                             *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.io.*;
import java.net.*;

// ----------------------------------------------------------------------------
// This class handles the network communication between two Laser Chess
// processes playing the same game on different machines.  It runs in a 
// separate thread because it needs to be constantly listening for new
// messages that come in, while the main thread responds to user input.
// Incoming messages are dispatched to the appropriate local handler;
// outgoing messages are passed through the "Send..." methods of Network.
// ----------------------------------------------------------------------------
class Network extends Thread
{
  static final int kDefaultPort=7600;
  static final int kMagicNumber=9763801; // DO NOT CHANGE!
  static final int kServerCheckFreq=750; // how often to retry server (in msec)
  static final int kTimeout=7000; // how long to wait for magic number exchange to succeed (in msec)
  static final String kLaserChessURL="http://www-cs-students.stanford.edu/~et/LaserChess/index.html";
  static final int kKeyPress=0, kKeyRelease=1, kMouseClick=2;
  static final int kMouseDoubleClick=3;
  LaserChess game;
  boolean connected=false;
  ServerSocket serversock=null;
  Socket sock=null;
  DataInputStream in;
  DataOutputStream out;
  RandomNumberServer randomNumberServer=null;
  GameConfigInfo config; // hold reference to object passed in

  // --------------------------------------------------------------------------
  // Network constructor.  configInfo parameter is in/out -- we set
  // localTeam, timerType, and the timerLengths appropriately if this end
  // is the client in a networked game.
  // --------------------------------------------------------------------------
  Network(LaserChess game, GameConfigInfo config)
  {
    this.game = game;
    this.config = config;
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public void run()
  {
    if (config.host)
      ForkRandomNumberServer();
    EstablishConnection();
    if (!VerifyCompatibility())
      System.exit(0);
    ExchangeConfigInfo();
    game.Continue();               // game finishes setting itself up
    EventLoop();
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SendKeyPress(int keycode)
  {
    try {
      out.writeInt(kKeyPress);
      out.writeInt(keycode);
    } catch (IOException e) { NetError(e); }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SendKeyRelease(int keycode)
  {
    try {
      out.writeInt(kKeyRelease);
      out.writeInt(keycode);
    } catch (IOException e) { NetError(e); }
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SendMouseClick(int row, int col)
  {
    try {
      out.writeInt(kMouseClick);
      out.writeInt(row);
      out.writeInt(col);
    } catch (IOException e) { NetError(e); }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SendMouseDoubleClick()
  {
    try {
      out.writeInt(kMouseDoubleClick);
    } catch (IOException e) { NetError(e); }
  }


  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  double GetSynchRandom()
  {
    if (config.host)
      return randomNumberServer.random(true);
    else {
      RandomNumberRequest req = new RandomNumberRequest(config.hostname);
      return req.GetSynchRandom();
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  int GetLocalTeam()
  {
    if (config.host || connected)
      // we know what color we're playing if we're the server and/or are
      // connected to the server (and have thus been told the answer)
      return config.localTeam;
    else
      // if we're not the server and we're not connected yet, we don't
      // know which team we're supposed to play yet!
      return LaserChess.kNoTeam;

  }

  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // Step 1: fork off a random number server (should only be called on one
  //         of the two machines)
  // --------------------------------------------------------------------------
  private void ForkRandomNumberServer()
  {
    randomNumberServer = new RandomNumberServer();
    randomNumberServer.start();
  }

  // --------------------------------------------------------------------------
  // Step 2: establish connection
  // --------------------------------------------------------------------------
  private void EstablishConnection()
  {
    if (config.host) {
      try {
	System.out.println("Network server: waiting for client to " +
			   "initiate connection...");
	serversock = new ServerSocket(kDefaultPort);
	sock = serversock.accept();
      } catch (IOException e) { NetError(e); } 
    }
    else {
      System.out.print("Network client: trying to connect to server \"" +
		       config.hostname + "\"...");
      while (true) {
	try {
	  sock = new Socket(config.hostname, kDefaultPort);
	} catch (ConnectException e) {
	  try {
	    Thread.sleep(kServerCheckFreq);
	  } catch(InterruptedException e2) { ; }
	  System.out.print(".");
	  continue;
	} catch (IOException e) { NetError(e); }
	break;
      }
      System.out.println("");
    }
    try {
      in = new DataInputStream(sock.getInputStream());
      out = new DataOutputStream(sock.getOutputStream());
    } catch (IOException e) { NetError(e); } 
    
    System.out.println("Network connection established!");
    connected = true;
  }

  // --------------------------------------------------------------------------
  // Step 3: make sure that both ends of the connection are running the same
  //         version of Laser Chess.  If not, tell each user what the story
  //         is, and who needs to upgrade to which version.
  // --------------------------------------------------------------------------
  private boolean VerifyCompatibility()
  {
    char v1='_', v2='_', v3='_', v4='_'; // version number of opponent
    int magicNum=0;
    boolean okay;

    System.out.print("Verifying version compatibility... ");

    // Exchange a magic number to check if we are talking to the latest
    // version of Laser Chess.  If not, we can't proceed with the rest of
    // the protocol to compare version numbers.
    try {
      // need a timeout because if opponent is running something earlier
      // than version 1.2a, they won't send a magic number and we'll
      // wait forever!
      sock.setSoTimeout(kTimeout);
      out.writeInt(kMagicNumber);
      magicNum = in.readInt();
      sock.setSoTimeout(0); // don't need this anymore
    } catch (IOException e) { ; }

    if (magicNum != kMagicNumber) {
      System.out.println("failed!\n" +
			 "\nTo play over a network, both computers must be running the same version" +
			 "\nof Laser Chess.  You are running version " + LaserChess.kVersionString + 
			 " and your opponent has\na version earlier than 1.2a.  The latest " +
			 "version can be downloaded from:\n\n\t" + kLaserChessURL);
      return false;
    }

    // We are both at least Laser Chess 1.2a or later.
    // Verify exact version number match...
    try {
      out.writeChar(LaserChess.kVersionString.charAt(0));
      out.writeChar(LaserChess.kVersionString.charAt(1));
      out.writeChar(LaserChess.kVersionString.charAt(2));
      out.writeChar(LaserChess.kVersionString.charAt(3));
      v1 = in.readChar();
      v2 = in.readChar();
      v3 = in.readChar();
      v4 = in.readChar();
    } catch (IOException e) { NetError(e); }

    if (v1 == LaserChess.kVersionString.charAt(0) &&
	v2 == LaserChess.kVersionString.charAt(1) &&
	v3 == LaserChess.kVersionString.charAt(2) &&
	v4 == LaserChess.kVersionString.charAt(3))
      okay = true;
    else {
      System.out.println("failed!\n" +
			 "\nTo play over a network, both computers must be running the same version" +
			 "\nof Laser Chess.  You are running version " + LaserChess.kVersionString + 
			 " and your opponent has\nversion " + v1 + v2 + v3 + v4 + ".  The latest " +
			 "version can be downloaded from:\n\n\t" + kLaserChessURL);
      okay = false;
    }

    // One more exchange is needed here to make sure both ends have figured out
    // what's going on before anyone breaks the connection by exiting.
    try {
      out.writeInt(0);
      in.readInt();
    } catch (IOException e) { NetError(e); }

    if (okay)
      System.out.println("okay.");
    return okay;
  }
  
  // --------------------------------------------------------------------------
  // Step 4: server tells client what color client is playing and what time
  //         limits (if any) have been established.
  // --------------------------------------------------------------------------
  private void ExchangeConfigInfo()
  {
    try {
      if (config.host) {
	out.writeInt(1-config.localTeam);
	out.writeInt(config.timerType);
	out.writeInt(config.redTimerLength);
	out.writeInt(config.greenTimerLength);
      }
      else {
	config.localTeam = in.readInt();
	config.timerType = in.readInt();
	config.redTimerLength = in.readInt();	
	config.greenTimerLength = in.readInt();	
      }
    } catch (IOException e) { NetError(e); }
  }

  // --------------------------------------------------------------------------
  // Endless event loop: we listen for messages over the network and dispatch
  // them to the game or the board, as appropriate.  If anyone wants to send
  // messages out, they can call a Network method for that, but it will happen
  // in their own thread.  When messages are received, they are dispatched from
  // the network's thread.  Normally, this would give rise to synchronization
  // issues, but because the player turns are lockstep, it is always the case
  // that anyone receiving a message isn't really doing anything else anyway.
  //
  // All network messages are simply a string of one or more integers.  The
  // first integer says what kind of a message it is, and thus determines how
  // many more integers are coming as part of the same message.
  // --------------------------------------------------------------------------
  private void EventLoop()
  {
    while (game.winner != LaserChess.kLostContact) {
      try {
	int type = in.readInt();
	switch (type) {
	case kKeyPress:
	  game.board.ProcessKeyPress(in.readInt(), false);
	  break;
	case kKeyRelease:
	  game.board.ProcessKeyRelease(in.readInt(), false);
	  break;
	case kMouseClick:
	  game.board.ProcessMouseClick(in.readInt(), in.readInt(), false);
	  break;
	case kMouseDoubleClick:
	  game.board.ProcessMouseDoubleClick(false);
	  break;
	}
      } catch (IOException e) { NetError(e); }
    }
  }

  // --------------------------------------------------------------------------
  // A network exception has occurred.  If it's a SocketException, we assume
  // it's probably something like "connection reset by peer", which would
  // most likely be caused by the opponent quitting the game.  We don't deal
  // gracefully with any other problems at this time.
  // --------------------------------------------------------------------------
  private void NetError(IOException e)
  {
    if (e instanceof java.net.SocketException ||
	e instanceof java.io.EOFException) {
      if (game.winner == LaserChess.kGameNotOver && game.board != null)
	game.board.EndGame(LaserChess.kLostContact, false);
    }
    else {
      System.out.println("\nNetwork error: " + e);
      System.exit(0);
    }
  }
}
