// ****************************************************************************
// * MessageBox.java                                                          *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;
import java.awt.event.*;

// ----------------------------------------------------------------------------
// We can't extend the dialog class because then there would be no reliable
// way to encapsulate returning input focus back to the caller!  This was a
// bug I encountered in JDK 1.1.1 -- I haven't checked to see if it is fixed
// in later versions.
// ----------------------------------------------------------------------------
class MessageBox implements ActionListener
{
  Dialog dlg;
  Panel messages;
  Button ok, cancel;
  Panel buttonPanel;
  Frame parent;
  int options;
  ResultProcessor rp;

  // Possible value for "options" parameter in constructor
  static final int kOK=0, kOKCancel=1;

  // --------------------------------------------------------------------------
  // A quick, easy-to-use version that only displays one line of text.
  // --------------------------------------------------------------------------
  MessageBox(Frame parent, String title, String text)
  {
    options = kOK;
    rp = null;
    Initialize1(parent, title);
    messages.setLayout(new GridLayout(1, 1));
    messages.add(new Label(text, Label.CENTER));
    Initialize2();
  }

  // --------------------------------------------------------------------------
  // A harder-to-use but more flexible version that can display any number
  // of lines of text.  Too bad this is so difficult, but AWT labels don't
  // allow for multi-line labels, so we have to do our own kludges.
  // --------------------------------------------------------------------------
  MessageBox(Frame parent, String title, String[] text, int align)
  {
    GridLayout gl;

    options = kOK;
    rp = null;
    Initialize1(parent, title);
    gl = new GridLayout(text.length, 1);
    messages.setLayout(gl);
    for (int i=0; i<text.length; ++i)
      messages.add(new Label(text[i], align));
    Initialize2();
  }

  // --------------------------------------------------------------------------
  // The ultimate message box -- in addition to arbitrary lines of text,
  // lets the caller choose what set of options should be offered to the
  // user.  The result of the dialog is passed to the provided
  // ResultProcessor in the form of a MessageBoxResult object.
  // --------------------------------------------------------------------------
  MessageBox(Frame parent, String title, String[] text, int align,
	     int options, ResultProcessor resultProcessor)
  {
    GridLayout gl;

    this.options = options;
    rp = resultProcessor;
    Initialize1(parent, title);
    gl = new GridLayout(text.length, 1);
    messages.setLayout(gl);
    for (int i=0; i<text.length; ++i)
      messages.add(new Label(text[i], align));
    Initialize2();
  }

  // --------------------------------------------------------------------------
  // The only action the user can perform is to dismiss the dialog, so we
  // just assume that's why we were called.
  // --------------------------------------------------------------------------
  public void actionPerformed(ActionEvent e)
  {
    Object src;
    src = e.getSource();
    int buttonPressed;

    if (src == ok)
      buttonPressed = MessageBoxResult.kOK;
    else if (src == cancel)
      buttonPressed = MessageBoxResult.kCancel;
    else
      buttonPressed = MessageBoxResult.kUnknown;

    if (rp != null) {
      rp.ProcessResult(parent,
		       new MessageBoxResult(buttonPressed));
    }

    dlg.dispose();
  }

  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public class WindowEventHandler extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    { dlg.dispose(); }

    public void windowClosed(WindowEvent e)
    { parent.toFront(); parent.requestFocus(); }
  }

  // --------------------------------------------------------------------------
  // Initialization common to both constructors
  // --------------------------------------------------------------------------
  private void Initialize1(Frame parent, String title)
  {
    this.parent = parent;
    dlg = new Dialog(parent, title, true);
    dlg.setLayout(new BorderLayout());
    messages = new Panel();
  }

  // --------------------------------------------------------------------------
  // Initialization common to both constructors
  // --------------------------------------------------------------------------
  private void Initialize2()
  {
    dlg.add("Center", messages);
    buttonPanel = new Panel();

    if (options == kOK) {
      buttonPanel.add(ok = new Button("OK"));
      ok.addActionListener(this);
    }

    else if (options == kOKCancel) {
      buttonPanel.add(ok = new Button("OK"));
      ok.addActionListener(this);
      buttonPanel.add(cancel = new Button("Cancel"));
      cancel.addActionListener(this);
    }

    // Workaround for JDK bug #8 (see BUGS.txt)
    ok.setBackground(Color.lightGray);
    dlg.setBackground(Color.lightGray);

    dlg.add("South", buttonPanel);
    dlg.addWindowListener(new WindowEventHandler());
    dlg.pack();
    MyUtils.CenterDialogWithinParent(dlg, parent);
    dlg.setResizable(false);
    dlg.show();
  }

}
