// ****************************************************************************
// * LaserChess.java                                                          *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;
import java.awt.event.*;
import java.lang.*;

interface Blockable
{
  public void Continue();
}

class LaserChess implements Blockable, ResultProcessor
{
  // must be exactly four characters for version number
  // comparison in Network.java to work, so pad if necessary!
  static final String kVersionString =  "1.3a";

  // what to show to the user -- doesn't have to be padded.
  static final String kVersionStringDisplay =  "1.3a";

  static final int kTitleBarHeight =       20;
  static final int kNoTeam =                2;
  static final int kLostContact =           3;
  static final int kGameNotOver =           4;
  static final int kFireLaserCost =         1;
  static final int kTimerWidth =           10;
  static final int kTimerHorizMargin =      5;

  Frame frame;
  Readout readout;
  Board board;
  ButtonPanel buttons;
  Network net=null;
  Menus menu;
  SoundManager sounds;
  GameConfigDialog configDialog;
  GameConfigInfo configInfo;
  int localTeam, winner;

  // --------------------------------------------------------------------------
  // Constructor for use when user entered command-line parameters
  // --------------------------------------------------------------------------
  LaserChess(GameConfigInfo configInfo)
  {
    this.configInfo = configInfo;
    ContinueAfterConfig();
  }

  // --------------------------------------------------------------------------
  // Constructor for use when user did not enter any command-line parameters:
  // we invoke the dialog and let them make choices there.
  // --------------------------------------------------------------------------
  LaserChess()
  {
    board = null;
    configDialog = new GameConfigDialog(this);
    configDialog.show();
  }

  // --------------------------------------------------------------------------
  // Called by game configuration dialog when user dismisses it.
  // --------------------------------------------------------------------------
  public void ProcessResult(Window source, Object result)
  {
    if (source == configDialog) {
      if (result != null) { // if user entered information, use it
	configInfo = (GameConfigInfo) result;
	ContinueAfterConfig();
      }
      else // otherwise, user hit cancel, so exit gracefully
	System.exit(0);
    }
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void ContinueAfterConfig()
  {
    winner = kGameNotOver;
    if (configInfo.netplay) {
      net = new Network(this, configInfo);
      net.start();
    }
    else
      Continue();
  }

  // --------------------------------------------------------------------------
  // If we're playing a networked game, we have to start the network before
  // doing all this stuff... hence the separate function.
  // --------------------------------------------------------------------------
  public void Continue()
  {
    if (configInfo.netplay) {
      localTeam = net.GetLocalTeam();
      System.out.println("Local color is " +
			 ColorManager.TeamColorCodeToString(localTeam) + ".");
    }
    else
      System.out.println("Both players using this computer.");

    PrintTimeLimits();

    frame = new Frame("Laser Chess");
    frame.setBackground(Color.black);
    frame.setLayout(new BorderLayout());
    frame.addWindowListener(new WindowEventHandler());

    board = new Board(this, configInfo.netplay, localTeam,
		      configInfo.timerType, configInfo.redTimerLength,
		      configInfo.greenTimerLength);
    readout = new Readout(board);
    buttons = new ButtonPanel(this, board);
    menu = new Menus(this, board);
    sounds = new SoundManager(menu);

    buttons.DisableActionButtons();
    menu.DisableActionItems();
    BoardConfiguration.PopulateBoard(board, BoardConfiguration.kStandard);

    frame.add("North", readout);
    frame.add("Center", board);
    frame.add("South", buttons);
    MyUtils.SizeWindowToFitScreen(frame, 0, kTitleBarHeight);

    MyUtils.CenterWindowOnScreen(frame);
    frame.setMenuBar(menu);
    board.SetTurn(0);
    frame.show();

  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SetKeys()
  {
    MessageBox keys;

    String[] text;
    text = new String[9];
    text[0] = "  In a future version, this dialog will let you set which keys";
    text[1] = "  perform which actions.  For now, it's more of a help screen.";
    text[2] = "";
    text[3] = "  To move the cursor: arrow keys";
    text[4] = "  To pick up or put down a piece: ENTER key";
    text[5] = "  To rotate a piece you have picked up: SPACE bar";
    text[6] = "  To fire a laser you have picked up: 'F' key";
    text[7] = "  To put a piece back how you found it: ESCAPE key";
    text[8] = "  To pause the game: 'P' key";
    board.GrabFocusNext();
    keys = new MessageBox(frame, "Set Keyboard Controls", text, Label.LEFT);
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  double GetSynchRandom()
  {
    if (!configInfo.netplay)
      return Math.random();
    else
      return net.GetSynchRandom();
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Quit()
  {
    frame.dispose();
  }

  // --------------------------------------------------------------------------
  // Ends the game.  There are four possible outcomes:
  //   - red player wins (winner == team code for red)
  //   - green player wins (winner == team code for green)
  //   - stalemate (winner = kNoTeam)
  //   - lost contact with opponent on remote computer (winner == kLostContact)
  // --------------------------------------------------------------------------
  void EndGame(int winner)
  {
    MessageBox announce;
    
    menu.EnableActionItems(false, false, false, false, false);
    buttons.EnableActionButtons(false, false, false, false, false);
    this.winner = winner;
    if (winner == kNoTeam)
      announce = new MessageBox(frame, "Game Over",
				"Game over!  It is a stalemate!");
    else if (winner == kLostContact) {
      String[] text;
      text = new String[2];
      text[0] = "Laser Chess lost contact with your opponent's computer.";
      text[1] = "Either he/she quit the game, " +
	        "or there was a problem with the network.";
      announce = new MessageBox(frame, "Lost Contact", text, Label.CENTER);
    }
    else
      announce = new MessageBox(frame, "Game Over", "Game over!  The " +
				ColorManager.TeamColorCodeToString(winner) +
				" team has won!");
  }

  // --------------------------------------------------------------------------
  // Entry point to the program.  If there are no command-line arguments, we
  // start the game without any parameters, and the game will prompt the user.
  // If there are arguments, we parse them, and either bail if they are 
  // inconsistent or incomplete, or start the game if they are valid.
  // --------------------------------------------------------------------------
  public static void main(String[] args)
  {
    LaserChess game;

    System.out.println("\nLaser Chess version " + kVersionStringDisplay +
		       ": (c) 1997-2002 by Eric Tucker.");
    if (args.length == 0)
      game = new LaserChess();
    else
      game = new LaserChess(ParseCommandLine(args));
  }

  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void PrintTimeLimits()
  {
    int gt, rt;

    gt = configInfo.greenTimerLength;
    rt = configInfo.redTimerLength;

    switch (configInfo.timerType) {
    case GameConfigInfo.kNoTimeLimit:
      System.out.println("No time limits.");
      break;
    case GameConfigInfo.kTimeLimitPerTurn:
      if (gt == rt) {
	System.out.println("Time limit: " + rt + " second(s) per turn.");
      }
      else {
	System.out.println("Red time limit: " + rt + " second(s) per turn.");
	System.out.println("Green time limit: " + gt + " second(s) per turn.");
      }
      break;
    case GameConfigInfo.kTimeLimitPerGame:
      gt /= 60;
      rt /= 60;
      if (gt == rt) {
	System.out.println("Time limit: " + rt + " minute(s) per player per game.");
      }
      else {
	System.out.println("Red time limit: " + rt + " minute(s) for the whole game.");
	System.out.println("Green time limit: " + gt + " minute(s) for the whole game.");
      }
      break;
    }
  }


  // --------------------------------------------------------------------------
  // Parses command-line arguments, if any.  If they are incorrect, we print
  // out help on invocation syntax; otherwise, we start the game.
  // --------------------------------------------------------------------------
  private static GameConfigInfo ParseCommandLine(String[] args)
  {
    GameConfigInfo cfg = new GameConfigInfo();

    for (int i=0; i<args.length; ++i) {

      if (args[i].equals("-s")) {
	if (cfg.netplay || i+1 >= args.length)
	  cfg.inconsistent = true;
	else {
	  cfg.netplay = true;
	  cfg.host = true;
	  if (args[i+1].equalsIgnoreCase("red"))
	    cfg.localTeam = ColorManager.TeamColorStringToCode("red");
	  else if (args[i+1].equalsIgnoreCase("green"))
	    cfg.localTeam = ColorManager.TeamColorStringToCode("green");
	  else
	    cfg.inconsistent = true;
	  ++i;
	}
      }

      else if (args[i].equals("-c")) {
	if (cfg.netplay || i+1 >= args.length)
	  cfg.inconsistent = true;
	else {
	  cfg.netplay = true;
	  cfg.host = false;
	  cfg.hostname = args[i+1];
	  ++i;
	}
      }

      else if (args[i].equals("-b")) {
	if (cfg.netplay)
	  cfg.inconsistent = true;
      }

      else if (args[i].equals("-lt")) {
	if (i+1 >= args.length)
	  cfg.inconsistent = true;
	else {
	  cfg.timerType = GameConfigInfo.kTimeLimitPerTurn;
	  cfg.redTimerLength = Integer.parseInt(args[i+1]);
	  cfg.greenTimerLength = cfg.redTimerLength;
	  ++i;
	}
      }

      else if (args[i].equals("-lg")) {
	if (i+1 >= args.length)
	  cfg.inconsistent = true;
	else {
	  cfg.timerType = GameConfigInfo.kTimeLimitPerGame;
	  cfg.redTimerLength = Integer.parseInt(args[i+1]) * 60;
	  cfg.greenTimerLength = cfg.redTimerLength;
	  ++i;
	}
      }

      else
	Usage();
    }

    if (cfg.inconsistent)
      Usage();
    return cfg;
  }
   
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private static void Usage()
  {
    System.out.print("\n" +
		     "Invoke with no command-line parameters to use configuration dialog box.\n" +
		     "Syntax for invoking with command-line parameters:\n\n" +
		     "   java LaserChess [networking options] [time limit options]\n\n" +
		     "Networking options (mutually exclusive):\n\n" + 
		     "   -b              Both teams playing on the same machine (default)\n\n" +
		     "   -s [red,green]  Network play with local machine as server, playing team\n" +
		     "                   specified by \"red\" or \"green\"\n\n" +
		     "   -c [hostname]   Network play with local machine as client, connecting\n" +
		     "                   to [hostname] as server.\n\n" +
		     "Time limit options (mutually exclusive; ignored if -c option specified):\n\n" +
		     "   -tn             No time limits (default)\n\n" +
		     "   -lt [seconds]   Limit each turn to [seconds] seconds.\n\n" +
		     "   -lg [minutes]   Limit sum of each player's turns to [minutes] minutes.\n\n" +
		     "Example:\n\n" +
		     "   java LaserChess -s red -lt 15\n\n" +
		     "   Initiate a networked game, with this machine acting as the server.\n" +
		     "   Player on this machine will play red.  Each turn limited to 15 seconds.\n\n");
    System.exit(0);
  }

  // --------------------------------------------------------------------------
  // Use an inner class so I don't have to put in stubs for all the different
  // window events.  Not sure why I had to do windowActivated, but before
  // I did, when the window lost and regained activation, NOBODY was left
  // with the input focus, so the user would have to click on a button to get
  // back in the game.
  // --------------------------------------------------------------------------
  public class WindowEventHandler extends WindowAdapter {
    public void windowClosing(WindowEvent e)
    {
      frame.dispose();
    }

    public void windowClosed(WindowEvent e)
    {
      System.exit(0);
    }

    public void windowActivated(WindowEvent e)
    {
      board.requestFocus();
    }
  }
  
}
