// ****************************************************************************
// * LaserBeam.java                                                           *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;

class LaserBeam
{
  static final int kMaxHops=100;  // for loop detection
  static final int kNoEffect=0, kVulnerable=1, kReflect=2;
  static final int kDeflect=3, kSplit=4, kAbsorb=5;

  Board board;
  Piece laser;
  int TTL;  // time-to-live counter, for loop detection

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  LaserBeam(Board board, Piece laser)
  {
    this.board = board;
    this.laser = laser;
  }

  // --------------------------------------------------------------------------
  // Just wrappers to the recursive laser propagation method.
  // --------------------------------------------------------------------------
  void Fire()
  { PropagateLaser(laser, new Direction(laser.ori), false, true); }

  void Draw()
  { PropagateLaser(laser, new Direction(laser.ori), false, false); }

  void Erase()
  { PropagateLaser(laser, new Direction(laser.ori), true, false); }

  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // This method figures out the path that a laser takes, starting from a 
  // specified piece (from which we derive a location) in a specified 
  // direction.  Depending on how it is invoked, we may draw or erase the
  // laser as its path is traced, and we may or may not keep track of pieces
  // we hit, in order to destroy them when the beam is stopped.
  //
  // Normally, we just hop from piece to piece, deflecting as needed, until we
  // hit something that doesn't allow us to continue.  If we hit a beam 
  // splitter, one of the two beams deflects as normal, but the other one is
  // handled by a recursive call to continue from that point as a separate
  // beam.
  //
  // The "dir" parameter is passed in as the starting direction, but it will
  // be changed throughout the course of this function, so the caller should
  // be prepared for this.
  // 
  // Note that if the beam hits a reflective piece (i.e. one that deflects 
  // by 180 degrees), we cannot simply kill the laser and stop right there;
  // if the beam was deflected by a beam splitter on its way, it might get
  // split on its way back and do additional damage!
  // --------------------------------------------------------------------------
  private void PropagateLaser(Piece startingPiece, Direction dir,
			      boolean erase, boolean destroyPieces)
  {
    Piece piece = startingPiece;
    Direction split_dir;
    int col=piece.col, row=piece.row, result;
    Graphics g = board.getGraphics();

    if (erase)
      g.setColor(ColorManager.kBoardBackground);
    else
      g.setColor(ColorManager.kLaserColor);
    for (TTL=kMaxHops; TTL>0; --TTL) { // main loop, one iteration per bounce
      piece.SpitOutBeam(dir, g);
      while (true) {  // move along until you hit something...
	col += dir.dx;
	row += dir.dy;
	if (!board.InRangeBlocks(col, row))
	  return; // went off the board
	if (board.pieces[col][row] != null)
	  break; // hit a piece!
	DrawLaserBeamSegment(g, col, row, dir);
      }

      // now we have hit a piece... figure out what happens next
      piece = board.pieces[col][row];
      split_dir = new Direction();
      result = piece.AcceptBeam(dir, g, dir, split_dir);
      switch(result) {
      case kVulnerable:
	if (destroyPieces)
	  board.morgue.Doom(piece);
	return;
      case kAbsorb:
	return;
      case kSplit:
	PropagateLaser(piece, split_dir, erase, destroyPieces);
	// we flow into the next case on purpose, since a kSplit really
	// includes a kDeflect with it.
      case kDeflect:  // direction already changed by AcceptBeam.
	break;
      case kReflect:  // direction already changed by AcceptBeam.
	break;
      case kNoEffect: // nothing to do here either!
	break;
      }
      // if we get here, we deflected (or are one of the two results of 
      // a split.  We can just loop back up to the top and continue!
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void DrawLaserBeamSegment(Graphics g, int col, int row,
				    Direction dir)
  {
    if (dir.dx == 0)   // draw a vertical line
      g.drawLine(board.GetMiddleX(col), board.GetY(row),
		 board.GetMiddleX(col), board.GetY(row+1));
    else               // draw a horizontal line
      g.drawLine(board.GetX(col), board.GetMiddleY(row),
		 board.GetX(col+1), board.GetMiddleY(row));
  }

}
