// ****************************************************************************
// * GameConfigDialog.java                                                    *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;
import java.awt.event.*;

// ----------------------------------------------------------------------------
// This class is a dialog box that lets the user configure the game 
// parameters (e.g. network play, time limits) in a more friendly way than
// command-line options.  It has to be a frame, rather than a dialog, because
// a dialog requires a parent frame, which we don't have at this point!
// ----------------------------------------------------------------------------
class GameConfigDialog extends Frame implements ActionListener, ItemListener
{
  RadioButton rbBothLocal, rbNetworked, rbClient, rbServer;
  RadioButton rbRed, rbGreen, rbTimeLimitNone, rbTimeLimitPerTurn;
  RadioButton rbTimeLimitPerGame;
  CheckboxGroup cbgHow, cbgHost, cbgColor, cbgTime;
  Panel p1, p2, pNone, pPerTurn, pPerGame;
  Label lWelcome, lEmpty, lTimeLimit;
  TextField tfCompName, tfTimeLimitPerTurn, tfTimeLimitPerGame;
  Button okButton, cancelButton;
  ResultProcessor rp;
  GridBagLayout gbl;
  GridBagConstraints gbc;
  static final int kHorizMargin=50, kVertMargin=35;
  static final int kCompNameWidth=40, kNumberWidth=4;

  // @@ We need this extra space padding because of a long-standing bug in the
  // JDK for Windows NT: once you set a label to a certain string, any future
  // strings it is set to will be truncated to the length of the first string.
  // This is somewhat unpredictable because of the cross-platform differences
  // between widths of characters in proportional-spaced fonts.  I have
  // verified that this bug still exists in JDK 1.2.2.
  static final String kTimeLimitPrompt1 = "Would you like time limits in " +
                                          "your game?                        ";
  static final String kTimeLimitPrompt2 = "Time limits for this game will " +
                                          "be set by your opponent.";
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  GameConfigDialog(ResultProcessor rp)
  {
    super("Laser Chess Game Setup");
    setBackground(Color.lightGray);
    addWindowListener(new WindowEventHandler());
    this.rp = rp;

    p1 = new Panel();
    lWelcome = new Label("How would you like to play Laser Chess?");
    lEmpty = new Label(" ");
    tfCompName = new TextField(kCompNameWidth);

    lTimeLimit = new Label(kTimeLimitPrompt1);
    CreateRadioButtons();

    gbl = new GridBagLayout();
    gbc = new GridBagConstraints();
    p1.setLayout(gbl);
    IndentProperly(p1);

    gbc.gridx = 0;
    gbc.gridy = 0;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    gbc.anchor = GridBagConstraints.WEST;
    gbl.setConstraints(lWelcome, gbc);
    p1.add(lWelcome);

    gbc.gridx = 0;
    gbc.gridy = 9;
    gbl.setConstraints(lTimeLimit, gbc);
    p1.add(lTimeLimit);

    gbc.gridx = 0;
    gbc.gridy = 8;
    gbl.setConstraints(lEmpty, gbc);
    p1.add(lEmpty);

    AddRadioButtons();
    DisableInappropriateChoices();

    p2 = new Panel();
    okButton = new Button("Continue");
    cancelButton = new Button("Quit");
    okButton.addActionListener(this);
    cancelButton.addActionListener(this);
    p2.add(okButton);
    p2.add(cancelButton);
    add("South", p2);
    add("Center", p1);

    setResizable(false);

    pack();
    MyUtils.EnlargeWindow(this, kHorizMargin, kVertMargin);
    MyUtils.CenterWindowOnScreen(this);
   }

  // --------------------------------------------------------------------------
  // Deal with radio button states being changed
  // --------------------------------------------------------------------------
  public void itemStateChanged(ItemEvent e)
  {
    DisableInappropriateChoices();
  }

  // --------------------------------------------------------------------------
  // Deal with button presses.
  // --------------------------------------------------------------------------
  public void actionPerformed(ActionEvent evt)
  {
    String hostName;
    Object src;
    int localTeam, timerType, redTimerLength=0, greenTimerLength=0;
    MessageBox msg;

    src = evt.getSource();
    if (src == okButton) {
      localTeam = (rbRed.getState()) ? 0 : 1;
      if (tfCompName.getText().length() > 0)
	hostName = tfCompName.getText();
      else
	hostName = "localhost";
      if (rbTimeLimitNone.getState())
	timerType = GameConfigInfo.kNoTimeLimit;
      else if (rbTimeLimitPerGame.getState()) {
	timerType = GameConfigInfo.kTimeLimitPerGame;
	try {
	  redTimerLength = greenTimerLength = Integer.parseInt(tfTimeLimitPerGame.getText()) * 60;
	} catch (NumberFormatException e) {
	  msg = new MessageBox(this, "Error", "Please enter a number for " +
			       "your time limit.");
	  return;
	}
      }
      else if (rbTimeLimitPerTurn.getState()) {
	timerType = GameConfigInfo.kTimeLimitPerTurn;
	try {
	  redTimerLength = greenTimerLength = Integer.parseInt(tfTimeLimitPerTurn.getText());
	} catch (NumberFormatException e) {
	  msg = new MessageBox(this, "Error", "Please enter a number for " +
			       "your time limit.");
	  return;
	}
      }
      else
	return; // how did we get here?

      rp.ProcessResult(this,
		       new GameConfigInfo(rbNetworked.getState(),
					  rbServer.getState(),
					  hostName, localTeam, timerType,
					  redTimerLength, greenTimerLength));
      dispose();
    }
    else if (src == cancelButton) {
      rp.ProcessResult(this, null);
      dispose();
    }
  }
  
  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void CreateRadioButtons()
  {
    cbgHow = new CheckboxGroup();
    rbBothLocal = new RadioButton("Both teams local (on this computer)",
				  cbgHow, true);
    rbBothLocal.addItemListener(this);
    rbNetworked = new RadioButton("One team local, one team remote (over " +
				  "the network)", cbgHow, false);
    rbNetworked.addItemListener(this);
    
    cbgHost = new CheckboxGroup();
    rbServer = new RadioButton("Server mode: game hosted on this computer",
			       cbgHost, true);
    rbServer.addItemListener(this);
    rbClient = new RadioButton("Client mode: game hosted by remote " +
			       "computer:", cbgHost, false);
    rbClient.addItemListener(this);
    
    cbgColor = new CheckboxGroup();
    rbRed = new RadioButton("Red team played on this computer", cbgColor,
			    true);
    rbRed.addItemListener(this);
    rbGreen = new RadioButton("Green team played on this computer", cbgColor,
			      false);
    rbGreen.addItemListener(this);

    // Set up time limit controls.  We had to use a panel with FlowLayout
    // for the two radio buttons with text boxes, to get the text boxes
    // to line up properly.  But panels have insets (borders) that cannot
    // be changed, so this misaligned those radio buttons with "no time limit",
    // which didn't have a panel.  So now everyone has panels whether they
    // need it or not...

    cbgTime = new CheckboxGroup();
    rbTimeLimitNone = new RadioButton("No time limit", cbgTime, true);
    rbTimeLimitPerTurn = new RadioButton("Time limit in seconds per move:",
					 cbgTime, false);
    rbTimeLimitPerGame = new RadioButton("Time limit in minutes per player " +
					 "per game (like chess):",
					 cbgTime, false);
    tfTimeLimitPerTurn = new TextField(kNumberWidth);
    tfTimeLimitPerGame = new TextField(kNumberWidth);

    pNone = new Panel();
    pPerTurn = new Panel();
    pPerGame = new Panel();
    pNone.setLayout(new FlowLayout());
    pPerTurn.setLayout(new FlowLayout());
    pPerGame.setLayout(new FlowLayout());
    rbTimeLimitNone.addItemListener(this);
    rbTimeLimitPerTurn.addItemListener(this);
    rbTimeLimitPerGame.addItemListener(this);
    pNone.add(rbTimeLimitNone);
    pPerTurn.add(rbTimeLimitPerTurn);
    pPerGame.add(rbTimeLimitPerGame);
    pPerTurn.add(tfTimeLimitPerTurn);
    pPerGame.add(tfTimeLimitPerGame);
  } 

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  private void AddRadioButtons()
  {
    gbc.gridx = 1; gbc.gridy = 1;
    gbl.setConstraints(rbBothLocal, gbc); p1.add(rbBothLocal);

    gbc.gridx = 1; gbc.gridy = 2;
    gbl.setConstraints(rbNetworked, gbc); p1.add(rbNetworked);

    gbc.gridx = 2; gbc.gridy = 3;
    gbl.setConstraints(rbServer, gbc); p1.add(rbServer);

    gbc.gridx = 3; gbc.gridy = 4;
    gbl.setConstraints(rbRed, gbc); p1.add(rbRed);

    gbc.gridx = 3; gbc.gridy = 5;
    gbl.setConstraints(rbGreen, gbc); p1.add(rbGreen);

    gbc.gridx = 2; gbc.gridy = 6;
    gbl.setConstraints(rbClient, gbc); p1.add(rbClient);

    gbc.gridx = 3; gbc.gridy = 7;
    gbl.setConstraints(tfCompName, gbc); p1.add(tfCompName);

    gbc.gridx = 1; gbc.gridy = 10;
    gbl.setConstraints(pNone, gbc); p1.add(pNone);

    gbc.gridx = 1; gbc.gridy = 11;
    gbl.setConstraints(pPerTurn, gbc); p1.add(pPerTurn);

    gbc.gridx = 1; gbc.gridy = 12;
    gbl.setConstraints(pPerGame, gbc); p1.add(pPerGame);
  }

  // --------------------------------------------------------------------------
  // Adds padding (Label objects) in just the right places to indent choices
  // as we want them for this dialog.
  // --------------------------------------------------------------------------
  private void IndentProperly(Panel p)
  {
    int i;
    Label[] pad = new Label[15];
    for (i=0; i<15; ++i)
      pad[i] = new Label("   ");
    gbc.gridwidth = 1;

    gbc.gridx = 0;
    for (i=0; i<=6; ++i) {
      gbc.gridy = i+1;
      gbl.setConstraints(pad[i], gbc);
      p.add(pad[i]);
    }


    gbc.gridx = 1;
    for (i=7; i<=11; ++i) {
      gbc.gridy = i-4;
      gbl.setConstraints(pad[i], gbc);
      p.add(pad[i]);
    }

    gbc.gridx = 2; gbc.gridy = 4;
    gbl.setConstraints(pad[12], gbc); p.add(pad[12]);
    
  }

  // --------------------------------------------------------------------------
  // Disables whichever choices are not available given current
  // radio button settings.
  // --------------------------------------------------------------------------
  private void DisableInappropriateChoices()
  {
    rbBothLocal.setEnabled(true);
    rbNetworked.setEnabled(true);
    rbClient.setEnabled(rbNetworked.getState());
    rbServer.setEnabled(rbNetworked.getState());
    rbRed.setEnabled(rbNetworked.getState() && rbServer.getState());
    rbGreen.setEnabled(rbNetworked.getState() && rbServer.getState());
    tfCompName.setEnabled(rbNetworked.getState() && rbClient.getState());
    tfTimeLimitPerTurn.setEnabled(rbTimeLimitPerTurn.getState());
    tfTimeLimitPerGame.setEnabled(rbTimeLimitPerGame.getState());
    rbTimeLimitNone.setVisible(!(rbClient.getState()));
    rbTimeLimitPerTurn.setVisible(!(rbClient.getState()));
    rbTimeLimitPerGame.setVisible(!(rbClient.getState()));
    tfTimeLimitPerTurn.setVisible(!(rbClient.getState()));
    tfTimeLimitPerGame.setVisible(!(rbClient.getState()));
    lTimeLimit.setText((rbClient.getState()) ? kTimeLimitPrompt2 :
		       kTimeLimitPrompt1);
  }
  
  // --------------------------------------------------------------------------
  // Deal with the user hitting the close button.  Note the use of 
  // "GameConfigDialog.this" refers to the current instance of the dialog.
  // Just plain "this" refers to the event handler, which isn't what we want.
  // --------------------------------------------------------------------------
  public class WindowEventHandler extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      rp.ProcessResult(GameConfigDialog.this, null);
      dispose();
    }
  }

}
