// ****************************************************************************
// * Board.java                                                               *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;
import java.awt.event.*;
import java.io.*;

class Board extends Canvas implements Timed
{
  // --------------------------------------------------------------------------
  // Constant declarations.
  // --------------------------------------------------------------------------

  static final int kNumRows = 9;
  static final int kNumCols = 9;
  static final int kBorderX = 30, kBorderY = 5;
  static final int kTimerWidth = 10, kTimerHorizMargin = 10;
  static final int kMovesPerTurn = 2;
  static final int kDoubleClickInterval = 400;  // in milliseconds
  static final int kMinLaserHangTime = 1500;    // in milliseconds
  static final int kKeyLaserFire = KeyEvent.VK_F;

  // --------------------------------------------------------------------------
  // Instance variables.
  // --------------------------------------------------------------------------
  int left, right, top, bottom, width, height, actualRight, actualBottom;
  int turn, movesLeft;  // which player's turn is it?
  int timeLeft[];
  Cursor cursor;
  Piece pickedUpPiece;
  Piece[][] pieces;
  LaserChess game;
  LaserBeam beam;      // laser beam in the air, if any
  Morgue morgue;       // pieces hit by a laser, waiting to be destroyed
  long whenLastClick, whenLaserFired;
  int colLastClick, rowLastClick;
  boolean netplay;
  int localTeam, timerType, redTimerLength, greenTimerLength;
  ProgressBar[] timeDisplays;
  boolean eip; // explosion in progress;
  boolean paused, pausedByMe, timeRanOut;
  Rectangle bounds;

  // What a hack!  If we put up a dialog, when that dialog is dismissed,
  // the board gets the focus back (because we request it) but then
  // promptly loses it "permanently" to no one!  So we listen for focus
  // lost events and grab it back... just once.  (If we continue to 
  // grab it back, it makes it impossible for the user to press any of
  // the buttons on the button bar!)  Then revert to normal behavior
  // until the next GrabFocusNext() is called.
  boolean grabFocusNext;

  // --------------------------------------------------------------------------
  // Constructor.
  // --------------------------------------------------------------------------
  Board(LaserChess game, boolean netplay, int localTeam, int timerType,
	int redTimerLength, int greenTimerLength)
  {
    this.game = game;
    this.netplay = netplay;
    this.localTeam = localTeam;
    this.timerType = timerType;
    this.redTimerLength = redTimerLength;
    this.greenTimerLength = greenTimerLength;
    timeRanOut = false;
    pieces = new Piece[kNumCols][kNumRows];
    turn = LaserChess.kNoTeam;
    eip = false;
    paused = false;
    pausedByMe = false;
    movesLeft = 0;
    cursor = new Cursor(this, 4, 4);
    pickedUpPiece = null;
    beam = null;
    grabFocusNext = false;
    morgue = new Morgue();
    addKeyListener(new KeyPressHandler());
    addMouseListener(new MousePressHandler());
    addFocusListener(new FocusHandler());
    timeDisplays = new ProgressBar[2];
    if (timerType != GameConfigInfo.kNoTimeLimit) {
      new Timer(this, 1000).start();
      timeDisplays[0] = new ProgressBar(this, new Rectangle(0, 0, 0, 0), false,
					ColorManager.PlayerColor(0), 1);
      timeDisplays[1] = new ProgressBar(this, new Rectangle(0, 0, 0, 0), false,
					ColorManager.PlayerColor(1), 1);
    }
    timeLeft = new int[2];
    timeLeft[ColorManager.kRedIndex] = redTimerLength;
    timeLeft[ColorManager.kGreenIndex] = greenTimerLength;
  }


  // --------------------------------------------------------------------------
  // Repaint self.
  // --------------------------------------------------------------------------
  public void paint(Graphics g)
  {
    if (g == null) return;
    int row, col;
    SetUsableBounds();

    if (paused) {
      g.setColor(Color.white);
      g.setFont(new Font("Serif", Font.BOLD, 24));
      if (pausedByMe) {
	g.drawString("Game paused.", 35, 30);
	g.drawString("Press 'P' to resume.", 35, 60);
      }
      else {
	g.drawString("Your opponent has paused the game.", 35, 30);
	g.drawString("Play will resume when he/she is ready.", 35, 60);
      }
    }

    else {
      DrawGrid();
      cursor.Draw();
      
      for (row=0; row<kNumRows; ++row)
	for (col=0; col<kNumCols; ++col)
	  if (pieces[col][row] != null)
	    pieces[col][row].Draw(g);
      if (beam != null)
	beam.Draw();
      
      if (timerType != GameConfigInfo.kNoTimeLimit) {
	timeDisplays[0].paint(g);
	timeDisplays[1].paint(g);
      }
    }
  }
  
  // --------------------------------------------------------------------------
  // Draws grid: helper method to paint().
  // --------------------------------------------------------------------------
  void DrawGrid()
  {
    Graphics g = getGraphics(); if (g == null) return;
    g.setColor(ColorManager.kBoardGrid);
    for (int col=0; col<=kNumCols; ++col)
      g.drawLine(GetX(col), GetY(0), GetX(col), GetY(kNumRows));
    for (int row=0; row<=kNumRows; ++row)
      g.drawLine(GetX(0), GetY(row), GetX(kNumCols), GetY(row));
  }

  // --------------------------------------------------------------------------
  // Figures out what part of the screen we can actually use.
  // --------------------------------------------------------------------------
  void SetUsableBounds()
  {
    bounds = getBounds();
    left = kBorderX;
    top = kBorderY;
    right = bounds.width - kBorderX;
    bottom = bounds.height - kBorderY;
    width = right-left;
    height = bottom-top;
    actualRight = GetX(kNumCols);
    actualBottom = GetY(kNumRows);

    if (timerType != GameConfigInfo.kNoTimeLimit) {
      timeDisplays[0].SetRect(left - kTimerHorizMargin - kTimerWidth, top,
			      kTimerWidth, actualBottom - top);
      timeDisplays[1].SetRect(actualRight + kTimerHorizMargin, top,
			      kTimerWidth, actualBottom - top);
    }
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public void ExplosionInProgress(boolean eip)
  {
    this.eip = eip;
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public void keyTyped(KeyEvent e)
  {
    e.consume();
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public void GrabFocusNext()
  {
    grabFocusNext = true;
  }

  public class FocusHandler extends FocusAdapter
  {
    public void focusGained(FocusEvent e)
    {
    }

    public void focusLost(FocusEvent e)
    {
      if (e.isTemporary() == false && grabFocusNext == true) {
	requestFocus();
	grabFocusNext = false;
      }
    }
  }

  public class KeyPressHandler extends KeyAdapter
  {

  // --------------------------------------------------------------------------
  // Remote play is done by sending every keystroke over the net and letting
  // the other computer deal with it as if it had originated locally.  There
  // are just a few caveats:
  //    - ignore keystrokes that initiate locally if it's not your turn
  //    - if a keystroke did not initiate locally, don't beep if it was illegal
  //
  // Note: we used to consume any events that contained the laser firing
  // key, when that key was TAB.  There was a comment saying that this
  // prevented keyboard access to buttons. But starting in JDK 1.4, this
  // stopped working; instead, there was keyboard access to buttons and
  // we did not receive TAB key events.  That's why the laser fire key
  // was changed to 'F'.
  // --------------------------------------------------------------------------
  public void keyPressed(KeyEvent e)
  {
    if (netplay && turn != localTeam)
      return;
    int key = e.getKeyCode();
    if (netplay)
      game.net.SendKeyPress(key);
    ProcessKeyPress(key, true);
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public void keyReleased(KeyEvent e)
  {
    if (netplay && turn != localTeam)
      return;
    int key = e.getKeyCode();
    if (netplay)
      game.net.SendKeyRelease(key);
    ProcessKeyRelease(key, true);
  }

  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void ProcessKeyPress(int keycode, boolean local)
  {
    if (beam != null || turn == LaserChess.kNoTeam)
      return;

    if (keycode == KeyEvent.VK_P) {
      TogglePause(local);
      return;
    }

    if (paused)
      return;
    
    if (keycode == kKeyLaserFire)           FireLaser(local);
    else if (keycode == KeyEvent.VK_LEFT)   cursor.Move(-1, 0);
    else if (keycode == KeyEvent.VK_RIGHT)  cursor.Move(1, 0);
    else if (keycode == KeyEvent.VK_UP)     cursor.Move(0, -1);
    else if (keycode == KeyEvent.VK_DOWN)   cursor.Move(0, 1);
    else if (keycode == KeyEvent.VK_ENTER)  ToggleSelect(false, false, local, false);
    else if (keycode == KeyEvent.VK_SPACE)  RotatePiece(local);
    else if (keycode == KeyEvent.VK_ESCAPE) ToggleSelect(true, false, local, false);

    game.readout.Update();
  }
  
  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void ProcessKeyRelease(int keycode, boolean local)
  {
    if (keycode == kKeyLaserFire)
      FinishLaser(local);
    if ((turn == 0 || turn == 1) && timerType == 
	GameConfigInfo.kTimeLimitPerGame && timeLeft[1-turn] <= 0)
      EndGame(turn, true);
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  public class MousePressHandler extends MouseAdapter
  {
    public void mousePressed(MouseEvent e)
    {
      if ((netplay && turn != localTeam) || beam != null)
	return;
      
      int x=e.getX(), y=e.getY(), col=GetCol(x), row=GetRow(y);
      if (InRangeBlocks(col, row) && turn != LaserChess.kNoTeam) {
	if (netplay)
	  game.net.SendMouseClick(row, col);
	ProcessMouseClick(row, col, true);
	if (e.getWhen() - whenLastClick < kDoubleClickInterval && col ==
	    colLastClick && row == rowLastClick) {
	  if (netplay)
	    game.net.SendMouseDoubleClick();
	  ProcessMouseDoubleClick(true);
	}
	whenLastClick = e.getWhen();
	colLastClick = col;
	rowLastClick = row;
      }
    }
  }

  void ProcessMouseDoubleClick(boolean local)
  {
    if (!paused)
      ToggleSelect(false, false, local, false);
  }

  void ProcessMouseClick(int row, int col, boolean local)
  {
    if (!paused) {
      cursor.MoveTo(col, row);
      game.readout.Update();
    }
  }

  // --------------------------------------------------------------------------
  // Indicates our timer has gone off.  Because we set the timer to do this
  // once per second, we can just lop a second off the clock.  (@@ We should
  // be just looking at the real clock, to avoid skew, and to make this
  // function work independently of how often the clock is to be updated!)
  // --------------------------------------------------------------------------
  public void tick(Timer t)
  {
    int timerLength;

    if (turn == ColorManager.kRedIndex)
      timerLength = redTimerLength;
    else
      timerLength = greenTimerLength;

    if ((turn == 0 || turn == 1) && !paused && !eip) {
      --timeLeft[turn];
      timeDisplays[turn].SetPercentageComplete((float)timeLeft[turn] /
					       timerLength);
      game.readout.Update();

      if (timeLeft[turn] <= 0 && !eip && beam == null && timerType !=
	  GameConfigInfo.kNoTimeLimit) {
	if (pickedUpPiece != null) {
	  if (netplay)
	    game.net.SendKeyPress(KeyEvent.VK_ESCAPE);
	  ProcessKeyPress(KeyEvent.VK_ESCAPE, true);
	}
	repaint();
	if (timerType == GameConfigInfo.kTimeLimitPerGame)
	  EndGame(1-turn, true);
	else { // timerType == GameConfigInfo.kTimeLimitPerTurn
	  SetTurn(1-turn);
	  game.readout.Update();
	}
      }
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void EndGame(int winner, boolean timeRanOut)
  {
    paused = false;
    repaint();
    game.readout.DeclareWinner(winner);
    this.timeRanOut = timeRanOut;
    SetTurn(LaserChess.kNoTeam);
    game.EndGame(winner);
  }

  // --------------------------------------------------------------------------
  // This function was originally thus named because the user has taken an
  // action that toggles whether or not he is holding a piece.  This can be
  // one of three cases:
  //    - picking up a new piece.
  //    - putting down a piece where the cursor is.
  //    - putting a piece back where you found it.
  //
  // The third case is invoked with a different command than the first two,
  // and the "cancel" parameter tells us which command was used.
  //
  // If the piece was a laser, it may have died during its turn, and we need
  // to know about this, hence "pieceDied".
  //
  // "local" just tells us if this action initiated locally or on a remote
  // computer, mostly so we know whether or not to beep if it was erroneous.
  // --------------------------------------------------------------------------
  void ToggleSelect(boolean cancel, boolean pieceDied, boolean local,
		    boolean afterLaserFire)
  {
    int movesTaken=0;

    if (pickedUpPiece == null && !cancel) {
      Piece target = pieces[cursor.sq.x][cursor.sq.y];
      if (target == null || target.team != turn)
	game.sounds.Beep(local);
      else {
	pickedUpPiece = target;
	cursor.SetPieceHeld(true);
	pickedUpPiece.PickedUp();
	pickedUpPiece.EnableAppropriateActions(game);
	game.sounds.Click();
      }
    }
    else if (pickedUpPiece != null) {
      if (cancel) {
	cursor.MoveTo(pickedUpPiece.col, pickedUpPiece.row);
	pickedUpPiece.ResetRotation();
	game.sounds.Click();
      }
      if (!pieceDied)
	movesTaken = pickedUpPiece.AttemptDrop
	  (cursor.sq, pieces[cursor.sq.x][cursor.sq.y], movesLeft,
	   afterLaserFire, local);
      if (movesTaken > -1) {
	pickedUpPiece.DisableInappropriateActions(game);
	pickedUpPiece = null;
	movesLeft -= movesTaken;
	cursor.SetPieceHeld(false);
	if (movesLeft == 0 && turn != LaserChess.kNoTeam)
	  SetTurn(1 - turn);
	else
	  cursor.Draw();
      }
    }
    game.readout.Update();
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void RotatePiece(boolean local)
  {
    if (pickedUpPiece != null)
      pickedUpPiece.Rotate(true, false);
  }

  // --------------------------------------------------------------------------
  // Can I get from (x0,y0) to (x1,y1) with the specified number of moves 
  // left without passing through another piece?  (for the purposes of this
  // function it is okay to land on another piece, just not to pass through
  // one en route).
  // --------------------------------------------------------------------------
  boolean CanGetThere(int x0, int y0, int x1, int y1, int movesLeft, Piece me)
  {
    if (x0<0 || x0>=kNumCols || y0<0 || y0>=kNumRows)
      return false;
    if (x0 == x1 && y0 == y1)
      return true;
    else if (movesLeft == 0)
      return false;
    else if (pieces[x0][y0] != null && pieces[x0][y0] != me)
      return false;
    else 
      return (CanGetThere(x0-1, y0, x1, y1, movesLeft-1, me) ||
	      CanGetThere(x0+1, y0, x1, y1, movesLeft-1, me) ||
	      CanGetThere(x0, y0-1, x1, y1, movesLeft-1, me) ||
	      CanGetThere(x0, y0+1, x1, y1, movesLeft-1, me));
  }

  // --------------------------------------------------------------------------
  // Teleports the piece to a new random unoccupied location, and rotates
  // it in a random direction.  (For when a piece encounters a hypercube)
  // --------------------------------------------------------------------------
  void TeleportPiece(Piece piece)
  {
    int newx, newy, rot;
    
    // rotate in a random direction.  since rotation periods of all pieces
    // are factors of 4, we can achieve the right effect by rotating any
    // of them 0-3 times 
    for (rot=(int)(game.GetSynchRandom() * 4); rot>0; --rot)
      piece.Rotate(false, true);
    
    // move to random location
    do {
      newx = (int)(game.GetSynchRandom() * kNumCols);
      newy = (int)(game.GetSynchRandom() * kNumRows);
    } while (pieces[newx][newy] != null);
    PieceMoved(piece.col, piece.row, newx, newy);
    piece.col = newx;
    piece.row = newy;
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void PieceMoved(int oldx, int oldy, int newx, int newy)
  {
    pieces[newx][newy] = pieces[oldx][oldy];
    pieces[oldx][oldy] = null;
  }

  void AddPiece(Piece newPiece)
  { int col=newPiece.col, row=newPiece.row; pieces[col][row] = newPiece; }

  int GetX(int col)
  { return left + (width/kNumCols * col); }

  int GetY(int row)
  { return top + (height/kNumRows * row); }

  int GetMiddleX(int col)
  { return left + (int)(width/kNumCols * (col+0.5)); }

  int GetMiddleY(int row)
  { return top + (int)(height/kNumRows * (row+0.5)); }

  int GetCol(int x)
  { return (x-left) / (width/kNumCols); }

  int GetRow(int y)
  { return (y-top) / (height/kNumRows); }

  boolean InRangePixels(int x, int y)
  { return (x >= left && x <= right && y >= top && y <= bottom); }

  boolean InRangeBlocks(int col, int row)
  { return (col >= 0 && col < kNumCols && row >= 0 && row < kNumRows); }

  void SetTurn(int TURN)
  {
    turn = TURN;
    movesLeft = kMovesPerTurn;
    if (timerType == GameConfigInfo.kTimeLimitPerTurn &&
	(turn == 0 || turn == 1)) {
      timeLeft[ColorManager.kRedIndex] = redTimerLength;
      timeLeft[ColorManager.kGreenIndex] = greenTimerLength;
      timeDisplays[turn].SetPercentageComplete(1);
      timeDisplays[1-turn].SetPercentageComplete(1);
    }
    game.readout.Update();
    cursor.SetTurn(turn, false);
    cursor.MoveTo(4, 4);
    for (int row=0; row<kNumRows; ++row)
      for (int col=0; col<kNumCols; ++col)
	if (pieces[col][row] != null)
	  pieces[col][row].Refresh();
    if (netplay && turn != localTeam) {
      game.buttons.EnableActionButtons(false, false, false, false, false);
      game.menu.EnableActionItems(false, false, false, false, false);
    }
    else {
      game.buttons.EnableActionButtons(true, false, false, false, true);
      game.menu.EnableActionItems(true, false, false, false, true);
    }
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void TogglePause(boolean local)
  {
    Graphics g = getGraphics();

    paused = !paused;
    pausedByMe = local;

    game.menu.TogglePause(local, paused);
    game.buttons.ChangePauseMessage(paused);
    if (paused) {
      game.buttons.EnableActionButtons(false, false, false, false, local);
      game.menu.EnableActionItems(false, false, false, false, local);
    }
    else if (pickedUpPiece != null)
      pickedUpPiece.EnableAppropriateActions(game);
    else {
      game.buttons.EnableActionButtons(true, false, false, false, local);
      game.menu.EnableActionItems(true, false, false, false, local);
    }

    repaint();
  }


  // ##########################################################################
  // ##########################################################################
 
  // --------------------------------------------------------------------------
  // If we're holding a laser piece, there's not already a laser beam in the
  // air, the laser has not already been fired this turn, and we have enough
  // moves left to fire the laser, then fire off a beam!
  // --------------------------------------------------------------------------
  private void FireLaser(boolean local)
  {
    if (pickedUpPiece == null || beam != null)
      return;

    if (pickedUpPiece.type != Piece.kLaser || 
	(pickedUpPiece.MovesUsed(cursor.sq) +
	LaserChess.kFireLaserCost < 0) ||
	pickedUpPiece.spent) {
      game.sounds.Beep(local);
      return;
    }
    else { 
      game.sounds.StartFiringLaser();
      beam = new LaserBeam(this, pickedUpPiece);
      game.readout.Update();
      beam.Fire();
      pickedUpPiece.spent = true;
      whenLaserFired = System.currentTimeMillis();
    }
  }
  
  // --------------------------------------------------------------------------
  // Blow up dead pieces and clean up the screen.
  // --------------------------------------------------------------------------
  private void FinishLaser(boolean local)
  {
    if (beam != null && turn != LaserChess.kNoTeam) {
      long sleepTime = Math.max(1, kMinLaserHangTime -
			       (System.currentTimeMillis() - whenLaserFired));
      try { Thread.sleep(sleepTime); } catch(InterruptedException e) { ; }
      game.sounds.StopFiringLaser();
      movesLeft -= LaserChess.kFireLaserCost;
      beam.Erase();
      beam = null;
      repaint();
      boolean aggressiveLaserDied = pickedUpPiece.doomed;
      morgue.DestroyAll(this);
      if (turn != LaserChess.kNoTeam)
	ToggleSelect(false, aggressiveLaserDied, local, true);
      else {
	game.buttons.DisableActionButtons();
	game.menu.DisableActionItems();
	pickedUpPiece.held = pickedUpPiece.doomed = false;
	pickedUpPiece.Draw();
      }
      game.frame.repaint();
    }
  }
  
}
