/***************************************************************************
 *                                                                         *
 *   GALTNG.C                                                              *
 *                                                                         *
 *   Copyright (C) 1990-1994 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is the Tele-Games Tingo Game code.                               *
 *                                                                         *
 *                                            - J. Kobal 11/28/89          *
 *                                            - M. Donnelly 11/8/93        *
 *                                                  Stand alone DLL          *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "flash.h"
#include "bbsutils.h"
#include "galtng.h"
#include "etlapi.h"

/* GALTNG.C 26/03/92 20.09.00 */
void delayed_init(void);
void initng(void);
void tingo(void);
void tngcall(void);
void tshwcrd(void);
void tngon(void);
void tngoff(void);
void tngexit(void);
void tngshtdwn(void);
void tnghlp(void);
void tngscan(void);
void tngpot(void);
int tnghook(int code);
void tnginp(void);

STATIC void tprfall(void);
STATIC void tcards(void);
STATIC void tgencrd(struct plyr *player);
STATIC void tcalls(void);
STATIC void tngnxt(void);
STATIC void tngmark(int num);
STATIC int tchkwn(struct plyr *player,int a,int b);
STATIC int tcol(struct plyr *player,int num);
STATIC int trow(struct plyr *player,int num);
STATIC int tldiag(struct plyr *player);
STATIC int trdiag(struct plyr *player);
STATIC void tshowin(void);
STATIC void psttc(long amtcrd);
STATIC void tnginf(void);

static
struct plyr {                 /* Tingo player structure                    */
     int nums[5][5];          /* Tingo card (0=marked, 1-75=space number)  */
     int request;             /* Flag when person has requested to play    */
     int playing;             /* Flag for when person is currently playing */
} *tingarr,                   /* dynamically allocated player array        */
  *tingptr;                   /* handy plyr structure pointer              */

static
int calls[75],                /* Randomly-ordered TINGO space calls        */
    callnum,                  /* Index to calls[] for current call number  */
    requests,                 /* Number of people requesting to play       */
    winner,                   /* Flag for when the game has been won       */
    players;                  /* Number of people currently playing        */

static
long credprz;                 /* Credit prize for game                     */

/*--- OPTIONS FROM GALETL.MSG ---*/
static
int plytng,                   /* is Tingo available to users now           */
    minply,                   /* minimum number of people to play game     */
    maxply,                   /* maximum number of people to play game     */
    btwngm,                   /* time to wait between games before checking*/
    btwncl,                   /* time to wait between tingo calls          */
    crdzcd,                   /* each players entrance fee to play tingo   */
    credml,                   /* credit multiplier for tingo for wins      */
    logwin;                   /* log winnings to audit trail?              */

int gbldbt;                   /* can users gamble with their debt?         */

FILE *tngmb;

void EXPORT
init__tng(void)               /* initialize tingo when system comes up     */
{
     tngmb=opnmsg("GALTNG.MCV");
     setmbk(tngmb);
     tingarr=(struct plyr *)alczer(nterms*sizeof(struct plyr));
     plytng=ynopt(PLYTING);
     gbldbt=ynopt(GBLDBT);
     minply=numopt(TMINPLY,1,256);
     maxply=numopt(TMAXPLY,1,256);
     btwngm=numopt(TBTWNGM,15,600);
     btwncl=numopt(TBTWNCL,5,60);
     crdzcd=numopt(TCRDZCD,0,32766);
     credml=numopt(TCREDML,1,100);
     logwin=ynopt(LOGWIN);
     winner=0;
     rtkick(1,delayed_init);
}

void
delayed_init(void)                 /* delayed initialization routine       */
{
     if (addhook("tingo",tnghook) != -1) {
          rtkick(btwngm,tingo);
     }
}

int
tnghook(int code)                  /* main input handler                   */
{
     switch (code) {
     case TH_INFO:
          tnginf();
          return(1);
     case TH_INPUT:
          tnginp();
          return(1);
     case TH_EXIT:
          tngexit();
          return(1);
     case TH_SHTDWN:
          tngshtdwn();
          return(1);
     }
     return(0);
}

void
tnginp(void)                       /* handle tingo commands                */
{
     setmbk(tngmb);
     if (margc == 1) {
          tnghlp();
          rstmbk();
          return;
     }
     if (sameas(margv[1],"off")) {
          tngoff();
     }
     else if (sameas(margv[1],"on")) {
          tngon();
     }
     else if (sameas(margv[1],"card")) {
          tshwcrd();
     }
     else if (sameas(margv[1],"scan")) {
          tngscan();
     }
     else if (sameas(margv[1],"pot")) {
          tngpot();
     }
     else {
          tnghlp();
     }
}

void
tingo(void)                   /* handle initiating a game of tingo         */
{
     setmbk(tngmb);
     if (requests < minply) {
          prfmlt(NDPLYRS);
          tprfall();
          rtkick(btwngm,tingo);
     }
     else {
          prfmlt(PLAYMSG);
          tprfall();
          tcards();
          tcalls();
          rtkick(btwncl,tngcall);
     }
}

STATIC void
tprfall(void)                 /* send a tingo message to requesting users  */
{
     int i;

     tlcpmt();
     for (i=0 ; i < nterms ; i++) {
          if (tingarr[i].request) {
               outmlt(i);
          }
     }
     clrmlt();
}

STATIC void
tcards(void)                  /* set users up for playing a game of tingo  */
{
     int num;

     credprz=0;
     players=0;
     winner=0;
     for (num=0 ; num < nterms ; num++) {
          if (tingarr[num].request) {
               if (!odedcrd(num,crdzcd,!gbldbt,0)) {
                    prfmlt(NOTENOC);
               }
               else {
                    tingarr[num].playing=1;
                    players++;
                    credprz+=crdzcd;
                    tgencrd(&tingarr[num]);
                    prfmlt(GIVCARD,crdzcd);
               }
               tlcpmt();
               outmlt(num);
          }
     }
     credprz*=credml;
}

STATIC void
tgencrd(player)               /* generate a tingo card                     */
struct plyr *player;
{
     int a,b,temp,swap,num[15];

     for (a=0 ; a < 5 ; a++) {
          for (b=0 ; b < 15 ; b++) {
               num[b]=15*a+b+1;
          }
          for (b=0 ; b < 15 ; b++) {
               temp=num[b];
               swap=genrdn(0,15);
               num[b]=num[swap];
               num[swap]=temp;
          }
          for (b=0 ; b < 5 ; b++) {
               player->nums[a][b]=num[b];
          }
     }
     player->nums[2][2]=0;
}

STATIC void
tcalls(void)                  /* generate the calls for the next game      */
{
     int i,temp,swap;

     for (i=0 ; i < 75 ; i++) {
          calls[i]=i+1;
     }
     for (i=0 ; i < 75 ; i++) {
          temp=calls[i];
          swap=genrdn(0,75);
          calls[i]=calls[swap];
          calls[swap]=temp;
     }
     callnum=0;
}

void
tngcall(void)                 /* call a tingo number and check for winner  */
{
     int num;
     char *letters="TINGO";

     num=calls[callnum++];
     setmbk(tngmb);
     prfmlt(NEXTNUM,letters[(num - 1)/15],num);
     tprfall();
     tngmark(num);
     if (!winner) {
          if (players > 1) {
               rtkick(btwncl,tngcall);
          }
          else {
               for (num=0 ; num < nterms ; num++) {
                    if (tingarr[num].playing) {
                         usrnum=num;
                         tshowin();
                         tngnxt();
                         break;
                    }
               }
          }
     }
     else {
          tngnxt();
     }
}

STATIC void
tngnxt(void)                  /* reset things in prep for next tingo game  */
{
     int num;

     credprz=0;
     winner=0;
     prfmlt(NEXTGAM);
     tprfall();
     for (num=0 ; num < nterms ; num++) {
          tingarr[num].playing=0;
     }
     rtkick(btwngm,tingo);
}

STATIC void
tngmark(num)                  /* mark a spot on a tingo card               */
int num;
{
     int a,b;

     a=(num-1)/15;
     for (usrnum=0 ; usrnum < nterms ; usrnum++) {
          if (tingarr[usrnum].playing) {
               for (b=0 ; b < 5 ; b++) {
                    if (tingarr[usrnum].nums[a][b] == num) {
                         tingarr[usrnum].nums[a][b]=0;
                         break;
                    }
               }
               if (b < 5) {
                    pmlt("\7");
                    outmlt(usrnum);
                    if (tchkwn(&tingarr[usrnum],a,b)) {
                         tshowin();
                    }
               }
          }
     }
}

STATIC int
tchkwn(player,a,b)            /* check to see if this card is a winner     */
struct plyr *player;
int a,b;
{
     return(tcol(player,a) || trow(player,b) || (a == b && tldiag(player))
       || (a+b == 4 && trdiag(player)));
}

STATIC int
tcol(player,num)              /* does this person have a column win        */
struct plyr *player;
int num;
{
     int i;

     for (i=0 ; i < 5 ; i++) {
          if (player->nums[num][i] > 0) {
               return(0);
          }
     }
     return(1);
}

STATIC int
trow(player,num)              /* does this person have a row win           */
struct plyr *player;
int num;
{
     int i;

     for (i=0 ; i < 5 ; i++) {
          if (player->nums[i][num] > 0) {
               return(0);
          }
     }
     return(1);
}

STATIC int
tldiag(player)                /* does this person have a left diagonal win */
struct plyr *player;
{
     int i;

     for (i=0 ; i < 5 ; i++) {
          if (player->nums[i][i] > 0) {
               return(0);
          }
     }
     return(1);
}

STATIC int
trdiag(player)                /* does this person have a right diagonal win*/
struct plyr *player;
{
     int i;

     for (i=0 ; i < 5 ; i++) {
          if (player->nums[i][4-i] > 0) {
               return(0);
          }
     }
     return(1);
}

STATIC void
tshowin(void)                 /* somebody won, tell everyone about it      */
{
     int unum;

     winner=1;
     usaptr=uacoff(usrnum);
     usrptr=&user[usrnum];
     psttc(credprz);
     prfmlt(WONTING,usaptr->userid);
     tlcpmt();
     for (unum=0 ; unum < nterms ; unum++) {
          if (unum != usrnum && tingarr[unum].request) {
               outmlt(unum);
          }
     }
     prfmlt(YOUTING,l2as(credprz));
     tlcpmt();
     outmlt(usrnum);
}

void
tshwcrd(void)                 /* display a tingo card to a user            */
{
     int a;
     char buf[8];

     tingptr=&tingarr[usrnum];
     if (tingptr->playing) {
          dspmlt(CARDHDR);
          for (a=0 ; a < 5 ; a++) {
               dspmlt(CARDSEP);
               sprintf(buf,"%2d",tingptr->nums[4][a]);
               dspmlt(CARDROW,
                      (tingptr->nums[0][a] == 0) ? " " : spr("%2d",tingptr->nums[0][a]),
                      (tingptr->nums[1][a] == 0) ? " " : spr("%2d",tingptr->nums[1][a]),
                      (tingptr->nums[2][a] == 0) ? " " : spr("%2d",tingptr->nums[2][a]));
               dspmlt(CARDRW2,
                      (tingptr->nums[3][a] == 0) ? " " : spr("%2d",tingptr->nums[3][a]),
                      (tingptr->nums[4][a] == 0) ? " " : buf);
          }
          dspmlt(CARDBTM);
     }
     else {
          prfmlt(WHTCARD);
     }
}

void
tngon(void)                   /* user is requesting to play tingo          */
{
     if (otstcrd(usrnum,1L,!gbldbt)) {
          if (!(tingarr[usrnum].request)) {
               if (usrptr->flags&INVISB) {
                    prfmlt(YOUINV);
               }
               else if (plytng && (requests < maxply)) {
                    prfmlt(TINGOON);
                    tingarr[usrnum].request=1;
                    requests++;
               }
               else {
                    prfmlt(TBLFULL);
               }
          }
          else {
               prfmlt(TINGOON);
          }
     }
     else {
          prfmlt(LIVONLY);
     }
}

void
tngoff(void)                  /* user wants to exit from tingo             */
{
     prfmlt(TINGOFF);
     tngexit();
}

void
tngexit(void)                 /* user exiting tingo for any reason         */
{
     if (tingarr[usrnum].request) {
          tingarr[usrnum].request=0;
          requests--;
     }
     if (tingarr[usrnum].playing) {
          tingarr[usrnum].playing=0;
          players--;
     }
}

void
tngshtdwn(void)               /* shutdown routine for Tingo                */
{
     clsmsg(tngmb);
}

void
tnghlp(void)                  /* display help message for tingo            */
{
     prfmlt(TNGHELP);
}

void
tngscan(void)                 /*  display all involved in tingo            */
{
     int num;

     prfmlt(TSCNHDR);
     for (num=0 ; num < nterms ; num++) {
          if (tingarr[num].request) {
               prfmlt(TSCNLIN,uacoff(num)->userid,
                      tingarr[num].playing ? "*" : " ");
          }
     }
}

void
tngpot(void)                  /* show current tingo pot                    */
{
     prfmlt(TINGPOT,l2as(credprz));
}

STATIC void
psttc(amtcrd)                 /* Post credits to an account                */
long amtcrd;
{
     crdusr(usaptr->userid,l2as(amtcrd),0,1);
     if (logwin) {
          shocst("TELE-TINGO WINNER","%s won %s credits",usaptr->userid,
                                     l2as(amtcrd));
     }
}

STATIC void
tnginf(void)
{
     setmbk(tngmb);
     prfmsg(TNGINF);
}
