/***************************************************************************
 *                                                                         *
 *   GALPOK.C                                                              *
 *                                                                         *
 *   Copyright (C) 1990-1994 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is the Tele-Games Poker Game code.                               *
 *                                                                         *
 *                                            - J. Kobal 2/8/90            *
 *                                            - M. Donnelly 11/8/93        *
 *                                                  Stand alone DLL          *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "flash.h"
#include "bbsutils.h"
#include "galpok.h"
#include "etlapi.h"

/* GALPOK.C 06/04/92 10.48.06 */
void delayed_init(void);
void initpv(void);
void initpok(void);
void poker(void);
void pprfall(void);
void pprfbt(int num);
void pdeal(void);
void pshuffle(void);
void plypok(void);
void pfindwin(void);
void pshowin(int num);
void pshwhnd(void);
void pshowh(int num);
int pnxtrn(int num);
void pnext(void);
void pokon(void);
void pokfold(void);
void pokoff(void);
void pokexit(void);
void pokshtdwn(void);
void poklow(void);
void pokhigh(void);
void pokcall(void);
void pokbet(void);
void pokrais(void);
void pdspcrd(int num);
int pgvcrds(void);
void pokdscrd(void);
void pokhlp(void);
void pokscan(void);
void pokpot(void);
void pokturn(void);
int pokhook(int code);
void pokinp(void);

STATIC void ppsttc(int num,long amtcrd);
STATIC void psrthnd(struct pokplyr *player);
STATIC void pdealhnd(struct pokplyr *player);
STATIC void pchkhnd(struct pokplyr *player,int *s1,int *s2,int *s3,int *s4,int *s5);
STATIC void pokinf(void);

static
struct pokplyr {              /* Poker player structure                    */
     int cards[5];            /* The players hand (5 cards)                */
     long bet;                /* How much player has bet on current hand   */
     int flags;               /* Various bit flags                         */
} *pokarr,                    /* dynamically allocated player array        */
  *pokptr;                    /* handy pokplyr structure pointer           */

                              /* Bit flag definitions for each player      */
#define REQUEST 0x0001        /*   Player has requested to play            */
#define PLAYING 0x0002        /*   Player is currently playing the game    */
#define DISCARD 0x0004        /*   Player has discarded already            */
#define DISCRD1 0x0008        /*   Player has discarded first card in hand */
#define DISCRD2 0x0010        /*   Player has discarded second card in hand*/
#define DISCRD3 0x0020        /*   Player has discarded third card in hand */
#define DISCRD4 0x0040        /*   Player has discarded fourth card in hand*/
#define DISCRD5 0x0080        /*   Player has discarded fifth card in hand */

static
int deck[52],                 /* Randomly-ordered deck of playing cards    */
    cardnum,                  /* Index to deck[] for next card to be dealt */
    putback,                  /* Index to deck[] for putting back cards    */
    minbet,                   /* Minimum bet for the game                  */
    maxbet,                   /* Maximum bet for the game                  */
    maxraises,                /* Maximum number of raises per bet round    */
    wait,                     /* Number of time cycles game has waited     */
    raises,                   /* Number of times the bet has been raised   */
    prequests,                /* Number of people requesting to play       */
    turn,                     /* Whose turn it is                          */
    first,                    /* First player in betting cycle             */
    pplayers,                 /* Number of people currently playing        */
    pflags;                   /* Various bit flags for the game            */

                              /* Bit flag definitions for game flag        */
#define HIGHSTAKES 0x0001          /* Game is for high stakes              */
#define DISCARDING 0x0002          /* Players are discarding cards         */
#define DISCARDED  0x0004          /* Players are finished discarding      */
#define WINNER     0x0008          /* Game has been won                    */

static
long plybet,                  /* Bet amount needed to "stay in the game"   */
     pot;                     /* Current "pot" for the game                */

/*--- OPTIONS FROM GALPOK.MSG ---*/
static
int pokply,                   /* is poker available to users now           */
    minply,                   /* minimum number of people to play game     */
    maxply,                   /* maximum number of people to play game     */
    btwngm,                   /* time to wait between games before checking*/
    btwnst,                   /* time to wait between in game checks       */
    minbtl,                   /* minimum bet in a low stakes poker game    */
    maxbtl,                   /* maximum bet in a low stakes poker game    */
    minbth,                   /* minimum bet in a high stakes poker game   */
    maxbth,                   /* maximum bet in a high stakes poker game   */
    maxrsl,                   /* maximum number of raises in low stakes    */
    maxrsh,                   /* maximum number of raises in high stakes   */
    resetl,                   /* after each game reset to low stakes? 1=YES*/
    housct,                   /* percent of pot taken by the house         */
    dlygam,                   /* number of checks before prodding a user   */
    kckout,                   /* number of checks before tossing a user    */
    logwin;                   /* log winnings to audit trail?              */

int gbldbt;                   /* can users gamble with their debt?         */

                              /* Poker hand status                         */
#define NOTHING   0           /*   User has only a high card               */
#define ONEPAIR   1           /*   User has one pair                       */
#define TWOPAIR   2           /*   User has two pair                       */
#define THREEKIND 3           /*   User has three of a kind                */
#define STRAIGHT  4           /*   User has a straight                     */
#define FLUSH     5           /*   User has a flush                        */
#define FULLHOUSE 6           /*   User has a full house                   */
#define FOURKIND  7           /*   User has four of a kind                 */

FILE *pokmb;

void EXPORT
init__poker(void)                 /* initialize poker when system comes up     */
{
     pokmb=opnmsg("GALPOK.MCV");
     setmbk(pokmb);
     pokarr=(struct pokplyr *)alczer(nterms*sizeof(struct pokplyr));
     initpv();
     pokply=ynopt(PLYPOKR);
     gbldbt=ynopt(GBLDBT);
     minply=numopt(PMINPLY,1,8);
     maxply=numopt(PMAXPLY,1,8);
     btwngm=numopt(PBTWNGM,15,600);
     btwnst=numopt(PBTWNST,5,60);
     minbtl=numopt(PMINBTL,0,32766);
     maxbtl=numopt(PMAXBTL,0,32766);
     minbth=numopt(PMINBTH,0,32766);
     maxbth=numopt(PMAXBTH,0,32766);
     maxrsl=numopt(PMAXRSL,0,100);
     maxrsh=numopt(PMAXRSH,0,100);
     resetl=ynopt(PRESETL);
     housct=numopt(PHOUSCT,0,100);
     dlygam=numopt(PDLYGAM,1,100);
     kckout=numopt(PKCKOUT,1,100);
     logwin=ynopt(LOGWIN);
     pflags=0;
     rtkick(1,delayed_init);
}

void
delayed_init(void)                 /* delayed initialization routine       */
{
     if (addhook("poker",pokhook) != -1) {
          rtkick(btwngm,poker);
     }
}


int
pokhook(int code)                  /* status 3 handler                     */
{
     switch (code) {
     case TH_INFO:
          pokinf();
          return(1);
     case TH_INPUT:
          pokinp();
          return(1);
     case TH_EXIT:
          pokexit();
          return(1);
     case TH_SHTDWN:
          pokshtdwn();
          return(1);
     }
     return(0);
}

void
initpv(void)                  /* initialize things for a new game          */
{
     pflags&=~(DISCARDING|DISCARDED|WINNER);
     pot=0;
     raises=0;
     first=-1;
     turn=-1;
     if (resetl && (pflags&HIGHSTAKES)) {
          pflags&=~HIGHSTAKES;
          prfmlt(LOWSTKS);
          pprfall();
     }
}

void
pokinp(void)                       /* main input handler                   */
{
     setmbk(pokmb);
     if (margc == 1) {
          pokhlp();
          rstmbk();
          return;
     }
     if (sameas(margv[1],"off")) {
          pokoff();
     }
     else if (sameas(margv[1],"on")) {
          pokon();
     }
     else if (sameas(margv[1],"hand")) {
          pshwhnd();
     }
     else if (sameas(margv[1],"scan")) {
          pokscan();
     }
     else if (sameas(margv[1],"turn")) {
          pokturn();
     }
     else if (sameas(margv[1],"fold")) {
          pokfold();
     }
     else if (sameas(margv[1],"pot")) {
          pokpot();
     }
     else if (sameas(margv[1],"call")) {
          pokcall();
     }
     else if (sameas(margv[1],"bet")) {
          pokbet();
     }
     else if (sameas(margv[1],"raise")) {
          pokrais();
     }
     else if (sameas(margv[1],"discard")) {
          pokdscrd();
     }
     else if (sameas(margv[1],"low")) {
          poklow();
     }
     else if (sameas(margv[1],"high")) {
          pokhigh();
     }
     else {
          pokhlp();
     }
}

void
poker(void)                   /* handle initiating a game of poker         */
{
     setmbk(pokmb);
     if (prequests < minply) {
          prfmlt(NEEDMOR);
          pprfall();
          rtkick(btwngm,poker);
     }
     else {
          prfmlt(PLAYPOK);
          pprfall();
          pshuffle();
          pdeal();
          if (pplayers > 0) {
               pnext();
               rtkick(btwnst,plypok);
          }
          else {
               initpv();
               rtkick(btwngm,poker);
          }
     }
}

void
pprfall(void)                 /* send a poker message to requesting users  */
{
     int i;

     tlcpmt();
     for (i=0 ; i < nterms ; i++) {
          if (pokarr[i].flags&REQUEST) {
               outmlt(i);
          }
     }
     clrmlt();
}

void
pprfbt(num)                   /* send poker message to all but 'num'       */
int num;
{
     int i;

     tlcpmt();
     for (i=0 ; i < nterms ; i++) {
          if ((pokarr[i].flags&REQUEST) && (i != num)) {
               outmlt(i);
          }
     }
     clrmlt();
}

void
pdeal(void)                   /* set users up for playing a game of poker  */
{
     int num;

     pot=0;
     pplayers=0;
     minbet=((pflags&HIGHSTAKES) ? minbth : minbtl);
     maxbet=((pflags&HIGHSTAKES) ? maxbth : maxbtl);
     maxraises=((pflags&HIGHSTAKES) ? maxrsh : maxrsl);
     plybet=minbet;
     for (num=0 ; num < nterms ; num++) {
          pokptr=&pokarr[num];
          if (pokptr->flags&REQUEST) {
               if (!odedcrd(num,(long)minbet,!gbldbt,0)) {
                    prfmlt(NOTENUF);
               }
               else {
                    pokptr->flags=(REQUEST+PLAYING);
                    pplayers++;
                    pot+=minbet;
                    pokptr->bet=minbet;
                    pdealhnd(pokptr);
                    prfmlt(ANTEPOT,minbet);
               }
               tlcpmt();
               outmlt(num);
          }
     }
}

STATIC void
psrthnd(player)               /* sort a players poker hand                 */
struct pokplyr *player;
{
     int a,temp,ok=0;

     while (!ok) {
          ok=1;
          for (a=0 ; a < 4 ; a++) {
               if ((player->cards[a]%13) >= (player->cards[a+1]%13)) {
                    if (((player->cards[a]%13) > (player->cards[a+1]%13))
                      || ((player->cards[a]/13) > (player->cards[a+1]/13))) {
                         ok=0;
                         temp=player->cards[a];
                         player->cards[a]=player->cards[a+1];
                         player->cards[a+1]=temp;
                    }
               }
          }
     }
}

STATIC void
pdealhnd(player)              /* deal five cards to a player               */
struct pokplyr *player;
{
     int a;

     for (a=0 ; a < 5 ; a++) {
          player->cards[a]=deck[cardnum++];
     }
     psrthnd(player);
}

void
pshuffle(void)                /* shuffle the card deck                     */
{
     int i,temp,swap;

     for (i=0 ; i < 52 ; i++) {
          deck[i]=i;
     }
     for (i=0 ; i < 51 ; i++) {
          temp=deck[i];
          swap=genrdn(i,52);
          deck[i]=deck[swap];
          deck[swap]=temp;
     }
     cardnum=0;
     putback=0;
}

void
plypok(void)                  /* handle the current poker game             */
{
     int num;

     setmbk(pokmb);
     if (pplayers == 0) {
          initpv();
          prfmlt(NXTGAME);
          pprfall();
          for (num=0 ; num < nterms ; num++) {
               pokarr[num].flags&=~PLAYING;
          }
          rtkick(btwngm,poker);
     }
     else if (pplayers <= 1) {
          for (num=0 ; num < nterms ; num++) {
               if (pokarr[num].flags&PLAYING) {
                    pshowin(num);
                    break;
               }
          }
          initpv();
          rtkick(btwngm,poker);
     }
     else if (turn == -1) {
          initpv();
          rtkick(btwngm,poker);
     }
     else {
          if (!(pokarr[turn].flags&PLAYING)) {
               pnext();
          }
          else if (++wait > dlygam) {
               if (!otstcrd(turn,(plybet-pokarr[turn].bet),!gbldbt)) {
                    prfmlt(YOUROUT);
                    tlcpmt();
                    outmlt(turn);
                    pokarr[turn].flags&=~PLAYING;
                    pplayers--;
                    prfmlt(OUTCRED,uacoff(turn)->userid);
                    prfmlt(FOLDING,uacoff(turn)->userid);
                    pprfbt(turn);
                    prfmlt(YOUFOLD);
                    tlcpmt();
                    outmlt(turn);
                    num=turn;
                    pnext();
                    if (first == num) {
                         first=pnxtrn(num);
                    }
                    tlcpmt();
                    outmlt(num);
               }
               else if (wait > kckout) {
                    prfmlt(PBYEBYE);
                    outmlt(turn);
                    pokarr[turn].flags&=~PLAYING;
                    pplayers--;
                    prfmlt(FOLDING,uacoff(turn)->userid);
                    pprfbt(turn);
                    prfmlt(YOUFOLD);
                    outmlt(turn);
                    num=turn;
                    pnext();
                    if (first == num) {
                         first=pnxtrn(num);
                    }
                    tlcpmt();
                    outmlt(num);
               }
               else {
                    prfmlt(WAITING);
                    tlcpmt();
                    outmlt(turn);
               }
          }
          if (pflags&WINNER) {
               initpv();
               rtkick(btwngm,poker);
          }
          else {
               rtkick(btwnst,plypok);
          }
     }
}

STATIC void
pchkhnd(                           /* determine the players high hand      */
struct pokplyr *player,
int *s1,int *s2,int *s3,
int *s4,int *s5)
{
     int a,b,c,d,e,f;

     f=((player->cards[0]/13) == (player->cards[1]/13))
       && ((player->cards[1]/13) == (player->cards[2]/13))
       && ((player->cards[2]/13) == (player->cards[3]/13))
       && ((player->cards[3]/13) == (player->cards[4]/13));
     a=player->cards[0]%13;
     b=player->cards[1]%13;
     c=player->cards[2]%13;
     d=player->cards[3]%13;
     e=player->cards[4]%13;
     *s1=(f ? FLUSH : NOTHING);
     *s1+=(((a == b-1) && (b == c-1)&&(c == d-1) && (d == e-1)) ? STRAIGHT
                                                                : NOTHING);
     if (*s1 > NOTHING) {      /* Straight, Flush, or Straight Flush   */
          *s2=e;
          *s3=(player->cards[0]/13);
          *s4=(player->cards[1]/13);
          *s5=(player->cards[2]/13);
     }
     else if (a == d) {
          *s1=FOURKIND;
          *s2=a;
          *s3=e;
          *s4=0;
          *s5=0;
     }
     else if (b == e) {
          *s1=FOURKIND;
          *s2=e;
          *s3=a;
          *s4=0;
          *s5=0;
     }
     else if (a == c) {
          *s1=((d == e) ? FULLHOUSE : THREEKIND);
          *s2=a;
          *s3=e;
          *s4=d;
          *s5=0;
     }
     else if (c == e) {
          *s1=((a == b) ? FULLHOUSE : THREEKIND);
          *s2=e;
          *s3=b;
          *s4=a;
          *s5=0;
     }
     else if (b == d) {
          *s1=THREEKIND;
          *s2=d;
          *s3=e;
          *s4=a;
          *s5=0;
     }
     else if (a == b) {
          if (c == d) {
               *s1=TWOPAIR;
               *s2=d;
               *s3=b;
               *s4=e;
               *s5=player->cards[3]/13;
          }
          else if (d == e) {
               *s1=TWOPAIR;
               *s2=e;
               *s3=b;
               *s4=c;
               *s5=player->cards[4]/13;
          }
          else {
               *s1=ONEPAIR;
               *s2=b;
               *s3=e;
               *s4=d;
               *s5=c;
          }
     }
     else if (b == c) {
          if (d == e) {
               *s1=TWOPAIR;
               *s2=e;
               *s3=c;
               *s4=a;
               *s5=player->cards[4]/13;
          }
          else {
               *s1=ONEPAIR;
               *s2=c;
               *s3=e;
               *s4=d;
               *s5=a;
          }
     }
     else if (c == d) {
          *s1=ONEPAIR;
          *s2=d;
          *s3=e;
          *s4=b;
          *s5=a;
     }
     else if (d == e) {
          *s1=ONEPAIR;
          *s2=e;
          *s3=c;
          *s4=b;
          *s5=a;
     }
     else {
          *s1=NOTHING;
          *s2=e;
          *s3=d;
          *s4=c;
          *s5=b;
     }
}

void
pfindwin(void)                /* find the winner of this poker game        */
{
     int num1,num2,sw;
     int a1,a2,a3,a4,a5;
     int b1,b2,b3,b4,b5;

     num1=a1=a2=a3=a4=a5=-1;
     for (num2=0 ; num2 < nterms ; num2++) {
          if ((pokptr=&pokarr[num2])->flags&PLAYING) {
               sw=0;
               pchkhnd(pokptr,&b1,&b2,&b3,&b4,&b5);
               if (b1 > a1) {
                    sw=1;
               }
               else if (b1 == a1) {
                    if (b2 > a2) {
                         sw=1;
                    }
                    else if (b2 == a2) {
                         if (b3 > a3) {
                              sw=1;
                         }
                         else if (b3 == a3) {
                              if (b4 > a4) {
                                   sw=1;
                              }
                              else if (b4 == a4) {
                                   if (b5 > a5) {
                                        sw=1;
                                   }
                              }
                         }
                    }
               }
               if (sw) {
                    a1=b1;
                    a2=b2;
                    a3=b3;
                    a4=b4;
                    a5=b5;
                    num1=num2;
               }
          }
     }
     if (num1 >= 0) {
          pshowin(num1);
     }
}

void
pshowin(                      /* show the poker winner                     */
int num)
{
     int unum;

     pflags|=WINNER;
     unum=usrnum;
     ppsttc(num,pot);
     for (usrnum=0 ; usrnum < nterms ; usrnum++) {
          if ((pokarr[usrnum].flags&REQUEST) && (usrnum != num)) {
               usaptr=uacoff(usrnum);
               prfmlt(WONPOKR,uacoff(num)->userid);
               pshowh(num);
               tlcpmt();
               outmlt(usrnum);
          }
     }
     usrnum=unum;
     usaptr=uacoff(usrnum);
     prfmlt(YOUPOKR,l2as(pot));
     tlcpmt();
     outmlt(num);
     initpv();
     prfmlt(NXTGAME);
     pprfall();
     for (num=0 ; num < nterms ; num++) {
          pokarr[num].flags&=~PLAYING;
     }
}

void
pshwhnd(void)                 /* show a poker hand                         */
{
     pshowh(usrnum);
}

void
pshowh(                       /* show the hand of 'num'                    */
int num)
{
     int a;
     int h1,h2,h3,h4,h5;
     static char *crds[13]={
          "Deuce",
          "Three",
          "Four",
          "Five",
          "Sixe",
          "Seven",
          "Eight",
          "Nine",
          "Ten",
          "Jack",
          "Queen",
          "King",
          "Ace"
     };
     static char *suits[4]={
          "Hearts",
          "Clubs",
          "Diamonds",
          "Spades"
     };

     pokptr=&pokarr[num];
     if (pokptr->flags&PLAYING) {
          dspmlt(INSERT6);
          for (a=0 ; a < 5 ; a++) {
               dspmlt(PCARD0+2*(pokptr->cards[a]));
          }
          pchkhnd(pokptr,&h1,&h2,&h3,&h4,&h5);
          switch (h1) {
          case NOTHING:
               prfmlt(POKNOT,crds[h2]);
               break;
          case ONEPAIR:
               prfmlt(HSPAIR, crds[h2]);
               break;
          case TWOPAIR:
               prfmlt(TWPAIR,crds[h2],crds[h3]);
               break;
          case THREEKIND:
               prfmlt(THREEO,crds[h2]);
               break;
          case STRAIGHT:
               prfmlt(STRAIG,crds[h2]);
               break;
          case FLUSH:
               prfmlt(AFLUSH,suits[h3]);
               break;
          case FULLHOUSE:
               prfmlt(FHOUSE,crds[h2],crds[h3]);
               break;
          case FOURKIND:
               prfmlt(FOAKND,crds[h2]);
               break;
          case STRAIGHT+FLUSH:
               prfmlt(STRFLU,crds[h2]);
               break;
          }
     }
     else {
          prfmlt(WHTHAND);
     }
}

int
pnxtrn(                       /* determine who goes next                   */
int num)
{
     int n;

     if (pplayers <= 0) {
          n=-1;
     }
     else {
          n=num;
          do {
               n=((n+1)%nterms);
          } while (!(pokarr[n].flags&PLAYING) && (n != num));
          if (!(pokarr[n].flags&PLAYING)) {
               n=-1;
          }
     }
     return(n);
}

void
pnext(void)                   /* determine the next event to happen        */
{
     int num;

     wait=0;
     turn=pnxtrn(turn);
     if (turn == -1) {
          initpv();
          prfmlt(NXTGAME);
          pprfall();
          for (num=0 ; num < nterms ; num++) {
               pokarr[num].flags&=~PLAYING;
          }
     }
     else if (first == turn) {
          if (pflags&DISCARDING) {
               pflags&=~DISCARDING;
               pflags|=DISCARDED;
               raises=0;
               first=-1;
               turn=-1;
               pnext();
          }
          else if (pflags&DISCARDED) {
               if (pplayers > 1) {
                    pfindwin();
               }
               else if (pplayers == 1) {
                    pshowin(turn);
               }
          }
          else {
               pflags|=DISCARDING;
               first=-1;
               turn=-1;
               pnext();
          }
     }
     else {
          if (pflags&DISCARDING) {
               prfmlt(YOURDSC);
          }
          else {
               prfmlt(YOURBET,l2as(plybet-pokarr[turn].bet));
          }
          tlcpmt();
          outmlt(turn);
     }
     if (first == -1) {
          first=turn;
     }
     clrmlt();
}

void
pokon(void)                   /* user requested to play poker              */
{
     if (otstcrd(usrnum,1L,!gbldbt)) {
          if (!(pokarr[usrnum].flags&REQUEST)) {
               if (usrptr->flags&INVISB) {
                    prfmlt(YOUINV);
               }
               else if (pokply && (prequests < maxply)) {
                    prfmlt(DEALINM,usaptr->userid);
                    pprfbt(usrnum);
                    prfmlt(POKERON,(pflags&HIGHSTAKES) ? ", for high stakes"
                                                       : "");
                    pokarr[usrnum].flags|=REQUEST;
                    prequests++;
               }
               else {
                    prfmlt(TBLFULL);
               }
          }
          else {
               prfmlt(POKERON,(pflags&HIGHSTAKES) ? ", for high stakes" : "");
          }
     }
     else {
          prfmlt(LIVEPLY);
     }
}

void
pokfold(void)                 /* user folded from current hand             */
{
     if (pokarr[usrnum].flags&PLAYING) {
          pokarr[usrnum].flags&=~PLAYING;
          pplayers--;
          prfmlt(FOLDING,usaptr->userid);
          pprfbt(usrnum);
          prfmlt(YOUFOLD);
          tlcpmt();
          outmlt(usrnum);
          if (turn == usrnum) {
               pnext();
          }
          if (first == usrnum) {
               first=pnxtrn(usrnum);
          }
          pmlt("");
     }
     else {
          prfmlt(CNTFOLD);
     }
}

void
pokoff(void)                  /* user exited from current status of poker  */
{
     if (pokarr[usrnum].flags&PLAYING) {
          prfmlt(DEALOUT,usaptr->userid);
          pprfbt(usrnum);
          pokfold();
          prfmlt(POKEROF);
          pokexit();
     }
     else if (pokarr[usrnum].flags&REQUEST) {
          prfmlt(RQSTOFF);
          pokexit();
     }
     else {
          prfmlt(NTPLING);
     }
}

void
pokexit(void)                 /* user forced to exit poker completely      */
{
     setmbk(pokmb);
     if (pokarr[usrnum].flags&REQUEST) {
          pokarr[usrnum].flags&=~REQUEST;
          prequests--;
     }
     if (pokarr[usrnum].flags&PLAYING) {
          pokarr[usrnum].flags&=~PLAYING;
          if (--pplayers == 0) {
               turn=-1;
               first=-1;
          }
          else {
               if (turn == usrnum) {
                    pnext();
               }
               if (first == usrnum) {
                    first=pnxtrn(usrnum);
               }
          }
          pmlt("");
     }
     rstmbk();
}

void
pokshtdwn(void)               /* shutdown routine for Poker                */
{
     clsmsg(pokmb);
}

void
poklow(void)                  /* set poker to low stakes game              */
{
     if (hasmkey(STKKEY)) {
          pflags&=~HIGHSTAKES;
          prfmlt(LOWSTKS);
          pprfbt(usrnum);
          prfmlt(LOWSTAK);
     }
     else {
          prfmlt(POKHELP);
     }
}

void
pokhigh(void)                 /* set poker to high stakes game             */
{
     if (hasmkey(STKKEY)) {
          pflags|=HIGHSTAKES;
          prfmlt(HISTAKS);
          pprfbt(usrnum);
          prfmlt(HIGHSTK);
     }
     else {
          prfmlt(POKHELP);
     }
}

void
pokcall(void)                 /* user is "calling"                         */
{
     long betamt;

     if ((pokarr[usrnum].flags&PLAYING) && (turn == usrnum)
       && !(pflags&DISCARDING)) {
          betamt=plybet-pokarr[usrnum].bet;
          if (!odedcrd(usrnum,betamt,!gbldbt,0)) {
               prfmlt(CANTBET);
          }
          else {
               pot+=betamt;
               pokarr[usrnum].bet=plybet;
               prfmlt(BETCALL,usaptr->userid);
               pprfbt(usrnum);
               prfmlt(YOUCALL);
               tlcpmt();
               outmlt(usrnum);
               pnext();
               pmlt("");
          }
     }
     else {
          prfmlt(NOTYOUR);
     }
}

void
pokbet(void)                  /* user is placing a bet                     */
{
     long b,r;

     if ((pokarr[usrnum].flags&PLAYING) && (turn == usrnum)
       && !(pflags&DISCARDING)) {
          if (margc == 2) {
               pokcall();
          }
          else {
               b=atol(margv[2]);
               if (b < minbet) {
                    prfmlt(BETOOLW,minbet);
               }
               else if (b > maxbet) {
                    prfmlt(BETOOHI,maxbet);
               }
               else if (!odedcrd(usrnum,b,!gbldbt,0)) {
                    prfmlt(CANTBET);
               }
               else {
                    pokarr[usrnum].bet+=b;
                    pot+=b;
                    r=pokarr[usrnum].bet-plybet;
                    if (r < 0) {
                         prfmlt(BETSOME,usaptr->userid,b);
                         pprfbt(usrnum);
                         prfmlt(BETMORE);
                         tlcpmt();
                         outmlt(usrnum);
                    }
                    else {
                         if (r == 0) {
                              prfmlt(BETCALL,usaptr->userid);
                              pprfbt(usrnum);
                              prfmlt(YOUCALL);
                         }
                         else if (raises < maxraises) {
                              plybet+=r;
                              raises++;
                              first=turn;
                              prfmlt(BETRAIS,usaptr->userid,l2as(r));
                              pprfbt(usrnum);
                              prfmlt(YOURAIS,l2as(r));
                         }
                         else {
                              pokarr[usrnum].bet-=r;
                              crdusr(usaptr->userid,l2as(r),0,0);
                              pot-=r;
                              prfmlt(BETCALL,usaptr->userid);
                              pprfbt(usrnum);
                              prfmlt(NORAISE);
                         }
                         tlcpmt();
                         outmlt(usrnum);
                         pnext();
                    }
               }
          }
          pmlt("");
     }
     else {
          prfmlt(NOTYOUR);
     }
}

void
pokrais(void)                 /* user is "raising"                         */
{
     int b=0;
     long betamt;

     if ((pokarr[usrnum].flags&PLAYING) && (turn == usrnum)
       && !(pflags&DISCARDING)) {
          if (margc == 2) {
               pokcall();
          }
          else {
               b=atoi(margv[2]);
               betamt=(plybet-pokarr[usrnum].bet)+b;
               if (b < minbet) {
                    prfmlt(BETOOLW,minbet);
               }
               else if (b > maxbet) {
                    prfmlt(BETOOHI,maxbet);
               }
               else if (!odedcrd(usrnum,betamt,!gbldbt,0)) {
                    prfmlt(CANTBET);
               }
               else {
                    pot+=betamt;
                    pokarr[usrnum].bet=plybet+b;
                    if (b == 0) {
                         prfmlt(BETCALL,usaptr->userid);
                         pprfbt(usrnum);
                         prfmlt(YOUCALL);
                    }
                    else if (raises < maxraises) {
                         plybet+=b;
                         raises++;
                         first=turn;
                         prfmlt(BETRAIS,usaptr->userid,spr("%d",b));
                         pprfbt(usrnum);
                         prfmlt(YOURAIS,spr("%d",b));
                    }
                    else {
                         pokarr[usrnum].bet-=b;
                         crdusr(usaptr->userid,l2as(b),0,0);
                         pot-=b;
                         prfmlt(BETCALL,usaptr->userid);
                         pprfbt(usrnum);
                         prfmlt(NORAISE);
                    }
                    tlcpmt();
                    outmlt(usrnum);
                    pnext();
               }
          }
          pmlt("");
     }
     else {
          prfmlt(NOTYOUR);
     }
}

void
pdspcrd(num)                  /* display a poker card                      */
int num;
{
     static char *crds[13]={
          " Two",
          " Three",
          " Four",
          " Five",
          " Six",
          " Seven",
          "n Eight",
          " Nine",
          " Ten",
          " Jack",
          " Queen",
          " King",
          "n Ace"
     };

     static char *suits[4]={
          "Hearts",
          "Clubs",
          "Diamonds",
          "Spades"
     };

     prfmlt(DEALCRD,crds[num%13],suits[num/13]);
}

int
pgvcrds(void)                 /* deal cards to a player if need to         */
{
     int f=0;

     pokptr=&pokarr[usrnum];
     if (pokptr->flags&DISCRD1) {
          pdspcrd(pokptr->cards[0]=deck[cardnum++]);
          f++;
     }
     cardnum=cardnum%52;
     if (pokptr->flags&DISCRD2) {
          pdspcrd(pokptr->cards[1]=deck[cardnum++]);
          f++;
     }
     cardnum=cardnum%52;
     if (pokptr->flags&DISCRD3) {
          pdspcrd(pokptr->cards[2]=deck[cardnum++]);
          f++;
     }
     cardnum=cardnum%52;
     if (pokptr->flags&DISCRD4) {
          pdspcrd(pokptr->cards[3]=deck[cardnum++]);
          f++;
     }
     cardnum=cardnum%52;
     if (pokptr->flags&DISCRD5) {
          pdspcrd(pokptr->cards[4]=deck[cardnum++]);
          f++;
     }
     cardnum=cardnum%52;
     if (f == 0) {
          prfmlt(YOUSTND);
     }
     else {
          psrthnd(pokptr);
          pmlt("");
     }
     return(f);
}

void
pokdscrd(void)                /* discard a card from a players hand        */
{
     int a,c;

     pokptr=&pokarr[usrnum];
     if ((pokptr->flags&PLAYING) && (turn == usrnum) && (pflags&DISCARDING)) {
          pokptr->flags|=DISCARD;
          if (margc > 2) {
               if (margc > 6) {
                    margc=6;
               }
               for (a=2 ; a < margc ; a++) {
                    switch (atoi(margv[a])) {
                    case 1:
                         if (!(pokptr->flags&DISCRD1)) {
                              deck[putback++]=pokptr->cards[0];
                         }
                         pokptr->flags|=DISCRD1;
                         break;
                    case 2:
                         if (!(pokptr->flags&DISCRD2)) {
                              deck[putback++]=pokptr->cards[1];
                         }
                         pokptr->flags|=DISCRD2;
                         break;
                    case 3:
                         if (!(pokptr->flags&DISCRD3)) {
                              deck[putback++]=pokptr->cards[2];
                         }
                         pokptr->flags|=DISCRD3;
                         break;
                    case 4:
                         if (!(pokptr->flags&DISCRD4)) {
                              deck[putback++]=pokptr->cards[3];
                         }
                         pokptr->flags|=DISCRD4;
                         break;
                    case 5:
                         if (!(pokptr->flags&DISCRD5)) {
                              deck[putback++]=pokptr->cards[4];
                         }
                         pokptr->flags|=DISCRD5;
                         break;
                    }
               }
          }
          c=pgvcrds();
          tlcpmt();
          outmlt(usrnum);
          if (c == 0) {
               prfmlt(STNDPAT,usaptr->userid);
          }
          else {
               prfmlt(TOOKCRD,usaptr->userid,c,(c > 1) ? "s" : "");
          }
          pprfbt(usrnum);
          pnext();
          pmlt("");
     }
     else {
          prfmlt(NOTYOUR);
     }
}

void
pokhlp(void)                  /* display poker help message                */
{
     prfmlt(POKHELP);
}

void
pokscan(void)                 /* display people involved in poker          */
{
     int num;

     prfmlt(PSCNHDR);
     for (num=0 ; num < nterms ; num++) {
          if (pokarr[num].flags&REQUEST) {
               prfmlt(PSCNLIN,uacoff(num)->userid,
                     (pokarr[num].flags&PLAYING) ? "*" : " ");
          }
     }
}

void
pokpot(void)                  /* display current poker pot                 */
{
     prfmlt(POKRPOT,l2as(pot));
}

void
pokturn(void)                 /* display poker turn status information     */
{
     if (turn == usrnum) {
          if (pflags&DISCARDING) {
               prfmlt(YOURDSC);
          }
          else {
               prfmlt(YOURBET,l2as(plybet-pokarr[usrnum].bet));
          }
     }
     else {
          if (turn == -1) {
               prfmlt(NOGMYET);
          }
          else {
               prfmlt(WHOTURN,uacoff(turn)->userid,
                     (pflags&DISCARDING) ? "discard" : "bet");
          }
     }
}

STATIC void
ppsttc(num,amtcrd)            /* Post credits to an account                */
int num;
long amtcrd;
{
     pot=amtcrd=(amtcrd*(100-housct))/100;
     crdusr(uacoff(num)->userid,l2as(amtcrd),0,1);
     usrnum=-1;
     if (logwin) {
          shocst("TELE-POKER WINNER","%s won %s credits",uacoff(num)->userid,
                 l2as(amtcrd));
     }
}

STATIC void
pokinf(void)
{
     setmbk(pokmb);
     prfmsg(POKINF);
}
