/*
*********************************************************************
* hwmanl.c
* This source file is part of HighWaterMark-Account-Number. It is meant
* to be "included" into the source programs that need it. All variables
* and other inclusions must be done in the source program that uses this
* file.
*
* Copyright 1994 by HighWaterMark Communication Systems,
* Frank LaPiana, Highland Park, New Jersey, USA
*********************************************************************
*/




/*
*********************************************************************
* an_OpenInit
* Open up data and message files as necessary. Should only be called
* once at start of program.
*
* parameters
* ----------
*	none
*
* return values
* -------------
*	none
*********************************************************************
*/
void EXPORT an_OpenInit (void)
{
	char				*pTemp1;

	// open up message and btrieve data files if not already done
	if (an_MsgFile == NULLPTR)
		an_MsgFile = opnmsg ("hwman.mcv");
	setmbk (an_MsgFile);

	pTemp1 = stgopt (ANDATAFL);
	if (an_DataFile == NULLPTR)
		an_DataFile = opnbtv (pTemp1, sizeof(AN_RECORD));
	free (pTemp1);
}
/*
*********************************************************************
* an_DataKey
* make the appropriate key structure for a record.
*
* parameters
* ----------
*	Record - a pointer to a structure (memory area) that contains
*		the record with the key fields to be used already set.
*	KeyBuff - memory area to contain the key structure, must be at
*		LEAST the size of "an_KeyBuff".
*	IndexValue - an integer specifying which of the indices to
*		use when building the key.
*
* return values
* -------------
*	KeyBuff is altered.
*********************************************************************
*/
void EXPORT an_DataKey (AN_RECORD *Record, AN_KEY_BUFF *KeyBuff, int IndexValue)
{
	setmem (KeyBuff, sizeof(AN_KEY_BUFF), NULLCHAR);
//	makhdl (Record->Name);
	switch (IndexValue)
	{
		default:
		case AN_NAME:
			strcpy (KeyBuff->Key0.Name, Record->Name);
			KeyBuff->Key0.Number = Record->Number;
			break;
		case AN_NUMBER:
			KeyBuff->Key1.Number = Record->Number;
			break;
		case AN_ORIG:
			KeyBuff->Key2.OrigNumber = Record->OrigNumber;
			break;
	}
}
/*
*********************************************************************
* an_DataNext
* Read the NEXT record using the passed buffer to build the key,
* and the specified read method.
*
* parameters
* ----------
*	Record - a pointer to a structure (memory area) that contains
*		the record with the key fields to be used already set.
*		This memory area will be over-written on return with a
*		new value if a record was retrieved.
*	IndexValue - an integer specifying which of the indices to
*		use when building the key.
*
* return values
* -------------
*	TRUE if function has retrieved and returned a record in Record.
*********************************************************************
*/
int EXPORT an_DataNext (AN_RECORD *Record, int IndexValue)
{
	int			iTemp1;

	an_DataKey (Record, &an_DataKeyBuff, IndexValue);
	setbtv (an_DataFile);
	iTemp1 = agtbtv ((char *)Record, (char *)&an_DataKeyBuff, IndexValue);
	rstbtv();
	return (iTemp1);
}
/*
*********************************************************************
* an_DataPrev
* Read the PREVIOUS record using the passed buffer to build the key,
* and the specified read method.
*
* parameters
* ----------
*	Record - a pointer to a structure (memory area) that contains
*		the record with the key fields to be used already set.
*		This memory area will be over-written on return with a
*		new value if a record was retrieved.
*	IndexValue - an integer specifying which of the indices to
*		use when building the key.
*
* return values
* -------------
*	TRUE if function has retrieved and returned a record in Record.
*********************************************************************
*/
int EXPORT an_DataPrev (AN_RECORD *Record, int IndexValue)
{
	int			iTemp1;

	an_DataKey (Record, &an_DataKeyBuff, IndexValue);
	setbtv (an_DataFile);
	iTemp1 = altbtv ((char *)Record, (char *)&an_DataKeyBuff, IndexValue);
	rstbtv();
	return (iTemp1);
}
/*
*********************************************************************
* an_DataRead
* Read a record using the passed buffer to build the key,
* and the specified read method.
*
* parameters
* ----------
*	Record - a pointer to a structure (memory area) that contains
*		the record with the key fields to be used already set.
*		This memory area will be over-written on return with a
*		new value if a record was retrieved.
*	IndexValue - an integer specifying which of the indices to
*		use when building the key.
*
* return values
* -------------
*	TRUE if function has retrieved and returned a record in Record.
*********************************************************************
*/
int EXPORT an_DataRead (AN_RECORD *Record, int IndexValue)
{
	int		iTemp1;

	an_DataKey (Record, &an_DataKeyBuff, IndexValue);
	setbtv (an_DataFile);
	iTemp1 = acqbtv((char *)Record, (char *)&an_DataKeyBuff, IndexValue);
	rstbtv();
	return (iTemp1);
}
/*
*********************************************************************
* an_DataWrite
* Write a record using the passed buffer to build the key,
* and the specified write method.
*
* parameters
* ----------
*	Record - a pointer to a structure (memory area) that contains
*		the record with the key fields to be used already set.
*	IndexValue - an integer specifying which of the indices to
*		use when building the key.
*	WriteType - an integer specifying how the record should be
*		written -
*		1 = insert new record,
*		2 = overwrite existing record,
*		3 = don't care but write it!
*
* return values
* -------------
*	TRUE if function has written the record.
*********************************************************************
*/
int EXPORT an_DataWrite (AN_RECORD *Record, int IndexValue, int WriteType)
{
	int							RetValue;
	static AN_RECORD 			TempRecord;

	memcpy (&TempRecord, Record, sizeof (AN_RECORD));
	setmem (Record, sizeof(AN_RECORD), NULLCHAR);
	strncpy (Record->Name, TempRecord.Name, UIDSIZ - 1);
//	makhdl (Record->Name);
	Record->Number = TempRecord.Number;
	Record->OrigNumber = TempRecord.OrigNumber;
	Record->CreateDate = TempRecord.CreateDate;
	Record->DeleteDate = TempRecord.DeleteDate;

	setbtv (an_DataFile);
	RetValue = FALSE;

	switch (WriteType)
	{
		case 1:
			memcpy (&TempRecord, Record, sizeof(AN_RECORD));
			RetValue = an_DataRead (&TempRecord, IndexValue);
			if (RetValue == TRUE)
			{
				RetValue = FALSE;
				goto EndFunction;
			}
			insbtv ((char *)Record);
			RetValue = TRUE;
			goto EndFunction;
		case 2:
			memcpy (&TempRecord, Record, sizeof(AN_RECORD));
			RetValue = an_DataRead (&TempRecord, IndexValue);
			if (RetValue == TRUE)
			{
				updbtv ((char *)Record);
				RetValue = TRUE;
				goto EndFunction;
			}
			RetValue = FALSE;
			goto EndFunction;
		case 3:
			memcpy (&TempRecord, Record, sizeof(AN_RECORD));
			RetValue = an_DataRead (&TempRecord, IndexValue);
			if (RetValue == TRUE)
			{
				updbtv ((char *)Record);
				RetValue = TRUE;
				goto EndFunction;
			}
			insbtv ((char *)Record);
			RetValue = TRUE;
			goto EndFunction;
	}

EndFunction:
	rstbtv();
	return (RetValue);
}
/*
*********************************************************************
* an_AcctID
* Return the user-ID (handle/name) given a caller's account number
* Designed to be accessed from other modules.
*
* parameters:
* -----------
*	AcctNum - account number to look up
*
* returns
* -------
*	"User-ID", if found
*	"*not found*", if not found
*	"<User-ID>", if deleted
*********************************************************************
*/
char * EXPORT an_AcctID (long AcctNum)
{
	static AN_RECORD	Record;
	static char			DelName[UIDSIZ];
	int					iTemp1;

	setmem (&Record, sizeof(Record), NULLCHAR);
	Record.Number = AcctNum;
	setmem (DelName, UIDSIZ, NULLCHAR);

	// search for primary references
	iTemp1 = an_DataRead (&Record, AN_NUMBER);
	if (iTemp1 == 1)
	{
		if (Record.DeleteDate == 0)
			return (Record.Name);

		DelName[0] = '<';
		strncat (DelName, Record.Name, UIDSIZ - 2);
		iTemp1 = strlen(DelName);
		if (iTemp1 >= (UIDSIZ - 2))
			iTemp1 = UIDSIZ - 2;
		DelName[iTemp1] = '>';
		return (DelName);
	}
	return ("*not found*");
}
/*
*********************************************************************
* an_AcctNumberFile
* Find the account number for a given user-id, and optionally create
* one if it doesn't exist. Won't create a record if the account does
* not exist.
*
* parameters
* ----------
*   UserID - the name string to find the account number for
*	CreateFlag - TRUE if account-number should be created, if it does
*		NOT exist; FALSE to find existing account-number only
*
* returns
* -------
*	Account-Number > 0 if record found/created
*	0 if an error occurred
*********************************************************************
*/
long EXPORT an_AcctNumberFile (char *UserID, int CreateFlag)
{
	AN_RECORD		Record;
	int				iTemp1;

	// see if record exists in file
	setmem (&Record, sizeof(Record), NULLCHAR);
	strncpy (Record.Name, UserID, UIDSIZ - 1);

	// search for primary references
	while ( (an_DataNext (&Record, AN_NAME)) == 1)
	{
		if (stricmp (Record.Name, UserID) != 0)
			break;
		if (Record.OrigNumber != 0)
			continue;
		if (Record.DeleteDate != 0)
			continue;
		return (Record.Number);
	}

	// create record if not found
	if (CreateFlag != TRUE)
		return (0);

	// make sure account record exists
	setbtv (accbb);
	iTemp1 = acqbtv(&acctmp,UserID,0);
	rstbtv();
	if ((iTemp1 != 1) OR (acctmp.flags BITAND DELTAG))
		return (0);

	setmem (&Record, sizeof(AN_RECORD), NULLCHAR);
	setbtv (an_DataFile);
	iTemp1 = ahibtv (&Record, AN_NUMBER);
	rstbtv ();
	if (iTemp1 != 1)
		Record.Number = 1;
	else
		Record.Number++;
	setmem (Record.Name, UIDSIZ, NULLCHAR);
	strncpy (Record.Name, acctmp.userid, UIDSIZ - 1);
	Record.OrigNumber = 0;
	Record.CreateDate = today();
	Record.DeleteDate = 0;
	an_DataWrite (&Record, AN_NUMBER, 3);
	return (Record.Number);
}
