/*
*********************************************************************
* HWMAN.C
* HWM "Account Number"
*
* This source code and all associated material is proprietary and
* copyright (c) 1992, 1993 by the author, Frank LaPiana, "HighWater Mark"
*********************************************************************
* AN is used to alias an "account number" to an "account name". The
* whole purpose is to provide a reduction of redundancy and database
* access searching.
*
* A simple API allows a program to find either the acccount NUMBER or
* NAME, given one or the other.
*
* Module-level menu:
*	General Information
*		displays text file to user.
*	Browse
*		allows the user to step through records sequentially or by
*		key. MASTER has the ability to change, save, delete records.
*		Non-MASTER cannot see alias records.
*	List
*		List in either numeric or name sequence. Non-Master cannot
*		see alias records.
* API functions:
*	FindNumber - find user number from name
*	FindName - find user name from number
*	CreateNumber - create a new account number/name record
*	AliasCreate - create an alias ID number for an account
*	AliasDelete - delete an alias ID number for an account
*	AliasRename - changes the name on an alias record
*********************************************************************
*/
#include "hwmcomp.h"		// compiler directives

#define HWMAN_GLOBAL		// allocate our global variabes (defined in hwmkr.inc)
#define HWMSE_EXTERNAL		// need to access SE variables

#include "gcomm.h"			// generic MBBS proto's
#include "majorbbs.h"	 	// internal MBBS proto's and data
#include "message.h"		// SIG/form proto's and data
#include "filexfer.h"		// for listing doc files
//	#include "flash.h"

#include "hwmse.inc"		// Screen Editor
#include "hwman.inc"		// stuff for AccountNumber
#include "hwmutil.inc"		// some HWM-supplied "utility" functions

#include "hwman.h"			// MCV message numbers


/*
*********************************************************************
* AN internal variables, usually not referenced outside of this
* module
*********************************************************************
*/
static char 		an_Version[64];			// version string

// main menu selection words
static char			*an_MainInfo;			// info doc file display
static char			*an_MainBrowse;			// rename a record
static char			*an_MainList;			// list records
static char			*an_MainVer;			// show version
static char			*an_MainDocFile;		// info doc file name

// screen titles, field prompts, commands
static char			*an_BrowseTitle;		// browse SE title

static char			*an_BCmdQuit;			// "quit" command
static char			*an_BCmdGet;			// "get" command
static char			*an_BCmdNext;			// "next" command
static char			*an_BCmdPrev;			// "prev" command
static char			*an_BCmdDel;			// "del" command
static char			*an_BCmdSave;			// "save" command
static char			*an_KeyEdit;			// key required to save or del a record

static char			*an_GlobalNumber; 		// what number for this user?
static char			*an_GlobalName;			// what name for this number?

// "atomic" processing variables, used mostly for user input
#define MAX_FIELD_LEN 75
static char 						FieldInput[MAX_FIELD_LEN + 1];
static char 						CharInput;

/*
*********************************************************************
* init__AccountNumber
* Initialize for processing at BBS startup time.
*********************************************************************
*/
void EXPORT init__AccountNumber (void)
{
	unsigned int				iTemp1;

	// module description/name and number
	stzcpy (an_Module.descrp, gmdnam("HWMAN.MDF"), MNMSIZ);
	an_ModuleNumber = register_module (&an_Module);

	// open up message and btrieve data files if not already done
	an_OpenInit();

	// get any run-time options from the config/msg file
	an_MainDocFile				= stgopt (ANDOCFL);
	an_MainInfo					= stgopt (ANMINFO);
	an_MainBrowse				= stgopt (ANMBROW);
	an_MainList					= stgopt (ANMLIST);
	an_MainVer					= stgopt (ANMVER);

	an_BrowseTitle				= stgopt (ANBTIT);
	an_BCmdQuit					= stgopt (ANCQUIT);
	an_BCmdGet					= stgopt (ANCGET);
	an_BCmdNext					= stgopt (ANCNEXT);
	an_BCmdPrev					= stgopt (ANCPREV);
	an_BCmdDel					= stgopt (ANCDEL);
	an_BCmdSave					= stgopt (ANCSAVE);

	an_KeyEdit					= stgopt (ANKEDIT);

	an_GlobalNumber				= stgopt (ANGLNUM);
	an_GlobalName 				= stgopt (ANGLNAM);

	// our version information, displayed in the "general info" banner
	strcpy (an_Version, "HWMAN ");
	strcat (an_Version, __DATE__);

	// install global handler if our global commands are defined
	globalcmd (an_GlobalCmd);

	dclvda (sizeof(AN_USER));

	iTemp1 = nterms * sizeof (AN_CONTROL);
	an_Control = (AN_CONTROL *) alcmem (iTemp1);
	setmem (an_Control, iTemp1, NULLCHAR);

	rstmbk();
}
/*
*********************************************************************
* an_Exit
* Check if user wants to abort/exit current process or menu. Always
* clears any pending output to user.
*
* parameters
* ----------
*	Status - new usrptr->substt status to be assigned if exit
*		entry ("X") is made.
*	Message1 - if non-zero, the first message to be displayed to the
*		user.
*	Message2 - if non-zero, the second message to be displayed to the
*		user.
* returns
* -------
*	TRUE if current function should exit.
*********************************************************************
*/
int an_Exit (int Status, int Message1, int Message2)
{
	// preprocess a possible "X" entry.  "X" ALWAYS backs up!
	if ((margc == 1) AND (stricmp(margv[0],"X") == 0) )
	{
		cncall();								// eat remaining input
		an_substt (Status, Message1, Message2); // display any messages to user
		return(TRUE);
	}
	return (FALSE);
}
/*
*********************************************************************
* an_substt
* set the next usrptr->substt, displaying any necessary prompts
*
* parameters
* ----------
*	Status - next user substate
*	Message1 - if non-zero, print this message to the user.
*	Message2 - if non-zero, print this message to the user.
* returns
* -------
*	none
*********************************************************************
*/
void an_substt (int Status, int Message1, int Message2)
{
	// always make sure we point at the correct message file, just in case
	setmbk (an_MsgFile);

	// set new "status" for user
	usrptr->state = an_ModuleNumber;
	usrptr->substt = Status;

	// display any messages appropriate for the new status
	if (Message1 == ANMENU0)
		an_MainMenuDisplay();
	else if (Message1 != 0)
		prfmsg (Message1);

	if (Message2 == ANMENU0)
		an_MainMenuDisplay();
	else if (Message2 != 0)
		prfmsg (Message2);

	// output message buffer to user, and clear it for next use
	outprf (usrnum);
	clrprf();
	rstmbk();
}
/*
*********************************************************************
* an_MainMenuDisplay
* display the appropriate main menu depending on user.
*********************************************************************
*/
void an_MainMenuDisplay (void)
{
	setmbk (an_MsgFile);
	prfmsg (ANMENU00, an_MainInfo, an_MainBrowse, an_MainList, an_MainVer);
	rstmbk();
}
/*
*********************************************************************
* an_ListFileDone
* set the next usrptr->substt, displaying any necessary prompts
*
* parameters
* ----------
*	Status - file displayed to completion (not used)
*********************************************************************
*/
void an_ListFileDone (int Status)
{
	an_UserSet();
	an_substt (ANMENU1, ANMENU0, 0);
}
/*
*********************************************************************
* an_UserSet
* set the user to point at the correct element.
*********************************************************************
*/
void an_UserSet (void)
{
	an_UserPtr = (AN_USER *) vdaptr;
	an_UserRecord = &an_UserPtr->Record;
	return;
}
/*
*********************************************************************
* an_UserClear
* Clear the user variables to "null" and empty.
*********************************************************************
*/
void an_UserClear (void)
{
	// clear all structures for this user
	an_UserSet();
	setmem (an_UserPtr, sizeof (AN_USER), NULLCHAR);
}
/*
*********************************************************************
* an_LogOn
* supplemental user logon routine
*********************************************************************
*/
int EXPORT an_LogOn (void)
{
	AN_RECORD		*RecPtr;
	AN_RECORD		TempRecord;
	int				iTemp1;

	RecPtr = &an_Control[usrnum].Record;
	setmem (RecPtr, sizeof (AN_RECORD), NULLCHAR);
	strncpy (RecPtr->Name, usaptr->userid, UIDSIZ - 1);

	while (an_DataNext (RecPtr, AN_NAME) == 1)
	{
		if (stricmp (RecPtr->Name, usaptr->userid) != 0)
			break;
		if (RecPtr->DeleteDate != 0)
			continue;
		return (0);
	}

	// record not found; create one for this user
	setmem (RecPtr, sizeof (AN_RECORD), NULLCHAR);
	strncpy (RecPtr->Name, usaptr->userid, UIDSIZ - 1);
	RecPtr->OrigNumber = 0;
	RecPtr->CreateDate = today();
	RecPtr->DeleteDate = 0;
	setmem (&TempRecord, sizeof(AN_RECORD), NULLCHAR);
	setbtv (an_DataFile);
	iTemp1 = ahibtv (&TempRecord, AN_NUMBER);
	rstbtv ();
	if (iTemp1 != 1)
		RecPtr->Number = 1;
	else
		RecPtr->Number = TempRecord.Number + 1;
	an_DataWrite (RecPtr, AN_NUMBER, 3);
	return (0);
}
/*
*********************************************************************
* an_Exec
* Handle any input from the user, if necessary passing it off to any
* lower level functions for processing.
* This function is called automatically by the MAJORBBS handling
* routines.
*
* parameters:
*	NONE
*
* globals:
* --------
*	uses global input variables:
*		margc
*		margv
*
* return(s)
* ---------
*	1 if input was handled;
*	0 to use DEFAULT input handler
*********************************************************************
*/
int EXPORT an_Exec (void)
{
	static int		RepromptMenu;
	static int		iTemp1;

	// suggest G'comm fix
	if (usrptr->class != ACTUSR)
		return (0);

	RepromptMenu = FALSE;		// don't need to redisplay menu
	an_UserSet();				// set up for current user

	// process any/all input waiting in the buffer
	do
	{
		bgncnc();	// begin "input crunching"

		// process according to user's current "state"
		switch (usrptr->substt)
		{
			// first time entry to this module
			case 0:
				an_UserClear();			// clear out info for this user
				cncchr();				// make sure we eat main-menu selection character
				if ( !morcnc() )		// no "type ahead", so...
										// display the main menu for the module
					RepromptMenu = TRUE;
				else					// more input
					an_substt (ANMENU1,0,0);
										// we'll be at "long menu" waiting for it
				break;

			// re-display long menu and wait for more input...
			case ANMENU0:
				RepromptMenu = TRUE;
				an_substt (ANMENU1,0,0);
				break;

			// long menu has been displayed, awaiting input...
			case ANMENU1:
				// check for exit from module
				if ( an_Exit(ANEXIT,ANEXIT,0) == TRUE)
					return (0);

				// get menu option
				iTemp1 = CncWord (FieldInput, 20);
				if (iTemp1 < 1)
				{
					// if nothing entered, redisplay the menu and wait again
					RepromptMenu = TRUE;
					break;
				}
				strupr (FieldInput);

				// display information
				if ((stricmp (FieldInput, an_MainInfo) == 0) OR ((iTemp1 == 1) AND (FieldInput[0] == *an_MainInfo)) )
				{
					clrprf();
					listing (an_MainDocFile, an_ListFileDone);
					return (1);
				}

				// browse responses
				if ((stricmp (FieldInput, an_MainBrowse) == 0) OR ((iTemp1 == 1) AND (FieldInput[0] == *an_MainBrowse)) )
				{
					an_BrowseStart();
					return (1);
				}

				// listing out of records
				if ((stricmp (FieldInput, an_MainList) == 0) OR ((iTemp1 == 1) AND (FieldInput[0] == *an_MainList)) )
				{
					an_UserClear();		// clear any junk from user vars
					an_substt (ANLSEQ,ANLSEQ,0);
					goto OptionDone;
				}

				// display version information
				if ((stricmp (FieldInput, an_MainVer) == 0) OR ((iTemp1 == 1) AND (FieldInput[0] == *an_MainVer)) )
				{
					clrprf();
					prf ("\n%s\n", an_Version);
					outprf(usrnum);
					RepromptMenu = TRUE;
					goto OptionDone;
				}

				// end of menu command handler
				// bad input from user
				setmbk (an_MsgFile);
				prfmsg (ANMENUB, FieldInput);
				rstmbk ();
				outprf (usrnum);
				clrprf();
				RepromptMenu = TRUE;
				cncall();	// eat any remaining input
				break;

			// listing of responses
			case ANLSEQ:		an_ListSequence();		break;
			case ANLNAME:
			case ANLNUM:
			case ANLONUM:		an_ListStart();		break;

			// listing responses
			case ANLINE:
				RepromptMenu = TRUE;
				break;

			// bad sub-state code for this user!
			default:
				// eat all remaining input, someone might
				// have press a key while a "cycling" process was
				// occuring.
				cncall();
				break;
		}

		// re-display the menu if necessary
OptionDone:
		if (RepromptMenu == TRUE)
		{
			an_substt (ANMENU1, ANMENU0, 0);
			clrprf ();
			*prfptr = NULLCHAR;
			cncall();
		}

	// keep cycling as long as there is more input buffered
	} while ( !endcnc() );

	cncall();					// eat all remaining input
	clrprf();
	return (1);					// we've handled input
}
/*
*********************************************************************
* an_Status
* Handle any "cycling" as necessary from the user.
*********************************************************************
*/
void EXPORT an_Status (void)
{
	static int			iTemp1;

	// only process "CYCLE" requests
	if ((status != CYCLE) OR (usrptr->class != ACTUSR))
	{
		// let the default status handler decide what to do, we don't!
		dfsthn();
		return;
	}

	// check if we have enuff room to do anything in the output buffer
	iTemp1 = btuoba(usrnum);
	if (iTemp1 < 1000)				// less then 1000 bytes (approx 1/2 page) available
	{
		btuinj(usrnum,CYCLE);		// "cycle" again, waiting...
		return;
	}

	// make sure we are pointing to the correct stuff
	an_UserSet();

	// handle the cycling depending on where we are...
	switch (usrptr->substt)
	{
		// re-prompt with menu, done with cycling
		case 0:
			an_substt (ANMENU1, ANMENU0, 0);
			break;

		// listing responses
		case ANLINE:
			an_ListRecord();
			break;

		// bad sub-state code for this user!
		default:
//			clrprf();
//			setmbk (an_MsgFile);
//			prfmsg (ANSTTERR, usrptr->substt);
//			rstmbk ();
//			outprf (usrnum);
//			an_substt (ANEXIT, ANEXIT, 0);
			an_substt (ANMENU1, ANMENU0, 0);
			break;
	}

	// clear anything that we may have sent to the user
	clrprf();						// reset the prf buffer pointer
	return;
}
/*
*********************************************************************
* an_LogOff
* supplemental user hangup routine
*********************************************************************
*/
int EXPORT an_LogOff (void)
{
	return (0);
}
/*
*********************************************************************
* an_HangUp
* supplemental user hangup routine
*********************************************************************
*/
void EXPORT an_HangUp (void)
{
	an_UserClear();
	return;
}
/*
*********************************************************************
* an_AcctDel
* Account deletion vector, called by module interface.
* Records are not actually deleted; the "delete date" is set however.
*
* parameters
* ----------
*	UserName - user ID of person to be deleted.
*
* returns
* -------
*	none
*********************************************************************
*/
void an_AcctDel (char *UserName)
{
	AN_RECORD		Record;
	AN_RECORD		Alias;

	setmem (&Record, sizeof(Record), NULLCHAR);
	strcpy (Record.Name, UserName);

	// search for primary references
	while ( (an_DataNext(&Record, AN_NAME) ) == TRUE)
	{
		if (stricmp(Record.Name, UserName) != 0)
			break;
		if (Record.DeleteDate != 0)
			continue;

		Record.DeleteDate = today();
		an_DataWrite (&Record, AN_NAME, 3);

		// search for an alias records for this user
		setmem (&Alias, sizeof(Alias), NULLCHAR);
		Alias.OrigNumber = Record.Number;
		while ( (an_DataNext (&Alias, AN_ORIG)) == TRUE)
		{
			if (Alias.OrigNumber != Record.Number)
				break;
			Alias.DeleteDate = today();
			an_DataWrite (&Alias, AN_NAME, 3);
		}
	}
	return;
}
/*
*********************************************************************
* an_ShutDown.c
* shutdown the module, done with processing...
*********************************************************************
*/
void EXPORT an_ShutDown(void)
{
	if (an_DataFile != NULLPTR)
		clsbtv (an_DataFile);
	an_DataFile = NULLPTR;

	// close the message file
	if (an_MsgFile != NULLPTR)
		clsmsg (an_MsgFile);
	an_MsgFile = NULLPTR;

	return;
}
/*
*********************************************************************
* an_GlobalCmd
* global command handler
*
* parameters
* ----------
*	none
*
* global:
* -------
*	global input variables are evaluated for processing
*
* returns
* -------
*	1 if input was a global command handled by this function
*********************************************************************
*/
int EXPORT an_GlobalCmd (void)
{
	static int 				iTemp1;
	static int				iTemp2;
	static int				RetValue;
	static AN_RECORD		Record;


	// return zero if global command was not handled
	RetValue = 0;
	/*
	*******************************************************
	* who is this user by number?
	* form is /cmd number
	*******************************************************
	*/
	// check all running HWM products and current versions at one shot
	if ((margc >= 1) AND (stricmp(margv[0],"-hwmver") == 0))
	{
		prf ("%s\n", an_Version);
		outprf (usrnum);
		clrprf ();
		goto EndFunction;
	}

	iTemp1 = 0;
	if ((margc >= 1) AND (stricmp(margv[0], an_GlobalNumber) == 0))
		iTemp1 = 1;
	if ((margc >= 1) AND (stricmp(margv[0], an_GlobalName) == 0))
		iTemp1 = 2;
	if (iTemp1 == 0)
		goto EndFunction;

	// check to make sure user has access to this
	if ((usrptr->flags BITAND MASTER) OR (hasmkey(ANGLKEY)))
		;
	else
		goto EndFunction;
	RetValue = 1;				// global command handled

	if (margc < 2)				// wrong number of arguements
	{
		// print help message
		setmbk (an_MsgFile);
		prfmsg (ANGLHELP);
		rstmbk ();
		outprf (usrnum);
		clrprf ();
		goto EndFunction;
	}

	setmem (&Record, sizeof(Record), NULLCHAR);
	if (iTemp1 == 1)
	{
		Record.Number = atol(margv[1]);
		iTemp2 = an_DataRead (&Record, AN_NUMBER);
		if (iTemp2 != 1)
		{
			// error reading record, print message and abort
			setmbk (an_MsgFile);
			prfmsg (ANGLERR);
			rstmbk ();
			outprf (usrnum);
			clrprf ();
			goto EndFunction;
		}
	}
	if (iTemp1 == 2)
	{
		rstrin();
		strncpy (Record.Name, margv[1], UIDSIZ - 1);
		iTemp2 = 1;
		while (an_DataNext (&Record, AN_NAME) == 1)
		{
			if (stricmp (Record.Name, margv[1]) != 0)
				break;
			if (Record.DeleteDate != 0)
				continue;
			iTemp2 = 0;
			break;
		}
		if (iTemp2 == 1)
		{
			// error reading record, print message and abort
			setmbk (an_MsgFile);
			prfmsg (ANGLERR);
			rstmbk ();
			outprf (usrnum);
			clrprf ();
			goto EndFunction;
		}
	}
	setmbk (an_MsgFile);
	prfmsg (ANGLDIS, Record.Name, l2as(Record.Number), l2as(Record.OrigNumber), ncdate(Record.CreateDate), ncdate(Record.DeleteDate));
	rstmbk ();
	outprf (usrnum);
	clrprf ();
	goto EndFunction;

EndFunction:
	return (RetValue);
}
/*
*********************************************************************
* an_ListSequence
* Prompt for starting values, store it in global user-area, prepare
* for listing
*********************************************************************
*/
void an_ListSequence (void)
{
	if (an_Exit (ANMENU1,ANMENU0,0) == TRUE)
		return;

	// enter in the listing "sequence"
	CharInput = cncchr();
	CharInput = toupper(CharInput);
	switch (CharInput)
	{
		// user name
		case '1':
			an_UserPtr->ListType = '1';
			an_substt (ANLNAME, ANLNAME, 0);
			break;
		// numeric
		case '2':
			an_UserPtr->ListType = '2';
			an_substt (ANLNAME, ANLNUM, 0);
			break;
		// by alias number
		case '3':
			if ((usrptr->flags BITAND MASTER) OR (hasmkey(ANGLKEY)))
			{
				an_UserPtr->ListType = '3';
				an_substt (ANLNAME, ANLONUM, 0);
				break;
			}

		// bad value, or wants help
		case '?':
		default:
			an_substt (ANLSEQ, ANLSEQH, ANLSEQ);
			break;
	}
	return;
}
/*
*********************************************************************
* an_ListStart
* Prompt for starting listing value
*
* global
* ------
*	an_UserPtr->Edit - set to the key-access to be used
*	an_UserRecord - starting key values are set up
*	an_UserPtr->Count1 - used as "number of reponses" printed - set to 0
*********************************************************************
*/
void an_ListStart (void)
{
	if (an_Exit (ANMENU1,ANMENU0,0) == TRUE)
		return;

	// get any values into our temporary input area
	setmem (FieldInput, MAX_FIELD_LEN + 1, NULLCHAR);
	strncpy (FieldInput, cncall(), MAX_FIELD_LEN);
	setmem (an_UserRecord, sizeof(AN_RECORD), NULLCHAR);

	switch (an_UserPtr->ListType)
	{
		// they entered the user's name
		case '1':
			strncpy (an_UserRecord->Name, FieldInput, UIDSIZ - 1);
			break;
		// list by account number
		case '2':
			if ( (stricmp(FieldInput, ".") == 0) OR (strlen(FieldInput) < 1))
				an_UserRecord->Number = 0;
			else
				an_UserRecord->Number = atoi(FieldInput);
			break;
		// list by original number
		case '3':
			if ( (stricmp(FieldInput, ".") == 0) OR (strlen(FieldInput) < 1))
				an_UserRecord->OrigNumber = 0;
			else
				an_UserRecord->OrigNumber = atoi(FieldInput);
			break;
		// bad listing type!
		default:
			an_substt (ANLSEQ, ANLSEQH, ANLSEQ);
			return;
	}
	// set the current output record number count
	an_UserPtr->Count = 0;

	// print the heading
	clrprf();
	setmbk (an_MsgFile);
	prfmsg (ANLHEAD);
	rstmbk ();
	outprf (usrnum);
	clrprf();

	// set new state and start printing
	an_substt (ANLINE, 0, 0);
	btuinj(usrnum,CYCLE);		// "cycle" again, waiting...
	return;
}
/*
*********************************************************************
* an_ListRecord
* continue printing records to the user
*********************************************************************
*/
void an_ListRecord (void)
{
	int				IndexValue;
	int				iTemp1;
	char			Date1[10];
	char			Date2[10];

	switch (an_UserPtr->ListType)
	{
		case '1':
			IndexValue = AN_NAME;
			break;
		// read by number
		case '2':
			IndexValue = AN_NUMBER;
			break;
		// read by original number
		case '3':
			IndexValue = AN_ORIG;
			break;
	}

	// retrieve the record
	iTemp1 = an_DataNext (an_UserRecord, IndexValue);
	if (iTemp1 != 1)
	{
		an_substt(ANMENU1, ANLEND, ANMENU0);
		return;
	}

	// display the record
	setmbk (an_MsgFile);
	strcpy (Date1, ncdate(an_UserRecord->CreateDate));
	strcpy (Date2, ncdate(an_UserRecord->DeleteDate));
	prfmsg (ANLINE, an_UserRecord->Name, l2as(an_UserRecord->Number), l2as(an_UserRecord->OrigNumber), Date1, Date2);
	rstmbk ();
	outprf (usrnum);
	clrprf ();
	an_substt (ANLINE, 0, 0);
	btuinj(usrnum,CYCLE);
	return;
}
/*
*********************************************************************
* an_BrowseFieldAlloc
* Allocate the basic fields for browse/enter response
*
* parameters
* ----------
*	ScreenTitle - pointer to a STATIC string containing the "title"
*		for this form.
*
* returns
* -------
*	TRUE if fields allocated without error.
*	FALSE fields not allocated, new status set to menu already
*********************************************************************
*/
int an_BrowseFieldAlloc (void)
{
	int					iTemp1;
	SE_FIELD_STRUCT		*FieldArray;
	SE_FIELD_STRUCT		*FieldPtr;
	SE_SCREEN_STRUCT	Screen;

	/*
	*****************************************************************
	* set up screen-field information for use with SE
	*****************************************************************
	*/
	// find out how many fields to process
	iTemp1 =
		+ 1									// retrievel index
		+ 1									// user name
		+ 1									// account number
		+ 1									// original number
		+ 1									// create date
		+ 1									// delete date
		+ 1;								// NULL terminating entry

	iTemp1 = iTemp1 * sizeof(SE_FIELD_STRUCT);
	FieldArray = (SE_FIELD_STRUCT *) getml (iTemp1);
	if (FieldArray == NULLPTR)
	{
		an_substt (ANBSEMR, ANMENU1, ANMENU0);
		return (FALSE);
	}
	setmem (FieldArray, iTemp1, NULLCHAR);
	FieldPtr = FieldArray;

	// index name to use in get/next/prev
	#define TAG_INDEX 1
	FieldPtr->Tag = TAG_INDEX;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_INT;
	FieldPtr->DataStore = &an_UserPtr->Count;
	FieldPtr->DisLen = 20;
	FieldPtr->PromptMsg = ANBIX;
	FieldPtr->HelpLineMsg = ANBIXH;
	FieldPtr->HelpScreenMsg = ANBIXH;
	FieldPtr->Translate = an_BrowseIndexIntercept;
	an_UserPtr->Count = AN_NAME;

	// user name
	FieldPtr++;
	#define TAG_NAME (TAG_INDEX + 1)
	FieldPtr->Tag = TAG_NAME;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_STRING;
	FieldPtr->DataStore = an_UserRecord->Name;
	FieldPtr->DisLen = UIDSIZ - 1;
	FieldPtr->PromptMsg = ANBNAM;
	FieldPtr->HelpLineMsg = ANBNAMH;
	FieldPtr->HelpScreenMsg = ANBNAMH;

	// account number
	FieldPtr++;
	#define TAG_NUMBER (TAG_NAME + 1)
	FieldPtr->Tag = TAG_NUMBER;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_LONG;
	FieldPtr->DataStore = &an_UserRecord->Number;
	FieldPtr->DisLen = 8;
	FieldPtr->PromptMsg = ANBNUM;
	FieldPtr->HelpLineMsg = ANBNUMH;
	FieldPtr->HelpScreenMsg = ANBNUM;

	// original account number
	FieldPtr++;
	#define TAG_ORIG (TAG_NUMBER + 1)
	FieldPtr->Tag = TAG_ORIG;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_LONG;
	FieldPtr->DataStore = &an_UserRecord->OrigNumber;
	FieldPtr->DisLen = 8;
	FieldPtr->PromptMsg = ANBONUM;
	FieldPtr->HelpLineMsg = ANBONUMH;
	FieldPtr->HelpScreenMsg = ANBONUMH;
	FieldPtr->BlankType = 'Y';

	// assign date-created
	FieldPtr++;
	#define TAG_CREATE (TAG_ORIG + 1)
	FieldPtr->Tag = TAG_CREATE;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_DATE;
	FieldPtr->DataStore = &an_UserRecord->CreateDate;
	FieldPtr->DisLen = 8;
	FieldPtr->PromptMsg = ANBCD;
	FieldPtr->HelpLineMsg = ANBCDH;
	FieldPtr->HelpScreenMsg = ANBCDH;

	// assign date-deleted
	FieldPtr++;
	#define TAG_DELETE (TAG_CREATE + 1)
	FieldPtr->Tag = TAG_DELETE;
	FieldPtr->Status = SE_FIELD_ON;
	FieldPtr->Type = SE_FT_DATE;
	FieldPtr->DataStore = &an_UserRecord->DeleteDate;
	FieldPtr->DisLen = 8;
	FieldPtr->PromptMsg = ANBDD;
	FieldPtr->HelpLineMsg = ANBDDH;
	FieldPtr->HelpScreenMsg = ANBDDH;
	FieldPtr->BlankType = 'Y';

	// make sure to mark the end of the list
	FieldPtr++;
	FieldPtr->Tag = (-1);

	/*
	*****************************************************************
	* set up screen-control information for use with SE
	*****************************************************************
	*/
	setmem (&Screen, sizeof(SE_SCREEN_STRUCT), NULLCHAR);
	Screen.DisOnly = FALSE;
	Screen.CmdFunction = an_BrowseCommand;
	Screen.MsgFile = an_MsgFile;
	Screen.FieldList = FieldArray;
	Screen.Title = an_BrowseTitle;

	iTemp1 = se_InitScreen (&Screen, NULLPTR);
	free (FieldArray);
	if (iTemp1 == FALSE)
	{
		an_substt (ANMENU1, ANMENU0, 0);
		return (FALSE);
	}
	return (TRUE);
}
/*
*********************************************************************
* an_BrowseStart
* set-up for BROWSE mode
*********************************************************************
*/
void an_BrowseStart (void)
{
	int					iTemp1;
	SE_USER_STRUCT		*NewScreen;

	iTemp1 = an_BrowseFieldAlloc ();
	if (iTemp1 == FALSE)
		return;

	setmem (an_UserRecord, sizeof(AN_RECORD), NULLCHAR);
	an_DataNext (an_UserRecord, AN_NAME);

	/*
	*****************************************************************
	* set up screen-control information for use with SE
	*****************************************************************
	*/
	NewScreen = se_UserAddress(usrnum);
	NewScreen->Screen->CmdFunction = an_BrowseCommand;
	an_UserPtr->Count = AN_NAME;			// index to use!

	se_CmdOnOff (an_BCmdGet, TRUE);
	se_CmdOnOff (an_BCmdNext, TRUE);
	se_CmdOnOff (an_BCmdPrev, TRUE);
	se_CmdOnOff (an_BCmdQuit, TRUE);
	if (haskey(an_KeyEdit))
	{
		se_CmdOnOff (an_BCmdDel, TRUE);
		se_CmdOnOff (an_BCmdSave, TRUE);
	}

	iTemp1 = se_StartScreen (SE_MODE_CMD, (-1), (-1) );
	if (iTemp1 == FALSE)
	{
		an_substt (ANMENU1, ANMENU0, 0);
		return;
	}
	return;
}
/*
*********************************************************************
* an_BrowseCommand
* Sub-menu used when someone is browsing responses.
* Handle any input from the user, if necessary passing it off to any
* lower level functions for processing.
*
* parameters:
*	NONE
*
* globals:
* --------
*	uses global input variables:
*		margc
*		margv
*
* return(s)
* ---------
*	none
*********************************************************************
*/
int an_BrowseCommand (char *CmdName)
{
	int					iTemp1;

	an_UserSet();

	if (stricmp(CmdName, an_BCmdQuit) == 0)
	{
		se_DeleteScreen ();
		an_substt (ANMENU1, ANMENU0, 0);
		return (0);
	}

	if (stricmp (CmdName, an_BCmdGet) == 0)
	{
		// check to see whether this person already has a response
		iTemp1 = an_DataRead (an_UserRecord, an_UserPtr->Count);
		// record was found
		if (iTemp1 == TRUE)
			return (3);

		// no record found
		clrprf();
		setmbk (an_MsgFile);
		prfmsg (ANEGET, an_BCmdGet);
		rstmbk ();
		outprf(usrnum);
		return (1);
	}

	if (stricmp (CmdName, an_BCmdNext) == 0)
	{
		iTemp1 = an_DataNext (an_UserRecord, an_UserPtr->Count);
		// record was found
		if (iTemp1 == TRUE)
			return (3);
		// no record found
		clrprf();
		setmbk (an_MsgFile);
		prfmsg (ANEGET, an_BCmdNext);
		rstmbk ();
		outprf(usrnum);
		return (1);
	}

	if (stricmp (CmdName, an_BCmdPrev) == 0)
	{
		iTemp1 = an_DataPrev (an_UserRecord, an_UserPtr->Count);
		// record was found
		if (iTemp1 == TRUE)
			return (3);
		// no record found
		clrprf();
		setmbk (an_MsgFile);
		prfmsg (ANEGET, an_BCmdPrev);
		rstmbk ();
		outprf(usrnum);
		return (1);
	}

	if (stricmp (CmdName, an_BCmdDel) == 0)
	{
		if (haskey (an_KeyEdit))
		{
			// check to see whether this person already has a response
			iTemp1 = an_DataRead (an_UserRecord, AN_NAME);
			if (iTemp1 == TRUE)
			{
				setbtv (an_DataFile);
				delbtv ();
				rstbtv();

				setmbk (an_MsgFile);
				prfmsg (ANEDEL, an_BCmdDel);
				rstmbk ();
				outprf (usrnum);
				clrprf();
				return (1);
			}
		}
		setmbk (an_MsgFile);
		prfmsg (ANEGET, an_BCmdDel);
		rstmbk ();
		outprf(usrnum);
		clrprf();
		return (1);
	}

	if (stricmp(CmdName, an_BCmdSave) == 0)
	{
		if (haskey (an_KeyEdit))
		{
			iTemp1 = an_DataWrite (an_UserRecord, AN_NUMBER, 3);
			if (iTemp1 != TRUE)
			{
				clrprf();
				setmbk (an_MsgFile);
				prfmsg (ANSAVERR, "writing your response!");
				rstmbk ();
				outprf(usrnum);
				return (1);
			}
		}
		return (3);
	}
	return (4);
}
/*
*********************************************************************
* an_BrowseIndexIntercept
* toggle the caller's INDEX method for reading the response file
* This function is called only by SE.
*
* parameters
* ----------
*	Tag - field tag number
*	KeyValue - specifies key that was pressed, or special processing
*	UserStructPtr - pointer to SE USER structure for this person
*
* returns
* -------
*	KeyValue = 0 (convert field for output)
*		TRUE if converted without error
*	KeyValue = (-1) (convert field to internal format based on entire input buffer)
*		TRUE if converted without error,
*		FALSE bad value was entered
*	KeyValue - anything else - FS in use, use "toggle" method
*
* start up the SE for registry maintenance/display/inquire
*********************************************************************
*/
int an_BrowseIndexIntercept (int Tag, int KeyValue)
{
	SE_USER_STRUCT		*UserStructPtr;

	if (Tag != TAG_INDEX)
		return (FALSE);

	UserStructPtr = se_UserAddress(usrnum);
	if (UserStructPtr == NULLPTR)
    	return (FALSE);

	an_UserSet();

	if (KeyValue == (-1))
	{
		if (stricmp (UserStructPtr->Buffer,"name") == 0)
			an_UserPtr->Count = AN_NAME;
		else if (stricmp (UserStructPtr->Buffer,"number") == 0)
			an_UserPtr->Count = AN_NUMBER;
		else if (stricmp (UserStructPtr->Buffer,"original") == 0)
			an_UserPtr->Count = AN_ORIG;
		else
			an_UserPtr->Count = AN_NAME;
	}
	if (KeyValue > 0)
	{
		switch (an_UserPtr->Count)
		{
			case AN_NAME:
				an_UserPtr->Count = AN_NUMBER;
				break;
			case AN_NUMBER:
				an_UserPtr->Count = AN_ORIG;
				break;
			case AN_ORIG:
				an_UserPtr->Count = AN_NAME;
				break;
			default:
				strcpy (UserStructPtr->Buffer,"NAME");
				an_UserPtr->Count = AN_NAME;
				return (FALSE);
		}
	}

	switch (an_UserPtr->Count)
	{
		case AN_NAME:
			strcpy (UserStructPtr->Buffer,"NAME");
			break;
		case AN_NUMBER:
			strcpy (UserStructPtr->Buffer,"NUMBER");
			break;
		case AN_ORIG:
			strcpy (UserStructPtr->Buffer,"ORIGINAL");
			break;
		default:
			strcpy (UserStructPtr->Buffer,"NAME");
			an_UserPtr->Count = AN_NAME;
			break;
	}
	return (TRUE);
}
/*
*********************************************************************
* an_CurrentAcctNum
* Return the account number for the current online user.
* Designed to be accessed from other modules.
*
* parameters:
* -----------
*	none
*
* returns
* -------
*	account-number for the user,
*	0 if the user doesn't have one or invalid current user
*********************************************************************
*/
long EXPORT an_CurrentAcctNum (void)
{
	AN_RECORD 		*RPtr;

	if ((usrnum < 0) OR (usrnum >= nterms))
		return (0);
	switch (usrptr->class)
	{
		case SUPLON:
		case SUPLOF:
		case ACTUSR:
			break;
		default:
			return (0);
	}

	// should correspond to the information in the control array
	RPtr = &an_Control[usrnum].Record;
	if (strlen(RPtr->Name) == 0)
		return (0);

	// shouldn't ever happen, but... check anyway
	if ( stricmp (usaptr->userid, RPtr->Name) != 0)
		return (0);

	return (RPtr->Number);
}
/*
*********************************************************************
* an_AcctNumber
* Find the account number for a given user-id, and optionally create
* one if it doesn't exist. Won't create a record if the account does
* not exist.
*
* parameters
* ----------
*   UserID - the name string to find the account number for
*	CreateFlag - TRUE if account-number should be created, if it does
*		NOT exist; FALSE to find existing account-number only
*
* returns
* -------
*	Account-Number > 0 if record found/created
*	0 if an error occurred
*********************************************************************
*/
long EXPORT an_AcctNumber (char *UserID, int CreateFlag)
{
	int				iTemp1;

	// check for an online account, first
	for (iTemp1 = 0; iTemp1 < nterms; iTemp1++)
	{
		switch (user[iTemp1].class)
		{
			case ACTUSR:
			case SUPLON:
			case SUPLOF:
				break;
			default:
				continue;
		}
		if (stricmp(UserID, an_Control[iTemp1].Record.Name) == 0)
			return (an_Control[iTemp1].Record.Number);
	}

    // check our database, and create the record if necessary
	return ( an_AcctNumberFile(UserID, CreateFlag) );
}

#include "hwmanl.c"
