/***************************************************************************/
/*   FSPACE.C  - Flash SpaceWars                     12/10/91 - Les Bird   */
/*                                                          & Ron Struempf */
/*             I generally wrote this code to test out the keyboard        */
/*             "make" and "break" control but eventually turned it         */
/*             into a full-fledged space war game.          - Les          */
/*                                                                         */
/*   Thanks to Ron Struempf for his help with some of the formulas.        */
/***************************************************************************/
/*                                                                         */
/*  Copyright (C) 1991 GALACTICOMM, Inc.                                   */
/*                                                                         */
/*  This source is for use by Galacticomm "FLASH" Protocol Licensees ONLY. */
/*  Any use of the contents of this file, or any functional derivative or  */
/*  part or portion thereof, outside of the provisions of the Galacticomm  */
/*  "FLASH" Protocol Developers' License Agreement is strictly prohibited. */
/*                                                                         */
/***************************************************************************/

#include "stdio.h"
#include "dos.h"
#include "fkcode.h"
#include "fscomm.h"
#include "fssnds.h"
#include "portable.h"
#include "costab.h"
#include "sintab.h"
#include "tantab.h"

#define MINSX       0         /* minimum screen X coordinate               */
#define MAXSX       640       /* maximum screen X coordinate               */
#define MINSY       0         /* minimum screen Y coordinate               */
#define MAXSY       350       /* maximum screen Y coordinate               */

#define scale(n,s)  (n >= 0 ? ((n+(s/2))/s) : ((n-(s/2))/s))

#define SCALER      10        /* 1/10th's scale value (ie. 34=3.4)         */
#define GCONST      100       /* gravity scaler (no larger than 327!)      */
#define MAXACC      900       /* maximum acceleration due to gravity       */
#define ACCBND      600       /* accel boundary value for velocity control */

#define MINBX       10*SCALER /* minimum border X value                    */
#define MAXBX       630*SCALER /* maximum border X value                   */
#define MINBY       20*SCALER /* minimum border Y value                    */
#define MAXBY       330*SCALER /* maximum border Y value                   */

#define MAXTORS     60        /* maximum torpedoes to update               */

#define NUMFIELD    3         /* number of star fields to update           */
#define MAXSTARS    20        /* maximum stars per star field              */

#define NSHIPS      3         /* number of ships defined                   */
#define SHPSIZ      80        /* ship size for collision detection         */

#define EXPLBITS    32        /* # of pixels of animated explosion         */
#define HYPEXPLO    30000     /* rnd() must be >= this for bad hyperspace  */
                              /* GAME-TYPE flags                           */
#define FRICT       1         /*   friction computed on ships              */
#define SUN         2         /*   sun gravitational pull computed on ships*/
#define NOPARM      4         /*   no parameters                           */
                              /* SHIP-TYPE flags                           */
#define SHIELDS     1         /*   shields up for this ship                */
#define CLOAK       2         /*   cloaking device activated               */
#define CLOAKC      8         /*   cloaked color                           */
                              /* WEAPONS CONTROL                           */
#define FIRETOR     1         /*   fire a torpedo                          */
#define FIREPHA     2         /*   fire a phaser                           */
                              /* JOYSTICK parameters                       */
#define JOYBUT1     16        /*   joystick #1 button #1 depressed         */
#define JOYBUT2     32        /*   joystick #1 button #2 depressed         */

#ifdef DEBUG
FILE    *dbgfp=NULL;
#endif

extern
int  joyb,                    /* joystick button status (jstick.asm)       */
     joyx,                    /* joystick X position (jstick.asm)          */
     joyy,                    /* joystick Y position (jstick.asm)          */
     onscrn,
     self;

extern
long rndnum;

union REGS regs;
struct SREGS sregs;

void interrupt (*oldkeyi)();
void interrupt newkeyi(void);

int  *bholec,                 /* color for current blackhole               */
     bholen,                  /* current leg of blackhole being drawn      */
     gentck,                  /* general purpose ticker variable           */
     gmoc,                    /* "game over" message color                 */
     gtimr[2],                /* game timers                               */
     gtmrc,                   /* game timer color                          */
     gtmrc1,                  /* less than 10 second timer "flash" color   */
     gtype,                   /* game-type in progress                     */
     joyflg,                  /* joystick flag (0=off, 1=on)               */
     joymaxx,                 /* joystick maximum X threshold              */
     joymaxy,                 /* joystick maximum Y threshold              */
     joyminx,                 /* joystick minimum X threshold              */
     joyminy,                 /* joystick minimum Y threshold              */
     lv=0,                    /* last v from newkeyi() - prevent repeat    */
     omode,                   /* old video mode (use for restoring later)  */
     starfn,                  /* star field number being updated           */
     starx[NUMFIELD][MAXSTARS], /* star field X coordinates                */
     stary[NUMFIELD][MAXSTARS], /* star field Y coordinates                */
     starc[NUMFIELD][MAXSTARS]; /* star colors for each star in each field */

int shipobj[][NPTS]={
     /*   object definition for the Warbird                      */
      -4, -4,  0,  5,  0,  5,  4, -4,  4, -4,  1, -2,  1, -2, -1, -2,
      -1, -2, -4, -4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,

     /*   Object definition for the Heavy Cruiser                */
      -5, -5, -5,  1,  5, -5,  5,  1, -5, -1, -1, -1,  5, -1,  1, -1,
      -1, -3, -1,  1,  1, -3,  1,  1, -1,  1, -3,  1,  1,  1,  3,  1,
      -3,  1, -3,  3,  3,  1,  3,  3, -3,  1, -1,  7,  3,  1,  1,  7,
      -1,  7,  1,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,

     /*   Object definition for the Battle Cruiser               */
      -5, -5, -5,  3,  5, -5,  5,  3, -5, -5, -1, -1,  5, -5,  1, -1,
      -1, -1,  1, -1, -5,  1,  5,  1, -1,  1, -1,  4,  1,  1,  1,  4,
      -1,  4, -2,  5,  1,  4,  2,  5, -2,  5, -2,  6,  2,  5,  2,  6,
      -2,  6, -1,  7,  2,  6,  1,  7, -1,  7,  1,  7,  0,  0,  0,  0
};

int shldobj[NPTS]={
      -9, -5, -9,  5, -9,  5, -5,  9, -5,  9,  5,  9,  5,  9,  9,  5,
       9,  5,  9, -5,  9, -5,  5, -9,  5, -9, -5, -9, -5, -9, -9, -5,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
       0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};

int shldc[]={            /* shield color for given strength                */
     8,7,15,15
};

int shlds[]={            /* shield strength for ships 1-3                  */
     180,480,360
};

int cloakd[]={           /* cloaking device flags for ships 1-3            */
     1,0,1
};

int hyperd[]={           /* hyperspace device flags for ships 1-3          */
     1,1,0
};

int shipm[NSHIPS]={      /* ship masses (affect acceleration & top speed)  */
     1,1,1
};

int shipc[MAXPYR]={           /* player colors                             */
      7, 9,10,11,12,13,14,15, 6, 5
};

int sx[MAXPYR]={              /* start X coordinate                        */
     MINSX+10,MAXSX-10,256,384,384,256,512,128,128,512
};

int sy[MAXPYR]={              /* start Y coordinate                        */
     MAXSY/2,MAXSY/2,MINSY+10,MAXSY-10,MINSY+10,MAXSY-10,MINSY+10,MAXSY-10,MINSY+10,MAXSY-10
};

int sa[MAXPYR]={              /* start angle (polar) 0=facing down,90=left */
     270, 90,  0,180,  0,180,  0,180,  0,180
};

int plrsx[MAXPYR]={           /* score X position for each player          */
      10,590,256,384,384,256,512,128,128,512
};

int plrsy[MAXPYR]={           /* score Y position for each player          */
       5,340,  5,340,  5,340,  5,340,  5,340
};

int alpha[][48]={             /* object definitions for alphabet characters*/
/*0*/ 3,-3,-3, 3,-3, 3,-4, 2,-4, 2,-4,-2,-4,-2,-2,-4,-2,-4, 2,-4, 2,-4, 4,-2,
      4,-2, 4, 2, 4, 2, 2, 4, 2, 4,-2, 4,-2, 4,-3, 3, 0, 0, 0, 0, 0, 0, 0, 0,
/*1*/ 1,-3, 2,-4, 2,-4, 2, 4, 1, 4, 3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*2*/-4,-2,-2,-4,-2,-4, 2,-4, 2,-4, 4,-2, 4,-2, 4,-1, 4,-1,-3, 2,-3, 2,-4, 3,
     -4, 3,-4, 4,-4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*3*/-4,-4, 4,-4, 4,-4,-1,-1,-1,-1, 2,-1, 2,-1, 4, 1, 4, 1, 4, 2, 4, 2, 2, 4,
      2, 4,-4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*4*/ 2, 4, 2,-4, 2,-4,-4, 0,-4, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*5*/ 4,-4,-4,-4,-4,-4,-4, 0,-4, 0, 2, 0, 2, 0, 4, 1, 4, 1, 4, 2, 4, 2, 2, 4,
      2, 4,-4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*6*/ 4,-4,-2,-4,-2,-4,-4,-2,-4,-2,-4, 2,-4, 2,-2, 4,-2, 4, 2, 4, 2, 4, 4, 2,
      4, 2, 4, 1, 4, 1, 2,-1, 2,-1,-2,-1,-2,-1,-4, 1, 0, 0, 0, 0, 0, 0, 0, 0,
/*7*/-4,-4, 4,-4, 4,-4,-4, 4,-4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*8*/-2, 0,-4,-2,-4,-2,-2,-4,-2,-4, 2,-4, 2,-4, 4,-2, 4,-2, 2, 0, 2, 0, 4, 2,
      4, 2, 2, 4, 2, 4,-2, 4,-2, 4,-4, 2,-4, 2,-2, 0,-2, 0, 2, 0, 0, 0, 0, 0,
/*9*/-4, 2,-2, 4,-2, 4, 2, 4, 2, 4, 4, 2, 4, 2, 4,-2, 4,-2, 2,-4, 2,-4,-2,-4,
     -2,-4,-4,-2,-4,-2,-4,-1,-4,-1,-3, 0,-3, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*A*/-4, 4,-4,-1,-4,-1,-1,-4,-1,-4, 1,-4, 1,-4, 4,-1, 4,-1, 4, 4, 4, 1,-4, 1,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*B*/-4, 4,-4,-4,-4,-4, 2,-4, 2,-4, 3,-3, 3,-3, 3,-2, 3,-2, 2,-1, 2,-1, 4, 1,
      4, 1, 4, 2, 4, 2, 2, 4, 2, 4,-4, 4,-4,-1, 2,-1, 0, 0, 0, 0, 0, 0, 0, 0,
/*C*/ 4, 4,-2, 4,-2, 4,-4, 2,-4, 2,-4,-2,-4,-2,-2,-4,-2,-4, 4,-4, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*D*/-4,-4, 2,-4, 2,-4, 4,-2, 4,-2, 4, 2, 4, 2, 2, 4, 2, 4,-4, 4,-4, 4,-4,-4,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*E*/ 4, 4,-4, 4,-4, 4,-4, 0,-4, 0, 1, 0,-4, 0,-4,-4,-4,-4, 4,-4, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*F*/-4, 4,-4, 0,-4, 0,-1, 0,-4, 0,-4,-4,-4,-4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*G*/ 4,-3, 3,-4, 3,-4,-2,-4,-2,-4,-4,-2,-4,-2,-4, 2,-4, 2,-2, 4,-2, 4, 4, 4,
      4, 4, 4, 1, 4, 1, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*H*/-4,-4,-4, 4,-4, 0, 4, 0, 4,-4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*I*/ 1,-4, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*J*/-4, 3,-3, 4,-3, 4, 0, 4, 0, 4, 1, 3, 1, 3, 1,-4, 4,-4,-3,-4, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*K*/-4, 4,-4,-4,-4, 0, 4,-4,-4, 0, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*L*/-4,-4,-4, 4,-4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*M*/-4, 4,-4,-4,-4,-4,-1, 0,-1, 0, 4,-4, 4,-4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*N*/-4, 4,-4,-4,-4,-4, 4, 4, 4, 4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*O*/-2, 4,-4, 2,-4, 2,-4,-2,-4,-2,-2,-4,-2,-4, 2,-4, 2,-4, 4,-2, 4,-2, 4, 2,
      4, 2, 2, 4, 2, 4,-2, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*P*/-4, 4,-4,-4,-4,-4, 1,-4, 1,-4, 2,-3, 2,-3, 2,-1, 2,-1, 1, 0, 1, 0,-4, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*Q*/-2, 4,-4, 2,-4, 2,-4,-2,-4,-2,-2,-4,-2,-4, 2,-4, 2,-4, 4,-2, 4,-2, 4, 2,
      4, 2, 2, 4, 2, 4,-2, 4, 2, 2, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*R*/-4, 4,-4,-4,-4,-4, 2,-4, 2,-4, 3,-3, 3,-3, 3,-1, 3,-1, 2, 0,-2, 0,-4, 0,
     -1, 0, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*S*/-4, 4, 2, 4, 2, 4, 3, 3, 3, 3, 3, 2, 3, 3, 1, 0, 1, 0,-1, 0, 1, 0,-1, 0,
     -1, 0,-3,-2,-3,-2,-3,-3,-3,-3,-2,-4,-2,-4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0,
/*T*/-4,-4, 4,-4, 0,-4, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*U*/-4,-4,-4, 2,-4, 2,-2, 4,-2, 4, 2, 4, 2, 4, 4, 2, 4, 2, 4,-4, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*V*/-4,-4,-1, 4,-1, 4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*W*/-4,-4,-2, 4,-2, 4, 0,-1, 0,-1, 2, 4, 2, 4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*X*/-4,-4, 4, 4,-4, 4, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*Y*/-4,-4,-1, 0,-1, 0,-1, 4,-1, 0, 4,-4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*Z*/-4,-4, 4,-4, 4,-4,-4, 4,-4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*(*/ 1,-4,-1,-2,-1,-2,-1, 2, 1, 2, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/*)*/-1,-4, 1,-2, 1,-2, 1, 2, 1, 2,-1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

char dbgmsg[80],
     evgamsg[]={
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,'E',0xCF,'G',0xCF,'A',0xCF,' ',0xCF,'O',0xCF,
          'R',0xCF,' ',0xCF,'V',0xCF,'G',0xCF,'A',0xCF,' ',0xCF,'R',0xCF,'E',0xCF,
          'Q',0xCF,'U',0xCF,'I',0xCF,'R',0xCF,'E',0xCF,'D',0xCF,' ',0xCF,'T',0xCF,
          'O',0xCF,' ',0xCF,'P',0xCF,'L',0xCF,'A',0xCF,'Y',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,
          ' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF,' ',0xCF
     },
     gipmsg[]={
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,'G',0x9F,'A',0x9F,'M',0x9F,'E',0x9F,' ',0x9F,
          'A',0x9F,'L',0x9F,'R',0x9F,'E',0x9F,'A',0x9F,'D',0x9F,'Y',0x9F,' ',0x9F,
          'I',0x9F,'N',0x9F,' ',0x9F,'P',0x9F,'R',0x9F,'O',0x9F,'G',0x9F,'R',0x9F,
          'E',0x9F,'S',0x9F,'S',0x9F,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,
          ' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10,' ',0x10
     },
     onhelp[]={
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,'R',0x70,'E',0x70,'A',0x70,'D',0x70,
          'I',0x70,'N',0x70,'G',0x70,' ',0x70,'T',0x70,'H',0x70,'E',0x70,' ',0x70,
          'H',0x70,'E',0x70,'L',0x70,'P',0x70,' ',0x70,'S',0x70,'C',0x70,'R',0x70,
          'E',0x70,'E',0x70,'N',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,
          ' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70,' ',0x70
     },
     msg[80],
     oldchw[MAXPYR][CHWSIZ*2];

struct torps {                /* photon torpedo structure (shared)         */
     int  a;                  /*   angle torp is travelling                */
     int  o;                  /*   owner of this torp                      */
     int  t;                  /*   time left for this torp                 */
     int  type;               /*   type of projectile (TORP or PHA)        */
     int  v;                  /*   initial velocity for this torp          */
     int  x1,y1;              /*   current x,y for this torp               */
     int  x2,y2;              /*   end of vector if phaser                 */
     int  xo,yo;              /*   combined ship & torp velocity           */
} torp[MAXTORS],*tptr;

struct star {                 /* star information (games with a SUN)       */
     int  x,xs;               /*   screen x and scaled x position          */
     int  y,ys;               /*   screen y and scaled y position          */
     int  g;                  /*   gravitational pull of this star         */
};

struct star star={
     320,3200,
     175,1750,
     0
};

int bholexy[4][24]={     /* object definition for blackhole (star)    */
      2, 1, 3, 0, 3,-1, 3,-2, 2,-3, 1,-4, 0,-5,-1,-5,-2,-5,-3,-5,-4,-5,
     -1, 2, 0, 3, 1, 3, 2, 3, 3, 2, 4, 1, 5, 0, 5,-1, 5,-2, 5,-3, 5,-4,
     -2,-1,-3, 0,-3, 1,-3, 2,-2, 3,-1, 4, 0, 5, 1, 5, 2, 5, 3, 5, 4, 5,
      1,-2, 0,-3,-1,-3,-2,-3,-3,-2,-4,-1,-5, 0,-5, 1,-5, 2,-5, 3,-5, 4
};

int posstar[4][11]={     /* positive gravity blackhole color          */
      7, 9, 9, 9, 9, 9, 1, 1, 1, 1, 1,
      7, 9, 9, 9, 9, 9, 1, 1, 1, 1, 1,
      7, 9, 9, 9, 9, 9, 1, 1, 1, 1, 1,
      7, 9, 9, 9, 9, 9, 1, 1, 1, 1, 1
};

int negstar[4][11]={     /* negative gravity blackhole color          */
      7,12,12,12,12,12, 4, 4, 4, 4, 4,
      7,12,12,12,12,12, 4, 4, 4, 4, 4,
      7,12,12,12,12,12, 4, 4, 4, 4, 4,
      7,12,12,12,12,12, 4, 4, 4, 4, 4
};

struct explo {                /* animated explosion info                   */
     int  x[EXPLBITS];
     int  y[EXPLBITS];
     int  xo[EXPLBITS];
     int  yo[EXPLBITS];
     int  c;
} explo[MAXPYR],*expptr;

struct player demoshp[NSHIPS];     /* struct for ship preview data         */

void interrupt                /* this is the new keyboard interrupt code   */
newkeyi(void)
{
     int  v;

     if (inportb(0x64)&0x01) {
          v=inportb(0x60);
          outportb(0x20,0x20);
          if (v == 129) {
               setvect(0x09,oldkeyi);
          }
          if (v != lv) {
               ecoutp(v);     /* send to the Flash pool                    */
               lv=v;
          }
     }
     enable();
}

void                          /* setup the new interrupt vector            */
setkeyi(void)
{
     disable();
     oldkeyi=getvect(0x09);
     setvect(0x09,newkeyi);
     enable();
}

void                          /* reset keyboard interrupt vector           */
rstkeyi(void)
{
     disable();
     setvect(0x09,oldkeyi);
     enable();
}

void                          /* get starting video mode; restore later    */
getmode(void)
{
     regs.h.ah=0x0f;
     int86(0x10,&regs,&regs);
     omode=regs.h.al;
}

int                           /* change video mode, save original          */
setmode(char m)
{
     int p;
     char far *sysptr;

     regs.h.ah=0x1A;
     regs.h.al=0x00;
     int86(0x10,&regs,&regs);
     if (regs.h.al != 0x1A) {
          sysptr=MK_FP(0x0F000,0x0FFFE);
          if (*sysptr == 0x0FD || *sysptr == 0x0F9) {
               return(0);
          }
          sysptr=MK_FP(0x0040,0x0010);
          if ((*sysptr&0x30) == 0x30) {
               return(0);
          }
          regs.h.ah=18;
          regs.h.al=16;
          int86(0x10,&regs,&regs);
          if (regs.h.bl == 16) {
               return(0);
          }
     }
     regs.h.ah=0;
     regs.h.al=m;
     int86(0x10,&regs,&regs);
     return(1);
}

int                           /* turn on video mode 16 if possible         */
setvid6(void)
{
     getmode();
     if (setmode(0x10) != 0) {
          onscrn=0;
          setkeyi();
          return(1);
     }
     return(0);
}

void                          /* reset display to original state           */
rstvid6(void)
{
     gtype=0;
     setmode(omode);
     shocha();
}

void                          /* set pixel at x,y to color c (egavga.asm)  */
pixelset(int x,int y,int c)
{
     pixel_set(x,y,c);
}

void                          /* draw vector from x1,y1 to x2,y2 color=c   */
drawline(int x1,int y1,int x2,int y2,int c)
{
     line(x1,y1,x2,y2,c);
}

void                          /* draw object using vectors pointed by *pts */
drawobj(int *pts,int color)
{
     int  n,x1,y1,x2,y2;

     for (n=0 ; n < NPTS ; n+=4) {
          x1=*pts++;
          y1=*pts++;
          x2=*pts++;
          y2=*pts++;
          if (x1 == 0 && y1 == 0) {
               break;
          }
          drawline(x1,y1,x2,y2,color);
     }
}

void                          /* draw shields around ship if activated     */
drawshd(int x,int y,int color)
{
     int  n,x1,y1,x2,y2;

     for (n=0 ; n < 32 ; n+=4) {
          x1=x+shldobj[n];
          y1=y+shldobj[n+1];
          x2=x+shldobj[n+2];
          y2=y+shldobj[n+3];
          drawline(x1,y1,x2,y2,color);
     }
}

void                          /* lower shields..                           */
shldlwr(int pn)
{
     player[pn].flags&=~SHIELDS;
     drawshd(player[pn].x/SCALER,player[pn].y/SCALER,0);
     if (pn == self) {
          noise(SHIELDDN);
     }
}

void                          /* raise the shields for player pn           */
shldsup(int pn)
{
     player[pn].flags|=SHIELDS;
     if (pn == self) {
          noise(SHIELDUP);
     }
}

void                          /* cloaking stuff enabled                    */
cloakon(int pn)
{
     player[pn].flags|=CLOAK;
     if (pn == self) {
          noise(CLOAKON);
     }
}

void                          /* cloaking stuff disabled                   */
cloakof(int pn)
{
     if (player[pn].flags&CLOAK) {
          player[pn].flags&=~CLOAK;
          if (pn == self) {
               noise(CLOAKOFF);
          }
     }
}

int                           /* decrease shield strength if shields up    */
shldhit(struct player *ptr)
{
     if (ptr->flags&SHIELDS) {
          if (--ptr->shield > 0) {
               if (ptr->shield%10 == 0 && ptr->pyrn == player[self].pyrn) {
                    pixelset(plrsx[ptr->pyrn]+ptr->shield/10,plrsy[ptr->pyrn]+8,0);
               }
               return(1);
          }
          else if (ptr->pyrn == player[self].pyrn) {
               pixelset(plrsx[ptr->pyrn],plrsy[ptr->pyrn]+8,0);
          }
     }
     return(0);
}

void                          /* draw alpha chars at x,y using color c     */
drawchr(int *pts,int x,int y,int color)
{
     int  n,x1,y1,x2,y2;

     for (n=0 ; n < 48 ; n+=4) {
          x1=*pts++;
          y1=*pts++;
          x2=*pts++;
          y2=*pts++;
          if (x1 == 0 && y1 == 0 || x2 == 0 && y2 == 0) {
               break;
          }
          drawline(x+x1,y+y1,x+x2,y+y2,color);
     }
}

void                          /* display alpha character at x,y            */
dspchr(int x,int y,int color,char chr)
{
     if (isdigit(chr)) {
          chr=chr-'0';
     }
     else if (isalpha(chr)) {
          chr=toupper(chr)-55;
     }
     else if (chr == '(' || chr == ')') {
          chr=chr-4;
     }
     else if (chr == '.') {
          pixelset(x,y,color);
          return;
     }
     else if (chr == ':') {
          pixelset(x,y-2,color);
          pixelset(x,y+2,color);
          return;
     }
     else if (chr == '-') {
          drawline(x-3,y,x+3,y,color);
          return;
     }
     else {
          return;
     }
     drawchr(&alpha[chr][0],x,y,color);
}

void                          /* display a string at x,y                   */
shostr(int x,int y,int color,char *stg)
{
     int n,len;

     len=strlen(stg);
     for (n=0 ; n < len ; n++) {
          dspchr(x,y,color,*stg++);
          x+=10;
     }
}

int                           /* adjust angle value if > 360 or < 0        */
adjang(int *ang)
{
     if (*ang > 359) {
          *ang-=360;
     }
     else if (*ang < 0) {
          *ang+=360;
     }
}

void                          /* compute vector endpoints for angle "ang"  */
rot2d(int *xr,int *yr,int ang,int x,int y,int xo,int yo)
{
     int xc,xs,yc,ys;

     xc=xo*costab[ang];
     xs=xo*sintab[ang];
     yc=yo*costab[ang];
     ys=yo*sintab[ang];
     *xr=(x+((xc-ys)/SCALER))/SCALER;
     *yr=(y+((yc+xs)/SCALER))/SCALER;
}

void                          /* adjust xoff and yoff given ang & accel    */
mov2d(int *axp,int *ayp,int ang,int accel)
{
     int ax,ay;

     ax=*axp-(accel*sintab[ang])/SCALER;
     if (ax > ACCBND) {
          ax=ACCBND;
     }
     else if (ax < -ACCBND) {
          ax=-ACCBND;
     }
     *axp=ax;
     ay=*ayp+(accel*costab[ang])/SCALER;
     ay*=2;
     if (ay > ACCBND) {
          ay=ACCBND;
     }
     else if (ay < -ACCBND) {
          ay=-ACCBND;
     }
     *ayp=ay/2;
}

void                          /* mov2d without velocity limitation         */
mov2dl(int *axp,int *ayp,int ang,int accel)
{
     *axp-=((accel*sintab[ang])/SCALER);
     *ayp+=((accel*costab[ang])/SCALER);
}

int                           /* compute polar angle given X & Y distance  */
rec2pol(int dx,int dy)
{
     int ang,high,low;
     unsigned tang;
     #ifdef DEBUG
     unsigned cnt=0;
     #endif

     low=0;
     high=900;
     ang=450;
     if (dx == 0) {
          if (dy > 0) {
               ang=180;
          }
          else {
               ang=0;
          }
     }
     else if (dy == 0) {
          if (dx > 0) {
               ang=90;
          }
          else {
               ang=270;
          }
     }
     else {
          tang=(abs(dy)*100)/abs(dx);
          while (1) {
               if (tang >= tantab[ang] && tang <= tantab[ang+1]) {
                    break;
               }
               if (tang > tantab[ang]) {
                    low=ang;
                    ang+=(high-low)/2;
               }
               else if (tang < tantab[ang]) {
                    high=ang;
                    ang-=(high-low)/2;
               }
               #ifdef DEBUG
               if (cnt++ > 100) {
                  fprintf(dbgfp,"***********************TOO MANY ITERATIONS IN REC2POL()****************\n");
                  fprintf(dbgfp,"DX: %d, DY: %d, HIGH: %d, LOW: %d, ANG: %d, TANG: %u\n",
                                 dx,dy,high,low,ang,tang);
                  fprintf(dbgfp,"***********************************************************************\n");
                  break;
               }
               #endif
          }
          ang/=10;
          if (dx > 0) {
               if (dy > 0) {       /* quadrant 4 (dx > 0 && dy > 0)   */
                    ang=90-ang;
               }
               else {              /* quadrant 3 (dx > 0 && dy < 0)   */
                    ang+=90;
               }
          }
          else {
               if (dy > 0) {       /* quadrant 1 (dx < 0 && dy > 0)   */
                    ang+=270;
               }
               else {              /* quadrant 2 (dx < 0 && dy < 0)   */
                    ang=270-ang;
               }
          }
     }
     return(ang);
}

void                          /* compute friction according to ship mass   */
friction(struct player *pyr)
{
     int  ix,iy,xm,ym;

     xm=(pyr->xo*pyr->mass);
     ix=xm>>3;
     if (ix == 0) {
          ix=xm>>2;
          if (ix == 0) {
               ix=xm>>1;
               if (ix == 0) {
                    ix=pyr->xo;
               }
          }
     }
     ym=(pyr->yo*pyr->mass);
     iy=ym>>3;
     if (iy == 0) {
          iy=ym>>2;
          if (iy == 0) {
               iy=ym>>1;
               if (iy == 0) {
                    iy=pyr->yo;
               }
          }
     }
     pyr->xo-=ix;
     pyr->yo-=iy;
}

void                          /* compute gravitational pull on ship        */
gravity(struct player *pyr)
{
     int a,distx,disty,gp,gxo,gyo,d;

     distx=scale(star.xs-pyr->x,SCALER);
     disty=scale(star.ys-pyr->y,SCALER);
     a=rec2pol(distx,disty<<1);
     distx=scale(distx,SCALER);
     disty=scale(disty,SCALER);
     d=distx*distx+disty*disty;
     if (d == 0 || (gp=((star.g-1)*GCONST)/d) > MAXACC ) {
          pyr->hyper=18;
          pyr->xo=0;
          pyr->yo=0;
     }
     else {
          if (gp <= 330) {                   // small pull                
               gxo=scale(gp*sintab[a],100);
               gyo=scale(gp*costab[a],100);
          }
          else if (gp <= 3300) {             // medium pull
               gxo=scale(scale(gp,10)*sintab[a],10);
               gyo=scale(scale(gp,10)*costab[a],10);
          }
          gxo+=pyr->xo;
          if (gxo > ACCBND) {
               gxo=ACCBND;
          }
          else if (gxo < -ACCBND) {
               gxo=-ACCBND;
          }
          pyr->xo=gxo;
          gyo+=pyr->yo*2;
          if (gyo > ACCBND) {
               gyo=ACCBND;
          }
          else if (gyo < -ACCBND) {
               gyo=-ACCBND;
          }
          pyr->yo=gyo/2;
     }
     #ifdef DEBUG
     fprintf( dbgfp, "%6u (%5d,%5d)  %4d (%5d,%5d)   %3d     %d\n",
        d,distx,disty,gp,gxo,gyo,a,(star.g-1)*GCONST);
     fflush(dbgfp);
     #endif
}

void
defpyr(int pn)
{
     int  n,slvlc[]={12,14,10,10,10};

     player[pn].accel=0;
     player[pn].ang=sa[player[pn].pyrn];
     player[pn].color=(player[pn].state == GAMOVER) ? 0 : shipc[player[pn].pyrn];
     player[pn].explo=0;
     player[pn].flags=SHIELDS;
     player[pn].firec=0;
     player[pn].fstorp=player[pn].pyrn*6;
     player[pn].hyper=0;
     player[pn].lstorp=player[pn].fstorp+5;
     player[pn].mass=shipm[player[pn].objn];
     player[pn].nxtorp=player[pn].fstorp;
     player[pn].rrate=0;
     player[pn].shield=(player[pn].state == GAMOVER) ? 0 : shlds[player[pn].objn];
     player[pn].x=sx[player[pn].pyrn]*SCALER;
     player[pn].y=sy[player[pn].pyrn]*SCALER;
     player[pn].xo=0;
     player[pn].yo=0;
     setmem(&player[pn].pts[0],NPTS,0);
     if (pn == self) {
          for (n=0 ; n < player[pn].shield/10 ; n++) {
               pixelset(plrsx[player[pn].pyrn]+n,plrsy[player[pn].pyrn]+8,
                    slvlc[n/12]);
          }
          noise(SHIELDUP);
     }
}

void                          /* initialize player explosion data          */
iniexp(int pn)
{
     int  n;

     expptr=&explo[pn];
     setmem(expptr,sizeof(struct explo),0);
     for (n=0 ; n < EXPLBITS ; n++) {
          expptr->x[n]=player[pn].x/SCALER;
          expptr->y[n]=player[pn].y/SCALER;
          mov2d(&expptr->xo[n],&expptr->yo[n],rnd()%360,rnd()%8);
          expptr->xo[n]/=SCALER;
          expptr->yo[n]/=SCALER;
     }
     noise(KABOOM);
}

void                          /* collision detection routine               */
chkhit(struct torps *ptr)
{
     int  n,sr,x,y;

     sr=SHPSIZ;
     x=ptr->x1;
     y=ptr->y1;
     if (ptr->type == FIREPHA) {
          if (ptr->x1 > ptr->x2) {
               x-=(ptr->x1-ptr->x2)/2;
          }
          else if (ptr->x1 < ptr->x2) {
               x+=(ptr->x2-ptr->x1)/2;
          }
          if (ptr->y1 > ptr->y2) {
               y-=(ptr->y1-ptr->y2)/2;
          }
          else if (ptr->y1 < ptr->y2) {
               y+=(ptr->y2-ptr->y1)/2;
          }
     }
     for (n=0 ; n < npyrs ; n++) {
          if (ingame(player[n].state)) {
               if (abs(x-player[n].x) < sr && abs(y-player[n].y) < sr
                  && player[n].explo == 0 && player[n].hyper == 0) {
                    ptr->t=1;
                    if (!shldhit(&player[n])) {
                         if (player[n].flags&SHIELDS) {
                              shldlwr(n);
                         }
                         drawobj(&player[n].pts[0],0);
                         player[n].explo=36;
                         iniexp(n);
                         if (ptr->o != n) {
                              sprintf(msg,"%5d",player[ptr->o].score);
                              shostr(plrsx[player[ptr->o].pyrn],
                                   plrsy[player[ptr->o].pyrn],0,msg);
                                   player[ptr->o].score+=1;
                              sprintf(msg,"%5d",player[ptr->o].score);
                              shostr(plrsx[player[ptr->o].pyrn],
                                   plrsy[player[ptr->o].pyrn],
                                   shipc[player[ptr->o].pyrn],msg);
                         }
                         return;
                    }
                    else if (ptr->type == FIRETOR) {
                         if (rnd() > 20000) {
                              player[n].ang+=(rnd()%8);
                         }
                         else {
                              player[n].ang-=(rnd()%8);
                         }
                         if (n == self) {
                              noise(SHIELDHIT);
                         }
                    }
                    else if (ptr->o == self) {
                         noise(HITSHIELD);
                    }
                    drawshd(player[n].x/SCALER,player[n].y/SCALER,11);
               }
          }
     }
}

int                           /* adjust boundaries (MINBX < x < MAXBX, etc)*/
adjbnds(int xy,int min,int max)
{
     int mag;

     if (xy > max) {
          xy=((xy-min)%(max-min))+min;
     }
     else if (xy < min) {
          xy=max-(abs(xy-min)%(max-min));
     }
     return(xy);
}

void                          /* move photon torpedoes                     */
movtor(void)
{
     int  n,x1,y1,x2,y2;

     for (n=0 ; n < MAXTORS ; n++) {
          tptr=&torp[n];
          if (tptr->t > 1) {
               x1=tptr->x1/SCALER;
               y1=tptr->y1/SCALER;
               if (tptr->type == FIREPHA) {
                    x2=tptr->x2/SCALER;
                    y2=tptr->y2/SCALER;
                    drawline(x1,y1,x2,y2,0);
               }
               else {
                    pixelset(x1,y1,0);
               }
               tptr->x1+=(tptr->xo>>1);
               tptr->y1+=(tptr->yo>>1);
               if (tptr->type == FIREPHA) {
                    tptr->x2+=(tptr->xo>>1);
                    tptr->y2+=(tptr->yo>>1);
                    if (tptr->x1 < MINBX || tptr->x2 < MINBX) {
                         tptr->x1=(MAXBX-abs(MINBX-tptr->x1));
                         tptr->x2=(MAXBX-abs(MINBX-tptr->x2));
                    }
                    else if (tptr->x1 > MAXBX || tptr->x2 > MAXBX) {
                         tptr->x1=(MINBX+abs(MAXBX-tptr->x1));
                         tptr->x2=(MINBX+abs(MAXBX-tptr->x2));
                    }
                    if (tptr->y1 < MINBY || tptr->y2 < MINBY) {
                         tptr->y1=(MAXBY-abs(MINBY-tptr->y1));
                         tptr->y2=(MAXBY-abs(MINBY-tptr->y2));
                    }
                    else if (tptr->y1 > MAXBY || tptr->y2 > MAXBY) {
                         tptr->y1=(MINBY+abs(MAXBY-tptr->y1));
                         tptr->y2=(MINBY+abs(MAXBY-tptr->y2));
                    }
               }
               else {
                    tptr->x1=adjbnds(tptr->x1,MINBX,MAXBX);
                    tptr->y1=adjbnds(tptr->y1,MINBY,MAXBY);
               }
               x1=tptr->x1/SCALER;
               y1=tptr->y1/SCALER;
               if (tptr->type == FIREPHA) {
                    x2=tptr->x2/SCALER;
                    y2=tptr->y2/SCALER;
                    drawline(x1,y1,x2,y2,shipc[player[tptr->o].pyrn]);
               }
               else {
                    pixelset(x1,y1,shipc[player[tptr->o].pyrn]);
               }
               tptr->t-=1;
               chkhit(tptr);
          }
          else if (tptr->t == 1) {
               x1=tptr->x1/SCALER;
               y1=tptr->y1/SCALER;
               if (tptr->type == FIREPHA) {
                    x2=tptr->x2/SCALER;
                    y2=tptr->y2/SCALER;
                    drawline(x1,y1,x2,y2,0);
               }
               else {
                    pixelset(x1,y1,0);
               }
               tptr->t=0;
          }
     }
}

void                          /* update starfield                          */
starf(void)
{
     int  n,x,y;

     for (n=0 ; n < MAXSTARS ; n++) {
          pixelset(starx[starfn][n],stary[starfn][n],starc[starfn][n]);
     }
     if (++starfn == NUMFIELD) {
          starfn=0;
     }
     if (gtype&SUN) {
          for (n=0 ; n < 11 ; n++) {
               x=star.x+bholexy[bholen][n*2];
               y=star.y+bholexy[bholen][(n*2)+1];
               pixelset(x,y,bholec[(bholen*11)+n]);
          }
          if (++bholen == 4) {
               bholen=0;
          }
     }
}

void                          /* compute new rotated object                */
compobj(struct player *ptr,int scale)
{
     int  epts[NPTS],n,ox,oy;

     movmem(&ptr->pts[0],epts,NPTS*sizeof(int));
     mov2d(&ptr->xo,&ptr->yo,ptr->ang,ptr->accel);
     if (gtype&SUN) {
          gravity(ptr);
     }
     if (gtype&FRICT) {
          friction(ptr);
     }
     ox=ptr->x/SCALER;
     oy=ptr->y/SCALER;
     ptr->x+=(ptr->xo>>1);
     ptr->y+=(ptr->yo>>1);
     ptr->x=adjbnds(ptr->x,MINBX,MAXBX);
     ptr->y=adjbnds(ptr->y,MINBY,MAXBY);
     drawobj(epts,0);
     if (ptr->flags&SHIELDS) {
          drawshd(ox,oy,0);
     }
     if (ptr->hyper == 0) {   // added this to stop the calc & redraw if I 
                              //     threw them into hyperspace
          for (n=0 ; n < NPTS ; n+=2) {
               if (shipobj[ptr->objn][n] == 0 && shipobj[ptr->objn][n+1] == 0) {
                    break;
               }
               rot2d(&ptr->pts[n],&ptr->pts[n+1],ptr->ang,ptr->x,ptr->y,
                    shipobj[ptr->objn][n]*scale,shipobj[ptr->objn][n+1]*scale);
          }
          if (ptr->flags&SHIELDS) {
               drawshd(ptr->x/SCALER,ptr->y/SCALER,shldc[ptr->shield/120]);
          }
          if (!(ptr->flags&CLOAK) || ptr->flags&SHIELDS) {
               drawobj(&ptr->pts[0],ptr->color);
          }
          else if (ptr->pyrn == player[self].pyrn) {
               drawobj(&ptr->pts[0],CLOAKC);
          }
     }
}

void                          /* fire a photon torpedo                     */
firetor(int pn)
{
     tptr=&torp[player[pn].nxtorp];
     if (player[pn].nxtorp++ == player[pn].lstorp) {
          player[pn].nxtorp=player[pn].fstorp;
     }
     if (tptr->t == 0) {
          tptr->a=player[pn].ang;
          tptr->o=pn;
          tptr->type=player[pn].firec;
          tptr->t=50;
          tptr->v=6;
          tptr->x1=tptr->x2=player[pn].x;
          tptr->y1=tptr->y2=player[pn].y;
          tptr->xo=tptr->yo=0;
          mov2dl(&tptr->xo,&tptr->yo,tptr->a,10);
          tptr->x1+=tptr->xo;
          tptr->y1+=tptr->yo;
          if (tptr->type == FIREPHA) {
               noise(NLASER);
               tptr->t/=10;
               tptr->v*=4;
               tptr->x2+=(tptr->xo*2);
               tptr->y2+=(tptr->yo*2);
          }
          else {
               noise(NTORPE);
          }
          tptr->xo=player[pn].xo;
          tptr->yo=player[pn].yo;
          mov2dl(&tptr->xo,&tptr->yo,tptr->a,tptr->v);
     }
}

void                          /* process player explosion                  */
prcxpl(int n)
{
     int  i;

     if (--player[n].explo == 0) {
          expptr=&explo[n];
          for (i=0 ; i < EXPLBITS ; i++) {
               pixelset(expptr->x[i],expptr->y[i],0);
          }
          defpyr(n);
     }
     else if (player[n].explo > 0) {
          expptr=&explo[n];
          for (i=0 ; i < EXPLBITS ; i++) {
               pixelset(expptr->x[i],expptr->y[i],0);
               expptr->x[i]+=expptr->xo[i];
               expptr->y[i]+=expptr->yo[i];
               expptr->x[i]=adjbnds(expptr->x[i],MINSX,MAXSX);
               expptr->y[i]=adjbnds(expptr->y[i],MINSY,MAXSY);
               pixelset(expptr->x[i],expptr->y[i],expptr->c);
               if (expptr->c < 16) {
                    expptr->c+=1;
               }
               else {
                    expptr->c=0;
               }
          }
     }
     else {
          player[n].explo=0;
     }
}

void                          /* move players around the screen            */
fpmovp2(void)
{
     int  i,n;

     for (n=0 ; n < npyrs ; n++) {
          if (ingame(player[n].state)) {
               if (player[n].explo > 0) {
                    prcxpl(n);
               }
               else if (player[n].hyper == 0) {
                    if (player[n].firec != 0 && !(player[n].flags&CLOAK)) {
                         firetor(n);
                    }
                    player[n].ang+=player[n].rrate;
                    adjang(&player[n].ang);
                    compobj(&player[n],1);
               }
               else if (player[n].hyper > 0) {
                    if (--player[n].hyper == 0) {
                         player[n].x=rnd()%MAXBX;
                         player[n].y=rnd()%MAXBY;
                         if (player[n].firec == 0 && rnd() > HYPEXPLO) {
                              player[n].explo=36;
                              iniexp(n);
                         }
                    }
               }
               if (!shldhit(&player[n])) {
                    if (player[n].flags&SHIELDS) {
                         shldlwr(n);
                    }
               }
          }
     }
}

void                          /* show joystick status - ON or OFF          */
shojoy(int color,int onoff)
{
     if (pyrp->state == PIKSHIP) {
          setmem(msg,sizeof(msg),0);
          sprintf(msg,"F10: JOYSTICK CONTROL IS %s",(onoff == 0) ? "OFF" : "ON");
          shostr((MAXSX/2)-((strlen(msg)*10)/2),(MAXSY/2)+100,
               (color == 0) ? 0 : (onoff == 0) ? 12 : 10,msg);
          sprintf(msg,"F10 AT ANYTIME WILL TOGGLE AND RECALIBRATE JOYSTICK");
          shostr((MAXSX/2)-((strlen(msg)*10)/2),(MAXSY/2)+120,
               (color == 0) ? 0 : (onoff == 0) ? 0 : 10,msg);
          sprintf(msg,"BE SURE STICK IS CENTERED FOR RECALIBRATION");
          shostr((MAXSX/2)-((strlen(msg)*10)/2),(MAXSY/2)+135,
               (color == 0) ? 0 : (onoff == 0) ? 0 : 10,msg);
     }
     if (joyflg) {
          jstick();
          joyminx=joyx-28;
          joymaxx=joyx+28;
          joyminy=joyy-28;
          joymaxy=joyy+28;
     }
}

void                          /* divide screen for ship preview            */
shogrid(int color)
{
     setmem(msg,sizeof(msg),0);
     strcpy(msg,"PRESS  F1  OR  F2  OR  F3  TWICE TO SELECT A SHIP");
     shostr((MAXSX/2)-((strlen(msg)*10)/2),10,color,msg);
     drawline(MAXSX/3,20,MAXSX/3,(MAXSY/2)+90,color);
     drawline((MAXSX/3)*2,20,(MAXSX/3)*2,(MAXSY/2)+90,color);
     drawline(MINSX,MAXSY/2,MAXSX,MAXSY/2,color);
     shostr(15,155,color,"F1:WARBIRD");
     shostr(10,(MAXSY/2)+20,(color == 0) ? 0 : 12,"CLOAK....... YES");
     shostr(10,(MAXSY/2)+50,(color == 0) ? 0 : 14,"SHIELDS..... WEAK");
     shostr(10,(MAXSY/2)+80,(color == 0) ? 0 : 9,"HYPERSPACE.. YES");
     shostr(238,155,color,"F2:HEAVY CRUISER");
     shostr(233,(MAXSY/2)+20,(color == 0) ? 0 : 12,"CLOAK....... NO");
     shostr(233,(MAXSY/2)+50,(color == 0) ? 0 : 14,"SHIELDS..... STRONG");
     shostr(233,(MAXSY/2)+80,(color == 0) ? 0 : 9,"HYPERSPACE.. YES");
     shostr(441,155,color,"F3:BATTLE CRUISER");
     shostr(436,(MAXSY/2)+20,(color == 0) ? 0 : 12,"CLOAK....... YES");
     shostr(436,(MAXSY/2)+50,(color == 0) ? 0 : 14,"SHIELDS..... MEDIUM");
     shostr(436,(MAXSY/2)+80,(color == 0) ? 0 : 9,"HYPERSPACE.. NO");
     shojoy(color,joyflg);
}

void                          /* initialize "demo" ship parameters         */
inishp(void)
{
     int  n,
          x[NSHIPS]={
               1050,3200,5350
          },
          y[NSHIPS]={
                880, 880, 880
          };

     setmem(&demoshp[0],sizeof(struct player)*NSHIPS,0);
     for (n=0 ; n < NSHIPS ; n++) {
          if (n == player[self].objn) {
               demoshp[n].color=15;
          }
          else {
               demoshp[n].color=13;
          }
          demoshp[n].mass=shipm[n];
          demoshp[n].objn=n;
          demoshp[n].rrate=4;
          demoshp[n].x=x[n];
          demoshp[n].y=y[n];
     }
     joyflg=0;
     shogrid(15);
}

void                          /* preview ships (rotate them, scale = 4)    */
shoshps(void)
{
     int  n;

     for (n=0 ; n < NSHIPS ; n++) {
          demoshp[n].ang+=demoshp[n].rrate;
          adjang(&demoshp[n].ang);
          compobj(&demoshp[n],4);
     }
}

void                          /* save chat-window & display message        */
savchw(int pn,char *msg)
{
     movmem(&player[pn].chawin,&oldchw[player[pn].chapos],CHWSIZ*2);
     movmem(msg,&player[pn].chawin,CHWSIZ*2);
     dspchw(pn);
}

void                          /* restore chat-window contents              */
rstchw(int pn)
{
     movmem(&oldchw[player[pn].chapos],&player[pn].chawin,CHWSIZ*2);
     dspchw(pn);
}

int                           /* set all players to a certain state        */
allset(int chkstt,int newstt)
{
     int  n;

     for (n=0 ; n < npyrs ; n++) {
          if (player[n].state == chkstt) {
               player[n].pyrn=n;
               player[n].state=newstt;
               if (newstt == 0) {
                    movmem(&oldchw[player[n].chapos],&player[n].chawin,CHWSIZ*2);
               }
               else if (newstt == GAMOVER) {
                    player[n].accel=player[n].firec=player[n].flags=player[n].rrate=0;
                    drawshd(player[n].x/SCALER,player[n].y/SCALER,0);
               }
          }
     }
}

int                           /* see if all players ready..                */
allrdy(int chkval)
{
     int  n,prdy;

     prdy=0;
     for (n=0 ; n < npyrs ; n++) {
          if (chkval == WAITING && !ingame(player[n].state)) {
               prdy+=1;
          }
          else if (player[n].state > 0 && player[n].state == chkval) {
               prdy+=1;
          }
     }
     return(prdy == npyrs);
}

int                           /* all players ready? 1=yes, 0=no            */
chkrdy(void)
{
     int  n;

     if (allrdy(pyrp->state)) {
          if (!setvid6()) {
               player[self].state=NOEVGA;
               savchw(self,evgamsg);
               ecoutp(0x1B);
          }
          else {
               allset(pyrp->state,PIKSHIP);
               for (n=0 ; n < npyrs ; n++) {
                    savchw(n,gipmsg);
                    player[n].objn=-1;
               }
               inishp();
          }
          return(1);
     }
     return(0);
}

void                          /* set game-type parameters                  */
setgame(void)
{
     int  i,n;

     inirtk();
     rndnum=rseed;
     switch (gtype) {
     case F5:
          gtype=FRICT;
          break;
     case F7:
          gtype=FRICT;
     case F6:
          star.g=(rnd()%10)+1;
          if (rnd() > 30000) {
               star.g=-star.g;
          }
          bholec=(star.g < 0) ? &negstar[0][0] : &posstar[0][0];
          gtype|=SUN;
          break;
     case F8:
          gtype=NOPARM;
          break;
     }
     for (i=0 ; i < NUMFIELD ; i++) {
          for (n=0 ; n < MAXSTARS ; n++) {
               starx[i][n]=rnd()%MAXSX;
               stary[i][n]=rnd()%MAXSY;
               starc[i][n]=(rnd()%7)+1;
               pixelset(starx[i][n],stary[i][n],starc[i][n]);
          }
     }
     for (n=0 ; n < npyrs ; n++) {
          if (ingame(player[n].state)) {
               shostr(plrsx[player[n].pyrn],plrsy[player[n].pyrn],
                    shipc[player[n].pyrn],"    0");
               player[n].score=0;
               defpyr(n);
          }
     }
     bholen=starfn=0;
     gtimr[0]=2178;
     gtimr[1]=(gtimr[0]/18)-1;
     gtmrc1=1;
}

int                           /* now wait until everyone is ready..        */
waitplr(void)
{
     int  n;

     if (allrdy(WAITING)) {
          setgame();
          setmem(&torp[0],sizeof(struct torps)*MAXTORS,0);
          setmem(dbgmsg,sizeof(dbgmsg),0);

          #ifdef DEBUG
          if( !dbgfp ) {
               dbgfp=fopen("DEBUGFS","w");
          }
          fprintf( dbgfp, "\n      distance               gravity       angle   star\n" );
          fprintf( dbgfp, "*******************************************************\n" );
          #endif

          return(1);
     }
     return(0);
}

void                          /* process joystick movements                */
prcjoy(void)
{
     if (!joyflg) {
          return;
     }
     jstick();
     if (joyx < joyminx && pyrp->rrate >= 0) {
          ecoutp(30);
     }
     else if (joyx > joymaxx && pyrp->rrate <= 0) {
          ecoutp(31);
     }
     else if (joyx >= joyminx && joyx <= joymaxx && pyrp->rrate != 0) {
          ecoutp(30+128);
     }
     if (joyy < joyminy && pyrp->accel == 0) {
          ecoutp(37);
     }
     else if (joyy > joymaxy && pyrp->hyper == 0) {
          if (pyrp->accel != 0) {
               ecoutp(37+128);
          }
          ecoutp(57);
     }
     else if (joyy >= joyminy && joyy <= joymaxy && pyrp->accel != 0) {
          ecoutp(37+128);
     }
     if (!(joyb&JOYBUT1)) {
          if (!(joyb&JOYBUT2)) {
               if (!(pyrp->flags&CLOAK)) {
                    ecoutp(54);
               }
          }
          else if (pyrp->firec == 0) {
               if (pyrp->flags&CLOAK) {
                    ecoutp(54);
               }
               ecoutp(38);
          }
     }
     else if (!(joyb&JOYBUT2)) {
          if (pyrp->firec == 0) {
               if (pyrp->flags&CLOAK) {
                    ecoutp(54);
               }
               ecoutp(39);
          }
     }
     else if (pyrp->firec != 0) {
          ecoutp(38+128);
     }
}

void                          /* end the game, unconditionally             */
endgam(void)
{
     int  n;

     if (player[self].state == PLAYING) {
          allset(PLAYING,GAMOVER);
     }
     noise(SHIELDDN);
     if (!onscrn) {
          shostr(275,150,(gmoc == 0) ? 4 : 12,"GAME OVER");
          shostr(275,160,(gmoc == 0) ? 12 : 4,"PRESS ESC");
     }
     if (gmoc == 0) {
          noise(CHBLOP);
     }
     else {
          noise(CHBEEP);
     }
     gmoc=(gmoc == 0) ? 1 : 0;
}

void                          /* process -TICK of game timer               */
prctmr(void)
{
     if (--gtimr[0] == 180) {
          gmoc=0;
          gtmrc1=15;
          rtkick(180,endgam);
     }
     if (gtimr[0]%18 == 0) {
          gtmrc=gtmrc1;
          sprintf(msg,"%1d:%02d",gtimr[1]/60,gtimr[1]%60);
          shostr((MAXSX/2)-(strlen(msg)*10)/2,15,0,msg);
          gtimr[1]-=1;
          sprintf(msg,"%1d:%02d",gtimr[1]/60,gtimr[1]%60);
          shostr((MAXSX/2)-(strlen(msg)*10)/2,15,gtmrc,msg);
     }
     else if (gtimr[0] < 180 && gtmrc != 1) {
          gtmrc=1;
          sprintf(msg,"%1d:%02d",gtimr[1]/60,gtimr[1]%60);
          shostr((MAXSX/2)-(strlen(msg)*10)/2,15,gtmrc,msg);
     }
}

void
fpmovp1(void)
{
     pyrp=&player[self];
     switch (pyrp->state) {
     case PIKSHIP:
          shoshps();
          break;
     case WAITING:
          sprintf(msg,"WAITING TO TRANSPORT");
          if (waitplr()) {
               allset(WAITING,PLAYING);
               shostr(MAXSX/2-(strlen(msg)*10)/2,175,0,msg);
          }
          else {
               shostr(MAXSX/2-(strlen(msg)*10)/2,175,(++gentck)%16,msg);
          }
          break;
     case PLAYING:
          if (joyflg) {
               prcjoy();
          }
          prctmr();
          starf();
          fpmovp2();
          movtor();
          break;
     case GAMOVER:
          if ((++gtimr[0])%9 == 0) {
               endgam();
          }
          starf();
          fpmovp2();
          movtor();
          break;
     }
}

void                          /* show game selected on window              */
shosel(int pn)
{
     int gcolr[]={
          0x4F,0x5F,0x6F,0x3F
     },k;
     char gname[][21]={
          "F5: FRICTION        ",
          "F6: GRAVITY         ",
          "F7: FRICTION+GRAVITY",
          "F8: NON-FRICTION    "
     };

     if (onscrn) {
          if (player[pn].state >= F5 && player[pn].state <= F8) {
               switch (player[pn].state) {
               case F5:
                    k=1;
                    break;
               case F6:
                    k=2;
                    break;
               case F7:
                    k=3;
                    break;
               default:
                    k=4;
                    break;
               }
               locate(CHWSIZ-23,1+player[pn].chapos*2);
               setatr(gcolr[k-1]);
               printf(" %-20s ",gname[k-1]);
               rstloc();
               gtype=player[pn].state;
          }
     }
}

void                          /* remove selected game type from window     */
rmvsel(int pn)
{
     if (onscrn) {
          locate(CHWSIZ-23,1+player[pn].chapos*2);
          setatr(AT4CHW);
          if (player[pn].chapos == 0) {
               printf("");
          }
          else {
               printf("");
          }
          rstloc();
     }
}

int                           /* additional game-related chat F-keys       */
addchr(int c)
{
     int  k;

     k=0;
     if (!ingame(pyrp->state) && c == pyrp->state) {
          pyrp->state=0;
          rmvsel(pyrn);
          return(0);
     }
     switch (c) {
     case 0x1B:
          if (pyrp->state == NOEVGA) {
               savchw(pyrn,evgamsg);
          }
          else {
               setmem(&oldchw[pyrp->chapos],CHWSIZ*2,0);
               oldchw[pyrp->chapos][CHWSIZ*2-1]=AT4WRT;
               rstchw(pyrn);
          }
          break;
     case F5:
          k=1;
          break;
     case F6:
          k=2;
          break;
     case F7:
          k=3;
          break;
     case F8:
          k=4;
          break;
     case PIKSHIP:
     case WAITING:
     case PLAYING:
     case GAMOVER:
     case ONHELP:
          pyrp->state=c;
          if (onscrn) {
               if (c == ONHELP) {
                    savchw(pyrn,onhelp);
               }
               else {
                    savchw(pyrn,gipmsg);
               }
          }
          break;
     }
     if (k > 0) {
          pyrp->state=c;
          shosel(pyrn);
     }
     return(chkrdy());
}

void
rmvxpl(int pn)                /* remove explosion if player leaves game    */
{
     int  n;

     if (player[pn].explo > 0) {
          expptr=&explo[pn];
          for (n=0 ; n < EXPLBITS ; n++) {
               pixelset(expptr->x[n],expptr->y[n],0);
          }
     }
}

void                          /* handle keyboard input                     */
dwgchr(int c)
{
     int  n;

     if (c == 1) {
          rstchw(pyrn);
          if (pyrp->state == PIKSHIP || pyrp->state == WAITING) {
               allset(PIKSHIP,0);
               allset(WAITING,0);
               if (!onscrn) {
                    rstvid6();
                    if (pyrn != self) {
                         rstkeyi();
                    }
               }
          }
          else if (pyrn == self) {
               noise(SHIELDDN);
               rstvid6();
          }
          else if (onscrn) {
               dspchw(pyrn);
          }
          else if (ingame(player[self].state)) {
               drawobj(&pyrp->pts[0],0);
               if (pyrp->flags&SHIELDS) {
                    shldlwr(pyrn);
               }
               rmvxpl(pyrn);
               sprintf(msg,"%5d",pyrp->score);
               shostr(plrsx[pyrp->pyrn],plrsy[pyrp->pyrn],0,msg);
               shostr(plrsx[pyrp->pyrn],plrsy[pyrp->pyrn],shipc[pyrp->pyrn],"QUIT");
          }
          pyrp->state=0;
          return;
     }
     else if (!ingame(player[self].state)) {
          return;
     }
     if (c == 68) {           /* F10 key pressed, toggle joystick          */
          if (pyrn == self) {
               shojoy(0,joyflg);
               joyflg=(joyflg == 0) ? 1 : 0;
               shojoy(1,joyflg);
          }
          return;
     }
     switch (pyrp->state) {
     case PIKSHIP:
          switch (c) {
          case 59:
          case 60:
          case 61:
               if (pyrp->objn == c-59) {
                    if (pyrn == self) {
                         for (n=0 ; n < NSHIPS ; n++) {
                              demoshp[n].color=0;
                         }
                         shoshps();
                         shogrid(0);
                    }
                    pyrp->state=WAITING;
                    gentck=0;
               }
               else {
                    if (pyrn == self) {
                         if (pyrp->objn >= 0) {
                              demoshp[pyrp->objn].color=13;
                         }
                         demoshp[c-59].color=15;
                    }
                    pyrp->objn=c-59;
               }
          }
          break;
     case PLAYING:
          switch (c) {
          case 30:            /* rotate left                               */
          case 75:
               pyrp->rrate=-8;
               break;
          case 30+128:        /* stop rotation                             */
          case 31+128:
          case 32+128:
          case 75+128:
          case 77+128:
               pyrp->rrate=0;
               break;
          case 31:            /* rotate right                              */
          case 32:
          case 77:
               pyrp->rrate=8;
               break;
          case 37:            /* thrust                                    */
          case 72:
               pyrp->accel=3-pyrp->mass;
               break;
          case 37+128:        /* thrusters off                             */
          case 72+128:
               pyrp->accel=0;
               break;
          case 38:            /* fire lasers                               */
          case 71:
               pyrp->firec=FIREPHA;
               cloakof(pyrn);
               break;
          case 39:            /* fire torpedos                             */
          case 73:
               pyrp->firec=FIRETOR;
               cloakof(pyrn);
               break;
          case 38+128:        /* fire control off                          */
          case 39+128:
          case 71+128:
          case 73+128:
               pyrp->firec=0;
               break;
          case 42:            /* shields up                                */
               if (pyrp->shield > 0) {
                    shldsup(pyrn);
               }
               break;
          case 42+128:        /* shields down                              */
               shldlwr(pyrn);
               break;
          case 54:            /* cloaking device engaged                   */
               if (cloakd[pyrp->objn]) {
                    if (pyrp->flags&CLOAK) {
                         cloakof(pyrn);
                    }
                    else {
                         cloakon(pyrn);
                    }
               }
               break;
          case 57:            /* enter hyperspace                          */
          case 80:
               if (pyrp->hyper == 0 && pyrp->explo == 0 && hyperd[pyrp->objn]) {
                    pyrp->hyper=18;
                    pyrp->xo=0;
                    pyrp->yo=0;
                    drawobj(&pyrp->pts[0],0);
                    if (pyrp->flags&SHIELDS) {
                         drawshd(pyrp->x/SCALER,pyrp->y/SCALER,0);
                    }
               }
          }
     }
}

int                           /* player in a game? 1=yes, 0=no             */
ingame(int pstate)
{
     return(pstate >= PIKSHIP && pstate <= GAMOVER);
}

void                          /* -REMOVE received, take care of pyrn       */
rmvplr(void)
{
     int  n;

     if (player[pyrn].state == PLAYING && player[self].state == PLAYING) {
          drawobj(&player[pyrn].pts[0],0);
          drawshd(player[pyrn].x/SCALER,player[pyrn].y/SCALER,0);
          sprintf(msg,"%5d",pyrp->score);
          shostr(plrsx[player[pyrn].pyrn],plrsy[player[pyrn].pyrn],0,msg);
          shostr(plrsx[player[pyrn].pyrn],plrsy[player[pyrn].pyrn],
               shipc[pyrp->pyrn],"GONE");
          rmvxpl(pyrn);
          for (n=0 ; n < MAXTORS ; n++) {
               if (torp[n].o == pyrn) {
                    torp[n].t=1;
               }
          }
     }
     setmem(&player[pyrn],sizeof(struct player),0);
     for (n=0 ; n < npyrs ; n++) {
          if (!ingame(player[n].state) && player[n].state != ONHELP) {
               player[n].state=0;
               rmvsel(n);
          }
     }
}

void                          /* -LINEUP received..                        */
linem2(void)
{
     int  n;

     if (player[self].state >= PIKSHIP) {
          ecoutp(player[self].state);
     }
     for (n=0 ; n < npyrs ; n++) {
          if (!ingame(player[n].state)) {
               player[n].state=0;
          }
     }
}

void
iniplr(void)
{
     pyrp->objn=-1;
}
