
        PAGE    94,118
        TITLE   PROGA.ASM - Assembly Examples for EGA Programmers Guide
        NAME    PROGA

;************************************************************************
;                                                                       *
;       File:   PROGA.ASM                                               *
;       Author: George Sutty                                            *
;       Date:   March 19, 1988                                          *
;       Description: Assembly procedures with examples for programming  *
;               the EGA/VGA.  These examples are in support of the      *
;               text for the  Programmer's Guide to EGA/VGA book.       *
;                                                                       *
;************************************************************************


_DATA   SEGMENT PUBLIC BYTE 'DATA'
_DATA   ENDS

_TEXT   SEGMENT BYTE PUBLIC 'CODE'
        ASSUME  CS:_TEXT, DS:_DATA

;************************************************************************
; Video BIOS Data Area Locations                                        *
;************************************************************************

BIOS_Mode       EQU     449H            ;Location of current mode
BIOS_CRT_Addr   EQU     463H            ;Address of CRT controller
BIOS_Rows       EQU     484H            ;Number of text rows
BIOS_Columns    EQU     44AH            ;Number of text columns
BIOS_Height     EQU     485H            ;Character height
BIOS_Page_Size  EQU     44CH            ;Number of bytes in one page
BIOS_Curs_Mode  EQU     460H            ;Cursor start and end
BIOS_Equipment  EQU     487H            ;Type of equipment in system
BIOS_Switch     EQU     488H            ;Configuration switches
BIOS_Curs_Pos   EQU     450H            ;Current cursor address
BIOS_Curs_Start EQU     460H            ;Current cursor shape
BIOS_Curs_Stop  EQU     461H

;************************************************************************
; Wait for start of next horizontal retrace                             *
;************************************************************************

        PUBLIC  _Horizontal_Retrace

_Horizontal_Retrace     PROC FAR
        PUSH    ES
        XOR     AX,AX                   ;Get address of STATUS register
        MOV     ES,AX                   ;From segment 0
        MOV     DX,3BAH                 ;Assume monochrome addressing
        TEST    BYTE PTR ES:[BIOS_Equipment],2  ;Is mono display attached?
        JNZ     HR_Address_Ok           ;...Yes, address is ok
        MOV     DX,3DAH                 ;...No, must set color addressing
HR_Address_Ok:

        IN      AL,DX                   ;Read in status
        JMP     $+2
        TEST    AL,01H                  ;Is retrace ON? (if ON, bit = 1)
        JZ      Wait_For_HOn            ;...no, go wait for start
Wait_For_HOff:                          ;...yes, wait for last one to go OFF
        IN      AL,DX
        JMP     $+2
        TEST    AL,01H                  ;Is retrace OFF?
        JNZ     Wait_For_HOff           ;...No, keep waiting
Wait_For_HOn:
        IN      AL,DX
        JMP     $+2
        TEST    AL,01H                  ;Is retrace ON?
        JZ      Wait_For_HOn            ;...No, keep on waiting
        POP     ES
        RETF
_Horizontal_Retrace     ENDP

;************************************************************************
; Common definitions for the next five routines                         *
;************************************************************************

GRAPHICS_CTL    EQU     3CEH            ;Address of Graphics controller
SEQUENCE_CTL    EQU     3C4H            ;Address the Sequencer

XSIZE   EQU     640                     ;Assume 640 pixels across
HBYTES  EQU     (XSIZE/8)               ;Compute bytes per raster
GRAPH_SEG       EQU     0A000H          ;Segment of display buffer

;************************************************************************
; Set pixel at X,Y to color COLOR                                       *
; using SET-RESET method                                                *
;************************************************************************

Pparms    struc
          dw   ?
          dw   ?                        ; FAR requires 2 words for
          dw   ?                        ;    pushed return address
X         dw   ?
Y         dw   ?
COLOR     dw   ?
Pparms    ends

        PUBLIC  _Pixel_Set

_Pixel_Set PROC    FAR
        PUSH    BP
        MOV     BP,SP
        PUSH    ES

; Convert x,y address into OFFSET:SEGMENT and get MASK

        MOV     BX,[BP+X]                ;Fetch X coordinate
        MOV     AX,[BP+Y]                ;Fetch Y coordinate
        CALL    Get_Address              ;Compute SEGMENT:OFFSET address
                                         ;in ES:BX, Mask in CL

        MOV     DX,GRAPHICS_CTL

; Enable set/reset and load value into reset register

        MOV     AL,00H                  ;Select SET/RESET register
        OUT     DX,AL
        INC     DX
        MOV     AX,[BP+COLOR]           ;Fetch color passed on the stack
        OUT     DX,AX                   ;Set SET register to color
        DEC     DX
        MOV     AL,01H                  ;Select ENABLE RESET
        OUT     DX,AL
        INC     DX
        MOV     AL,0FH                  ;Enable all four planes for SET
        OUT     DX,AL
        DEC     DX

        ; Set mask register to preserve unused 7 bits in a byte

        MOV     AL,08H                  ;Select BIT MASK register
        OUT     DX,AL
        MOV     AL,CL                   ;Fetch mask returned earlier
        INC     DX                      ;and set MASK register to it
        OUT     DX,AL
        DEC     DX

        ; Set the pixel to new color

        MOV     AL,ES:[BX]              ;Must latch to preserve old bits
        MOV     ES:[BX],AL              ;Set pixel color to new value

        ; Restore SET/RESET and MASK registers

        MOV     AL,01H                  ;Select ENABLE RESET
        OUT     DX,AL
        INC     DX
        MOV     AL,00H                  ;Disable all four planes for SET
        OUT     DX,AL
        DEC     DX

        MOV     AL,08H                  ;Select BIT MASK register
        OUT     DX,AL
        MOV     AL,0FFH                 ;Enable all 8 bits for write
        INC     DX                      ;(default)
        OUT     DX,AL
        DEC     DX

        ; Restore segment registers and return

        POP     ES
        MOV     SP,BP
        POP     BP
        RETF
_Pixel_Set      ENDP

;************************************************************************
; Read value of the pixel at x,y and return value in AX                 *
;************************************************************************

Rparms  struc
        dw     ?
        dw     ?                        ; FAR requires 2 words for
        dw     ?                        ;    pushed return address
RX      dw     ?
RY      dw     ?

Rparms  ends

        PUBLIC  _Pixel_Read

_Pixel_Read PROC    FAR
        PUSH    BP
        MOV     BP,SP
        PUSH    ES
        PUSH    SI

        ; Convert x,y address into OFFSET:SEGMENT and get MASK

        MOV     BX,[BP+RX]              ;Fetch X coordinate
        MOV     AX,[BP+RY]              ;Fetch Y coordinate
        CALL    Get_Address             ;Compute SEGMENT:OFFSET address
                                        ;in ES:BX, Mask in CL

        ; Set MAP MASK register to read next plane and read next plane

        MOV     DX,GRAPHICS_CTL
        MOV     AL,04H                  ;Select MAP MASK register
        OUT     DX,AL
        INC     DX

        MOV     SI,BX                   ;Copy offset into register SI
        XOR     BH,BH                   ;Clear color
        MOV     BL,CL                   ;Copy mask into register CL
        MOV     CX,4                    ;Initialize loop counter
Read_Plane_Loop:
        SHL     BH,1
        MOV     AL,CL                   ;Select next plane for read
        DEC     AL
        OUT     DX,AL

        MOV     AH,ES:[SI]              ;Fetch values for 8 pixels in plane 0
        AND     AH,BL                   ;Mask of the bit for our pixel (mask in
        JZ      Zero_Value              ;Put 0 into next color bit
        OR      BH,1                    ;Put 1 into next color bit
Zero_Value:
        LOOP    Read_Plane_Loop

        MOV     AL,BH                   ;Put result into AX
        XOR     AH,AH

        ; Restore segment registers and return

        POP     SI
        POP     ES                      ;Restore registers
        MOV     SP,BP
        POP     BP
        RETF
_Pixel_Read     ENDP

;************************************************************************
; Compute SEGMENT:OFFSET pair from a given x,y address                  *
; of a pixel.  We know that there is HBYTES bytes in each               *
; raster and that each byte  contains eight pixels.                     *
; To find offest of pixel x,y we use the following formula              *
;                                                                       *
;       OFFSET = HBYTES * y + x/8                                       *
;                                                                       *
; To compute which bit within byte is to be changed we get              *
; (x mod 8) which is remainder when x is divided by 8.                  *
; Which is same as keeping last three bits of x.  We use                *
; position within a byte to rotate value 80H so that the                *
; single bit matches the position in a byte.                            *
; Recall that bit7 in a byte represents left most pixel.                *
; Thus MASK is computed as follows:                                     *
;                                                                       *
;       MASK = ROTATE 80H TO THE RIGHT BY (X AND 7)                     *
;                                                                       *
; Entry:        BX - X coordinate (pixel)                               *
;               AX - Y coordinate (raster)                              *
; Exit:         CL - Mask (x mod 8)                                     *
;               BX - Absolute offset in display buffer                  *
;************************************************************************

        ; Compute SEGMENT:OFFSET pair from x,y pair

Get_Address:
        MOV     CX,HBYTES               ;Fetch bytes per raster
        MUL     CX                      ;Compute offset past y rasters
        MOV     CL,BL                   ;Keep copy of x for later
        SHR     BX,1                    ;Get offset within raster as x mod 8
        SHR     BX,1
        SHR     BX,1
        ADD     BX,AX                   ;Add offsets together and keep in BX
        MOV     AX,GRAPH_SEG            ;Fetch segment and copy it into ES
        MOV     ES,AX

        ; Compute MASK within byte from x coordinate

        AND     CL,07H                  ;Compute which bit in a byte
        MOV     AL,80H                  ;and use it to rotate mask into positio
        ROR     AL,CL
        MOV     CL,AL                   ;Keep mask in CL

        RET

;************************************************************************
; Line (x0, y0, x1, y1,c ): draw line from (x0,y0) to (x1,y1)           *
;               with a color (c).                                       *
;               This routine is divided into three parts. Horizontal    *
;               lines are done in the first part, vertical lines in     *
;               second part, and rest in the third part.  The lines     *
;               in the third part are done using the Bresenhams         *
;               algorithm.                                              *
; entry : sp + 2 = x0                                                   *
;         sp + 4 = y0                                                   *
;         sp + 6 = x1                                                   *
;         sp + 8 = y1                                                   *
;         sp + 10= color                                                *
;************************************************************************

Lparms  struc
        dw     ?
        dw     ?
        dw     ?
X0      dw     ?
Y0      dw     ?
X1      dw     ?
Y1      dw     ?
LCOLOR  dw     ?

Lparms  ends

        PUBLIC  _Line

_Line   PROC    FAR
        PUSH    BP
        MOV     BP,SP
        SUB     SP,6                    ;Declare three local variables

        PUSH    DI
        PUSH    SI
        PUSH    DS
        PUSH    ES

        MOV     AX,[BP+X0]              ; make sure that x1 >= x0
        MOV     CX,[BP+X1]
        CMP     CX,AX
        JGE     Get_Offset
        MOV     BX,[BP+Y0]
        MOV     DX,[BP+Y1]
        MOV     [BP+X0],CX
        MOV     [BP+Y0],DX
        MOV     [BP+X1],AX
        MOV     [BP+Y1],BX

;-------COMPUTE ADDRESS AND MASK FOR FIRST PIXEL -------------
Get_Offset:
        ;--- Compute offset and save on stack
        MOV     CL,4                    ; offset = 80 * y + x/8
        MOV     AX,[BP+Y0]
        SHL     AX,CL
        MOV     BX,AX
        SHL     AX,1
        SHL     AX,1
        ADD     AX,BX
        MOV     BX,[BP+X0]              ; + x/8
        MOV     CL,3
        SHR     BX,CL
        ADD     AX,BX
        PUSH    AX                      ; save offset on stack, later pop to DI

        ;--- Compute mask and save on the stack

        MOV     CX,[BP+X0]              ; compute which bit (x mod 8) to modify
        AND     CL,7
        MOV     BX,80h
        SHR     BX,CL
        PUSH    BX                      ; save mask on stack, later pop to SI
        MOV     DX,03CEh                ; enable only the bit(within a byte)
        MOV     AL,08H                  ; to be changed
        OUT     DX,AL
        INC     DX
        MOV     AL,BL                   ; ... reg-bx has the correct bit
        OUT     DX,AL

        ;--- Load set/reset registers with current color

        MOV     DX,03CEh                ; move color into reset register
        XOR     AL,AL
        OUT     DX,AL
        INC     DX
        MOV     AX,[BP+LCOLOR]
        OUT     DX,AL
        DEC     DX                      ; enable use of reset register
        MOV     AL,1
        OUT     DX,AL
        INC     DX
        MOV     AL,0FH
        OUT     DX,AL
        MOV     DX,03C4h                ; enable all four planes for writting
        MOV     AL,2
        OUT     DX,AL
        INC     DX
        MOV     AL,0FH
        OUT     DX,AL

        ;--- Load segment register

        MOV     DX,0A000h
        MOV     DS,DX
        MOV     ES,DX

;-------COMPUTE  DX AND DY ----------------------------------------
;       DETERMINE IF HORIZONTAL, VERTICAL OR DIAGONNAL LINE

        MOV     AX,80                   ; set raster increment
        MOV     [bp-6],AX
        MOV     SI,[BP+X1]              ; compute dx           reg-si
        SUB     SI,[BP+X0]
        MOV     DI,[BP+Y1]              ; compute dy           reg-di
        SUB     DI,[BP+Y0]
        JGE     DyIsPos
        NEG     AX
        MOV     [bp-6],AX
        NEG     DI
DyIsPos:
        CMP     SI,0                    ; jump according to type of line
        JZ      Vert
        CMP     DI,0
        JZ      Horiz
        JMP     Diag

;-------GENERATE A VERTICAL LINE -----------------

Vert:   MOV     cx,di                   ; set up counter
        INC     cx
        MOV     bx,[bp-6]

        POP     SI                      ; fetch mask
        MOV     DX,03CEh                ; set mask
        MOV     AL,08h
        OUT     DX,AL
        INC     DX
        MOV     AX,SI
        OUT     DX,AL

        POP     DI                      ; fetch offset
LoopVert:
        MOV     AL,[DI]                 ; latch data(to preserve other 7 bits i
        MOV     [DI],AL                 ; write new data (only one bit will be
        ADD     DI,BX                   ; update offset
        LOOP    LoopVert
        JMP     LineDone

;-------GENERATE A HORIZONTAL LINE ---------------

Horiz:  MOV     CX,SI                   ; set counter of pixels
        POP     SI                      ; fetch mask
        POP     DI                      ; fetch offset

        ;---    draw pixels from the leading partial byte

        MOV     AX,[BP+X0]
        AND     AX,07h                  ; check for partial byte
        JZ      FullBytes
        MOV     BX,0FFh                 ; compute the mask
        PUSH    CX
        MOV     CX,AX
        SHR     BX,CL
        POP     CX
        ADD     CX,AX                   ; update counter
        SUB     CX,08h
        JGE     MaskSet                 ; modify mask if only one byte
        NEG     CX
        SHR     BX,CL
        SHL     BX,CL
        XOR     CX,CX                   ; restore counter
MaskSet:
        MOV     DX,03CEh                ; set the mask
        MOV     AL,08h
        OUT     DX,AL
        INC     DX
        MOV     AL,BL
        OUT     DX,AL
        MOV     AH,[DI]                 ; latch data
        MOV     [DI],AL                 ; write new data
        INC     DI                      ; update offset

        ;---    draw pixels from the middle complete bytes
FullBytes:                              ;
        MOV     BX,CX                   ; check if any bytes to set
        CMP     CX,8
        JL      TrailBytes
        SHR     CX,1                    ; compute count
        SHR     CX,1
        SHR     CX,1
        MOV     DX,03CEh                ; set the mask
        MOV     AL,08h
        OUT     DX,AL
        INC     DX
        MOV     AL,0FFh
        OUT     DX,AL

        REP     STOSB                   ; fill complete bytes

        ;---    draw pixels from the trailing partial byte
TrailBytes:
        AND     BX,07h
        JZ      HorizDone
        MOV     AX,0FFFFh                ; compute mask
        MOV     CX,BX
        SHR     AX,CL
        XOR     AH,0FFh                  ; set the mask
        MOV     DX,03CEh
        MOV     AL,08h
        OUT     DX,AL
        INC     DX
        MOV     AL,AH
        OUT     DX,AL
        MOV     AL,[DI]                  ; latch data
        MOV     [DI],AL                  ; set new data
HorizDone:
        JMP     LineDone

;----------------------- GENERATE A DIAGONAL LINE -----------------

        ;--- figure out which quarter does the line lie in

Diag:   CMP     SI,DI                   ; Is dy > dx
        JLE     oct12                   ; ...Yes, do processing in octants
                                        ;    1 and 2
        ; Compute constants for octant zero and three
        ; This is where x is the major direction and y is minor

oct03:  MOV     CX,SI                   ; set counter to dx+1
        INC     CX
        SAL     DI,1                    ; d1 = dy*2             reg-di
        MOV     BX,DI                   ; d  = dy*2-dx          reg-bx
        SUB     BX,SI
        NEG     SI                      ; d2 = dy*2-dx-dx       reg-si
        ADD     SI,BX

        MOV     [bp-2],di               ; save d1
        MOV     [bp-4],si               ; save d2
        MOV     DX,03CEh                ; select BIT MASK register
        MOV     AL,08h
        OUT     DX,AL
        INC     DX
        POP     AX                      ; fetch mask
        POP     DI                      ; fetch address

        ;-------------- GENERATE LINE IN THE OCTANT ZERO AND THREE ------

next0:  OUT     DX,AL                   ; enable a bit in a byte
        MOV     AH,[DI]                 ; latch old data
        MOV     [DI],AH                 ; modify (enabled bits)

        ROR     AL,1                    ; update mask
        ADC     DI,0                    ; update byte address

        TEST    BX,8000H                ; if d >= 0 then ...
        JNZ     dneg0
        ADD     BX,[BP-4]               ; ... d = d + d2
        ADD     DI,[BP-6]               ; update offset to next scan line
        LOOP    next0
        JMP     LineDone

dneg0:  ADD     BX,[BP-2]               ; if d < 0 then d = d + d1
        LOOP    next0
        JMP     LineDone

        ;----------------------------------------------------------------

        ;---    Compute constants for octant one and two

oct12:  MOV     CX,DI                   ; set counter to dy+1
        INC     CX
        SAL     SI,1                    ; d1 = dx * 2
        MOV     BX,SI                   ; d  = dx * 2 - dy
        SUB     BX,DI
        NEG     DI                      ; d2 = -dy + dx * 2 - dy
        ADD     DI,BX

        MOV     [BP-4],DI               ; save d2
        MOV     [BP-2],SI               ; save d1
        MOV     dx,03CEh                ; select BIT MASK register
        MOV     al,08h
        OUT     DX,AL
        INC     DX
        POP     AX                      ; fetch mask
        POP     DI                      ; fetch address
        OUT     DX,AL                   ; enable a bit in a byte

        ;-----GENERATE A LINE IN THE OCTANT ONE AND TWO -----------------

next1:  MOV     AH,[DI]                 ; latch old data
        MOV     [DI],AH                 ; modify (enabled bits)

        ADD     DI,[BP-6]               ; update offset (y = y+1)

        TEST    BX,8000H                ; if d >= 0 then ...
        JNZ     dneg1

        ADD     BX,[BP-4]               ; ... d = d + d2
        ROR     AL,1                    ; ... update mask (x = x+1)
        ADC     DI,0                    ; ... update offset
        OUT     DX,AL                   ; Enable next bit within a byte
        LOOP    next1
        JMP     LineDone

dneg1:  ADD     BX,[BP-2]               ; if d < 0 then d = d + d1
        LOOP    next1

        ;----------------------------------------------------------------
        ;--- Restore PLANE ENABLE and BIT MASK registers
LineDone:
        MOV     DX,03CEh                ; Enable all 8-bits in a byte for write
        MOV     AL,08h                  ; by setting BIT MASK register to Fhex
        OUT     DX,AL
        INC     DX
        MOV     AL,0FFh
        OUT     DX,AL

        DEC     DX                      ; Disable SET/RESET function
        MOV     AL,1
        OUT     DX,AL
        INC     DX
        XOR     AX,AX
        OUT     DX,AL

        POP     ES
        POP     DS
        POP     SI
        POP     DI
        MOV     SP,BP
        POP     BP
        RETF
_Line   ENDP
;
_TEXT   ENDS
        END
