/***************************************************************************
 *                                                                         *
 *   FTFVIEW.C                                                             *
 *                                                                         *
 *   Copyright (C) 1992-1993 GALACTICOMM, Inc.      All Rights Reserved.   *
 *                                                                         *
 *   File Transfer Software ZIP/ARC/ZOO/LZH/ICE/PSE/ZSE file viewer        *
 *                                                                         *
 *   With all operating-system-specific functions removed (I/O, disk,      *
 *   memory, time), the pure file transfer algorithm can be isolated.      *
 *                                                                         *
 *                 - Bob Stein (glue) and Rob Rose (technology)  1/21/92   *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "ftf.h"
#include "ftfview.h"

STATIC void arpli (char *fname,int fnleng,long length,char *meth,long size,
                   int date,int time,unsigned long crc,int cflag);
STATIC int zpprfr(void);
STATIC char *zipmth(void);
STATIC int arprfr(void);
STATIC int icprfr(void);
STATIC int zoprfr(void);
STATIC void vewini(void);
STATIC void vewsrt(void);
STATIC void vewins(char *stg);
STATIC int vewctn(void);
STATIC void z2fend(void);

/*--- View transmitter states ---*/
#define VEWOPN    1                               /* open file for transmit */
#define VEWFID    2                            /* check type, output header */
#define VEWFIL    3                             /* output info on each file */
#define VEWEND    4                                              /* wind up */
#define VEWFND    5           /* searching ZIP2 file for end of central dir */


/*--- ARC/ZIP/etc display structures ---*/

#define VEWCHUNK 128                /* amount of data to read for each file */
#define ZP2CHUNK 2048               /* data to read to check for ZIP2 file  */
#define CDRLEN   sizeof(struct endcdr)    /* valid end of central dir size  */

char *arcdsc[]={
     "",
     "ARC ",
     "ZIP ",
     "LZH/ICE ",
     "ZOO ",
     "self-extracting ZIP ",
     "self-extracting PAK ",
     "ZIP ",
     "self-extracting ZIP "};

static
char *arcerr[]={
     "Not a compressed file",
     "ARC file is not properly formatted",
     "ZIP file is not properly formatted",
     "LZH/ICE file is not properly formatted",
     "ZOO file is not properly formatted",
     "Self-extracting ZIP file is not properly formatted",
     "Self-extracting PAK file is not properly formatted",
     "ZIP file is not properly formatted",
     "Self-extracting ZIP file is not properly formatted"};

static
struct filhead {         /* 30 bytes+filename needed */
     char headr[4];      /* Should be 0x04034B50                      */
     int versneed;       /* version of pkzip needed to extract        */
     int flags;          /* general purpose bitflags                  */
     int method;         /* compression method                        */
     int ftime;          /* file time                                 */
     int fdate;          /* file date                                 */
     unsigned long crc32;/* file crc                                  */
     long compsz;        /* compressed size                           */
     long normsz;        /* uncompressed size                         */
     int namesz;         /* filename size                             */
     int xtrasz;         /* extra field size                          */
} *fhdr;

static
struct dirhead {         /* 46+filename */
     char headr[4];      /* Should be 0x02014B50                      */
     int versmade;       /* version of pkzip that created             */
     int versneed;       /* version of pkzip needed to extract        */
     int flags;          /* general purpose bitflags                  */
     int method;         /* compression method                        */
     int ftime;          /* file time                                 */
     int fdate;          /* file date                                 */
     unsigned long crc32;/* file crc                                  */
     long compsz;        /* compressed size                           */
     long normsz;        /* uncompressed size                         */
     int namesz;         /* filename size                             */
     int xtrasz;         /* extra field size                          */
     int commsz;         /* file comment length                       */
     int dsknst;         /* disk number start                         */
     int infatr;         /* internal file attributes                  */
     long extatr;        /* external file attributes                  */
     long offshd;        /* offset to local header                    */
} *dhdr;


static
struct endcdr {          /* end of central directory record           */
     char headr[4];      /* should be 0x06054b50                      */
     int diskno;         /* number of this disk                       */
     int ctrbeg;         /* number of disk with start of central dir  */
     int ctrcnt;         /* number of entries in central dir on disk  */
     int totcnt;         /* total number of central dir entries       */
     unsigned long ctrsiz;/* size of the central directory            */
     long ctroff;        /* offset of start of central directory      */
     int commsz;         /* zip file comment length                   */
} *ehdr;

                         /* Deflate compression flavors for PKZIP V2  */
#define NORCMP   0x0000  /*   Normal compression (-en)                */
#define MAXCMP   0x0002  /*   Maximum compression (-ex)               */
#define FSTCMP   0x0004  /*   Fast compression (-ef)                  */
#define SFSCMP   (MAXCMP+FSTCMP)   /* Super Fast compression (-es)    */

                         /* ZIP compression methods                   */
#define DEFLAT      8    /*   This file is Deflated                   */

static
struct archead {
     char signat;        /* Signature byte of 26 (1Ah)                */
     char hdrtyp;        /* Header ID                                 */
     char fname[13];     /* Filename                                  */
     long size;          /* Size of compressed file                   */
     int date;           /* File date                                 */
     int time;           /* File time                                 */
     unsigned int crc;   /* file CRC-16                               */
     long length;        /* uncompressed file length (header ID >1)   */
} *ahdr;

static
struct icehead {
     char hdrtyp[2];     /* header type                               */
     char encod[5];      /* encode type                               */
     long size;          /* bytes after compression                   */
     long length;        /* bytes originally                          */
     int  time;
     int  date;
     int  attir;         /* file attributes                           */
     char fnlen;         /* length of filename                        */
/*   char fname[fnlen];     filename (for fnlen bytes)                */
/*   unsigned int  crc16;   CRC-16 for file                           */
} *ihdr;

struct zoohead {
     unsigned long head;      /* should be 0xFDC4A7DCL                */
     unsigned char type;      /* directory type (should be 1)         */
     unsigned char meth;      /* packing method                       */
     long next;               /* offset to next directory entry       */
     long data;               /* offset that this file starts at      */
     int date;
     int time;
     unsigned int crc;
     long length;             /* bytes originally */
     long size;               /* bytes after compression */
     char majver;
     char minver;
     char delete;
     char struc;
     long comment;            /* offset to comment; zero if none      */
     unsigned int cmtsize;    /* length of comment; 0 if none         */
     char fname[13];          /* filename                             */
} *ohdr;

static
char *zoomth[2]={"Stored  ","LevZimp "};

static
char *methds[]={"Stored  ","Shrunk  ","Reduce1 ","Reduce2 ","Reduce3 ",
                "Reduce4 ","Implode ","--------","Deflat"};

static
char *arcmth[10]={"ENDOFARC","Stored  ","Stored  ","Packed  ","Squeezed",
                  "crunched","crunched","crunched","Crunched","Squashed"};
/* Chances of a false ID:
       ZOO format:  1 in       4,228,250,625
       ARC format:  1 in           1,658,137
       PSE format:  1 in       4,228,250,625
       ZIP format:  1 in   1,078,203,909,370
       ZSE format:  1 in 274,941,996,891,000
   ICE,LZH format:  1 in     107,820,390,937
*/

static
char vewhdr[]="\r\
 Length   Method    Size  Ratio   Date    Time     CRC     Name\r\
 ------  --------  ------ -----   ----    ----    ------   ----\r\
";                                 /* view ARC/ZIP/LZH file announce/header */

/*--- The Rob Rose ARC/ZIP/etc view routines ---*/

STATIC void
arpli(                                                  /* report on 1 file */
char *fname,                      /* name of file (may not be 0-terminated) */
int fnleng,             /* max length of file name (not incl 0-term if any) */
long length,                                    /* bytes before compression */
char *meth,                                        /* method of compression */
long size,                                       /* bytes after compression */
int date,                                          /* DOS-style date & time */
int time,
unsigned long crc,                                                   /* CRC */
int cflag)                                              /* 1=comments 0=not */
{
     char buff[80];
     int ratio;     /* percentage of reduction (e.g. 75% ==> 1/4 orig size) */

     if (length != 0) {
          ratio=100-(int)((100*size)/length);
     }             /* note: LHARC program actually displays 100%-this ratio */
     else {
          ratio=0;
     }
     sprintf(buff,"%7ld  %8.8s %7ld %3d%%  %8.8s  %5.5s  %8.8lx  ",
             length,meth,size,ratio,ncdate(date),nctime(time),crc);
     ftfous(buff);
     sprintf(buff,"%-12.*s",min(79,fnleng),fname);
     ftfous(buff);
     if (cflag) {
          ftfous(" C");
     }
     ftfous("\r");
}

STATIC int
zpprfr(void)                               /* output one PKZIP record entry */
{                                     /* returns 1=done -1=error 0=continue */
     int hdr;
     int rval=0;

     dhdr=(struct dirhead *)ftfbuf;
     fhdr=(struct filhead *)ftfbuf;
     hdr=fhdr->headr[2]+fhdr->headr[3]*256;
     switch (hdr) {
     case 0x01EF:                   /* Self-Extracting Header               */
          vewscb->pos=0x31F0L;
          break;
     case 0x0201:                   /* Directory record                     */
          arpli(ftfbuf+0x2E,
                dhdr->namesz,
                dhdr->normsz,
                zipmth(),
                dhdr->compsz,
                dhdr->fdate,
                dhdr->ftime,
                (long)dhdr->crc32,
                dhdr->commsz);
          vewscb->pos+=0x2e+dhdr->namesz+dhdr->xtrasz+dhdr->commsz;
          break;
     case 0x0403:                   /* File header record                   */
          vewscb->pos+=0x1e+fhdr->namesz+fhdr->xtrasz+fhdr->compsz;
          break;
     case 0x0605:                   /* End of Central Dir Record            */
          rval=1;
          break;
     default:
          rval=-1;
          break;
     }
     return(rval);
}

STATIC char *
zipmth(void)                            /* returns zip compression method  */
{                                       /* assumes dhdr is set             */
     static char mthbuf[10];

     strcpy(mthbuf,methds[dhdr->method]);
     if (dhdr->method == DEFLAT) {         /* PKZIP V2x deflate method */
          if ((dhdr->flags&SFSCMP) == SFSCMP) {
               strcat(mthbuf,"S");
          }
          else if (dhdr->flags&FSTCMP) {
               strcat(mthbuf,"F");
          }
          else if (dhdr->flags&MAXCMP) {
               strcat(mthbuf,"X");
          }
          else {
               strcat(mthbuf,"N");
          }
     }
     return(mthbuf);
}

STATIC int
arprfr(void)                                 /* output one ARC record entry */
{                                     /* returns 1=done -1=error 0=continue */
     ahdr=(struct archead *)ftfbuf;
     if (vewscb->pos == 0L && ftfbuf[0] == 'M' && ftfbuf[1] == 'Z') {
          vewscb->pos=0x3352L;
          return(0);
     }
     if (*ftfbuf != 0x1a) {
          return(-1);
     }
     if (ahdr->hdrtyp == 0) {
          return(1);
     }
     arpli(ahdr->fname,
           12,
           ahdr->length,
           arcmth[ahdr->hdrtyp],
           ahdr->size,
           ahdr->date,
           ahdr->time,
           (long)ahdr->crc,
           0);
     vewscb->pos+=0x1d+ahdr->size;
     return(0);
}

STATIC int
icprfr(void)                                 /* output one ICE record entry */
{                                     /* returns 1=done -1=error 0=continue */
     unsigned long crc;
     char mbf[10];

     ihdr=(struct icehead *)ftfbuf;
     if (ftfbuf[2] != '-' && ftfbuf[6] != '-') {
          return(-1);
     }
     if (ihdr->hdrtyp[0] == 0) {
          return(1);
     }
     setmem(mbf,10,'\0');
     strncpy(mbf,ihdr->encod,5);
     crc=ftfbuf[22+ihdr->fnlen]+256*ftfbuf[23+ihdr->fnlen];
     arpli(ftfbuf+22,
           min(ihdr->fnlen,18),
           ihdr->length,
           mbf,
           ihdr->size,
           ihdr->date,
           ihdr->time,
           crc,
           0);
     vewscb->pos+=ihdr->fnlen+ihdr->size+(ihdr->encod[3] == '5' ? 0x1B : 0x18);
     return(0);
}

STATIC int
zoprfr(void)                                 /* output one ZOO record entry */
{                                     /* returns 1=done -1=error 0=continue */
     ohdr=(struct zoohead *)ftfbuf;
     if (vewscb->pos == 0L) {            /* Skip header portion             */
          vewscb->pos=20L;
          return(0);
     }
     if (vewscb->pos == 20L) {           /* Directory header                */
          vewscb->pos=*(long *)(ftfbuf+4);
          return(0);
     }
     if (ohdr->head != 0xFDC4A7DCL) {    /* Look for                        */
          return(-1);
     }
     if (ohdr->length == 0
      && ohdr->size == 0
      && ohdr->date == 0
      && ohdr->time == 0
      && ohdr->meth == 0
      && ohdr->crc == 0) {
          return(1);
     }
     arpli(ohdr->fname,
           12,
           ohdr->length,
           zoomth[ohdr->meth],
           ohdr->size,
           ohdr->date,
           ohdr->time,
           (unsigned long)ohdr->crc,
           (int)ohdr->comment);
     vewscb->pos=ohdr->next;
     return(0);
}

int
zafilid(                                              /* Identify File type */
char *recbuf)                               /* first VIDCHUNK bytes of file */
{                                                        /* returns file id */
     int i;

     if (strncmp(recbuf,"ZOO ",4) == 0) {
          return(ZOOFILE);
     }
     if (*recbuf == 26) {
          if (recbuf[1] > 9) {
               return(UNKFILE);
          }
          for (i=2 ; i < 15 ; i++) {
               if (recbuf[i] == '\0') {
                    break;
               }
               if (recbuf[i] < ' ' || recbuf[i] > '}') {
                    return(UNKFILE);
               }
          }
          return(ARCFILE);
     }
     if (recbuf[0] == 'M'
      && recbuf[1] == 'Z'
      && recbuf[2] == 'R'
      && recbuf[3] == 0x01) {
          return(PSEFILE);
     }
     if (recbuf[0] == 'P'
      && recbuf[1] == 'K'
      && ((recbuf[2] == 0x03
      && recbuf[3] == 0x04)
      || (recbuf[2] == 0x07
      && recbuf[3] == 0x08))) {
          return(recbuf[4] < 0x14 ? ZIPFILE : ZP2FILE);
     }
     if (recbuf[0] == 'M'
      && recbuf[1] == 'Z') {
          if (recbuf[2] == 0xEF
            && recbuf[3] == 0x01
            && recbuf[0x32] == 'P'
            && recbuf[0x33] == 'K') {
               return(ZSEFILE);
          }
          if ((recbuf[2] == 0xBA
            && recbuf[3] == 0x01)
            || (recbuf[2] == 0xF4
            && recbuf[3] == 0x01
            && recbuf[0x1E] == 'P'
            && recbuf[0x1F] == 'K'
            && recbuf[0x20] == 'L')) {
               return(ZS2FILE);
          }
     }
     if (recbuf[2] == '-'
      && recbuf[3] == 'l'
      && recbuf[4] == 'h'
      && recbuf[5] >= '1' && recbuf[5] <= '9'
      && recbuf[6] == '-') {
          return(ICEFILE);
     }
     return(UNKFILE);
}

/*--- The FTF.H interface for 'V' protocol output ---*/

STATIC void
vewini(void)                                           /* Initialize Viewer */
{
     if (fbleng < max(VIDCHUNK,ZP2CHUNK)) {
          fbleng=max(VIDCHUNK,ZP2CHUNK);
     }
}

STATIC void
vewsrt(void)                                      /* Begin viewing sesssion */
{
     setmem(vewscb,sizeof(*vewscb),0);
     ftfnew(VEWOPN);
}

STATIC void
vewins(                         /* Handle incoming line during View session */
char *stg)
{
     ftfabt(stg == NULL ? "" : "Operator abort");
}

STATIC int
vewctn(void)                                            /* Continue Viewing */
{
     int rc;

     switch (ftfscb->state) {
     case VEWOPN:
          if (ftfxop() == 0) {
               ftfsrt();
               ftfnew(VEWFID);
          }
          else {
               ftfabt("Cannot find file");
          }
          break;
     case VEWFID:
          if (ftfxrd(ftfbuf,VIDCHUNK) < VIDCHUNK
           || (vewscb->filid=zafilid(ftfbuf)) == UNKFILE) {
               ftfabt(arcerr[0]);
          }
          else {
               if ((vewscb->filid == ZP2FILE || vewscb->filid != ZS2FILE)
                 && ftfbuf[2] == 0x07 && ftfbuf[3] == 0x08) {
                    ftfabt("Unable to view zip files spanning multiple disks");
                    return(1);
               }
               ftfous(vewhdr);
               vewscb->pos=0L;
               if (vewscb->filid != ZP2FILE && vewscb->filid != ZS2FILE) {
                    ftfnew(VEWFIL);
               }
               else {
                    if (ftfscb->estbyt > (long)ZP2CHUNK) {
                         vewscb->pos=ftfscb->estbyt-(long)ZP2CHUNK;
                    }
                    ftfnew(VEWFND);
               }
          }
          break;
     case VEWFIL:
          if (ftfoba() < 100) {
               break;
          }
          ftfxsk(vewscb->pos);
          if (ftfxrd(ftfbuf,VEWCHUNK) == 0) {
               ftfabt(arcerr[vewscb->filid]);
               break;
          }
          switch(vewscb->filid) {
          case ARCFILE:
          case PSEFILE:
               rc=arprfr();
               break;
          case ZIPFILE:
          case ZSEFILE:
          case ZP2FILE:
          case ZS2FILE:
               rc=zpprfr();
               break;
          case ICEFILE:
               rc=icprfr();
               break;
          case ZOOFILE:
               rc=zoprfr();
               break;
          default:
               ftfabt("Invalid type-code for compressed file");
               rc=0;
               break;
          }
          if (rc == -1) {
               ftfabt(arcerr[vewscb->filid]);
          }
          else if (rc == 1) {
               ftfnew(VEWEND);
          }
          ftfscb->tckact=ftftck;
          ftfact=1;
          break;
     case VEWFND:
          ftfxsk(vewscb->pos);
          if (ftfxrd(ftfbuf,
              ftfscb->estbyt > (long)ZP2CHUNK ? ZP2CHUNK
                                              : (int)ftfscb->estbyt) == 0) {
               ftfabt(arcerr[vewscb->filid]);
               break;
          }
          z2fend();
          break;
     case FTFABORT:
          ftfxca();
          return(-1);
     case VEWEND:
          if (ftfoba() == ftfomt) {
               ftfxcl(1);
               ftfscb->isopen=0;
               ftfscb->actfil++;
               return(0);
          }
          break;
     }
     return(1);
}

STATIC void
z2fend(void)                      /* find end of central dir record in ZIP2 */
{
     char *ptr;
     int cnt=0;
     long oldpos;
     int byts;

     oldpos=vewscb->pos;
     ptr=ftfbuf;
     byts=ftfscb->estbyt > (long)ZP2CHUNK ? ZP2CHUNK : (int)ftfscb->estbyt;
     while (cnt <= byts-CDRLEN) {
          if (*ptr == 'P'
            && *(ptr+1) == 'K'
            && *(ptr+2) == 5
            && *(ptr+3) == 6) {
               ehdr=(struct endcdr *)&ftfbuf[cnt];
               vewscb->pos=ehdr->ctroff;
               ftfnew(VEWFIL);
               return;
          }
          ptr++;
          cnt++;
     }
     if (oldpos != 0L) {
          vewscb->pos-=(long)(ZP2CHUNK-CDRLEN);
          if (vewscb->pos < 0L) {
               vewscb->pos=0L;
          }
     }
     else {
          ftfabt(arcerr[vewscb->filid]);
     }
}

struct ftfpsp ftpvew={                /* ARC/ZIP/etc file View transmitting */
     NULL,
     "V",                                  /* 1-3 code letters for protocol */
     "View compressed file",                            /* name of protocol */
     FTFXMT+FTFASC+FTF7BT+FTFAFN+FTFXTD,       /* protocol capability flags */
     sizeof(struct vewdat),        /* total length of session control block */
     0,         /* .byttmo                             default byte timeout */
     0,         /* .paktmo                           default packet timeout */
     0,         /* .retrys                              default max retries */
     0L,        /* .window   max window size (packets/bytes as appropriate) */
     0,         /* .paksiz                        packet size 0=auto-figure */
     vewini,    /* .initze()    Initialize this protocol (recompute scblen) */
     vewsrt,    /* .start()                                Start a transfer */
     vewctn,    /* .contin()              Continuously call, 1=more, 0=done */
     NULL,      /* .hdlinc()                       Handle one incoming byte */
     vewins,    /* .hdlins()                   Handle incoming line of text */
     ftfabt,    /* .term()        Initiate graceful termination of transfer */
     ftfxca,    /* .abort()  Immediately unconditionally abort the transfer */
     ftfinbc,   /* .hdlinb()              Handle an array of incoming bytes */
     "",        /* .secur                App-specific security of some kind */
     {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}
};
