/*
*********************************************************************
* hwman.inc
* definitions for the "hwman" module.
*
* To access function prototypes, just "include".
*
* To access external variables, define "HWMAN_EXTERNAL" before including.
*
* To allocate storage define "HWMAN_GLOBAL" before including.
*********************************************************************
*/
#ifndef hwman_inc
	#define hwman_inc

	/*
	****************************************************************
	* constants and structures
	****************************************************************
	*/

	/*
	*********************************************
	* HWMAN.DAT
	* This file contains the account number/name correspondence.
	* Multiple records may exist for the "same account", in order
	* to provide "alias" services; ie one account with multiple
	* names.
	*********************************************
	*/
	typedef struct
	{
		char		Name[UIDSIZ];
						// the name of the person.
		long		Number;
						// the alias number for the person. Values
						// are always positive, and are assigned
						// sequentially if necessary.
		long		OrigNumber;
						// if this is an "alias", then this is the
						// original account number.
						// If this is the "original number", then
						// this is zero.
		unsigned int
					CreateDate;
						// date this record was created. just FYI, not
						// processed in any way.
		unsigned int
					DeleteDate;
						// this is the date that the original account
						// was deleted. Records are NOT deleted immediately,
						// so that past history, statistics, etc, can be
						// found.
						// Every so often, an offline program can be used
						// to purge records that have been "deleted more then
						// NN days ago".
	} AN_RECORD;

	// methods (indices) of reading the above record from the btrieve data file
	#define AN_NUMBER 0
	typedef struct
	{
		unsigned long 	Number;
	} AN_NUMBER_KEY;

	#define AN_NAME (AN_NUMBER + 1)
	typedef struct
	{
		char			Name[UIDSIZ];
		unsigned long 	Number;
	} AN_NAME_KEY;

	#define AN_ORIG (AN_NAME + 1)
	typedef struct
	{
		unsigned long	OrigNumber;
	} AN_ORIG_KEY;

	typedef union
	{
		AN_NAME_KEY			Key0;
		AN_NUMBER_KEY		Key1;
		AN_ORIG_KEY			Key2;
	} AN_KEY_BUFF;

	/*
	*********************************************
	* AnControl
	* These are ram-resident control tables, used to keep current
	* info for on-line users.
	*********************************************
	*/
	typedef struct
	{
		AN_RECORD		Record;
							// This is a copy of the record as read from
							// disk
	} AN_CONTROL;

	/*
	*********************************************
	* user & control stuff
	* Each user has a copy of this structure, keeping track of
	* their present status and information.
	* This is part of the "vda" as defined by MBBS.
	*********************************************
	*/
	typedef struct
	{
		AN_RECORD		Record;
							// current record being listed, worked with, etc.
		int				Count;
		char			ListType;
	} AN_USER;


	/*
	****************************************************************
	* function prototypes
	****************************************************************
	*/
	void 				init__AccountNumber (void);
	int 				an_GlobalCmd (void);

	// module-level functions
	int 				an_LogOff (void);
	int 				an_LogOn (void);
	int 				an_Exec (void);
	void 				an_Status (void);
	void 				an_HangUp (void);
	void 				an_AcctDel (char *UserName);
	void 				an_ShutDown(void);
	void 				an_MainMenuDisplay (void);

	// general service routines
	int 				an_Exit (int Status, int Message1, int Message2);
	void 				an_ListFileDone (int Status);
	void 				an_substt (int Status, int Message1, int Message2);
	void 				an_UserSet (void);
	void 				an_UserClear (void);

	// list record
	void 				an_ListSequence (void);
	void 				an_ListStart (void);
	void 				an_ListRecord (void);

	int 				an_BrowseFieldAlloc (void);
	int 				an_BrowseCommand (char *CmdName);
	void 				an_BrowseStart (void);
	int 				an_BrowseIndexIntercept (int Tag, int KeyValue);

	void 				an_DataKey (AN_RECORD *Record, AN_KEY_BUFF *KeyBuff, int IndexValue);
	int 				an_DataNext (AN_RECORD *Record, int IndexValue);
	int 				an_DataPrev (AN_RECORD *Record, int IndexValue);
	int 				an_DataRead (AN_RECORD *Record, int IndexValue);
	int 				an_DataWrite (AN_RECORD *Record, int IndexValue, int WriteType);

	void 				an_OpenInit (void);
	long				an_AcctNumber (char *UserID, int CreateFlag);
	long 				an_AcctNumberFile (char *UserID, int CreateFlag);
	char				*an_AcctID (long AcctNum);
	long 				an_CurrentAcctNum (void);

	/*
	*****************************************************************
	* macro function/value definitions
	*****************************************************************
	*/

	/*
	*****************************************************************
	* global variable declarations
	*****************************************************************
	*/
	#ifdef HWMAN_GLOBAL
		int
			an_ModuleNumber;
		struct module an_Module =
		{
			"",							/* char *descrp */
										/*    description for main menu         */
			an_LogOn,					/* int (*lonrou)() */
										/*    user logon supplemental routine   */
			an_Exec,					/* int (*sttrou)() */
										/*    input routine if selected         */
			an_Status,					/* int (*stsrou)()
										/*    status-input routine if selected  */
			NULLPTR,					/* int (*injrou)() */
										/*    "inject other" channel routine" */
			an_LogOff,					/* int (*lofrou)() */
										/*    log-off routine     */
			an_HangUp,					/* int (*huprou)() */
										/*    hangup (lost carrier) routine     */
			NULLPTR,					/* int (*mcurou)() */
										/*    midnight cleanup routine          */
			an_AcctDel,					/* int (*dlarou)() */
										/*    delete-account routine            */
			an_ShutDown					/* int (*finrou)() */
										/*    finish-up (sys shutdown) routine  */
		};

		FILE
			*an_MsgFile = NULLPTR;

		AN_USER
			*an_UserPtr = NULLPTR;
		AN_RECORD
			*an_UserRecord = NULLPTR;

		BTVFILE
			*an_DataFile = NULLPTR;
		AN_KEY_BUFF
			an_DataKeyBuff;

		AN_CONTROL
			*an_Control;
	#endif

	#ifdef HWMAN_DOS
		FILE
			*an_MsgFile = NULLPTR;
		BTVFILE
			*an_DataFile = NULLPTR;
		AN_KEY_BUFF
			an_DataKeyBuff;
	#endif

	/*
	*****************************************************************
	* external variable declarations
	*****************************************************************
	*/
	#ifdef HWMAN_EXTERNAL
		extern int
			an_ModuleNumber;
		extern struct module
			an_Module;

		extern FILE
			*an_MsgFile;

		extern AN_USER
			*an_UserPtr;

		extern AN_CONTROL
			*an_Control;
	#endif
#endif
