/***************************************************************************
 *                                                                         *
 *   FTFZMOD.C                                                             *
 *                                                                         *
 *   Copyright (C) 1991-1993 GALACTICOMM, Inc.      All Rights Reserved.   *
 *                                                                         *
 *   File Transfer Software for ZMODEM                                     *
 *                                                                         *
 *   With all operating-system-specific functions removed (I/O, disk,      *
 *   memory, time), the pure file transfer algorithm can be isolated,      *
 *                                                                         *
 *                                               - R. Stein  9/04/91       *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "ftf.h"
#include "ftfzmod.h"
#include "crctab.h"               /* ZMODEM CRC tables from Omen Technology */

#undef  UPDC32            /* (recast from CRCTAB.H to avoid compiler error) */
#define UPDC32(b, c) (cr3tab[((int)c ^ b) & 0xff] ^ ((c >> 8) & 0x00FFFFFFL))


/*--- ZMODEM Transmit substates ---*/
#define ZDINIT    1                                     /* start of session */
#define ZDRQINIT  2                 /* very beginning, need to send ZRQINIT */
#define ZDRINIT   3              /* waiting for ZRINIT header from receiver */
#define ZDRPOS    4         /* waiting for first ZRPOS header from receiver */
#define ZDGOING   5   /* transmiting, but with ear cocked for receiver msgs */
#define ZDEOF     6                               /* transmitted last block */
#define ZDNEXT    7                 /* transmitted ZEOF, waiting for ZRINIT */
#define ZDFIN     8               /* transmitted ZFIN, waiting or ZFIN back */
#define ZDCRCING  9       /* computing file CRC in response to ZCRC command */
#define ZDEND     10     /* the end of a successful ZMODEM transmit session */
#define ZDFROWN   11      /* wait for goofy Telix ZFIN's to stop after xmit */

/*--- ZMODEM Receive substates ---*/
#define ZUINIT    1
#define ZURINIT   2                 /* Begin receiving file(s), send ZRINIT */
#define ZUFILE    3             /* Sent ZRINIT, waiting for ZFILE (or ZFIN) */
#define ZUSINIT   4                /* Got ZSINIT header, waiting for packet */
#define ZUGOING   5             /* Got ZDATA header, receiving data packets */
#define ZULOGGED  6                            /* Got ZEOF, logging in file */
#define ZUOOUT    7
#define ZUEND     8


/* ZMODEM constants (excerpts from Omen Technology's ZMODEM.H) */
/*-------------------------------------------------------------*/
#define ZGARBAGE 0xFF   /* header had bad CRC (or ZCRCW) */
#define ZRQINIT 0       /* Request receive init */
#define ZRINIT  1       /* Receive init */
#define ZSINIT 2        /* Send init sequence (optional) */
#define ZACK 3          /* ACK to above */
#define ZFILE 4         /* File name from sender */
#define ZSKIP 5         /* To sender: skip this file */
#define ZNAK 6          /* Last packet was garbled */
#define ZABORT 7        /* Abort batch transfers */
#define ZFIN 8          /* Finish session */
#define ZRPOS 9         /* Resume data trans at this position */
#define ZDATA 10        /* Data packet(s) follow */
#define ZEOF 11         /* End of file */
#define ZFERR 12        /* Fatal Read or Write error Detected */
#define ZCRC 13         /* Request for file CRC and response */
#define ZCHALLENGE 14   /* Receiver's Challenge */
#define ZCOMPL 15       /* Request is complete */
#define ZCAN 16         /* Other end canned session with CAN*5 */
#define ZFREECNT 17     /* Request for free bytes on filesystem */
#define ZCOMMAND 18     /* Command from sending program */
#define ZSTDERR 19      /* Output to standard error, data follows */

#define ZDLE '\x18'
/* other ZDLE sequences */
#define ZCRCE 'h'       /* CRC next, frame ends, header packet follows */
#define ZCRCG 'i'       /* CRC next, frame continues nonstop */
#define ZCRCQ 'j'       /* CRC next, frame continues, ZACK expected */
#define ZCRCW 'k'       /* CRC next, ZACK expected, end of frame */
#define ZRUB0 'l'       /* Translate to rubout 0177 */
#define ZRUB1 'm'       /* Translate to rubout 0377 */

#define ZF0     3       /* First flags byte */
#define ZF1     2
#define ZF2     1
#define ZF3     0

/* Bit Masks for ZRINIT flags byte ZF0 */
#define CANFC32 040     /* Receiver can use 32 bit Frame Check */
#define ESCCTL 0100     /* Receiver expects ctl chars to be escaped */

/* Parameters for ZSINIT frame */
/* Bit Masks for ZSINIT flags byte ZF0 */
#define TESCCTL 0100    /* Transmitter expects ctl chars to be escaped */

/* Parameters for ZFILE frame */
/* Conversion options one of these in ZF0 */
#define ZCBIN   1       /* Binary transfer - inhibit conversion */
#define ZCNL    2       /* Convert NL to local end of line convention */
#define ZCRESUM 3       /* Resume interrupted file transfer */
/* Management include options, one of these ored in ZF1 */
#define ZMSKNOLOC       0200    /* Skip file if not present at rx */
/* Management options, one of these ored in ZF1 */
#define ZMMASK  037     /* Mask for the choices below */
#define ZMNEWL  1       /* Transfer if source newer or longer */
#define ZMCRC   2       /* Transfer if different file CRC or length */
#define ZMAPND  3       /* Append contents to existing file (if any) */
#define ZMCLOB  4       /* Replace existing file */
#define ZMNEW   5       /* Transfer if source newer */
        /* Number 5 is alive ... */
#define ZMDIFF  6       /* Transfer if dates or lengths different */
#define ZMPROT  7       /* Protect destination file */
/* Transport options, one of these in ZF2 */
#define ZTLZW   1       /* Lempel-Ziv compression */
#define ZTCRYPT 2       /* Encryption */
#define ZTRLE   3       /* Run Length encoding */
/* Extended options for ZF3, bit encoded */
#define ZXSPARS 64      /* Encoding for sparse file operations */

#define ZBLOCK 1024               /* maximum nominal ZMODEM sub-packet size */
         /* IMPORTANT:  fbleng (size of ftfbuf[]) MUST BE AT LEAST ZBLOCK*2 */

#define ZBRECOVER 512           /* Block size after ZMODEM transmit recover */
 /* Note: Omen's DSZ seems to need 1st block 512 bytes to recover transmits */

#define ZBEOF  12     /* maximum size of xmit ZEOF header (type 'A' or 'C') */
#define ZCHUNK 1024     /* min number of bytes to receive & process at once */
                     /* also num of bytes chunked from file to compute ZCRC */
#define ZABWAIT  (1*16)       /* number of ticks to wait after ZMODEM abort */
#define ZURETRY (10*16)     /* # seconds before retry during ZMODEM receive */
#define ZOOWAIT (10*16)    /* number of seconds to wait for "OO" after ZFIN */
#define ZDRQWAIT (5*16)             /* repeat ZRQINIT messages for transmit */
#define ZDFRETRY (1*16)   /* pause before responding to 2nd ZRINIT for xmit */
#define ZDFWAIT  (1*16)   /* wait for goofy Telix ZFIN's after "OO" on xmit */
#define ZDEWAIT     (8)       /* wait for xmits to finish at end of session */

#define rawbuf (ftfbuf+ZBLOCK)              /* buffer for input to zsdata() */


/*--- ZMODEM header transmit variables and macros ---*/

static unsigned char dummych;         /* 1-char buffer for sendline() macro */
                              /* send a single character (no escape-coding) */
#define sendline(ch) (dummych=(ch),ftfout(&dummych,1))

                            /* zescape() - control character escaping macro */
       /* (note: no complex check for CR - '@' - CR:  all CR's are escaped) */
#define zescape(ch) ((ch&0x60) == 0 && (zctlesc || cetabl[ch&0x1F]))
static char cetabl[32]={0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,              /* ^M */
                        1,1,0,1,0,0,0,0,1,0,0,0,0,0,0,0};    /* ^P ^Q ^S ^X */
static char zctlesc;                          /* global for zescape() macro */


/*--- ZMODEM header receive variables and macros ---*/

unsigned char *zptr,*eptr;         /* pointers for scanning received header */
unsigned char rtype;                       /* frame type of received header */
unsigned char rhdr[8];                 /* received header (from zget?hdr()) */
int badcrc;                                 /* set if crc of header was bad */
#define ranout() (zptr > eptr)          /* Did readline() ever return a -1? */


/*--- ZMODEM header transmit utilities (used both for file xmt & file rcv) ---*/

STATIC void
zsendline(                  /* send a zmodem character (with escape coding) */
register unsigned char ch)
{
     if (zescape(ch)) {
          sendline(ZDLE);
          sendline(ch^0x40);
     }
     else {
          sendline(ch);
     }
}

STATIC unsigned
upfcrc(                      /* function form of updcrc() macro in CRCTAB.H */
unsigned char c,
unsigned crc)
{
     return(updcrc(c,crc));
}

STATIC long
upfc32(                      /* function form of UPDC32() macro in CRCTAB.H */
unsigned char c,
long crc)
{
     return(UPDC32(c,crc));
}

STATIC void
zsbhdr(                                 /* Send ZMODEM binary-format header */
unsigned char type,                                      /* Frame type code */
unsigned char *hdr)        /* ZF3,...,ZF0 or ZP0,...,ZP3 depending on frame */
{
     int i;
     unsigned crc;
     unsigned long lcrc;

     sendline('*');                                                    /* * */
     sendline(ZDLE);                                                /* ZDLE */
     zctlesc=(zmxscb->flags&ZCTLESC) != 0;
     if (zmxscb->flags&ZCRC32) {           /* binary header with 32-bit CRC */
          sendline('C');                                               /* C */
          lcrc=0xFFFFFFFFL;
          lcrc=upfc32(type,lcrc);
          zsendline(type);                                        /* <type> */
          for (i=0 ; i < 4 ; i++,hdr++) {
               lcrc=upfc32(*hdr,lcrc);
               zsendline(*hdr);                  /* <zf3> <zf2> <zf1> <zf0> */
          }
          lcrc=~lcrc;
          for (i=0 ; i < 4 ; i++) {
               zsendline((unsigned char)lcrc);   /* <crc-lo> . . . <crc-hi> */
               lcrc>>=8;
          }
     }
     else {                                /* binary header with 16-bit CRC */
          sendline('A');                                               /* A */
          crc=upfcrc(type,0);
          zsendline(type);                                        /* <type> */
          for (i=0 ; i < 4 ; i++,hdr++) {
               crc=upfcrc(*hdr,crc);
               zsendline(*hdr);                  /* <zf3> <zf2> <zf1> <zf0> */
          }
          crc=upfcrc(0,upfcrc(0,crc));
          zsendline(crc>>8);                                   /* <crc-hi> */
          zsendline(crc);                                      /* <crc-lo> */
     }
}

STATIC void
zshhdr(                                    /* Send ZMODEM HEX-format header */
unsigned char type,                                      /* Frame type code */
unsigned char *hdr)        /* ZF3,...,ZF0 or ZP0,...,ZP3 depending on frame */
{
     char rqbuff[40],*rp;
     int i;
     unsigned crc;

     crc=upfcrc(type,0);
     sprintf(rqbuff,"**%cB%02x",ZDLE,type);            /* * * ZDLE B <type> */
     rp=rqbuff+strlen(rqbuff);
     for (i=0 ; i < 4 ; i++,hdr++,rp+=2) {
          crc=upfcrc(*hdr,crc);
          sprintf(rp,"%02x",*hdr);               /* <zf3> <zf2> <zf1> <zf0> */
     }
     crc=upfcrc(0,upfcrc(0,crc));
     sprintf(rp,"%02x%02x\r\n",(crc>>8)&0xFF,crc&0xFF);
                                             /* <crc-hi> <crc-lo> <CR> <LF> */
     if (type != ZFIN && type != ZACK) {
          strcat(rqbuff,"\x11");                                   /* <XON> */
     }
     ftfout(rqbuff,strlen(rqbuff));
}


/*--- ZMODEM header receive utilities (used both for file xmt & file rcv) ---*/

STATIC int
readline(void)                      /* returns next char (or -1 if no more) */
{
     int c;

     c=*zptr;
     zptr++;
     return(ranout() ? -1 : c);
}

STATIC int
zdlread(void)                       /* read a header byte from input buffer */
{                             /* return de-escaped byte or -1=no more bytes */
     int c;

     if ((c=readline()) == ZDLE) {
          switch(c=readline()) {
          case -1:
               break;
          case ZRUB0:
               c=0x7F;
               break;
          case ZRUB1:
               c=0xFF;
               break;
          default:
               c^=0x40;
               break;
          }
     }
     return(c);
}

STATIC int
zdhread(void)                                      /* read hexadecimal byte */
{
     static char buf[3]="00";
     int c0,c1,byt;

     if ((c0=readline()) == -1 || (c1=readline()) == -1) {
          return(-1);
     }
     buf[0]=c0;
     buf[1]=c1;
     sscanf(buf,"%x",&byt);
     return(byt);
}

STATIC int
zgetahdr(void)             /* Decode ZMODEM "A" header (16-bit crc, binary) */
{                                                  /* returns 1=good, 0=bad */
     unsigned crc;
     int c;
     int i;

     zptr=zmxscb->hdrbuf;
     eptr=zptr+zmxscb->numhdr;
     if (readline() != 'A' || (c=zdlread()) == -1) {
          return(0);
     }
     rtype=(unsigned char)c;
     for (i=0 ; i < 6 ; i++) {
          if ((c=zdlread()) == -1) {
               return(0);
          }
          rhdr[i]=(unsigned char)c;
     }
     crc=upfcrc(rtype,0);
     for (i=0 ; i < 6 ; i++) {
          crc=upfcrc(rhdr[i],crc);
     }
     badcrc=crc != 0;
     return(!badcrc);
}

STATIC int
zgetbhdr(void)                /* Decode ZMODEM "B" header (16-bit crc, hex) */
{                                                  /* returns 1=good, 0=bad */
     unsigned crc;
     int c;
     int i;

     zptr=zmxscb->hdrbuf;
     eptr=zptr+zmxscb->numhdr;
     if (readline() != 'B' || (c=zdhread()) == -1) {
          return(0);
     }
     rtype=(unsigned char)c;
     for (i=0 ; i < 6 ; i++) {
          if ((c=zdhread()) == -1) {
               return(0);
          }
          rhdr[i]=c;
     }
     if ((readline()&0x7F) != '\r' || (readline()&0x7F) != '\n') {
          return(0);
     }
     crc=upfcrc(rtype,0);
     for (i=0 ; i < 6 ; i++) {
          crc=upfcrc(rhdr[i],crc);
     }
     badcrc=(crc != 0);
     if (badcrc) {
          return(0);
     }
     else {
          zmxscb->flags&=~ZGOTH32;
          return(1);
     }
}

STATIC int
zgetchdr(void)             /* Decode ZMODEM "C" header (32-bit crc, binary) */
{                                                  /* returns 1=good, 0=bad */
     unsigned long crc;
     int c;
     int i;

     zptr=zmxscb->hdrbuf;
     eptr=zptr+zmxscb->numhdr;
     if (readline() != 'C' || (c=zdlread()) == -1) {
          return(0);
     }
     rtype=(unsigned char)c;
     for (i=0 ; i < 8 ; i++) {
          if ((c=zdlread()) == -1) {
               return(0);
          }
          rhdr[i]=c;
     }
     crc=0xFFFFFFFFL;
     crc=upfc32(rtype,crc);
     for (i=0 ; i < 8 ; i++) {
          crc=upfc32(rhdr[i],crc);
     }
     badcrc=crc != 0xDEBB20E3L;
     if (badcrc) {
          return(0);
     }
     else {
          zmxscb->flags|=ZGOTH32;
          return(1);
     }
}

STATIC int
zmhchr(          /* returns:  1=got header (or cancel); 0=nothing to report */
unsigned char raw)                                   /* next raw input byte */
                        /* if returns 1, rtype is received header type code */
            /* zmxscb->numhdr is state code and has the following meanings: */
                                   /* n>=0 number of characters in hdrbuf[] */
                 /* -6,-5,-4,-3 means 4,3,2,1 ZDLE's (CAN's) have been rcvd */
                                          /* -2 is quiescent (wait for pad) */
                                        /* -1 means wait for ZDLE after pad */
{
     int state;                               /* zmxscb->numhdr temporarily */
     int rw7;                              /* raw input with bit 7 stripped */
     int rc=0;                                               /* return code */

     if ((rw7=raw&0x7F) == 'S'-64) {                  /* got XOFF character */
          zmxscb->flags|=ZXOFF;
          ftfxlk(1);                                    /* shut off output! */
          return(0);                                          /* (chuck it) */
     }
     else if (zmxscb->flags&ZXOFF) {     /* any character serves as un-XOFF */
          zmxscb->flags&=~ZXOFF;
          ftfxlk(0);                                       /* resume output */
     }
     if (rw7 == 0x11 || ((zmxscb->flags&ZCTLESC) &&
                                     !(rw7&0x60) &&
                                     raw != ZDLE && rw7 != '\r' && rw7 != '\n')) {
          return(0);              /* other than as an un-XOFF, ignore XON's */
     }                   /* and ignore ctrl chars if they should be escaped */
     switch(state=zmxscb->numhdr) {
     case -6:                                       /* got 4 ZDLEs in a row */
          switch(raw) {
          case '*':                                    /* a pad? start over */
               state=-1;
               break;
          case ZDLE:                           /* a fifth ZDLE means cancel */
               rtype=ZCAN;
               state=-2;
               rc=1;
               break;
          default:                             /* anything else, start over */
               state=-2;
               break;
          }
          break;
     case -5:                                       /* got 3 ZDLEs in a row */
     case -4:                                       /* got 2 ZDLEs in a row */
     case -3:                                 /* got 1 ZDLE out of the blue */
          if (raw == ZCRCW) {         /* detect sub-packet from receiver?!? */
               state=-2;
               rtype=ZCRCW;
               rc=1;
          }
          /* fall through */
     case -2:                    /* QUIESCENT STATE:  waiting for '*' (pad) */
          switch(raw) {
          case '*':                                       /* here's the pad */
               state=-1;
               break;
          case ZDLE:                                  /* multiple ZDLE's... */
               state--;                                     /* count 'em up */
               break;
          default:                             /* anything else, start over */
               state=-2;
               break;
          }
          break;
     case -1:                                  /* got '*', waiting for ZDLE */
          switch(raw) {
          case ZDLE:                                     /* here's the ZDLE */
               state=0;
               break;
          case '*':                               /* duplicate pad's are ok */
               break;
          default:                             /* anything else, start over */
               state=-2;
               break;
          }
          break;
     case 0:              /* got '*' and ZDLE, expecting 1st char of header */
          if (raw < 'A' || raw > 'C') {
               state=-2;                            /* this is not a header */
               break;
          }
          /* fall thru, store A/B/C in hdrbuf[] */
     default:                      /* receiving header, start with A B or C */
          if (state >= MAXHDR) {                       /* (header too long, */
               state=-2;                                     /* start over) */
               break;
          }
          zmxscb->hdrbuf[state++]=raw;
          zmxscb->numhdr=state;
          badcrc=0;
          if (!zgetahdr() && !ranout()      /* could ever be type A header? */
           && !zgetbhdr() && !ranout()      /* could ever be type B header? */
           && !zgetchdr() && !ranout()) {   /* could ever be type C header? */
               state=-2;     /* NO NO NO: header makes no sense, start over */
          }
          else if (!ranout()) {                    /* we got a good header! */
               state=-2;                     /* (start receiver over again) */
               rc=1;
          }        /* (otherwise, possibly good header is still incomplete) */
          break;
     }
     zmxscb->numhdr=state;
     return(rc);
}



/*--- ZMODEM Receive File routines ---*/

STATIC void
zmrini(void)                                  /* Initialize ZMODEM receiver */
{
     if (fbleng < max(ZBLOCK*2,ZCHUNK)) {
          fbleng=max(ZBLOCK*2,ZCHUNK);
     }
}

STATIC void
zmrsrt(void)                                /* Begin ZMODEM receive session */
{
     setmem(zmrscb,sizeof(*zmrscb),0);
     ftfstf();
     zmrscb->numhdr=-2;                    /* quiescent zmhchr() state code */
     ftfnew(ZUINIT);
}

unsigned char ptype;    /* global return var for zmpchr():   ZCRCE,..,ZCRCW */

STATIC int
zmpchr(                             /* receive next byte of data sub-packet */
int raw)                                             /* raw input character */
              /* zmrscb->pstate == 0 when not expecting packets, otherwise: */
   /* 1=receiving, 2=got ZCRCx terminator, 3-6=getting crc, -n=ZDLE escaped */
        /* zmrscb->numpkt == 0 to ZINSIZE-1 as packet collected in packet[] */
  /* if returns 1, ptype is packet terminator type (or ZGARBAGE if bad crc) */
{
     int ch;                                        /* un-escaped character */
     register int i;
     register unsigned char c;
     unsigned char *cp;
     unsigned crc;
     unsigned long lcrc;
     int iterm;                /* index of terminator character in packet[] */
     int badcrc;
     int gottrm=0;                       /* ZCRCx terminator just received! */

     if (zmrscb->pstate == 0) {                      /* no packets expected */
          return(0);
     }
     switch (ch=raw) {
     case '\x11':
     case '\x13':
     case '\x91':
     case '\x93':
          return(0);                    /* always ignore ^S=XOFF and ^Q=XON */
     default:
          if ((zmrscb->flags&ZCTLESC) && !(ch&0x60)) {
               return(0); /* ignore control chars if they should be escaped */
          }
     case ZDLE:
          break;
     }
     if (zmrscb->pstate < 0) {           /* pstate < 0 (last char was ZDLE) */
          zmrscb->pstate=-zmrscb->pstate;
          switch(raw) {
          case ZRUB0:
               ch=0x7F;
               break;
          case ZRUB1:
               ch=0xFF;
               break;
          case ZCRCE:
          case ZCRCG:
          case ZCRCQ:
          case ZCRCW:
               gottrm=1;
               break;         /* note that ZCRCx is stored in packet[] also */
          default:
               ch^=0x40;
          }
     }
     else {                          /* pstate > 0 (last char was not ZDLE) */
          if (raw == ZDLE) {                              /* THIS is a ZDLE */
               zmrscb->pstate=-zmrscb->pstate;
          }
     }
     if (zmrscb->numpkt >= ZINSIZE) {               /* input buffer overrun */
          zmrscb->pstate=0;
          zmrscb->numpkt=0;                  /* throw out all of the buffer */
          ptype=ZGARBAGE;                       /* and send a ZNAK or ZRPOS */
          return(1);
     }
     if (zmrscb->pstate > 0) {
          zmrscb->packet[zmrscb->numpkt++]=ch;    /* store the decoded char */
          if (gottrm || zmrscb->pstate >= 2) {  /* (part of the terminator) */
               zmrscb->pstate++;                         /* count crc bytes */
          }
          iterm=0;
          if (zmrscb->flags&ZGOTH32) {                     /* 32-bit CRC's: */
               if (zmrscb->pstate == 6) {                 /* end of packet? */
                    lcrc=0xFFFFFFFFL;
                    for (i=0,cp=zmrscb->packet ; i < zmrscb->numpkt ; i++,cp++) {
                         c=*cp;
                         lcrc=UPDC32(c,lcrc);
                    }
                    badcrc=(lcrc != 0xDEBB20E3L);
                    iterm=zmrscb->numpkt-5;
               }
          }
          else {                                           /* 16-bit CRC's: */
               if (zmrscb->pstate == 4) {                 /* end of packet? */
                    crc=0;
                    for (i=0,cp=zmrscb->packet ; i < zmrscb->numpkt ; i++,cp++) {
                         c=*cp;
                         crc=updcrc(c,crc);
                    }
                    badcrc=(crc != 0);
                    iterm=zmrscb->numpkt-3;
               }
          }
          if (iterm) {                     /* end of either type of packet? */
               if (badcrc) {                                        /* GACK */
                    ptype=ZGARBAGE;
                    zmrscb->numpkt=0;
               }
               else {                                       /* good packet! */
                    ptype=zmrscb->packet[iterm];           /* extract ZCRCx */
                    zmrscb->numpkt=iterm;               /* strip terminator */
               }
               return(1);
          }
     }
     return(0);
}

STATIC void
svcuhdr(void)              /* service recieved header during ZMODEM receive */
{
     if (ftfscb->state == FTFABORT
      || ftfscb->state == FTFABWAIT) {
          if (rtype == ZFIN) {     /* handle silly Telix's ZFIN after abort */
               zshhdr(ZFIN,"\0\0\0\0");
          }
          return;
     }
     zmrscb->pstate=0;    /* assume (for the moment) that no packets follow */
     zmrscb->numpkt=0;
     switch(rtype) {                              /* easily handled headers */
     case ZABORT:
          ftfabt("Sender aborted.");
          return;
     case ZCAN:
          ftfabt("Sender or operator CTRL-X abort.");
          return;
     case ZGARBAGE:
          ftfscb->garbag++;
          ftfscb->stecnt++;
          zshhdr(ZNAK,"\0\0\0\0");
          return;
     }
     switch(ftfscb->state) {
     case ZUSINIT:
     case ZUFILE:
          switch(rtype) {
          case ZRQINIT:              /* aha!  sender just now coming awake! */
               zshhdr(ZRINIT,"\0\0\0\43");        /* CANFC32+CANOVIO+CANFDX */
               ftfnew(ZUFILE);
               break;
          case ZSINIT:                       /* get ready for SINIT message */
               (rhdr[ZF0]&TESCCTL)  ? (zmrscb->flags|=ZCTLESC)
                                    : (zmrscb->flags&=~ZCTLESC);
               ftfnew(ZUSINIT);
               zmrscb->pstate=1;
               break;
          case ZFILE:                    /* sender sending a file and stuff */
               ftfscb->stecnt=0;
               zmrscb->sopt0=rhdr[ZF0];
               zmrscb->sopt1=rhdr[ZF1];
               zmrscb->pstate=1;
               break;
          case ZFIN:
               ftfscb->stecnt=0;
               zshhdr(ZFIN,"\0\0\0\0");
               ftfnew(ZUOOUT);
               break;
          }
          break;
     case ZUGOING:
          switch(rtype) {
          case ZDATA:
               if (*(long *)&rhdr[0] != ftfscb->actbyt) {
                    ftfout(zmrscb->attn,strlen(zmrscb->attn));
                    zshhdr(ZRPOS,(char *)&ftfscb->actbyt);
                    break;
               }
               else {
                    zmrscb->pstate=1;
                    ftfscb->stecnt=0;
               }
               break;
          case ZSKIP:
          case ZEOF:
               if (*(long *)&rhdr[0] != ftfscb->actbyt) {   /* pos mismatch */
                    /* ZMODEM says ignore wrong-position EOF */
                    /* force a ZUGOING timeout instead */
                    break;
               }
               ftfrcl(1);
               ftfscb->isopen=0;
               ftfscb->actfil++;
               ftfnew(ZULOGGED);
               break;
          case ZFIN:
               ftfabt("Sender abruptly terminated.");
               break;
          }
          break;
     case ZUOOUT:                             /* assume ZFIN's are retrying */
          zshhdr(ZFIN,"\0\0\0\0");
          zmrscb->pstate=1;                      /* (so we can detect "OO") */
          break;
     }
}

STATIC void
svcupkt(void)        /* service a recieved sub-packet during ZMODEM receive */
{
     char *cp;
     int doit;            /* should we transfer this file (versus skip it)? */
     int isfile;           /* does this (DOS) file exist on the BBS already */
     int apnd,asci,resum;
     char noack=0;     /* don't send ZACK if sending some other type of hdr */
     long newtim,newbyt;
     long oldtim,oldbyt;
     char *skipreason;

     if (ptype == ZGARBAGE) {
          ftfscb->garbag++;
          ftfscb->stecnt++;
          ftfout(zmrscb->attn,strlen(zmrscb->attn));
          zshhdr(ftfscb->state == ZUGOING ? ZRPOS
                                          : ZNAK,(char *)&ftfscb->actbyt);
          zmrscb->pstate=0;
          zmrscb->numpkt=0;
          return;
     }
     ftfscb->stecnt=0;
     ftfnwp();
     switch(ftfscb->state) {
     case ZUSINIT:                 /* receiving ZSINIT sub-packet: attn seq */
          stzcpy(zmrscb->attn,zmrscb->packet,ZATTNLEN+1);
          ftfnew(ZUFILE);
          break;
     case ZUFILE:           /* receiving ZFILE sub-packet: file information */
          ftfcli();                            /* toss any piled-up ZFILE's */
          noack=1;
          movmem(zmrscb->packet,ftfscb->fname,8+1+3+1);
          ftfscb->fname[8+1+3]='\0';      /* (crude but secure string copy) */
          cp=zmrscb->packet+strlen(zmrscb->packet)+1;
          newtim=0L;
          newbyt=0L;
          if (*cp != '\0') {
               sscanf(cp,"%ld%lo",&newbyt,&newtim);
          }
          isfile=ftfrex();
          oldtim=ftfscb->unxtim;
          oldbyt=ftfscb->estbyt;
          switch(zmrscb->sopt1&ZMMASK) {
          case ZMNEW:                                      /* DSZ -n option */
               doit=!isfile || newtim > oldtim;
               skipreason="file up to date";
               break;
          case ZMNEWL:                                     /* DSZ -N option */
               doit=!isfile || newtim > oldtim || newbyt > oldbyt;
               skipreason="file up to date, and big enough";
               break;
          case ZMPROT:                                     /* DSZ -p option */
               doit=!isfile;
               skipreason="file exists already";
               break;
          case ZMCLOB:                                     /* DSZ -y option */
          case ZMAPND:                                     /* DSZ -+ option */
          default:                                         /* DSZ no option */
               doit=1;
               break;
          }
          if (zmrscb->sopt1&ZMSKNOLOC) {                   /* DSZ -Y option */
               if (doit && !isfile) {
                    doit=0;
                    skipreason="-Y option skipped because file DOESN'T exist already";
               }
          }
          if (doit) {
               asci=zmrscb->sopt0 == ZCNL;
               apnd=(zmrscb->sopt1&ZMMASK) == ZMAPND;
               resum=isfile == 2 && zmrscb->sopt0 == ZCRESUM;    /* resume? */
                                                      /* keep old contents! */
               ftfscb->unxtim=newtim;
               ftfscb->estbyt=newbyt;
               if (ftfrop(apnd,asci,resum) != 0) {
                    ftfabt(ftfscb->abwhy);
                    break;
               }
               ftfsrt();
               if (resum && !apnd) {                              /* resume */
                    ftfrsk(ftfscb->actbyt=(oldbyt&~(ZBRECOVER-1)));
               }
               else {
                    ftfscb->actbyt=0L;
               }
               zmrscb->uplcnt++;
               zshhdr(ZRPOS,(char *)&ftfscb->actbyt);
               ftfnew(ZUGOING);
          }
          else {
               zshhdr(ZSKIP,"\0\0\0\0");
               ftfrsq(skipreason);
               ftfnew(ZUFILE);
          }
          zmrscb->fnucnt++;
          break;
     case ZUGOING:                               /* receiving ZDATA packets */
          if (ftfrwr(zmrscb->packet,zmrscb->numpkt) != zmrscb->numpkt) {
               ftfabt("Disk full.");
               noack=1;
          }
          else {                                        /* got that packet! */
               ftfscb->actbyt+=zmrscb->numpkt;
          }
          ftfnew(ZUGOING);
          ftfcbl();
          break;
     }
     if (!noack && (ptype == ZCRCW || ptype == ZCRCQ)) {
          zshhdr(ZACK,(char *)&ftfscb->actbyt);             /* ACK expected */
     }
     if (ptype == ZCRCE || ptype == ZCRCW) {
          zmrscb->pstate=0;                        /* no sub-packet follows */
     }
     else {
          zmrscb->pstate=1;                    /* more sub-packet(s) follow */
     }
     zmrscb->numpkt=0;
}

STATIC void
zmrinc(
char c)
{
     if (zmhchr(c) && rtype != ZCRCW) {
          svcuhdr();                             /* service incoming header */
     }
     else if (zmpchr(c)) {
          svcupkt();                         /* service incoming sub-packet */
     }
}

STATIC void
svctimeout(void)                          /* service ZMODEM receive timeout */
{
     switch(ftfscb->state) {
     case ZUSINIT:
     case ZUFILE:
          zshhdr(ZRINIT,"\0\0\0\43");             /* CANFC32+CANOVIO+CANFDX */
          break;
     case ZUGOING:
          zshhdr(ZRPOS,(char *)&ftfscb->actbyt);
          break;
     default:
          return;
     }
}

STATIC int
zmrctn(void)                                      /* Service ZMODEM receive */
{                                  /* returns 1=still working on it, 0=done */
     int n;

     switch(ftfscb->state) {                   /* constant service tasks... */
     case ZUINIT:
          zshhdr(ZRINIT,"\0\0\0\43");             /* CANFC32+CANOVIO+CANFDX */
          ftfnew(ZUFILE);
          break;
     case FTFABORT:
          ftfrca();
          ftfout(zmrscb->attn,strlen(zmrscb->attn));
          ftfcan();
          ftfnew(FTFABWAIT);
          break;
     case FTFABWAIT:
          if (ftftck-ftfscb->tckstt > ZABWAIT) {
               return(-1);
          }
          break;
     case ZUOOUT:                      /* sent ZFIN, wait a little for "OO" */
          if (((n=zmrscb->numpkt) >= 2
           &&  zmrscb->packet[n-1] == 'O'
           &&  zmrscb->packet[n-2] == 'O')
           || ftftck-ftfscb->tckstt >= ZOOWAIT) {
               ftfnew(ZUEND);
          }
          else {
               zmrscb->pstate=1;                   /* keep looking for "OO" */
          }
          break;
     case ZULOGGED:
          zshhdr(ZRINIT,"\0\0\0\43");        /* CANFC32+CANOVIO+CANFDX */
          ftfnew(ZUFILE);
          break;
     case ZUEND:
          return(0);
     }
     if (ftftck-ftfscb->tckstt >= ZURETRY) {
          svctimeout();
          ftfscb->tmouts++;
          ftfscb->stecnt++;
          ftfnew(ftfscb->state);
     }
     return(1);
}



/*--- ZMODEM Transmit File routines ---*/

STATIC void
zmxini(void)                               /* Initialize ZMODEM transmitter */
{
     if (fbleng < ZCHUNK) {
          fbleng=ZCHUNK;
     }
}

STATIC void
zsdata(                                /* transmit a zmodem data sub-packet */
int nbytes,                              /* number of bytes (could be zero) */
unsigned char framend)                     /* pre-CRC frame terminator code */
{                /* rawbuf is implicit input, ftfbuf is used to format data */
     int i;
     unsigned crc;
     unsigned long lcrc;
     unsigned char c;
     char *zp;
     char *buf;

     zctlesc=(zmxscb->flags&ZCTLESC) != 0;
     if (zmxscb->flags&ZCRC32) {           /* binary header with 32-bit CRC */
          lcrc=0xFFFFFFFFL;
          zp=ftfbuf;
          buf=rawbuf;
          for (i=0 ; i < nbytes ; i++,buf++) {
               c=*buf;
               lcrc=UPDC32(c,lcrc);
               if (zescape(c)) {  /* more efficient than nbytes*zsendline() */
                    *zp++=ZDLE;
                    *zp++=c^0x40;
               }
               else {
                    *zp++=c;
               }
          }
          ftfout(ftfbuf,(int)(zp-ftfbuf));
          sendline(ZDLE);
          sendline(framend);
          lcrc=upfc32(framend,lcrc);
          lcrc=~lcrc;
          for (i=0 ; i < 4 ; i++) {
               zsendline((unsigned char)lcrc);   /* <crc-lo> . . . <crc-hi> */
               lcrc>>=8;
          }
     }
     else {                                /* binary header with 16-bit CRC */
          crc=0;
          zp=ftfbuf;
          buf=rawbuf;
          for (i=0 ; i < nbytes ; i++,buf++) {
               c=*buf;
               crc=updcrc(c,crc);
               if (zescape(c)) {  /* more efficient than nbytes*zsendline() */
                    *zp++=ZDLE;
                    *zp++=c^0x40;
               }
               else {
                    *zp++=c;
               }
          }
          ftfout(ftfbuf,(int)(zp-ftfbuf));
          sendline(ZDLE);
          sendline(framend);
          crc=upfcrc(framend,crc);
          crc=upfcrc(0,upfcrc(0,crc));
          zsendline(crc>>8);                                   /* <crc-hi> */
          zsendline(crc);                                      /* <crc-lo> */
     }
     if (framend == ZCRCW) {
          sendline('\x11');                                        /* <XON> */
     }
}

STATIC void
zfhdr(void)                                          /* send a ZFILE header */
{                                                    /* wit all de trimmins */
     int n;
     static char filehdr[4]={0,0,0,0};

     filehdr[ZF0]=(zmxscb->flags&(ZRESUM1+ZRESUMA)) ? ZCRESUM : ZCBIN;
     zsbhdr(ZFILE,filehdr);
     sprintf(rawbuf,"%s%c%lu %lo 0 0 1 %lu",ftfscb->fname,'\0',
                                            ftfscb->estbyt,
                                            ftfscb->unxtim,
                                            ftfscb->estbyt);
     strlwr(rawbuf);
     n=strlen(rawbuf)+1;
     n+=strlen(rawbuf+n)+1;           /* then format stats and another '\0' */
     zsdata(n,ZCRCW);                                /* and transmit it all */
}

STATIC void
zdseek(void)                              /* respond to ZRPOS from receiver */
{
     zmxscb->confirm=                       /* everything starts over again */
     zmxscb->txpos=
     ftfscb->actbyt=*(long *)(&rhdr[0]);
     ftfxsk(ftfscb->actbyt);
     zsbhdr(ZDATA,rhdr);
     ftfnew(ZDGOING);
}

STATIC void
zhdlri(void)               /* handle flags imbedded in ZRINIT from receiver */
{
     (rhdr[ZF0]&ESCCTL)  ? (zmxscb->flags|=ZCTLESC) : (zmxscb->flags&=~ZCTLESC);
     (rhdr[ZF0]&CANFC32) ? (zmxscb->flags|=ZCRC32)  : (zmxscb->flags&=~ZCRC32);
}

STATIC void
zdgo(void)                               /* begin ZMODEM transmit of a file */
{
     zfhdr();
     ftfnew(ZDRPOS);
     zmxscb->txpos=0L;
}

STATIC void
zdnext(void)                                   /* next ZMODEM transmit file */
{
     if (ftfxop() == 0) {
          ftfsrt();
          zmxscb->flags|=ZDBLK1;
          zdgo();
     }
     else {
          zshhdr(ZFIN,"\0\0\0\0");
          ftfnew(ZDFIN);
     }
     zmxscb->flags&=~ZRESUM1;
}

STATIC void
zmxsrt(void)                               /* Begin ZMODEM transmit session */
{
     setmem(zmxscb,sizeof(*zmxscb),0);
     zmxscb->numhdr=-2;                    /* quiescent zmhchr() state code */
     ftfscb->paksiz=ftfpsp->paksiz;
     ftfscb->window=ftfpsp->window;
     while (ftfscb->paksiz*2+ZBOVHEAD > ftfomt) {
          ftfscb->paksiz>>=1;                          /* (but not too big) */
     }
     if (ftfxop() == 0) {
          ftfsrt();
          zmxscb->flags|=ZDBLK1;
          ftfnew(ZDINIT);
     }
     else {
          ftfnew(ZDEND);
     }
}

STATIC void
zxrsrt(void)           /* Begin ZMODEM transmit session w/resume-if-you-can */
{
     zmxsrt();
     zmxscb->flags|=ZRESUMA;
}

STATIC void
svchdr(void)              /* service received ZMODEM header during transmit */
{
     switch(rtype) {                              /* easily handled headers */
     case ZABORT:
          ftfabt("Receiver aborted.");
          return;
     case ZCAN:
          ftfabt("Operator CTRL-X abort.");
          return;
     case ZCRCW:                           /* ZMODEM says transmiter should */
     case ZGARBAGE:                    /* immediately ZNAK a received ZCRCW */
          ftfscb->garbag++;
          ftfscb->stecnt++;
          zshhdr(ZNAK,"\0\0\0\0");  /* for some kind of fast error recovery */
          return;
     }
     switch(ftfscb->state) {                  /* handle incoming header... */
     case ZDRINIT:                /* waiting for first ZRINIT from receiver */
          switch(rtype) {
          case ZRINIT:                                         /* got it... */
               ftfscb->stecnt=0;
               zhdlri();
               zdgo();                    /* send header for the first file */
               break;
          case ZCHALLENGE:                          /* respond to challenge */
               zshhdr(ZACK,rhdr);
               break;
          default:                                        /* anything else: */
               ftfout("rz\r",3);
               zshhdr(ZRQINIT,"\0\0\0\0");                /* repeat ZRQINIT */
               break;
          }
          break;
     case ZDRPOS:                  /* waiting for first ZRPOS from receiver */
          switch(rtype) {
          case ZRPOS:                                          /* got ZRPOS */
               ftfscb->stecnt=0;
               ftfstf();
               zdseek();
               break;
          case ZSKIP:                      /* skip this file, go on to next */
               ftfxsq("file skipped by receiver");
               ftfxcl(0);
               ftfscb->isopen=0;
               zdnext();
               break;
          case ZCRC:
               zmxscb->filcrc=0xFFFFFFFFL;
               ftfxsk(0L);
               ftfnew(ZDCRCING);
               break;
          case ZFIN:
               ftfabt("Receiver abruptly terminated.");
               break;
          case ZRINIT:                         /* if we get another ZRINIT, */
          default:                                     /* or anything else, */
               if (ftftck-ftfscb->tckstt > ZDFRETRY) {   /* after 2 seconds */
                    zfhdr();                      /* retry the ZFILE header */
                    ftfnew(ftfscb->state);
                    ftfscb->retrys++;
                    ftfscb->stecnt++;
               }
               break;
          }
          break;
     case ZDGOING:                /* some error reply to data subpackets... */
     case ZDEOF:      /* some error reply to last data subpacket in file... */
          switch(rtype) {
          case ZRPOS:                                      /* retry request */
               ftfclo();
               ftfscb->retrys++;
               ftfscb->stecnt++;
               zsdata(0,ZCRCE);               /* (term current frame first) */
               zdseek();
               break;
          case ZRINIT:                          /* new file req (oh yeah?) */
          case ZSKIP:                                     /* skip this file */
               ftfclo();
               zsdata(0,ZCRCE);               /* (term current frame first) */
               ftfxsq("file started and then skipped by receiver");
               ftfxcl(0);
               ftfscb->isopen=0;
               zdnext();
               break;
          case ZACK:                            /* listen to ZACK's address */
               ftfscb->stecnt=0;
               ftfscb->actbyt=*(long *)(&rhdr[0]);
               break;
          case ZFIN:                                         /* be that way */
               ftfabt("Receiver abruptly terminated.");
               break;
          }
          break;
     case ZDNEXT:                    /* sent ZEOF, expecting next ZRINIT... */
          switch(rtype) {
          case ZACK:                         /* ignore ZACK's from old file */
               break;
          case ZRPOS:                      /* retries of old file after eof */
               zdseek();
               break;
          case ZSKIP:                  /* skip the last file, go on to next */
               ftfxsq("file skipped by receiver even after it had all been sent");
               ftfxcl(0);
               ftfscb->isopen=0;
               zdnext();
               break;
          case ZRINIT:                  /* here's the request for next file */
               ftfscb->stecnt=0;
               zhdlri();
               ftfscb->actbyt=zmxscb->txpos;
               ftfxcl(1);
               ftfscb->isopen=0;
               ftfscb->actfil++;               /* count successful transmit */
               zdnext();
               break;
          default:                             /* anything else is gibbrish */
               ftfabt("Bad reply to ZEOF.");
               break;
          }
          break;
     case ZDFROWN:
          ftfout("OO",2);                         /* redundant Over and Out */
          ftfnew(ZDFROWN);
          break;
     case ZDFIN:                                       /* got reply to ZFIN */
          switch(rtype) {
          case ZFIN:                                /* expect another ZFIN: */
               ftfscb->stecnt=0;
               ftfout("OO",2);                              /* Over and Out */
               ftfnew(ZDEND);
               break;
          default:                              /* but anything else means: */
               zshhdr(ZFIN,"\0\0\0\0");                       /* retry ZFIN */
               ftfscb->retrys++;
               ftfscb->stecnt++;
               break;
          }
          break;
     }
}

STATIC void
svcdnl(void)               /* service routine checking for zmodem transmit */
{
     int n;
     int packend;
     long ztxwindow;
     int block;

     ztxwindow=ftfscb->window;
     if (ftfoba() >= ftfscb->paksiz*2+ZBOVHEAD
                                          /* room for next ZDATA subpacket? */
      && (zmxscb->flags&ZXOFF) == 0                      /* XOFF-throttled? */
      && (ztxwindow == 0 || zmxscb->txpos-ftfscb->actbyt < ztxwindow)) {
                                                    /* receiver keeping up? */

          block=((zmxscb->flags&(ZRESUMA+ZRESUM1)) && (zmxscb->flags&ZDBLK1))
                ? ZBRECOVER : ftfscb->paksiz;
          zmxscb->flags&=~ZDBLK1;
          n=ftfxrd(rawbuf,block);
          zmxscb->txpos+=n;
          if (n == block) {                         /* this is a full block */
               if (ztxwindow != 0
                && zmxscb->txpos-zmxscb->confirm >= (ztxwindow>>2)) {
                    packend=ZCRCQ;             /* time to get some feedback */
                    zmxscb->confirm=zmxscb->txpos;
               }
               else {
                    packend=ZCRCG;               /* don't need feedback yet */
               }
               zsdata(n,packend);
          }
          else {                         /* this is a (partial) final block */
               zsdata(n,ZCRCE);
               ftfnew(ZDEOF);
          }
          ftfnwp();                                /* count outgoing packet */
          if (ztxwindow == 0) {
               ftfscb->tckact=ftftck;
               ftfact=1;
          }
     }                  /* may never be ack'd!  may be retried & recounted! */
}

STATIC void
svceof(void)          /* service routine checking after ZMODEM transmit eof */
{
     if (ftfoba() >= ZBEOF) {                  /* room for ZEOF terminator? */
          zsbhdr(ZEOF,(char *)&zmxscb->txpos);
          ftfnew(ZDNEXT);
     }
}

STATIC void
svccrc(void)                                /* service file crc computation */
{
     unsigned char c,*zp;
     int i,n;
     unsigned long crc;

     n=ftfxrd(ftfbuf,ZCHUNK);
     crc=zmxscb->filcrc;
     for (i=0,zp=ftfbuf ; i < n ; i++) {
          c=*zp;
          crc=upfc32(c,crc);
     }
     zmxscb->filcrc=crc;
     if (n != ZCHUNK) {
          crc=~crc;
          ftfxsk(0L);
          zshhdr(ZCRC,(char *)&crc);
          ftfnew(ZDRPOS);
     }
}

STATIC void
zmxinc(                         /* handle incoming byte for ZMODEM transmit */
char c)
{
     if (zmhchr(c)) {
          svchdr();                           /* service incoming header(s) */
     }
}

STATIC int
zmxctn(void)                         /* continue processing zmodem transmit */
{                                        /* returns 1=still working, 0=done */
     switch(ftfscb->state) {                   /* constant service tasks... */
     case ZDINIT:
          ftfout("rz\r",3);
          zshhdr(ZRQINIT,"\0\0\0\0");               /* send ZRQINIT to rcvr */
          ftfnew(ZDRINIT);
          break;
     case ZDRINIT:
          if (ftftck-ftfscb->tckstt > ZDRQWAIT) {
               zshhdr(ZRQINIT,"\0\0\0\0");            /* send ZRQINIT again */
               ftfnew(ZDRINIT);
          }
          break;
     case ZDGOING:                                           /* transmiting */
          svcdnl();
          break;
     case ZDEOF:                                      /* sending eof header */
          svceof();
          break;
     case ZDCRCING:                                   /* computing file CRC */
          svccrc();
          break;
     case FTFABORT:
          ftfxca();
          ftfcan();
          ftfnew(FTFABWAIT);
          break;
     case FTFABWAIT:
          if (ftftck-ftfscb->tckstt > ZABWAIT) {
               return(-1);
          }
          break;
     case ZDFROWN:
          if (ftftck-ftfscb->tckstt > ZDFWAIT) {
               ftfcli();
               ftfnew(ZDEND);
          }
          break;
     case ZDEND:
          if (ftfoba() == ftfomt || ftftck-ftfscb->tckstt > ZDEWAIT) {
               if (zmxscb->numhdr != -2) {
                    ftfnew(ZDFROWN);       /* (hold out for redundant ZFIN) */
                    break;
               }
               zmxscb->flags&=~ZRESUM1;
               return(0);
          }
          break;
     }
     return(1);
}

int
ftfzad(                              /* check for automatic ZMODEM download */
char c)            /* call .start(), then pass all incoming characters here */
                                 /* returns 1 when download should be begun */
                                        /* global inputs: ftfpsp and ftfscb */
{
     static char zadtrg[]="*\x18\B00";

     if (c == '*') {
          zmrscb->zadctr=1;
     }
     else if (c == zadtrg[zmrscb->zadctr]) {
          if (++zmrscb->zadctr == 5) {
               zmrscb->zadctr=0;
               ftfpsp->start();
               ftfpsp->hdlinc('*');
               ftfpsp->hdlinc('\x18');
               ftfpsp->hdlinc('B');
               ftfpsp->hdlinc('0');
               ftfpsp->hdlinc('0');
               return(1);
          }
     }
     else {
          zmrscb->zadctr=0;
     }
     return(0);
}

struct ftfpsp ftpzmr={                                  /* ZMODEM receiving */
     NULL,
     "Z",                                  /* 1-3 code letters for protocol */
     "ZMODEM",                                          /* name of protocol */
     FTFMUL,                                   /* protocol capability flags */
     sizeof(struct zmrdat),        /* total length of session control block */
     3*16,      /* .byttmo                             default byte timeout */
     10*16,     /* .paktmo                           default packet timeout */
     10,        /* .retrys                              default max retries */
     2048L,     /* .window   max window size (packets/bytes as appropriate) */
     1024,      /* .paksiz                        packet size 0=auto-figure */
     zmrini,    /* .initze()    Initialize this protocol (recompute scblen) */
     zmrsrt,    /* .start()                                Start a transfer */
     zmrctn,    /* .contin()              Continuously call, 1=more, 0=done */
     zmrinc,    /* .hdlinc()                       Handle one incoming byte */
     NULL,      /* .hdlins()                   Handle incoming line of text */
     ftfabt,    /* .term()        Initiate graceful termination of transfer */
     ftfrca     /* .abort()  Immediately unconditionally abort the transfer */
};

struct ftfpsp ftpzmx={                               /* ZMODEM transmitting */
     NULL,
     "Z",                                  /* 1-3 code letters for protocol */
     "ZMODEM",                                          /* name of protocol */
     FTFXMT+FTFMUL,                            /* protocol capability flags */
     sizeof(struct zmxdat),        /* total length of session control block */
     3*16,      /* .byttmo                             default byte timeout */
     10*16,     /* .paktmo                           default packet timeout */
     10,        /* .retrys                              default max retries */
     2048L,     /* .window   max window size (packets/bytes as appropriate) */
     1024,      /* .paksiz                        packet size 0=auto-figure */
     zmxini,    /* .initze()    Initialize this protocol (recompute scblen) */
     zmxsrt,    /* .start()                                Start a transfer */
     zmxctn,    /* .contin()              Continuously call, 1=more, 0=done */
     zmxinc,    /* .hdlinc()                       Handle one incoming byte */
     NULL,      /* .hdlins()                   Handle incoming line of text */
     ftfabt,    /* .term()        Initiate graceful termination of transfer */
     ftfxca     /* .abort()  Immediately unconditionally abort the transfer */
};

struct ftfpsp ftpzrx={          /* ZMODEM transmitting w/attempted RESUMING */
     NULL,
     "ZR",                                 /* 1-3 code letters for protocol */
     "ZMODEM (resume after abort)",                     /* name of protocol */
     FTFXMT+FTFMUL,                            /* protocol capability flags */
     sizeof(struct zmxdat),        /* total length of session control block */
     3*16,      /* .byttmo                             default byte timeout */
     10*16,     /* .paktmo                           default packet timeout */
     10,        /* .retrys                              default max retries */
     2048L,     /* .window   max window size (packets/bytes as appropriate) */
     1024,      /* .paksiz                        packet size 0=auto-figure */
     zmxini,    /* .initze()    Initialize this protocol (recompute scblen) */
     zxrsrt,    /* .start()                                Start a transfer */
     zmxctn,    /* .contin()              Continuously call, 1=more, 0=done */
     zmxinc,    /* .hdlinc()                       Handle one incoming byte */
     NULL,      /* .hdlins()                   Handle incoming line of text */
     ftfabt,    /* .term()        Initiate graceful termination of transfer */
     ftfxca     /* .abort()  Immediately unconditionally abort the transfer */
};
