/* ////////////////////////////////////////////////////////////////////////

   HVSAUTOV.C -- Main code module for Auto Validator.
   (C) Copyright 1992, High Velocity Software, Inc.

   Release 1.13

   1.00  03AUG92 MMD  Creation and start of version control.
   1.00a 06AUG92 MMD  Fixed @A and @R ValidateUser mis-calls.
   1.01  06AUG92 MMD  Modified global handler to ignore short (1) input.

   1.10 07SEP92 MMD  Mailing label/class name addition modifications.
   1.10  15SEP92 MMD  Orphan accounts automatically deleted at @L and cleanup.
   1.10a 16SEP92 MMD  Recompiled with proper MAJORBBS.H (user struct)
   1.10b 17SEP92 MMD  Removed // comments, fixed lingering NORMAL key
   1.10c 23SEP92 MMD  Fixed NULL padding on orphan detection.
   1.10d 24SEP92 MMD  ShowActivity uses channel instead of user number
                      FindAndBusyEmptyChannel checks for W2KILL
   1.10e 28SEP92 MMD  Fixed numerous key-dropping bugs causing lingering stuff.
   1.10f 02OCT92 MMD  Fixed erroneous parameters to shocst() under V5
                      Modified channel-finder to look for AWAITC.

   1.11  08OCT92 MMD  Adjusted MailAll & MailEnabled[] to work better.
   1.11a 20OCT92 MMD  Fixed channel clearing.
                      Changed credit limits to 1,000,000 instead of 100,000.
   1.11b 09DEC92 MMD  Modified channel-finder to not shutdown if all channels
                      are busy.
                      Added long specifiers to lngopt calls.
   1.11c 17DEC92 MMD  Fixed auto-update silliness.

   1.12  13MAR93 MMD  Added options for area code dialing on toll and local.
   1.12a 12APR93 MMD  Fixed mailing label self-silliness and CNF bad hinges.

   //////////////////////////////////////////////////////////////////////// */

#define AUTOV_VERSION "1.13"

#include "gcomm.h"
#include "majorbbs.h"
#include "..\debug\hvsdebug.h"

#ifndef TRUE
#define TRUE  1
#define FALSE 0
#endif

#include "hvsavs.h"              /*  Structs/defines           */
#include "hvsautov.h"            /* .MCV constants             */

void EXPORT init__autov(void);
extern FILE * mjrmb;
static void  SetNewLegend(USHORT Channel, PSZ pStatus, CHAR Char);
static PSZ pSampleKey;

static int   AVLogon(void);
static void  AVHangup(void);
static int   AVMain(void);
static void  AVCleanup(void);
static void  AVDelete(PSZ pUserName);
static void  AVShutdown(void);

static struct module AVModule =
  {
  "*",                             /* String with menu descr          */
  AVLogon,                         /* User logged on                  */
  AVMain,                          /* Normal input routine            */
  dfsthn,                          /* Status input routine            */
  NULL,                            /* Injection entry point           */
  NULL,                            /* User logging off                */
  AVHangup,                        /* User hung up                    */
  AVCleanup,                       /* Cleanup                         */
  AVDelete,                        /* User deleted                    */
  AVShutdown,                      /* Shutdown                        */
  };

static void  AddLabel(PAVREC pUser);
static void  ShowActivity(void);
static void  LogEvent(PSZ pFormat, PSZ pUserID);
static void  LogScreen(PSZ pFormat, PSZ pUserID);
static void  ReadConfiguration(void);
static BOOL  GetConfigLine(void);
static PVOID ClearAlloc(USHORT Size);
static void  MakeDefaultUser(void);
static void  ParsePhoneNumber(PSZ pInput, PSZ pOutput);
static BOOL  IsSysop(void);
static void  ScreenPhone(PAVREC pUser);
static BOOL  SearchShortList(PUSHORT pList, USHORT Value, USHORT Max);
static BOOL  SearchLongList(PULONG pList, ULONG Value, USHORT Max);
static BOOL  OkToMail(PAVREC pUser);
static BOOL  OkToCall(PAVREC pUser);
static void  GenericRebuff(PAVREC pUser);
static void  StartPulse(void);
static USHORT FindAndBusyEmptyChannel(PAVREC pTarget);
static void SendDialString(USHORT Channel);
static void LogFailure(PCHANNELINFO pInfo, USHORT KeyChannel);
static BOOL LogValidation(PAVREC pUser);
static BOOL ValidateUser(PAVREC pUser);
static USHORT CheckForValidation(void);
static USHORT FindEmptyPreSlot(void);
static USHORT IncrementPhone(PSZ pPhone);
static USHORT DecrementPhone(PSZ pPhone);
static USHORT QueryPhone(PSZ pPhone);
static BOOL   HasTooManyPhones(PAVREC pUser);
static int    GlobalCommand(void);
static PAVREC GetUser(PSZ pUserID);
static void SaveUser(PAVREC pUser);
static PAVREC GetCommandUser(void);
static void RestartAttempt(USHORT Channel);
static void UpdateUser(PAVREC pUser);
static void NullPad(PSZ pString, USHORT Length);
static void RefreshAll(void);
static PSZ FormatPhone(PSZ pPhone);
static void PostMarkerCredit(PCHANNELINFO pInfo);
static void TakeMarkerCredit(PCHANNELINFO pInfo);
static void LogStartup(void);
static PSZ TempHandle(PSZ pName);
static PSZ TVMailCode(void);

static void Pulse(void);      /* rtkick routine */

static PAVREC pCurrentLabel;

static FILE *  pMessage;
static PULONG  pNeverNumbers     = NULL;
static PUSHORT pNeverAreaCodes   = NULL;
static PUSHORT pNeverPrefixes    = NULL;
static PUSHORT pAreaCodeList     = NULL;
static PUSHORT pPrefixList       = NULL;
static BOOL    PrefixListBad     = TRUE;
static BOOL    AreaCodeListBad   = FALSE;
static BOOL    AllowLongDistance = FALSE;
static BOOL    AllowToll         = FALSE;
static USHORT  MaxDialAttempts;
static USHORT  MaxRetries;
static USHORT  SystemAreaCode;
static FILE *  pConfig;
static CHAR    ConfigLine[BIG_STRING];
static PSZ     pLocalPrefix;
static PSZ     pTollPrefix;
static PSZ     pLongDistancePrefix;
static PSZ     pSuffix;
static PSZ     pSys1;
static PSZ     pSys2;
static PSZ     pSys3;
static PSZ     pBusyString;
static PSZ     pNoBusyString;
static BOOL    Disconnect[RANGES];
static BOOL    Enabled = TRUE;
static BOOL    CleanupRefresh;
static BOOL    LoggingScreens;
static BOOL    LogAnything;
static USHORT  MaxAutos;
static USHORT  MaxPhone;
static BOOL    AutoUpdate;
static BOOL    PostFirst = TRUE;

static BOOL    AreaCodeToll;
static BOOL    AreaCodeLong;

static CHAR    GlobalKick;

static BOOL    LogSuccess;
static BOOL    MailEnabled[RANGES];
static BOOL    MailAll;

static BOOL    PulseActive = FALSE;     /* Is the rtkick routine burning? */

static PUSHORT pCurrentList = NULL;
static USHORT  CurrentMax   = 0;
static PSZ     pCurrentAction = NULL;
static PSZ     pLogName;
static PSZ     pMailName;

static PSZ     pValidClass;
static PSZ     pClasses[RANGES];
static BOOL    ClassSwitch[RANGES];

static PCHANNELINFO pInfo = NULL;

static BTVFILE * pbPhoneRec;
static BTVFILE * pbUserRec;

BOOL SecureMode;
BOOL BypassScreen;
PSZ pSecureSource = NULL;
PSZ pSecureDest = NULL;

static void SetNewLegend(USHORT Channel, PSZ pStatus, CHAR Char)
  {
  USHORT OldUser;

  OldUser = usrnum;
  usrnum  = Channel;
  shochl(pStatus, Char, 0x1f);
  usrnum  = OldUser;
  }

BOOL ValidationInProgress(USHORT Channel)
  {
  if (pInfo[Channel].Status == CS_IDLE)
    return FALSE;
  else
    return TRUE;
  }

static BOOL IsSysop(void)
  {
  BOOL Sysop = FALSE;

  if (haskey("SYSOP"))
    Sysop = TRUE;
  else
    if (sameas(usaptr->userid, pSys1) ||
        sameas(usaptr->userid, pSys2) ||
        sameas(usaptr->userid, pSys3))
      Sysop = TRUE;

  return Sysop;
  }

static PVOID ClearAlloc(USHORT Size)
  {
  PVOID pRegion;

  pRegion = (PVOID) alcmem(Size);
  setmem(pRegion, Size, 0);

  return pRegion;
  }

void EXPORT init__autov(void)
  {
  pMessage   = opnmsg(AUTOV_MESSAGE);
  pbPhoneRec = opnbtv(AUTOV_PHONEREC, sizeof(PHONEREC) );
  pbUserRec  = opnbtv(AUTOV_USERREC, sizeof(AVREC) );

  setmbk(pMessage);

  stzcpy(AVModule.descrp, gmdnam("HVSAUTOV.MDF"), MNMSIZ);

  pClasses[0]       = stgopt(AVNCLNAM);  /* Index by AVR_ range */
  pClasses[1]       = stgopt(AVTCLNAM);
  pClasses[2]       = stgopt(AVLCLNAM);

  pValidClass       = stgopt(AVCLASS);

  ClassSwitch[0]    = ynopt(AVNCLASS);   /* Index by AVR_ range */
  ClassSwitch[1]    = ynopt(AVTCLASS);
  ClassSwitch[2]    = ynopt(AVLCLASS);

  MaxDialAttempts = numopt(AVMAXAT, 1, 32000);
  MaxRetries      = numopt(AVMAXRE, 1, 100);

  MaxRetries--;     /* Count starts at 0, not 1 */

  AllowToll         = ynopt(AVTOLLOK);
  AllowLongDistance = ynopt(AVLDOK);

  LogSuccess        = ynopt(AVAUDIT);
  CleanupRefresh    = ynopt(AVCLEAN);
  LoggingScreens    = ynopt(AVSCREEN);
  LogAnything       = ynopt(AVLOG);
  MailAll           = !(ynopt(AVMALL));
  MailEnabled[0]    = ynopt(AVNMAIL);    /* Index by AVR_ range */
  MailEnabled[1]    = ynopt(AVTMAIL);
  MailEnabled[2]    = ynopt(AVLMAIL);

  Disconnect[0]     = ynopt(AVNDISC);    /* Index by AVR_ range */
  Disconnect[1]     = ynopt(AVTDISC);
  Disconnect[2]     = ynopt(AVLDISC);

  AreaCodeToll      = ynopt(AVACTOL);
  AreaCodeLong      = ynopt(AVACLNG);

  AutoUpdate        = ynopt(AVUPDAT);
//  PostFirst         = ynopt(AVPOSTF);

  MaxAutos          = numopt(AVAUTOS, 1, 64);
  MaxPhone          = numopt(AVMAXPHO, 1, 32000);

  SecureMode        = ynopt(AVSECURE);
  BypassScreen      = ynopt(AVBYPASS);

  if (SecureMode)
    {
    pSecureSource   = stgopt(AVSECSRC);
    pSecureDest     = stgopt(AVSECDST);
    }

  pNeverNumbers     = ClearAlloc(sizeof(ULONG) * MAX_NEVER_NUMBERS);
  pNeverAreaCodes   = ClearAlloc(sizeof(USHORT) * MAX_NEVER_AREA_CODES);
  pNeverPrefixes    = ClearAlloc(sizeof(USHORT) * MAX_NEVER_PREFIXES);
  pAreaCodeList     = ClearAlloc(sizeof(USHORT) * MAX_AREA_CODE_LIST);
  pPrefixList       = ClearAlloc(sizeof(USHORT) * MAX_PREFIX_LIST);
  pInfo             = ClearAlloc(sizeof(CHANNELINFO) * (MaxAutos + nterms));

  GlobalKick = chropt(AVGLOB);
  globalcmd(GlobalCommand);

  pLocalPrefix        = stgopt(AVNPRE);
  xltctls(pLocalPrefix);
  pTollPrefix         = stgopt(AVTPRE);
  xltctls(pTollPrefix);
  pLongDistancePrefix = stgopt(AVLPRE);
  xltctls(pLongDistancePrefix);
  pSuffix             = stgopt(AVSUF);
  xltctls(pSuffix);

  pSys1     = stgopt(AVSYS1);
  pSys2     = stgopt(AVSYS2);
  pSys3     = stgopt(AVSYS3);
  pLogName  = stgopt(AVLOGFL);
  pMailName = stgopt(AVMAILFL);

  pBusyString = stgopt(AVBUSY);
  xltctls(pBusyString);
  pNoBusyString = stgopt(AVNBUSY);
  xltctls(pNoBusyString);

  ReadConfiguration();

  LogStartup();

  pSampleKey = stgopt(AVSAMPKY);

  register_module(&AVModule);
  register_textvar("MAILCODE", TVMailCode);
  }

/* The PreSlots are "channels" that are above nterms, used by the Auto
   Validator to get ready to dial someone.                                  */

static USHORT FindEmptyPreSlot(void)
  {
  BOOL   Found = FALSE;
  USHORT Index;

  for (Index = nterms; Index < nterms + MaxAutos; Index++)
    if (pInfo[Index].Status == CS_IDLE)
      {
      Found = TRUE;
      break;
      }

  if (!Found)
    return NO_CHANNEL;

  return Index;
  }

static int AVLogon(void)
  {
  USHORT OtherChannel;
  ULONG  Credits;
  BOOL   Found;

  if (pInfo[usrnum].Status == CS_HOLDING)
    return 1;

  setmbk(pMessage);

  if (pInfo[usrnum].Status == CS_CARRIER)
    {
    if (!strcmp(pCurrentUser->Name, usaptr->userid))
      {
      TakeMarkerCredit(pInfo + usrnum);

      if (SecureMode)
        {
        swtcls(usaptr, 1, pSecureDest, 4, 0);
        LogEvent("SECURE: Phone validated user %s.", pCurrentUser->Name);

        if (LogSuccess)
          shocst("AV: SECURE VALIDATION", "UserID: %s", pCurrentUser->Name);
        }
      else
        {
        switch (pCurrentUser->Range)
          {
          case AVR_LOCAL:
            Credits = lngopt(AVNCRED, 0L, 1000000L);
            break;

          case AVR_TOLL:
            Credits = lngopt(AVTCRED, 0L, 1000000L);
            break;

          case AVR_LONG_DISTANCE:
            Credits = lngopt(AVLCRED, 0L, 1000000L);
            break;

          default:            /* Severe internal confusion! */
            Credits = 0L;
            break;
          }

        if (Credits > 0L)
          prfmsg(DONECRED, spr("%ld", Credits) );
        else
          prfmsg(DONEVAL);

        outprf(usrnum);
        LogValidation(pCurrentUser);

        if (Disconnect[pCurrentUser->Range])
          {
          byenow(MUSTGO);

          pInfo[usrnum].Status = CS_HOLDING;
          }
        }
      }
    else
      {
      byenow(NOTYOU);

      pInfo[usrnum].Status = CS_HOLDING;
      }
    }
  else
    if (SecureMode)
      {
      if (sameas(usaptr->curcls, pSecureDest))   // Security breach!!
        {
        swtcls(usaptr, 1, pSecureSource, 4, 0);
        shocst("AV: UNEXPECTED PRIVILEGE LOGIN", "%s switched back to %s", usaptr->userid, pSecureSource);
        }
      }

  setbtv(pbUserRec);

  Found = acqbtv(pCurrentUser, TempHandle(usaptr->userid), 0);

  if (!Found)
    {
    MakeDefaultUser();
    insbtv(pCurrentUser);
    IncrementPhone(pCurrentUser->Phone);
    }
  else
    {
    OtherChannel = CheckForValidation();

    if (OtherChannel != NO_CHANNEL)
      {
      LogEvent("FAILED: User %s logged on while being validated.",
               pCurrentUser->Name);

      LogFailure(pInfo + usrnum, OtherChannel);
      }
    }

  if (pInfo[usrnum].Status != CS_HOLDING)
    {
    pInfo[usrnum].Status = CS_IDLE;
    return 0;
    }
  else
    return 1;
  }

static USHORT CheckForValidation(void)
  {
  PSZ pNameCheck = pCurrentUser->Name;
  USHORT Channel;
  USHORT Oldusrnum;
  BOOL   Found = FALSE;

  for (Channel = 0; Channel < nterms + MaxAutos; Channel++)
    if ((Channel != usrnum) && (pInfo[Channel].Status != CS_IDLE))
      if (!strcmp(pNameCheck, pUserOnChannel(Channel)->Name))
        {
        Oldusrnum = usrnum;
        usrnum = Channel;
        rstchn();               /* Undo the attempt in progress */
        usrnum = Oldusrnum;

        pInfo[Channel].Status = CS_IDLE;
        Found = TRUE;
        break;
        }

  if (!Found)
    Channel = NO_CHANNEL;

  return Channel;
  }

static void MakeDefaultUser(void)
  {
  PAVREC pUser = pCurrentUser;

  strcpy(pUser->Name, usaptr->userid);
  ParsePhoneNumber(usaptr->usrpho, pUser->Phone);
  strcpy(pUser->AuxPhone, pUser->Phone);

  pUser->Access   = AVA_AUTO;
  pUser->Attempts = 0;
  ScreenPhone(pCurrentUser);

  if (usaptr->totcreds > 0L)
    pCurrentUser->Status = AVS_VALIDATED;      /* Not a signup! */
  }

static void RestartAttempt(USHORT Channel)
  {
  pInfo[Channel].Status    = CS_DIALPENDING;
  pInfo[Channel].TicksLeft = PENDING_DELAY;
  pInfo[Channel].Retries   = 0;   /* Attempts to find open line */
  }

static void AVHangup(void)
  {
  if (SecureMode)
    if (sameas(usaptr->curcls, pSecureDest))   // Security breach!!
      swtcls(usaptr, 1, pSecureSource, 4, 0);

  if (pInfo[usrnum].Status == CS_CARRIER)
    {
    LogEvent("FAILED: Dialed user %s, but user hung up before log on.",
             pCurrentUser->Name);

    if (pInfo[usrnum].TotalAttempts == MaxRetries)
      {
      LogFailure(pInfo + usrnum, usrnum);
      pInfo[usrnum].Status = CS_IDLE;
      }
    else
      {
      pInfo[usrnum].TotalAttempts++;
      RestartAttempt(usrnum);
      }

    }
  else
    {
    setbtv(pbUserRec);
    if (acqbtv(NULL, pCurrentUser->Name, 0))
      updbtv(pCurrentUser);
    else
      {
      shocst("AV: HANGUP UNABLE TO ACQUIRE", "UserID: %s", usaptr->userid);

      // catastro("HANGUP UNABLE TO ACQUIRE %s", pCurrentUser->Name);
      }

    pInfo[usrnum].Status = CS_IDLE;
    }

  pCurrentUser->Name[0] = '\0';
  }

static void AVShutdown(void)
  {
  USHORT Index;

  for (Index = 0; Index < nterms + MaxAutos; Index++)
    if ((pInfo[Index].Status != CS_IDLE) && pInfo[Index].TakeAwayKey)
      TakeMarkerCredit(pInfo + Index);

  clsmsg(pMessage);
  clsbtv(pbPhoneRec);
  clsbtv(pbUserRec);
  }

static void EnterState(int NewState)
  {
  usrptr->substt = NewState;
  prfmsg(NewState, FormatPhone(pCurrentUser->Phone));  /* Menu here          */
  prfmsg(NewState + 1);                                /* New state's prompt */
  }

static BOOL StateEntry(void)
  {
  extern PSZ bbsttl;
  CHAR TempBuffer[BIG_STRING];
  BOOL CallBackAble;
  BOOL MailAble;

  prfmsg(ENTRY, bbsttl);

  if (IsSysop())
    {
    prfmsg(USYSOP, GlobalKick);
    return TRUE;
    }
  else
    {
    if (pCurrentUser->Phone[0] == '-')
      {
      PSZ pPhone = pCurrentUser->Phone;

      sprintf(TempBuffer, "%03d", SystemAreaCode);

      pPhone[0] = TempBuffer[0];
      pPhone[1] = TempBuffer[1];
      pPhone[2] = TempBuffer[2];

      IncrementPhone(pPhone);
      }

    if (!Enabled)
      {
      prfmsg(DISABLED);
      return TRUE;
      }

    if (SecureMode)
      {
      if (stricmp(pSecureSource, usaptr->curcls))
        return TRUE;
      }
    else
      if (strlen(pValidClass) != 0)
        {
        if (stricmp(pValidClass, usaptr->curcls))
          {
          prfmsg(UNOCLASS, pValidClass);
          return TRUE;
          }

        }

    switch (pCurrentUser->Access)
      {
      case AVA_RESTRICT:
        prfmsg(YOURESTR);
        return TRUE;

      case AVA_RESCREEN:                    /* AVBUILD made this record */
        pCurrentUser->Access = AVA_AUTO;

        ScreenPhone(pCurrentUser);
        break;
      }

    if (pCurrentUser->Status == AVS_MAILED)
      if ((pCurrentUser->Attempts == MaxDialAttempts) &&
          (pCurrentUser->Access   != AVA_OVERRIDE) )
        {
        prfmsg(MAIL2MNY);
        return TRUE;
        }
      else
        {
        EnterState(MAILCODE);
        return FALSE;
        }

    if (AutoUpdate)
      UpdateUser(pCurrentUser);

    MailAble = OkToMail(pCurrentUser);       /* No messages */
    CallBackAble = OkToCall(pCurrentUser);

    if (MailAble && CallBackAble)
      {
      EnterState(WEDOU);
      return FALSE;
      }

    if (MailAble)
      {
      EnterState(WEMAILU);
      return FALSE;
      }

    if (CallBackAble)
      {
      EnterState(WECALLU);
      return FALSE;
      }

    GenericRebuff(pCurrentUser);
    return TRUE;
    }
  }

static BOOL StateWeMail(void)
  {
  switch (cncchr())
    {
    case 'Y':
      cncall();
      prfmsg(MAILNOW);
      pCurrentUser->Status = AVS_MAILED;
      pCurrentUser->MailCode = rand() + rand() + rand() + rand();
      AddLabel(pCurrentUser);
      LogEvent("MAIL: Prepared label for user %s.", pCurrentUser->Name);
      return TRUE;

    case 'N':
    case 'X':
      cncall();
      prfmsg(NOMAIL);
      return TRUE;

    default:
      EnterState(WEMAILU);
      return FALSE;
    }

  }

static BOOL StateMailCode(void)
  {
  CHAR MailCode[6];
  PSZ  pAll;

  sprintf(MailCode, "%04x", pCurrentUser->MailCode);
  strupr(MailCode);

  pAll = cncall();

  if (*pAll == 'X')
    if (strlen(pAll) == 1)
      {
      prf("...Exiting!\r");
      return TRUE;
      }

  if (!stricmp(MailCode, pAll))
    {
    if (LogSuccess)
      shocst("AV: MAIL VALIDATION", "User-Id: %s", pCurrentUser->Name);

    prfmsg(UMAILVAL);
    pCurrentUser->Status = AVS_VALIDATED;
    ValidateUser(pCurrentUser);
    setmbk(pMessage);
    strcpy(pCurrentUser->AuxPhone, "000000MAIL");
    LogEvent("MAIL: User %s validated with proper mail code.",
              pCurrentUser->Name);
    return TRUE;
    }
  else
    {
    pCurrentUser->Attempts++;

    LogScreen("MAIL: User %s entered invalid mail code.",
              pCurrentUser->Name);

    prfmsg(BADMAILC);
    return TRUE;
    }
  }

static BOOL StateWeDo(void)
  {
  switch (cncchr())
    {
    case 'D':
      EnterState(WECALLU);
      return FALSE;

    case 'M':
      EnterState(WEMAILU);
      return FALSE;

    case 'N':
    case 'X':
      prfmsg(NONOTHN);
      return TRUE;

    default:
      EnterState(WEDOU);
      return FALSE;
    }
  }

static BOOL StateWeCall(void)
  {
  USHORT Channel;

  switch (cncchr())
    {
    case 'Y':
      cncall();
      Channel = FindEmptyPreSlot();

      if (Channel != NO_CHANNEL)
        {
        byenow(WECALLIN);
        pInfo[Channel].User          = pInfo[usrnum].User;
        pInfo[Channel].TotalAttempts = 0;   /* Total tries, period        */
        RestartAttempt(Channel);
        PostMarkerCredit(pInfo + Channel);
        StartPulse();
        return FALSE;
        }
      else
        {
        prfmsg(WEFULL);
        return TRUE;
        }

    case 'N':
    case 'X':
      cncall();
      prfmsg(NOCALL);
      return TRUE;

    default:
      EnterState(WECALLU);
      return FALSE;
    }
  }

static int AVMain(void)
  {
  BOOL Exiting   = FALSE;
  BOOL Processed = FALSE;

  setmbk(pMessage);

  if ((usrptr->substt != 0))      /* Check null input and ? first */
    {
    if (margc == 0)
      {
      Processed = TRUE;
      prfmsg(usrptr->substt + 1);
      }

    if ((margc == 1) && (input[0] == '?'))
      {
      Processed = TRUE;
      EnterState(usrptr->substt);
      }
    }

  if (!Processed)                 /* If not that, then handle the state */
    {

    do
      {
      bgncnc();
      clrprf();

      switch (usrptr->substt)
        {
        case 0:
          cncchr();
          Exiting = StateEntry();
          break;

        case WECALLU:
          Exiting = StateWeCall();
          break;

        case WEMAILU:
          Exiting = StateWeMail();
          break;

        case WEDOU:
          Exiting = StateWeDo();
          break;

        case MAILCODE:
          Exiting = StateMailCode();
          break;

        default:
          prf("! Unknown substt -> %d\r", usrptr->substt);
          Exiting = TRUE;

        }
      }
    while (!endcnc());

    }

  if (Exiting)
    prfmsg(USREXIT);

  outprf(usrnum);

  if (Exiting)
    return 0;
  else
    return 1;
  }

static BOOL GetConfigLine(void)
  {
  BOOL EndOfFile = FALSE;
  USHORT CharIndex;

  while (TRUE)
    {
    if (feof(pConfig))
      {
      EndOfFile = TRUE;
      break;
      }

    fgets(ConfigLine, BIG_STRING, pConfig);

    for (CharIndex = 0; CharIndex < BIG_STRING; CharIndex++)
      if (ConfigLine[CharIndex] == '\n')
        {
        ConfigLine[CharIndex] = '\0';       /* Get rid of newline      */
        break;
        }

    if ((ConfigLine[0] != ';') &&
        (ConfigLine[0] != '#') )
      {
      if (feof(pConfig))      /* This is last line in file! */
        EndOfFile = TRUE;

      break;
      }
    }

  return EndOfFile;
  }

static void ProcessSystemAreaCode(void)
  {
  GetConfigLine();     /* Get next line, since that's where it is! */

  SystemAreaCode = atoi(ConfigLine);

  if (SystemAreaCode == 0)
    {
    shocst("AV: NO SYS AREA CODE", "The Auto Validator needs to be configured!");
    Enabled = FALSE;          /* Turn it off before someone uses it! */
    }
  }

static BOOL AddToLongList(PULONG pList, ULONG Value, USHORT Max)
  {
  USHORT SlotCheck;
  BOOL   Added = FALSE;

  for (SlotCheck = 0; SlotCheck < Max; SlotCheck++)
    if ((pList[SlotCheck] == 0) || (pList[SlotCheck] == Value))
      {
      pList[SlotCheck] = Value;
      Added = TRUE;
      break;
      }

  return Added;
  }

static BOOL AddToNumberList(PUSHORT pList, USHORT Value, USHORT Max)
  {
  USHORT SlotCheck;
  BOOL   Added = FALSE;

  for (SlotCheck = 0; SlotCheck < Max; SlotCheck++)
    if ((pList[SlotCheck] == 0) || (pList[SlotCheck] == Value))
      {
      pList[SlotCheck] = Value;
      Added = TRUE;
      break;
      }

  return Added;
  }

static BOOL SearchShortList(PUSHORT pList, USHORT Value, USHORT Max)
  {
  USHORT SlotCheck;
  BOOL   Found = FALSE;

  for (SlotCheck = 0; (SlotCheck < Max) && (pList[SlotCheck] != 0); SlotCheck++)
    if (pList[SlotCheck] == Value)
      {
      Found = TRUE;
      break;
      }

  return Found;
  }

static BOOL SearchLongList(PULONG pList, ULONG Value, USHORT Max)
  {
  USHORT SlotCheck;
  BOOL   Found = FALSE;

  for (SlotCheck = 0; (SlotCheck < Max) && (pList[SlotCheck] != 0); SlotCheck++)
    if (pList[SlotCheck] == Value)
      {
      Found = TRUE;
      break;
      }

  return Found;
  }

static void ProcessNeverNumberList(void)
  {
  ULONG ConfigValue;
  BOOL  Added;
  BOOL  EndOfFile;

  while (TRUE)
    {
    EndOfFile = GetConfigLine();

    if ((ConfigLine[0] == '\0') || (ConfigLine[0] == ' '))
      break;

    ConfigValue = atol(ConfigLine);

    if (ConfigValue > 0)
      {
      Added = AddToLongList(pNeverNumbers, ConfigValue, MAX_NEVER_NUMBERS);

      if (!Added)
        catastro("AV: TOO MANY %s", pCurrentAction);
      }

    if (EndOfFile)
      break;
    }
  }

static void ProcessGenericList(void)
  {
  USHORT ConfigValue;
  BOOL   Added;
  BOOL   EndOfFile;

  while (TRUE)
    {
    EndOfFile = GetConfigLine();

    if ((ConfigLine[0] == '\0') || (ConfigLine[0] == ' '))
      break;

    ConfigValue = atoi(ConfigLine);

    if (ConfigValue > 0)
      {
      Added = AddToNumberList(pCurrentList, ConfigValue, CurrentMax);

      if (!Added)
        catastro("AV: TOO MANY %s", pCurrentAction);
      }

    if (EndOfFile)
      break;
    }
  }

static void ReadConfiguration(void)
  {
  BOOL EndOfFile;

  pConfig = fopen(AV_CONFIG_NAME, "rt");

  if (pConfig == NULL)
    catastro("MISSING CONFIG %s", AV_CONFIG_NAME);

  EndOfFile = GetConfigLine();

  while (!EndOfFile)
    {
    if (sameto("SystemAreaCode:", ConfigLine))
      ProcessSystemAreaCode();

    if (sameto("TollPrefixes:", ConfigLine))
      {
      pCurrentList   = pPrefixList;
      CurrentMax     = MAX_PREFIX_LIST;
      pCurrentAction = "BadPrefixes";
      PrefixListBad  = TRUE;

      ProcessGenericList();
      }

    if (sameto("LocalPrefixes:", ConfigLine))
      {
      pCurrentList   = pPrefixList;
      CurrentMax     = MAX_PREFIX_LIST;
      pCurrentAction = "GoodPrefixes";
      PrefixListBad  = FALSE;

      ProcessGenericList();
      }

    if (sameto("BadAreaCodes:", ConfigLine))
      {
      pCurrentList   = pAreaCodeList;
      CurrentMax     = MAX_AREA_CODE_LIST;
      pCurrentAction = "BadAreaCodes";
      AreaCodeListBad = TRUE;

      ProcessGenericList();
      }

    if (sameto("GoodAreaCodes:", ConfigLine))
      {
      pCurrentList   = pAreaCodeList;
      CurrentMax     = MAX_AREA_CODE_LIST;
      pCurrentAction = "GoodAreaCodes";
      AreaCodeListBad = FALSE;

      ProcessGenericList();
      }

    if (sameto("NeverPrefixes:", ConfigLine))
      {
      pCurrentList   = pNeverPrefixes;
      CurrentMax     = MAX_NEVER_PREFIXES;
      pCurrentAction = "NeverPrefixes";

      ProcessGenericList();
      }

    if (sameto("NeverAreaCodes:", ConfigLine))
      {
      pCurrentList   = pNeverAreaCodes;
      CurrentMax     = MAX_NEVER_AREA_CODES;
      pCurrentAction = "NeverAreaCodes";

      ProcessGenericList();
      }

    if (sameto("NeverNumbers:", ConfigLine))
      {
      pCurrentAction = "NeverNumbers";
      ProcessNeverNumberList();
      }

    EndOfFile = GetConfigLine();
    }

  fclose(pConfig);
  }

static void ParsePhoneNumber(PSZ pInput, PSZ pOutput)
  {
  CHAR   TempBuffer[BIG_STRING];
  USHORT Digits = 0;
  PSZ pStartOut = pOutput;

  while (*pInput != '\0')
    {
    if (isdigit(*pInput))
      {
      *pOutput = *pInput;
      pOutput++;
      Digits++;
      }

    pInput++;
    }

  switch (Digits)
    {
    case 7:       /* Prefix and suffix, but no area code - must fix */
      *pOutput = '\0';
      sprintf(TempBuffer, "%3d%s", SystemAreaCode, pStartOut);
      strcpy(pStartOut, TempBuffer);
      break;

    case 10:      /* Area code, prefix, suffix - everything ok      */
      *pOutput = '\0';
      break;

    default:      /* ?! */
      strcpy(pStartOut, "BADFORMAT!");
      break;
    }
  }

static void DisassemblePhone(PSZ pPhone,
                             PUSHORT pAreaCode,
                             PUSHORT pPrefix,
                             PUSHORT pSuffix,
                             PULONG  pNumber)
  {
  CHAR TempBuffer[8];

  if (pAreaCode != NULL)
    {
    strncpy(TempBuffer, pPhone, 3);
    TempBuffer[3] = '\0';

    *pAreaCode = atoi(TempBuffer);
    }

  if (pPrefix != NULL)
    {
    strncpy(TempBuffer, pPhone + 3, 3);
    TempBuffer[3] = '\0';

    *pPrefix = atoi(TempBuffer);
    }

  if (pSuffix != NULL)
    {
    strncpy(TempBuffer, pPhone + 6, 4);
    TempBuffer[4] = '\0';

    *pSuffix = atoi(TempBuffer);
    }


  /* The pNumber must be stored as a long, so go back and do it. */

  if (pNumber != NULL)
    {
    strncpy(TempBuffer, pPhone + 3, 7);
    TempBuffer[7] = '\0';

    *pNumber = atol(TempBuffer);
    }

  }

static BOOL IsToll(USHORT Prefix)
  {
  BOOL InList = SearchShortList(pPrefixList, Prefix, MAX_PREFIX_LIST);

  if (!PrefixListBad)
    InList = !InList;

  return InList;
  }

static BOOL OkLongDistance(USHORT AreaCode)
  {
  BOOL InList = SearchShortList(pAreaCodeList, AreaCode, MAX_AREA_CODE_LIST);
  BOOL DialIt = TRUE;

  if (AllowLongDistance)
    {
    if (AreaCodeListBad && InList)
      DialIt = FALSE;
    }
  else
    {
    DialIt = FALSE;       /* Default is NOT to do it this time */

    if ( (!AreaCodeListBad) && InList)
      DialIt = TRUE;
    }

  return DialIt;
  }

static void ScreenPhone(PAVREC pUser)
  {
  BYTE Range  = AVR_UNKNOWN;
  BYTE Status = AVS_NOT_TRIED;
  USHORT Index;
  USHORT AreaCode;
  USHORT Prefix;
  USHORT Suffix;
  ULONG  Number;
  PSZ    pPhone = pUser->Phone;

  for (Index = 0; Index < AV_PHONE - 1; Index++) /* Don't look at the null! */
    if (!isdigit(pPhone[Index]))
      {
      Status = AVS_BAD_FORMAT;
      break;
      }

  if (Status == AVS_NOT_TRIED)
    {
    DisassemblePhone(pPhone, &AreaCode, &Prefix, &Suffix, &Number);

    if (SearchLongList(pNeverNumbers, Number, MAX_NEVER_NUMBERS))
      Status = AVS_BAD_NUMBER;

    if ((Status == AVS_NOT_TRIED) &&
        SearchShortList(pNeverAreaCodes, AreaCode, MAX_NEVER_AREA_CODES))
      Status = AVS_BAD_AREA_CODE;

    if ((Status == AVS_NOT_TRIED) &&
        SearchShortList(pNeverPrefixes, Prefix, MAX_NEVER_PREFIXES))
      Status = AVS_BAD_PREFIX;

    if (Status == AVS_NOT_TRIED)
      {
      if (AreaCode != SystemAreaCode)
        {
        Range = AVR_LONG_DISTANCE;

        if (!OkLongDistance(AreaCode) )
          Status = AVS_LONG_DISTANCE;
        }
      else
        {
        if (IsToll(Prefix))
          {
          Range  = AVR_TOLL;

          if (!AllowToll)
            Status = AVS_TOLL;
          }
        }

      if (Range == AVR_UNKNOWN)
        Range = AVR_LOCAL;
      }

    }

  pUser->Status = Status;
  pUser->Range  = Range;
  }

static USHORT FindAndBusyEmptyChannel(PAVREC pTarget)
  {
  CHAR   ModemString[100];
  USHORT MinChannel;
  USHORT MaxChannel;
  USHORT ChannelNum;
  SHORT  User;
  BOOL   ValidChannel = FALSE;
  BOOL   Found = FALSE;

  switch (pTarget->Range)
    {
    case AVR_LOCAL:
      MinChannel = hexopt(AVNLOW,  0, 0xFF);
      MaxChannel = hexopt(AVNHIGH, 0, 0xFF);
      break;

    case AVR_TOLL:
      MinChannel = hexopt(AVTLOW,  0, 0xFF);
      MaxChannel = hexopt(AVTHIGH, 0, 0xFF);
      break;

    case AVR_LONG_DISTANCE:
      MinChannel = hexopt(AVLLOW,  0, 0xFF);
      MaxChannel = hexopt(AVLHIGH, 0, 0xFF);
      break;

    default:              /* Serious internal confusion */
      if (SecureMode && BypassScreen)
        {
        MinChannel = hexopt(AVNLOW,  0, 0xFF);
        MaxChannel = hexopt(AVNHIGH, 0, 0xFF);
        }
      else
        return NO_CHANNEL;
    }

  if (MinChannel > MaxChannel)
    {
    USHORT Temp = MinChannel;
    MinChannel  = MaxChannel;
    MaxChannel  = Temp;
    }

  for (ChannelNum = MaxChannel; ChannelNum >= MinChannel; ChannelNum--)
    {
    User = usridx(ChannelNum);

    if (User != -1)
      ValidChannel = TRUE;

    if ((User != -1) &&
        (user[User].class == VACANT) &&
        (user[User].state == AWAITC) )     // Busy-out modems will not
      {                                    // show up with AWAITC state!
      user[User].state = STATE_BUSY;
      strcpy(ModemString, pBusyString);
      strcat(ModemString, pSuffix);
      btuxmt(User, ModemString);
      SetNewLegend(User,
                   spr("AV: Busy-out for %s", pTarget->Name),
                   AUTOV_CHARACTER);
      Found = TRUE;
      break;
      }
    }

  if (!ValidChannel)
    {
    shocst("AV: INVALID CHANNELS", "%02X to %02X are bad! (check HVSAUTOV.MSG)", MinChannel, MaxChannel);
    setbtv(pbUserRec);
    Enabled = FALSE;    /* Disable module from user selection! */
    }

  if (!Found)
    User = NO_CHANNEL;

  return User;
  }

static void StartPulse(void)
  {
  if (!PulseActive)
    {
    PulseActive = TRUE;
    rtkick(1, Pulse);
    }
  }

static BOOL ValidateUser(PAVREC pUser)
  {
  BOOL Posted = FALSE;
  LONG Credits;
  BOOL Paid;

  struct usracc TempUser;
  struct usracc * pChange;

  switch (pUser->Range)
    {
    case AVR_LOCAL:
      Paid    = ynopt(AVNPAID);
      Credits = lngopt(AVNCRED, 0L, 1000000L);
      break;

    case AVR_TOLL:
      Paid    = ynopt(AVTPAID);
      Credits = lngopt(AVTCRED, 0L, 1000000L);
      break;

    case AVR_LONG_DISTANCE:
      Paid    = ynopt(AVLPAID);
      Credits = lngopt(AVLCRED, 0L, 1000000L);
      break;

    default:            /* Severe internal confusion! */
      Credits = 0L;
      break;
    }

  if (PostFirst)
    {
    if (Credits > 0L)
      {
      Posted = TRUE;
      NullPad(pUser->Name, UIDSIZ);
      addcrd(pUser->Name, spr("%ld", Credits), Paid);
      }
    else
      Posted = FALSE;
    }

  if (ClassSwitch[pUser->Range])
    {
    if (onsysn(TempHandle(pUser->Name), 1))
      pChange = othuap;
    else
      {
      setbtv(accbb);

      if (!acqbtv(&TempUser, TempHandle(pUser->Name), 0))
        catastro("COULD NOT GET USER %s FROM ACCOUNT DATABASE!", pUser->Name);

      pChange = &TempUser;
      }

    swtcls(pChange, 1, pClasses[pUser->Range], 4, 0);

    setmbk(pMessage);
    setbtv(pbUserRec);
    }

  if (!PostFirst)
    {
    if (Credits > 0L)
      {
      Posted = TRUE;
      NullPad(pUser->Name, UIDSIZ);
      addcrd(pUser->Name, spr("%ld", Credits), Paid);
      }
    else
      Posted = FALSE;
    }

  if (pUser->Access != AVA_RESTRICT)
    pUser->Access = AVA_AUTO;

  return Posted;
  }

static BOOL LogValidation(PAVREC pUser)
  {
  BOOL ValCode;

  LogEvent("SUCCESS: Phone validated user %s.", pUser->Name);

  strcpy(pUser->AuxPhone, pUser->Phone);
  pUser->Status = AVS_VALIDATED;
  pUser->Attempts++;

  if (LogSuccess)
    shocst("AV: AUTO VALIDATION", "User-Id: %s", pUser->Name);

  setbtv(pbUserRec);
  geqbtv(NULL, pUser->Name, 0);
  updbtv(pUser);

  ValCode = ValidateUser(pUser);
  setmbk(pMessage);

  return ValCode;
  }

static void LogFailure(PCHANNELINFO pChanInfo, USHORT Channel)
  {
  PAVREC pUser = &(pChanInfo->User);

  pUser->Status = AVS_FAILED;
  pUser->Attempts++;

  setbtv(pbUserRec);
  geqbtv(NULL, pUser->Name, 0);
  updbtv(pUser);

  TakeMarkerCredit(pInfo + Channel);
  }

static PSZ TempHandle(PSZ pName)
  {
  static CHAR TempBuffer[BIG_UIDSIZ];

  strcpy(TempBuffer, pName);
  NullPad(TempBuffer, UIDSIZ);

  makhdl(TempBuffer);

  return TempBuffer;
  }

static void PostMarkerCredit(PCHANNELINFO pInfo)
  {
  if (uidkey(pInfo->User.Name, pSampleKey))
    pInfo->TakeAwayKey = FALSE;
  else
    {
    pInfo->TakeAwayKey = TRUE;
    givkey(pInfo->User.Name, pSampleKey);
    }

  setbtv(pbUserRec);
  }

static void TakeMarkerCredit(PCHANNELINFO pInfo)
  {
  if (pInfo->TakeAwayKey)
    {
    pInfo->TakeAwayKey = FALSE;    /* Better safe than sorry! */
    rmvkey(pInfo->User.Name, pSampleKey);
    }

  setbtv(pbUserRec);
  }

static void Pulse(void)
  {
  static CHAR ModemString[100];
  BOOL KeepGoing = FALSE;
  USHORT Index;
  USHORT NewChannel;
  PCHANNELINFO pCurrent = pInfo;

  setmbk(pMessage);
  setbtv(pbUserRec);

  for (Index = 0; Index < nterms + MaxAutos; Index++, pCurrent++)
    switch (pCurrent->Status)
      {
      case CS_IDLE:
        break;

      case CS_DIALPENDING:
        KeepGoing = TRUE;
        pCurrent->TicksLeft--;

        if (pCurrent->TicksLeft == 0)
          {
          NewChannel = FindAndBusyEmptyChannel(&(pCurrent->User));

          if (NewChannel == NO_CHANNEL)
            {
            pCurrent->Retries++;

            if (pCurrent->Retries == MAX_LOOKS)
              {
              LogEvent("FAILURE: Unable to find vacant channel to validate user %s.",
                       pCurrent->User.Name);

              LogFailure(pCurrent, Index);
              pCurrent->Status = CS_IDLE;
              }

            pCurrent->TicksLeft = FIND_RETRY;
            }
          else
            {
            pCurrent->Status = CS_IDLE;

            pInfo[NewChannel].Status      = CS_BUSYOUT;
            pInfo[NewChannel].TicksLeft   = BUSY_DELAY;
            pInfo[NewChannel].Retries     = 0;
            pInfo[NewChannel].User        = pCurrent->User;

            pInfo[NewChannel].TakeAwayKey = pCurrent->TakeAwayKey;
            }
          }
        break;

      case CS_BUSYOUT:
        KeepGoing = TRUE;
        pCurrent->TicksLeft--;

        if (pCurrent->TicksLeft == 0)
          {
          strcpy(ModemString, pNoBusyString);
          strcat(ModemString, pSuffix);
          btuxmt(Index, ModemString);

          SetNewLegend(Index,
                      spr("AV: Pausing for %s", pUserOnChannel(Index)->Name),
                      AUTOV_CHARACTER);

          pCurrent->Status++;        /* Promote to next status - HANGUPWAIT */
          pCurrent->TicksLeft = HANGUP_DELAY;
          }
        break;

      case CS_HANGUPWAIT:
        KeepGoing = TRUE;
        pCurrent->TicksLeft--;

        if (pCurrent->TicksLeft == 0)
          {
          SetNewLegend(Index,
                      spr("AV: Dialing %s", pUserOnChannel(Index)->Name),
                      AUTOV_CHARACTER);

          pCurrent->Status++;        /* Promote to next status - DIALING */
          pCurrent->TicksLeft = DIAL_TIMEOUT;
          btuoes(Index, 0);
          user[Index].state = HARING;
          btucli(Index);

          if (SecureMode)
            LogEvent("SECURE: Dial  -> %s", pUserOnChannel(Index)->Name);

          SendDialString(Index);
          }
        break;

      case CS_DIALING:
        pCurrent->TicksLeft--;

        if ((user[Index].state != HARING) && (user[Index].state != STATE_BUSY))
          if (user[Index].state == JSTRST)
            {
            pCurrent->Status = CS_IDLE;

            if (pCurrent->TotalAttempts == MaxRetries)
              {
              LogEvent("FAILED: Dial attempts failed to connect with user %s.",
                       pCurrent->User.Name);

              LogFailure(pInfo + Index, Index);
              }
            else
              {
              pCurrent->TotalAttempts++;
              RestartAttempt(Index);
              KeepGoing = TRUE;
              break;
              }
            }
          else
            {
            pCurrent->Status++;        /* Promote to next status - CARRIER */
            pCurrent->TicksLeft++;     /* In case it connected last second */
            KeepGoing = TRUE;
            }

        if (pCurrent->TicksLeft == 0)
          {
          if (pCurrent->TotalAttempts == MaxRetries)
            {
            pCurrent->Status = CS_IDLE;

            LogEvent("FAILED: Dial attempts failed to connect with user %s.",
                     pCurrent->User.Name);

            LogFailure(pInfo + Index, Index);
            }
          else
            {
            pCurrent->TotalAttempts++;
            RestartAttempt(Index);
            KeepGoing = TRUE;
            }

          usrnum = Index;
          rstchn();
          }
        else
          KeepGoing = TRUE;

        break;

      case CS_CARRIER:
        KeepGoing = TRUE;

        if ((user[Index].state == JSTRST) && (user[Index].class == VACANT))
          {
          LogEvent("FAILED: Connected with user %s, but lost carrier before log on.", pCurrent->User.Name);
          if (pCurrent->TotalAttempts == MaxRetries)
            LogFailure(pInfo + Index, Index);
          else
            {
            pCurrent->TotalAttempts++;
            RestartAttempt(Index);
            break;
            }
          pCurrent->Status = CS_IDLE;
          }
        break;
      }

  if (KeepGoing)
    rtkick(1, Pulse);
  else
    PulseActive = FALSE;
  }

static void SendDialString(USHORT Channel)
  {
  CHAR ModemString[100];
  PSZ pPhone;

  if (SecureMode && BypassScreen)
    {
    struct usracc TempUser;

    setbtv(accbb);
    acqbtv(&TempUser, pInfo[Channel].User.Name, 0);
    rstbtv();

    strcpy(ModemString, pLocalPrefix);
    strcat(ModemString, TempUser.usrpho);
    }
  else
    {
    switch (pInfo[Channel].User.Range)
      {
      case AVR_LOCAL:
        strcpy(ModemString, pLocalPrefix);
        pPhone = pInfo[Channel].User.Phone + 3;    /* Skip area code       */
        break;

      case AVR_TOLL:
        strcpy(ModemString, pTollPrefix);

        if (AreaCodeToll)
          pPhone = pInfo[Channel].User.Phone;        /* Don't skip anything! */
        else
          pPhone = pInfo[Channel].User.Phone + 3;    /* Skip area code. */

        break;

      case AVR_LONG_DISTANCE:
        strcpy(ModemString, pLongDistancePrefix);

        if (AreaCodeLong)
          pPhone = pInfo[Channel].User.Phone;        /* Don't skip anything! */
        else
          pPhone = pInfo[Channel].User.Phone + 3;    /* Skip area code. */

        break;
      }

    strcat(ModemString, pPhone);
    }

  if (SecureMode)
    LogEvent("        Modem -> %s", ModemString);

  strcat(ModemString, pSuffix);
  btuxmt(Channel, ModemString);
  }

static USHORT IncrementPhone(PSZ pPhone)
  {
  USHORT   Phones = 1;
  PHONEREC TempRec;

  setbtv(pbPhoneRec);

  if (acqbtv(&TempRec, pPhone, 0))
    {
    TempRec.Count++;
    Phones = TempRec.Count;
    updbtv(&TempRec);
    }
  else
    {
    strcpy(TempRec.Phone, pPhone);
    TempRec.Count = 1;
    insbtv(&TempRec);
    }

  return Phones;
  }

static USHORT DecrementPhone(PSZ pPhone)
  {
  USHORT   Phones = 0;
  PHONEREC TempRec;

  setbtv(pbPhoneRec);

  if (acqbtv(&TempRec, pPhone, 0))
    {
    TempRec.Count--;

    if (TempRec.Count == 0)
      {
      delbtv();
      }
    else
      {
      Phones = TempRec.Count;
      updbtv(&TempRec);
      }
    }

  return Phones;
  }

static USHORT QueryPhone(PSZ pPhone)
  {
  PHONEREC TempRec;

  setbtv(pbPhoneRec);

  if (acqbtv(&TempRec, pPhone, 0))
    return TempRec.Count;
  else
    return 0;
  }

static PAVREC GetUser(PSZ pUserID)
  {
  static AVREC TempRec;

  PAVREC pUser = NULL;
  USHORT Index;
  BOOL   OnLine = FALSE;

  if (strlen(pUserID) < 3)      /* It's just not enough */
    return NULL;

  for (Index = 0; Index < nterms; Index++)
    if (sameas(pUserID, pUserOnChannel(Index)->Name))
      {
      OnLine = TRUE;
      pUser = pUserOnChannel(Index);
      break;
      }

  if (!OnLine)
    if (acqbtv(&TempRec, TempHandle(pUserID), 0))
      pUser = &TempRec;

  return pUser;
  }

static PAVREC GetCommandUser(void)
  {
  PAVREC pFound;

  nxtcmd = margv[1];
  rstrin();

  pFound = GetUser(cncuid());

  if (pFound == NULL)
    prfmsg(NOUSER);
  else               /* Check for orphaned account! */
    {
    setbtv(accbb);

    if (!acqbtv(NULL, TempHandle(pFound->Name), 0))   /* Forget it was even there! */
      {
      AVDelete(pFound->Name);
      pFound = NULL;
      prfmsg(NOUSER);
      }
    }

  return pFound;
  }

static void NullPad(PSZ pString, USHORT Length)
  {
  BOOL Padding = FALSE;

  while (Length > 0)
    {
    if (Padding)
      *pString = '\0';

    if (*pString == '\0')
      Padding = TRUE;

    pString++, Length--;
    }
  }

static void UpdateUser(PAVREC pUser)
  {
  CHAR TempPhone[BIG_STRING];
  struct usracc User;

  if (onsysn(pUser->Name, 0))
    User = *othuap;
  else
    {
    setbtv(accbb);
    acqbtv(&User, TempHandle(pUser->Name), 0);
    setbtv(pbUserRec);
    }

  if (pUser->Phone[0] == '-')
    {
    CHAR TempBuffer[6];
    PSZ pPhone = pUser->Phone;

    sprintf(TempBuffer, "%03d", SystemAreaCode);

    pPhone[0] = TempBuffer[0];
    pPhone[1] = TempBuffer[1];
    pPhone[2] = TempBuffer[2];

    IncrementPhone(pPhone);
    }

  if (pUser->Access == AVA_RESCREEN)
    pUser->Access = AVA_AUTO;

  ParsePhoneNumber(User.usrpho, TempPhone);

  if (!sameas(TempPhone, pUser->Phone))
    {
    DecrementPhone(pUser->Phone);
    strcpy(pUser->Phone, TempPhone);
    IncrementPhone(pUser->Phone);

    pUser->Attempts = 0;
    }

  if (pUser->Access == AVA_AUTO)
    ScreenPhone(pUser);

  setbtv(pbUserRec);
  }

static void SaveUser(PAVREC pUser)
  {
  USHORT Index;
  BOOL   OnLine = FALSE;

  for (Index = 0; Index < nterms; Index++)
    if (!strcmp(pUser->Name, pUserOnChannel(Index)->Name))
      {
      OnLine = TRUE;
      pInfo[Index].User = *pUser;
      break;
      }

  if (!OnLine)
    {
    geqbtv(NULL, pUser->Name, 0);
    updbtv(pUser);
    }
  }

static void ShowActivity(void)
  {
  BOOL   Some = FALSE;
  USHORT Index;

  prfmsg(SYSACTH);

  for (Index = 0; Index < nterms; Index++)
    if (pInfo[Index].Status != CS_IDLE)
      {
      Some = TRUE;
      prfmsg(SYSACT, spr("%02X", channel[Index]), pInfo[Index].User.Name);
      }

  if (!Some)
    prfmsg(SYSNOAC);
  else
    prf("\r");
  }

static int GlobalCommand(void)
  {
  PAVREC pUser;

  if ( (input[0] == GlobalKick) &&
       (strlen(margv[0]) == 2)  &&
       IsSysop()                &&
       (inplen > 1) )
    {
    setmbk(pMessage);
    setbtv(pbUserRec);

    switch (toupper(input[1]))
      {
      case 'M':
        if (SecureMode)
          {
          prfmsg(CANTSEC);
          break;
          }

        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            if (pUser->Status == AVS_VALIDATED)
              {
              prfmsg(LABLVAL);
              break;
              }

            if (pUser->Access == AVA_RESTRICT)
              {
              prfmsg(LABLRES);
              break;
              }

            if (pUser->Status != AVS_MAILED)
              {
              pUser->Status = AVS_MAILED;
              pUser->MailCode = rand() + rand() + rand() + rand();
              setbtv(pbUserRec);
              setmbk(pMessage);
              SaveUser(pUser);
              }

            AddLabel(pUser);
            prfmsg(LABELD, pUser->Name);
            }
          }
        break;

      case 'I':
        ShowActivity();
        break;

      case 'L':
        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            PSZ  pPhoneString = "Signup phone  .......";
            CHAR TempPhone[20];
            strcpy(TempPhone, FormatPhone(pUser->AuxPhone) );

            if (pUser->Status == AVS_VALIDATED)
              pPhoneString = "Validated phone  ....";

            prfmsg(USERINFO, pUser->Name,
                             FormatPhone(pUser->Phone),
                             pPhoneString,
                             TempPhone,
                             Ranges[pUser->Range],
                             UserStatuses[pUser->Status],
                             pUser->Attempts,
                             QueryPhone(pUser->Phone),
                             Accesses[pUser->Access] );

            if (pUser->Status == AVS_MAILED)
              {
              CHAR CodeBuffer[6];

              sprintf(CodeBuffer, "%04x", pUser->MailCode);
              strupr(CodeBuffer);

              prfmsg(MAILINFO, CodeBuffer);
              }
            }
          }
        break;

      case 'V':
        if (SecureMode)
          {
          prfmsg(CANTSEC);
          break;
          }

        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            pUser->Status = AVS_VALIDATED;
            shocst("AV: FORCE VALIDATION", "User-Id: %s", pUser->Name);
            ValidateUser(pUser);
            setbtv(pbUserRec);
            setmbk(pMessage);
            SaveUser(pUser);
            prfmsg(USERVALD);
            }
          }
        break;

      case 'D':
        if (SecureMode)
          {
          prfmsg(CANTSEC);
          break;
          }

        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            setbtv(pbUserRec);
            pUser->Status = AVS_NOT_TRIED;
            ScreenPhone(pUser);
            SaveUser(pUser);
            prfmsg(USERDEVD);
            }
          }
        break;

      case 'O':
        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          BOOL CanOverride = TRUE;

          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            switch (pUser->Status)
              {
              case AVS_BAD_FORMAT:
                prfmsg(SYSBADF);
                CanOverride = FALSE;
                break;

              case AVS_VALIDATED:
                prfmsg(SYSVALD);
                CanOverride = FALSE;
                break;

              case AVS_BAD_AREA_CODE:
              case AVS_BAD_PREFIX:
              case AVS_BAD_NUMBER:
                prfmsg(SYSBADN);
                CanOverride = FALSE;
                break;
              }

            if (CanOverride)
              {
              pUser->Access = AVA_OVERRIDE;

              if (pUser->Range == AVR_UNKNOWN)
                pUser->Range = AVR_LOCAL;

              setbtv(pbUserRec);
              SaveUser(pUser);
              prfmsg(USEROVER);
              }
            }
          }
        break;

      case 'R':
        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            setbtv(pbUserRec);
            pUser->Access = AVA_RESTRICT;
            SaveUser(pUser);
            prfmsg(USERREST);
            }
          }
        break;

      case 'A':
        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            setbtv(pbUserRec);
            pUser->Access = AVA_AUTO;
            SaveUser(pUser);
            prfmsg(USERAUTO);
            }
          }
        break;

      case 'U':
        if (margc < 2)
          prfmsg(NEEDUSER);
        else
          {
          pUser = GetCommandUser();

          if (pUser != NULL)
            {
            UpdateUser(pUser);
            SaveUser(pUser);
            prfmsg(USERUPD);
            }
          }
        break;

      case 'T':
        if (margc >= 2)
          {
          if (!stricmp(margv[1], "ON"))
            Enabled = TRUE;

          if (!stricmp(margv[1], "OFF"))
            Enabled = FALSE;
          }

        if (Enabled)
          prfmsg(AVTOGGL, "on");
        else
          prfmsg(AVTOGGL, "off");

        break;

      case 'C':
        if (margc >= 2)
          {
          if (!stricmp(margv[1], "ON"))
            CleanupRefresh = TRUE;

          if (!stricmp(margv[1], "OFF"))
            CleanupRefresh = FALSE;

          }

        if (CleanupRefresh)
          prfmsg(CLSTAT, "on");
        else
          prfmsg(CLSTAT, "off");

        break;

      case '?':
        prfmsg(SYSHELP, AUTOV_VERSION,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick,
                        GlobalKick);
        break;

      default:
        return 0;
      }

    outprf(usrnum);

    return 1;
    }
  else
    return 0;
  }

static void AVCleanup(void)
  {
  if (CleanupRefresh)
    RefreshAll();
  }

static void AVDelete(PSZ pUserName)
  {
  AVREC DeletedUser;
  BOOL  Found;

  setbtv(pbUserRec);

  Found = acqbtv(&DeletedUser, TempHandle(pUserName), 0);

  if (Found)
    {
    delbtv();
    DecrementPhone(DeletedUser.Phone);
    }
  }

static void RefreshAll(void)
  {
  BOOL  Found;
  AVREC CurrentUser;
  PSZ pFormat;
  struct usracc CurrentAccount;

  setbtv(pbUserRec);

  qlobtv(0);

  do
    {
    gcrbtv(&CurrentUser, 0);

    setbtv(accbb);
    Found = acqbtv(&CurrentAccount, TempHandle(CurrentUser.Name), 0);

    if (!Found)     /* Account deleted while AV was not on-line! */
      {
      AVDelete(CurrentUser.Name);
      }
    else
      {
      pFormat = FormatPhone(CurrentAccount.usrpho);

      if (strcmp(pFormat, CurrentUser.Phone))    /* Different! */
        {
        DecrementPhone(CurrentUser.Phone);

        ParsePhoneNumber(CurrentAccount.usrpho, CurrentUser.Phone);
        IncrementPhone(CurrentUser.Phone);

        if (CurrentUser.Access == AVA_RESCREEN)
          CurrentUser.Access = AVA_AUTO;

        if ((CurrentUser.Access  == AVA_AUTO) &&
            (CurrentUser.Status  != AVS_VALIDATED))
          ScreenPhone(&CurrentUser);

        CurrentUser.Attempts = 0;

        setbtv(pbUserRec);
        gcrbtv(NULL, 0);
        updbtv(&CurrentUser);
        }
      else
        setbtv(pbUserRec);
      }

    } while (qnxbtv());
  }

static PSZ FormatPhone(PSZ pPhone)
  {
  static CHAR FormatBuffer[30];
  PSZ    pCurrent = FormatBuffer;
  USHORT Position = 0;

  while (*pPhone != '\0')
    {
    switch (Position)
      {
      case 0:
        *pCurrent = '(';
        pCurrent++;
        break;

      case 3:
        *pCurrent = ')';
        pCurrent++;
        break;

      case 6:
        *pCurrent = '-';
        pCurrent++;
        break;
      }

    *pCurrent = *pPhone;
    Position++;
    pCurrent++;
    pPhone++;
    }

  *pCurrent = '\0';

  return FormatBuffer;
  }

static void LogScreen(PSZ pFormat, PSZ pUserID)
  {
  if (!LoggingScreens)
    return;

  LogEvent(pFormat, pUserID);
  }

static void LogEvent(PSZ pFormat, PSZ pUserID)
  {
  static char Temp[AVLOG_LIMIT];

  FILE * pOutput;
  int Time;

  if (!LogAnything)
    return;

  Time = now();

  sprintf(Temp, pFormat, pUserID);

  pOutput = fopen(pLogName, "at+");

  if (pOutput == NULL)       /* Unable to append - is file missing?       */
    {
    pOutput = fopen(pLogName, "wt+");

    if (pOutput == NULL)     /* Unable to create file for some reason (!) */
      return;
    }

  fprintf(pOutput, "%02d:%02d:%02d %s\n",
          dthour(Time),
          dtmin(Time),
          dtsec(Time),
          Temp);

  fclose(pOutput);  /* File must be closed every time to ensure integrity */

  return;
  }


static void LogStartup(void)
  {
  static char Temp[AVLOG_LIMIT];
  int Date;
  int Month;
  int Year;
  int Day;

  Date  = today();
  Month = ddmon(Date);
  Day   = ddday(Date);
  Year  = ddyear(Date);

  if (Year >= 2000)
    Year -= 2000;
  else
    Year -= 1900;

  sprintf(Temp, "-------- %d/%d/%d System initialized.",
          Month,
          Day,
          Year);

  LogEvent(Temp, NULL);
  }

static BOOL HasTooManyPhones(PAVREC pUser)
  {
  USHORT Phones = QueryPhone(pUser->Phone);

  if ((Phones > MaxPhone) && (pUser->Access != AVA_OVERRIDE))
    return TRUE;
  else
    return FALSE;
  }

static BOOL OkToMail(PAVREC pUser)
  {
  USHORT Phones;
  BOOL   MightMail;

  if (SecureMode)     // Never mail in secure mode.
    return FALSE;

  MightMail = MailEnabled[pUser->Range];

  Phones = QueryPhone(pUser->Phone);

  if ((Phones > MaxPhone) && (pUser->Access != AVA_OVERRIDE))
    return FALSE;

  if ((pUser->Access == AVA_RESTRICT) ||
      (!MightMail)                    ||
      HasTooManyPhones(pUser))
    return FALSE;

  switch (pUser->Status)
    {
    case AVS_VALIDATED:        /* We already called this guy!     */
      return FALSE;

    case AVS_BAD_NUMBER:       /* Should we call bad numbers?     */
    case AVS_BAD_PREFIX:
    case AVS_BAD_AREA_CODE:
    case AVS_BAD_FORMAT:
      if (!MailAll)
        return FALSE;
                               /* <--- Missing break intentional! */
    case AVS_TOLL:
    case AVS_LONG_DISTANCE:
    case AVS_NOT_TRIED:
    case AVS_FAILED:

      return TRUE;

    default:
      return FALSE;
    }
  }

static BOOL OkToCall(PAVREC pUser)
  {
  BOOL Call = FALSE;      /* Should we call this guy back, normally?  */
  BOOL Override = pUser->Access == AVA_OVERRIDE;

  if (SecureMode && BypassScreen)
    return TRUE;

  switch (pUser->Status)
    {
    case AVS_NOT_TRIED:
    case AVS_FAILED:
      Call = TRUE;
      break;

    case AVS_VALIDATED:
      if (SecureMode)
        {
        Call = TRUE;
        break;
        }

      // If not, fall through to the next guy.

    case AVS_MAILED:
    case AVS_BAD_FORMAT:
    case AVS_BAD_NUMBER:
    case AVS_BAD_PREFIX:
    case AVS_BAD_AREA_CODE:
      Override = FALSE;
      break;

    case AVS_LONG_DISTANCE:
      if (Override)
        Call = TRUE;

      break;

    case AVS_TOLL:
      if (Override)
        Call = TRUE;

      break;

    }

  if ((Call) && (!Override))
    if (HasTooManyPhones(pUser))
      {
      if (!SecureMode)
        Call = FALSE;
      }

  if (Call && (pUser->Range == AVR_UNKNOWN))
    {
    Call = FALSE;
    }

  if (Call && (pUser->Attempts == MaxDialAttempts) && (!Override))
    {
    if (!SecureMode)
      Call = FALSE;
    }

  return Call;
  }

static void GenericRebuff(PAVREC pUser)
  {
  BOOL Call = FALSE;      /* Should we call this guy back, normally?  */
  BOOL Override = pUser->Access == AVA_OVERRIDE;

  switch (pUser->Status)
    {
    case AVS_MAILED:
      break;

    case AVS_NOT_TRIED:
    case AVS_FAILED:
      Call = TRUE;
      break;

    case AVS_BAD_FORMAT:
      if (Override)
        pUser->Access = AVA_AUTO;

      LogScreen("SCREEN: Bad format on user %s.", pCurrentUser->Name);
      prfmsg(YOUBADF);
      break;

    case AVS_VALIDATED:
      if (Override)
        pUser->Access = AVA_AUTO;

      prfmsg(YOUVALD);
      break;

    case AVS_LONG_DISTANCE:
      if (Override)
        {
        Call = TRUE;
        break;
        }

      LogScreen("SCREEN: User %s is long distance.", pUser->Name);
      prfmsg(YOULD);
      break;

    case AVS_TOLL:
      if (Override)
        {
        Call = TRUE;
        break;
        }

      LogScreen("SCREEN: User %s is a toll call.", pUser->Name);
      prfmsg(YOUTOLL);
      break;

    case AVS_BAD_NUMBER:
      if (Override)
        pUser->Access = AVA_AUTO;

      LogScreen("SCREEN: User %s has phone number in NeverNumber list.",
                pUser->Name);

      prfmsg(YOUBADN, FormatPhone(pUser->Phone));
      break;

    case AVS_BAD_PREFIX:
      if (Override)
        pUser->Access = AVA_AUTO;

      LogScreen("SCREEN: User %s has prefix in NeverPrefix list.",
                pUser->Name);

      prfmsg(YOUBADP, FormatPhone(pUser->Phone));
      break;

    case AVS_BAD_AREA_CODE:
      if (Override)
        pUser->Access = AVA_AUTO;

      LogScreen("SCREEN: User %s has area code in NeverAreaCode list.",
                pUser->Name);

      prfmsg(YOUBADA, FormatPhone(pUser->Phone));
      break;
    }

  if ((Call) && (!Override))
    {
    if (HasTooManyPhones(pUser))
      {
      Call = FALSE;

      LogScreen("SCREEN: User %s has too many duplicate phone numbers.",
                pUser->Name);

      prfmsg(YOU2MANY,
             QueryPhone(pUser->Phone),
             FormatPhone(pUser->Phone),
             MaxPhone);
      }
    }

  if (Call && (pUser->Range == AVR_UNKNOWN))
    {
    LogScreen("SCREEN: Bad range on user %s.", pCurrentUser->Name);

    prfmsg(YOUBADF);
    Call = FALSE;
    }

  if (Call && (pUser->Attempts == MaxDialAttempts) && (!Override))
    {
    if (pUser->Status != AVS_MAILED)
      {
      LogScreen("SCREEN: Too many dial attempts on user %s.", pCurrentUser->Name);

      prfmsg(YOUATT);
      }

    Call = FALSE;
    }

  }

static void AddLabel(PAVREC pUser)
  {
  CHAR TempLabel[6];
  struct usracc TempAcc;
  FILE * pLabels;
  PSZ pMessage;
  PSZ pCurrent;
  struct usracc * SaveUsaptr = usaptr;

  sprintf(TempLabel, "%04x", pUser->MailCode);
  strupr(TempLabel);

  pLabels = fopen(pMailName, "at");

  if (pLabels == NULL)
    {
    pLabels = fopen(pMailName, "wt");

    if (pLabels == NULL)
      catastro("UNABLE TO OPEN LABEL FILE %s", pMailName);
    }

  setbtv(accbb);
  acqbtv(&TempAcc, TempHandle(pUser->Name), 0);

  usaptr = &TempAcc;
  pCurrentLabel = pUser;

  pMessage = getmsg(MLABEL);
  stpans(pMessage);

  usaptr = SaveUsaptr;

  pCurrent = pMessage;

  while (*pCurrent != '\0')
    {
    if (*pCurrent == '\r')
      *pCurrent = '\n';

    pCurrent++;
    }

  fprintf(pLabels,
          pMessage,
          TempLabel,
          pUser->Name);

  fclose(pLabels);
  }

static PSZ TVMailCode(void)
  {
  static char TempLabel[10];

  sprintf(TempLabel, "%04x", pCurrentLabel->MailCode);
  strupr(TempLabel);

  return TempLabel;
  }
