'****************************************************************************
'*                                                                          *
'*  GALMDB.BAS                                                              *
'*                                                                          *
'*  Copyright (c) 1995-1997 Galacticomm, Inc.    All Rights Reserved.       *
'*                                                                          *
'*  This file contains declarations and functions for reading and writing   *
'*  Galacticomm Message Database files.                                     *
'*                                                                          *
'*                                                  - J. Alvrus 10/13/95    *
'*                                                                          *
'****************************************************************************

Option Explicit

Global Const NOIDX = -1             ' no index code
Global Const NOHDL = -1             ' no handle code
Global Const MDBFEXT = "MDB"        ' message database default file extension
' message database ID string, DO NOT CHANGE COPYRIGHT DATE
Const MDBIDSTR = "Galacticomm Message Database - Copyright (c) 1996 Galacticomm, Inc.  All Rights Reserved."
Const OLDVER0 = "1.00"              ' old, still supported, version(s)
Const MDBVER = "1.01"               ' message database version
Const MDBIDSIZ = 95                 ' database header ID string size
Const MDBVERSIZ = 4                 ' database header version string size
Const MDBSPARE = 144                ' amount of spare space left in header
Const MDBIDSTRSIZ = 4               ' length of ID strings
Const MDBRECIDSTR = "REC0"          ' record header ID string
Const MDBMHSP = 2                   ' amount of spare space left in message header
Const MDBIDXIDSTR = "IDX0"          ' index header ID string
Const MDBIHSP = 2                   ' amount of spare space left in index header

' message database file create/open error codes
Global Const MDBERR_NOOVERWRITE = 1 ' couldn't overwrite existing file
Global Const MDBERR_FILEIO = 2      ' file I/O error
Global Const MDBERR_MEMORY = 3      ' out of memory
Global Const MDBERR_NOTFOUND = 4    ' file not found
Global Const MDBERR_INVALID = 5     ' file is not a message database file
' message database read error codes
Global Const MDBERR_BADREC = 27131  ' corrupted record or invalid position
Global Const MDBERR_DELETED = 27132 ' record has been deleted
Global Const MDBERR_EOF = 27133     ' end of file reached (on mdbnext)

Type mdbhdr                         ' message database file header
    idstr As String * MDBIDSIZ      '   packet ID string
    verstr As String * MDBVERSIZ    '   packet version string
    ctrlz As String * 1             '   end of text marker
    flags As Integer                '   packet type flags
    idxfpos As Long                 '   file position of start of index
    nmsgs As Long                   '   number of messages in database
    usrinfsiz As Integer            '   size of user info
    spare As String * MDBSPARE      '   padding for future enhancements
End Type                            ' (total size must be 256 bytes)

Type mdbmsghdr                      ' header for individual messages in message database
    flags As Integer                '   message status flags
    reclen As Long                  '   length of message record
    invlen As Long                  '   bitwise inverse of reclen (for error check)
    hdrid As String * MDBIDSTRSIZ   '   record header ID string
    spare As String * MDBMHSP       '   padding for future enhancements
End Type                            ' (total size must be 16 bytes)

' message database per-message header flags
Const MDBMH_DELETED = 1             ' message has been deleted

Type mdbidxhdr                      ' header for message database index (if any)
    flags As Integer                '   index type flags
    count As Long                   '   number of index entries
    invcnt As Long                  '   bitwise inverse of count (for error check)
    hdrid As String * MDBIDSTRSIZ   '   index header ID string
    spare As String * MDBIHSP       '   padding for future enhancements
End Type                            ' (total size must be 16 bytes)

Type mdbtrack                       ' structure for tracking open files
    f As Integer                    '   file handle
    outpos As Long                  '   current output position
    inpos As Long                   '   current input position
    hdr As mdbhdr                   '   header from file
End Type

Dim errhold As Integer              ' hold variable for last Err result
Dim mdbtrack() As mdbtrack          ' array for tracking open files

Function mdbaddmsg (ByVal f As Integer, msg As msgdpk) As Long
' write a message to a message database file at the current output position
' f:    file handle to write to
' msg:  message to add
' returns file position written to or 0 if error

    Dim idx As Integer
    Dim tmps As String
    Dim hdr As mdbmsghdr
    Dim msghdr As message

    mdbaddmsg = 0&
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    On Error GoTo addmsgfilerr
    mdbtrack(idx).hdr.nmsgs = mdbtrack(idx).hdr.nmsgs + 1
    Put f, 1, mdbtrack(idx).hdr
    LSet msghdr = msg
    tmps = RTrim$(msg.info)
    hdr.flags = 0
    hdr.reclen = Len(msghdr) + Len(tmps)
    hdr.invlen = Not hdr.reclen
    If mdbtrack(idx).hdr.verstr = MDBVER Then
        hdr.hdrid = MDBRECIDSTR
    Else
        hdr.hdrid = String$(Len(hdr.hdrid), 0)
    End If
    hdr.spare = String$(Len(hdr.spare), 0)
    Put f, mdbtrack(idx).outpos, hdr
    Put f, , msghdr
    Put f, , tmps
    mdbaddmsg = mdbtrack(idx).outpos
    mdbtrack(idx).outpos = Seek(f)
addmsgfilerr:
    errhold = Err
    Exit Function
End Function

Sub mdbclose (ByVal f As Integer)
' close a message database file
' f:    file handle to close

    Dim i As Integer

    Close f
    i = trackidx(f)
    If i <> NOIDX Then
        mdbtrack(i).f = NOHDL
    End If
End Sub

Function mdbdelete (ByVal f As Integer, ByVal pos As Long) As Integer
' read the next message from a message database file
' f:    file handle in use
' pos:  file position of record to delete or 0 for current position
' returns True if successful
' Note: physical and input file pointers are unchanged by this function

    Dim idx As Integer
    Dim savpos As Long
    Dim tmps As String
    Dim hdr As mdbmsghdr
    Dim msghdr As message

    mdbdelete = False
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    savpos = Seek(f)
    On Error GoTo mdbdelete_err
    If pos = 0 Then
        pos = Seek(f)
    End If
    Get f, pos, hdr
    If hdr.reclen < Len(msghdr) Or hdr.reclen > MAXDPKV Or hdr.invlen <> (Not hdr.reclen) Or (hdr.hdrid <> MDBRECIDSTR And mdbtrack(idx).hdr.verstr = MDBVER) Then
        errhold = MDBERR_BADREC
        Seek f, savpos
        Exit Function
    End If
    hdr.flags = hdr.flags Or MDBMH_DELETED
    Put f, pos, hdr
    mdbdelete = True
mdbdelete_err:
    errhold = Err
    Seek f, savpos
    Exit Function
End Function

Function mdberror () As Integer
' get error code associated with last MDB operation
' Note:  calling this function clears the error code

    mdberror = errhold
    errhold = 0
End Function

Function mdbidxsiz (ByVal f As Integer) As Long
' get number of index entries for a message database file
' f:    file handle to read from
' returns number of index entries or 0 if there is an error or the file has no index

    Dim idx As Integer
    Dim savpos As Long, cnt As Long
    Dim idxhdr As mdbidxhdr

    mdbidxsiz = 0&
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    savpos = Seek(f)
    On Error GoTo idxsizfilerr
    If mdbtrack(idx).hdr.idxfpos = 0& Then
        errhold = 0
        Seek f, savpos
        Exit Function
    End If
    Get f, mdbtrack(idx).hdr.idxfpos, idxhdr
    If idxhdr.invcnt <> (Not idxhdr.count) Or (idxhdr.hdrid <> MDBIDXIDSTR And mdbtrack(idx).hdr.verstr = MDBVER) Then
        errhold = MDBERR_BADREC
        Seek f, savpos
        Exit Function
    End If
    mdbidxsiz = idxhdr.count
idxsizfilerr:
    errhold = Err
    Seek f, savpos
    Exit Function
End Function

Function mdbnew (ByVal filnam As String, filhdl As Integer, ByVal userinfo As String) As Integer
' create a new message database file
' filnam:   name of file to create (overwrites existing file)
' filhdl:   buffer for file handle of opened file
' userinfo: buffer containing user-defined, per-file information
' returns 0 on success, error code on failure

    Dim f As Integer, idx As Integer

    f = FreeFile
    On Error Resume Next
    errhold = MDBERR_NOOVERWRITE
    mdbnew = errhold
    Kill filnam
    If Err <> 0 And Err <> 53 Then  'file not found
        Exit Function
    End If
    On Error GoTo newfilerr
    mdbnew = MDBERR_FILEIO
    Open filnam For Binary Access Read Write As f
    idx = newtrack()
    If idx = NOIDX Then
        Close f
        errhold = MDBERR_MEMORY
        mdbnew = errhold
        Exit Function
    End If
    mdbtrack(idx).f = f
    mdbtrack(idx).hdr.idstr = MDBIDSTR
    mdbtrack(idx).hdr.verstr = MDBVER
    mdbtrack(idx).hdr.ctrlz = Chr$(26)
    mdbtrack(idx).hdr.flags = 0
    mdbtrack(idx).hdr.idxfpos = 0&
    mdbtrack(idx).hdr.nmsgs = 0&
    mdbtrack(idx).hdr.usrinfsiz = Len(userinfo)
    mdbtrack(idx).hdr.spare = String$(MDBSPARE, 0)
    Put f, , mdbtrack(idx).hdr
    If Len(userinfo) Then
        Put f, , userinfo
    End If
    mdbtrack(idx).inpos = Seek(f)
    mdbtrack(idx).outpos = Seek(f)
    filhdl = f
    mdbnew = 0
newfilerr:
    errhold = Err
    Exit Function
End Function

Function mdbnext (ByVal f As Integer, msg As msgdpk) As Long
' read the next message from a message database file
' f:    file handle to read from
' msg:  message buffer to fill
' returns file position on success, 0 on end-of-file or error
' Note: This function advances the file pointer to the next message.
'       In general, this function should not be used on files with saved indices

    Dim idx As Integer
    Dim savpos As Long
    Dim tmps As String
    Dim hdr As mdbmsghdr
    Dim msghdr As message

    mdbnext = 0&
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    On Error GoTo nextfilerr
    Seek f, mdbtrack(idx).inpos
    Do
        savpos = Seek(f)
        Get f, , hdr
        If hdr.reclen < Len(msghdr) Or hdr.reclen > MAXDPKV Or hdr.invlen <> (Not hdr.reclen) Then
            errhold = MDBERR_BADREC
            Exit Function
        End If
        If hdr.hdrid <> MDBRECIDSTR And mdbtrack(idx).hdr.verstr = MDBVER Then
            errhold = MDBERR_EOF
            Exit Function
        End If
        tmps = Space$(hdr.reclen - Len(msghdr))
        Get f, , msghdr
        Get f, , tmps
    Loop While Not EOF(f) And (hdr.flags And MDBMH_DELETED) <> 0
    mdbtrack(idx).inpos = Seek(f)
    If EOF(f) Or (hdr.flags And MDBMH_DELETED) <> 0 Then
        errhold = MDBERR_EOF
        Exit Function
    End If
    LSet msg = msghdr
    msg.info = tmps
    mdbnext = savpos
nextfilerr:
    errhold = Err
    Exit Function
End Function

Function mdbnmsgs (ByVal f As Integer) As Long
' get number of messages stored in a message database file
' f:    file handle to read from
' returns number of messages in file (including deleted messages) or -1 if there is an error

    Dim idx As Integer

    idx = trackidx(f)
    If idx = NOIDX Then
        mdbnmsgs = -1&
    Else
        mdbnmsgs = mdbtrack(idx).hdr.nmsgs
    End If
End Function

Function mdbopen (ByVal filnam As String, filhdl As Integer) As Integer
' open an existing message database file
' filnam:   name of file to open
' filhdl:   buffer for file handle of opened file
' returns 0 on success, error code on failure

    Dim f As Integer, idx As Integer

    f = FreeFile
    On Error GoTo openfilerr
    errhold = MDBERR_NOTFOUND
    mdbopen = errhold
    Open filnam For Binary Access Read Write As f
    idx = newtrack()
    If idx = NOIDX Then
        Close f
        errhold = MDBERR_MEMORY
        mdbopen = errhold
        Exit Function
    End If
    mdbopen = MDBERR_FILEIO
    Get f, , mdbtrack(idx).hdr
    If RTrim$(mdbtrack(idx).hdr.idstr) <> MDBIDSTR Or (mdbtrack(idx).hdr.verstr <> MDBVER And mdbtrack(idx).hdr.verstr <> OLDVER0) Then
        Close f
        errhold = MDBERR_INVALID
        mdbopen = errhold
        Exit Function
    End If
    mdbtrack(idx).f = f
    mdbtrack(idx).inpos = Seek(f) + mdbtrack(idx).hdr.usrinfsiz
    If mdbtrack(idx).hdr.idxfpos Then
        mdbtrack(idx).outpos = mdbtrack(idx).hdr.idxfpos
    Else
        mdbtrack(idx).outpos = FileLen(filnam) + 1
    End If
    filhdl = f
    mdbopen = 0
openfilerr:
    errhold = Err
    Exit Function
End Function

Function mdbread (ByVal f As Integer, ByVal pos As Long, msg As msgdpk) As Integer
' read the next message from a message database file
' f:    file handle to read from
' pos:  file position to read from or 0 to read from current position
' msg:  message buffer to fill
' returns True if successful
' Note: this function advances the file pointer to the next message

    Dim idx As Integer
    Dim tmps As String
    Dim hdr As mdbmsghdr
    Dim msghdr As message

    mdbread = False
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    On Error GoTo readfilerr
    If pos = 0 Then
        Get f, , hdr
    Else
        Get f, pos, hdr
    End If
    If hdr.reclen < Len(msghdr) Or hdr.reclen > MAXDPKV Or hdr.invlen <> (Not hdr.reclen) Or (hdr.hdrid <> MDBRECIDSTR And mdbtrack(idx).hdr.verstr = MDBVER) Then
        errhold = MDBERR_BADREC
        Exit Function
    End If
    If hdr.flags And MDBMH_DELETED Then
        errhold = MDBERR_DELETED
        Exit Function
    End If
    tmps = Space$(hdr.reclen - Len(msghdr))
    Get f, , msghdr
    Get f, , tmps
    LSet msg = msghdr
    msg.info = tmps
    mdbtrack(idx).inpos = Seek(f)
    mdbread = True
readfilerr:
    errhold = Err
    Exit Function
End Function

Function mdbreadidx (ByVal f As Integer, idx() As Long, ByVal arrsiz As Long) As Long
' read index from a message database file
' f:        file handle to read from
' idx:      array to read index into
' arrsiz:   UBound() + 1 of idx array
' returns number of items read or 0 if there is an error or the file has no index

    Dim i As Integer
    Dim savpos As Long, cnt As Long
    Dim idxhdr As mdbidxhdr

    mdbreadidx = 0&
    i = trackidx(f)
    If i = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    savpos = Seek(f)
    On Error GoTo rdidxfilerr
    If mdbtrack(i).hdr.idxfpos = 0& Then
        errhold = 0
        Seek f, savpos
        Exit Function
    End If
    Get f, mdbtrack(i).hdr.idxfpos, idxhdr
    If idxhdr.invcnt <> (Not idxhdr.count) Or (idxhdr.hdrid <> MDBIDXIDSTR And mdbtrack(i).hdr.verstr = MDBVER) Then
        errhold = MDBERR_BADREC
        Seek f, savpos
        Exit Function
    End If
    If idxhdr.count < arrsiz Then
        arrsiz = idxhdr.count
    End If
    cnt = 0
    While cnt < arrsiz And Not EOF(f)
        Get f, , idx(cnt)
        cnt = cnt + 1
    Wend
    mdbreadidx = cnt
rdidxfilerr:
    errhold = Err
    Seek f, savpos
    Exit Function
End Function

Function mdbreadusr (ByVal f As Integer) As String
' read user-defined info
' f:        file to read from
' returns user info string or "" if no user info or error

    Dim idx As Integer
    Dim savpos As Long
    Dim tmps As String

    mdbreadusr = ""
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    savpos = Seek(f)
    On Error GoTo rdusrfilerr
    tmps = Space$(mdbtrack(idx).hdr.usrinfsiz)
    Get f, Len(mdbtrack(idx).hdr) + 1, tmps
    mdbreadusr = tmps
rdusrfilerr:
    errhold = Err
    Seek f, savpos
    Exit Function
End Function

Sub mdbrewind (ByVal f As Integer)
' set a message database file's read pointer to the beginning (also sets physical file pointer)
' f:    file handle

    Dim i As Integer

    i = trackidx(f)
    If i <> NOIDX Then
        mdbtrack(i).inpos = Len(mdbtrack(i).hdr) + mdbtrack(i).hdr.usrinfsiz + 1
        Seek f, mdbtrack(i).inpos
    End If
End Sub

Function mdbsaveidx (ByVal f As Integer, idx() As Long, ByVal nitems As Long) As Long
' write an index to a message database file
' f:        file handle to write to
' idx:      index array
' nitems:   number of entries in idx array
' returns number of items written or 0 if there is an error
' Note:  The index is written at the current output position (which is not updated).

    Dim i As Integer
    Dim savpos As Long, cnt As Long
    Dim idxhdr As mdbidxhdr

    mdbsaveidx = 0&
    savpos = Seek(f)
    On Error GoTo svidxfilerr
    i = trackidx(f)
    If i = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    idxhdr.flags = 0
    idxhdr.count = nitems
    idxhdr.invcnt = Not nitems
    If mdbtrack(i).hdr.verstr = MDBVER Then
        idxhdr.hdrid = MDBIDXIDSTR
    Else
        idxhdr.hdrid = String$(Len(idxhdr.hdrid), 0)
    End If
    idxhdr.spare = String$(Len(idxhdr.spare), 0)
    Put f, mdbtrack(i).outpos, idxhdr
    cnt = 0
    While cnt < nitems
        Put f, , idx(cnt)
        cnt = cnt + 1
    Wend
    mdbtrack(i).hdr.idxfpos = mdbtrack(i).outpos
    Put f, 1, mdbtrack(i).hdr
    Seek f, savpos
    mdbsaveidx = cnt
    errhold = 0
    Exit Function
svidxfilerr:
    errhold = Err
    mdbtrack(i).hdr.idxfpos = 0&
    Seek f, savpos
    Exit Function
End Function

Function mdbsaveusr (ByVal f As Integer, userinfo As String) As Integer
' save new version of user-defined info
' f:        file to save in
' userinfo: information to save
' returns True if successful
' Note:  the new user info must be the same length as (or shorter than) the existing user info

    Dim i As Integer
    Dim savpos As Long
    Dim idxhdr As mdbidxhdr

    mdbsaveusr = False
    savpos = Seek(f)
    On Error GoTo svusrfilerr
    i = trackidx(f)
    If i = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If
    Put f, Len(mdbtrack(i).hdr) + 1, userinfo
    mdbsaveusr = True
svusrfilerr:
    errhold = Err
    Seek f, savpos
    Exit Function
End Function

Function mdbupdate (ByVal f As Integer, ByVal pos As Long, msg As msgdpk) As Long
' update a message in a message database file
' f:    file handle to read from
' pos:  file position of message to update or 0 to update at the current position
' msg:  new message buffer to update with
' returns updated position if successful, zero if failure
' Note: this function does not change the current file position

    Dim idx As Integer, newreclen As Integer, flgPosChanged As Integer
    Dim savepos As Long
    Dim tmps As String
    Dim hdr As mdbmsghdr
    Dim msghdr As message

    ' initialize return value (returns zero by default/on error)
    mdbupdate = 0&

    ' find our tracking info on this file
    idx = trackidx(f)
    If idx = NOIDX Then
        errhold = 52    ' bad file number
        Exit Function
    End If

    ' save current file position
    savepos = Seek(f)
    On Error GoTo mdbupdate_err

    ' get correct position if position argument is zero
    If pos = 0 Then
        pos = savepos
    End If

    ' get record header
    Get f, pos, hdr
    
    ' check record header for errors and return error code if invalid
    If hdr.reclen < Len(msghdr) Or hdr.reclen > MAXDPKV Or hdr.invlen <> (Not hdr.reclen) Or (hdr.hdrid <> MDBRECIDSTR And mdbtrack(idx).hdr.verstr = MDBVER) Then
        errhold = MDBERR_BADREC
        Seek f, savepos
        Exit Function
    End If

    ' prepare new message for insertion
    LSet msghdr = msg
    tmps = RTrim$(msg.info)

    ' figure out size of new record
    newreclen = Len(msghdr) + Len(tmps)

    ' deal with changed record size
    flgPosChanged = (newreclen <> hdr.reclen)
    If flgPosChanged Then

        ' delete old record
        hdr.flags = hdr.flags Or MDBMH_DELETED
        Put f, pos, hdr
        hdr.flags = hdr.flags And Not MDBMH_DELETED

        ' update file header
        mdbtrack(idx).hdr.nmsgs = mdbtrack(idx).hdr.nmsgs + 1
        Put f, 1, mdbtrack(idx).hdr

        ' add new record (just add header and set new position here)
        hdr.reclen = newreclen
        hdr.invlen = Not hdr.reclen
        pos = mdbtrack(idx).outpos
        Put f, pos, hdr
    End If

    ' put updated message info into data file and return new position
    Put f, , msghdr
    Put f, , tmps
    mdbupdate = pos

    ' save new output position if position changed
    If flgPosChanged Then
        mdbtrack(idx).outpos = Seek(f)
    End If

    ' general exit-function handling... set error, restore old position
mdbupdate_err:
    errhold = Err
    Seek f, savepos
    Exit Function
End Function

Private Function newtrack () As Integer
' find or allocate an entry in the file tracking table
' returns index on success, NOIDX on failure

    Dim i As Integer, ntable As Integer

    newtrack = NOIDX
    errhold = 0
    On Error Resume Next
    ntable = UBound(mdbtrack)
    If Err Then
        On Error GoTo alcerr
        ReDim mdbtrack(0)
        newtrack = 0
    Else
        For i = 0 To ntable
            If mdbtrack(i).f = NOHDL Then
                newtrack = i
                Exit Function
            End If
        Next
        On Error GoTo alcerr
        ReDim Preserve mdbtrack(ntable + 1)
        newtrack = ntable + 1
    End If
alcerr:
    errhold = Err
    Exit Function
End Function

Private Function trackidx (ByVal f As Integer) As Integer
' find or allocate an entry in the file tracking table
' returns index on success, NOIDX on failure

    Dim i As Integer, ntable As Integer

    trackidx = NOIDX
    On Error GoTo notrack
    For i = 0 To UBound(mdbtrack)
        If mdbtrack(i).f = f Then
            trackidx = i
            Exit Function
        End If
    Next
notrack:
    Exit Function
End Function

