/***************************************************************************
 *                                                                         *
 *   THREADU.C                                                             *
 *                                                                         *
 *   Copyright (c) 1994-1995 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is a suite of utility routines for dealing with Forum message    *
 *   threads.                                                              *
 *                                                                         *
 *                                               - C. Robert  10/25/94     *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "threadu.h"

#define FILREV "$Revision:   1.0.1.0.1.0  $"

static
unsigned maxcch,                   /* max # of threads in in-memory cache  */
         swpnxt;                   /* in-memory thread to swap to disk next*/
static
int      cchhdl;                   /* handle to in-memory cache array      */

struct thrinf {                    /* per-thread information structure     */
     unsigned forum;               /*   Forum number thread resides in     */
     long thrid;                   /*   unique thread ID within Forum      */
     unsigned nmsgs;               /*   number of messages within thread   */
};

#define CCHISZ      40             /* increment size for resizing cache    */

BTVFILE *tctbb;                    /* thread-count Btrieve data file ptr   */

STATIC struct thrinf *sinthr(unsigned forum,long thrid);
STATIC void southr(struct thrinf *thrptr);

void
inithu(                            /* initialize thread utilities          */
unsigned maxinm)                   /*   max # of threads in in-memory cache*/
{
     maxcch=maxinm;
     swpnxt=0;
     cchhdl=newarr(CCHISZ,sizeof(struct thrinf));
     tctbb=opnbtv("galthrs.dat",sizeof(struct thrinf));
}

unsigned
ninthr(                            /* get number of messages in a thread   */
unsigned forum,                    /*   forum number thread resides in     */
long thrid)                        /*   unique thread ID within Forum      */
{
     int i;
     struct thrinf thrinf,*tp;

     for (i=0 ; i < ninarr(cchhdl) ; i++) {
          tp=arrelem(cchhdl,i);
          if (tp->nmsgs != 0 && tp->forum == forum && tp->thrid == thrid) {
               return(tp->nmsgs);
          }
     }
     setbtv(tctbb);
     thrinf.forum=forum;
     thrinf.thrid=thrid;
     thrinf.nmsgs=0;
     acqbtv(&thrinf,&thrinf,0);
     rstbtv();
     return(thrinf.nmsgs);
}

BOOL                               /*   did this just create a new thread? */
incthr(                            /* increment # of messages in a thread  */
unsigned forum,                    /*   forum number thread resides in     */
long thrid)                        /*   unique thread ID within Forum      */
{
     struct thrinf *thrptr;

     thrptr=sinthr(forum,thrid);
     if (++thrptr->nmsgs == 1) {
          return(TRUE);
     }
     return(FALSE);
}

BOOL                               /*   did this just destroy a thread?    */
decthr(                            /* decrement # of messages in a thread  */
unsigned forum,                    /*   forum number thread resides in     */
long thrid)                        /*   unique thread ID within Forum      */
{
     struct thrinf *thrptr;

     thrptr=sinthr(forum,thrid);
     if (thrptr->nmsgs == 0) {
          return(FALSE);
     }
     if (--thrptr->nmsgs == 0) {
          setbtv(tctbb);
          if (acqbtv(NULL,thrptr,0)) {
               delbtv();
          }
          rstbtv();
          return(TRUE);
     }
     return(FALSE);
}

void
setthrc(                           /* set number of messages in thread     */
unsigned forum,                    /*   forum ID thread is in              */
long thrid,                        /*   thread ID of thread                */
unsigned msgcnt)                   /*   number of messages in thread       */
{
     struct thrinf *thrptr;

     thrptr=sinthr(forum,thrid);
     if (msgcnt == 0) {
          thrptr->nmsgs=0;
          setbtv(tctbb);
          if (acqbtv(NULL,thrptr,0)) {
               delbtv();
          }
          rstbtv();
     }
     else {
          thrptr->nmsgs=msgcnt;
     }
}

void
cathrs(void)                       /* clear all thread counters (mcu only) */
{
     int i;

     for (i=0 ; i < ninarr(cchhdl) ; i++) {
          ((struct thrinf *)arrelem(cchhdl,i))->nmsgs=0;
     }
     setbtv(tctbb);
     while (slobtv(NULL)) {
          delbtv();
     }
     rstbtv();
}

void
cfthrs(                            /* clr Forum thread counters (mcu only) */
unsigned forum)                    /*   Forum to clear all threads for     */
{
     int i;
     struct thrinf *thrptr,tinf;

     for (i=0 ; i < ninarr(cchhdl) ; i++) {
          thrptr=arrelem(cchhdl,i);
          if (thrptr->forum == forum) {
               thrptr->nmsgs=0;
          }
     }
     setbtv(tctbb);
     tinf.forum=forum;
     tinf.thrid=0;
     if (qgebtv(&tinf,0)) {
          do {
               gcrbtv(&tinf,0);
               if (tinf.forum != forum) {
                    break;
               }
               delbtv();
          } while (qnxbtv());
     }
     rstbtv();
}

void
clsthu(void)                       /* close down thread utilities          */
{
     int i;
     struct thrinf *thrptr;

     for (i=0 ; i < ninarr(cchhdl) ; i++) {
          thrptr=arrelem(cchhdl,i);
          if (thrptr->nmsgs != 0) {
               southr(thrptr);
          }
     }
     clsbtv(tctbb);
}

STATIC struct thrinf *
sinthr(                            /* swap thread on disk into memory      */
unsigned forum,                    /*   forum number thread resides in     */
long thrid)                        /*   unique thread ID within Forum      */
{
     int i,freeth;
     struct thrinf *thrptr,*tp;

     thrptr=NULL;
     freeth=-1;
     for (i=0 ; i < ninarr(cchhdl) ; i++) {
          tp=arrelem(cchhdl,i);
          if (tp->nmsgs != 0 && tp->forum == forum && tp->thrid == thrid) {
               thrptr=tp;
               break;
          }
          if (tp->nmsgs == 0 && freeth == -1) {
               freeth=i;
          }
     }
     if (thrptr == NULL) {
          if (freeth != -1) {
               thrptr=arrelem(cchhdl,freeth);
          }
          else if (ninarr(cchhdl) < maxcch) {
               thrptr=add2arr(cchhdl,NULL);
          }
          else {
               if (swpnxt >= ninarr(cchhdl)) {
                    swpnxt=0;
               }
               thrptr=arrelem(cchhdl,swpnxt);
               southr(thrptr);
               swpnxt++;
          }
          setbtv(tctbb);
          thrptr->forum=forum;
          thrptr->thrid=thrid;
          if (!acqbtv(thrptr,thrptr,0)) {
               thrptr->nmsgs=0;
          }
          rstbtv();
     }
     return(thrptr);
}

STATIC void
southr(                            /* swap thread in cache out to disk     */
struct thrinf *thrptr)             /*   pointer to thread in cache         */
{
     setbtv(tctbb);
     if (acqbtv(NULL,thrptr,0)) {
          if (((struct thrinf *)tctbb->data)->nmsgs != thrptr->nmsgs) {
               updbtv(thrptr);
          }
     }
     else {
          insbtv(thrptr);
     }
     rstbtv();
     thrptr->nmsgs=0;
}

