/***************************************************************************
 *                                                                         *
 *   GMEUTL.C                                                              *
 *                                                                         *
 *   Copyright (c) 1994-1995 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This file contains common and utility functions used by and for GME   *
 *   both online and offline.                                              *
 *                                                                         *
 *                                           - J. Alvrus   6/9/94          *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "galme.h"
#include "gme.h"
#include "gmeutl.h"
#include "gmeloc.h"

#define FILREV "$Revision:   1.0.1.0.1.0  $"

#define GMEMBN "GALME.MCV"         /* GME CNF options file name            */

unsigned _reclen,                  /* message file record length           */
         _txtlen=0U;               /* max message text length              */

int _maxqsf=0,                     /* max forums in a user's quickscan     */
    qssiz;                         /* size of in-memory quickscan          */

FILE *gmemb=NULL;                  /* GME CNF options file block pointer   */

char *utltxt;                      /* GME utility message text buffer      */
struct message *utlmsg;            /* GME utility message header structure */
struct gmework *utlwork;           /* GME utility work area                */
struct fordef *utldef;             /* GME utility forum definition         */

char *extinf;                      /* extended return information buffer   */

unsigned hirqidx=0U;               /* highest request array index          */
char *gmerqarr=NULL;               /* pointer to open request array        */

unsigned                           /*   returns max message text length    */
txtlen(void)                       /* init__ callable get-text-length      */
{
     if (_txtlen == 0U) {
          if (gmemb == NULL) {
               gmemb=opnmsg(GMEMBN);
          }
          else {
               setmbk(gmemb);
          }
          _reclen=(unsigned)lngopt(MSGSIZ,1024L,16384L);
          _txtlen=_reclen-sizeof(struct msgdsk)-2*MAXADR;
          rstmbk();
     }
     return(_txtlen);
}

int                                /*   returns max # forums allowed in qs */
maxqsf(void)                       /* init__ callable get-max-qscan-forums */
{
     if (_maxqsf == 0) {
          if (gmemb == NULL) {
               gmemb=opnmsg(GMEMBN);
          }
          else {
               setmbk(gmemb);
          }
          _maxqsf=numopt(MAXQSF2,1,MAXNQSF);
          qssiz=sizeof(struct qscfg)-1+_maxqsf*sizeof(struct fmidky)
               +(_maxqsf+1)/2;
          rstmbk();
     }
     return(_maxqsf);
}

void
iniutl(void)                       /* initialize GME utilities             */
{
     ASSERT(NVFORSZ == sizeof(struct fordsk)-1);
     ASSERT(NVMSGSZ == sizeof(struct msgdsk));
     ASSERT(NVQSCSZ == sizeof(struct qscfg)-1);
     ASSERT(NVQIKSZ == sizeof(struct qikdat)-1);
     ASSERT(GMEWRKSZ >= sizeof(struct gmework));
     txtlen();                     /* these will open .MCV file and ensure */
     maxqsf();                     /* TXTLEN and MAXQSF are initialized    */
     setmbk(gmemb);
     gmerqarr=(char *)alczer(BIGBLK);
     hirqidx=BIGBLK;
     utlwork=(struct gmework *)alczer(sizeof(struct gmework));
     utlmsg=(struct message *)alcmem(sizeof(struct message));
     utltxt=alcmem(TXTLEN);
     utldef=(struct fordef *)alcmem(sizeof(struct fordef));
     extinf=alcmem(XINFSZ);
}

void
clsutl(void)                       /* close GME utilities                  */
{
     if (gmemb != NULL) {
          clsmsg(gmemb);
          gmemb=NULL;
     }
}

void
inigmerq(                          /* initialize GME request               */
void *workb)                       /*   work area to be used for request   */
{
     struct gmework *work=(struct gmework *)workb;

     ASSERT(work != NULL);
     ASSERTM(!gmerqopn(work),spr("rqid=%u st1=%d st2=%d st3=%d",
             work->rqid,work->state1,work->state2,work->state3));
     if (gmerqopn(work)) {
          clsgmerq(work);
     }
     setmem(work,sizeof(struct gmework),0);
     work->state1=START;
     work->state2=START;
     work->state3=START;
     addrq(work);
}

void
clsgmerq(                          /* close a GME request                  */
void *workb)                       /*   work area being used for request   */
{
     struct gmework *work=(struct gmework *)workb;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     if (gmerqopn(work)) {
          if (!gmeoffl()) {
               gmeulkr(work);
          }
          if (work->fp != NULL) {
               fclose(work->fp);
          }
          if (work->fpout != NULL) {
               fclose(work->fpout);
          }
          unlink(work->cpyatt);
          if (!gmeoffl()) {
               if (work->flags&(QIKLST|MASSLST|SYSLST)) {
                    clsdist(work);
               }
          }
          if (!((long)work->orgflg&FILIND) && istmp(work->orgatt)) {
               unlink(work->orgatt);
          }
          remrq(work);
     }
     setmem(work,sizeof(struct gmework),0);
     work->state1=START;
     work->state2=START;
     work->state3=START;
}

void
addrq(                             /* add request to list of requests      */
struct gmework *work)              /*   pointer to request work area       */
{
     unsigned idx,bit;

     ASSERT(work != NULL);
     ASSERT(!gmerqopn(work));
     for (idx=0 ; (idx < hirqidx) && (~gmerqarr[idx] == 0) ; ++idx) {
     }
     if ((long)idx*8L >= MAXGMERQ+1) {
          catastro("Too many simultaneous GME requests");
     }
     if (idx == hirqidx) {
          ASSERT(hirqidx+BIGBLK < MAXBLK);
          gmerqarr=(char *)alcrsz(gmerqarr,hirqidx,hirqidx+BIGBLK);
          setmem(&gmerqarr[hirqidx],BIGBLK,0);
          hirqidx+=BIGBLK;
     }
     for (bit=0 ; gmerqarr[idx]&(1<<bit) ; ++bit) {
          ASSERT(bit < 8);
     }
     gmerqarr[idx]|=1<<bit;
     work->rqid=idx*8+bit+1;
     work->antirq=~work->rqid;
}

void
remrq(                             /* remove request from list             */
struct gmework *work)              /*   pointer to request work area       */
{
     unsigned idx,bit;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     idx=(work->rqid-1)>>3;        /* idx = rqid / 8 */
     bit=(work->rqid-1)&7;         /* bit = rqid % 8 */
     gmerqarr[idx]&=~(1<<bit);
}

BOOL
gmerqopn(                          /* is this request open?                */
void *workb)                       /*   pointer to work area               */
{
     unsigned idx,bit;
     struct gmework *work=(struct gmework *)workb;

     ASSERT(work != NULL);
     if (work->rqid > 0 && work->rqid < hirqidx*8) {
          idx=(work->rqid-1)>>3;   /* idx = rqid / 8 */
          bit=(work->rqid-1)&7;    /* bit = rqid % 8 */
          return(work->antirq == ~work->rqid && (gmerqarr[idx]&(1<<bit)));
     }
     return(FALSE);
}

BOOL                               /*   returns FALSE if unable to add     */
sfinqs(                            /* set forum in quickscan               */
struct qscfg *qsc,                 /*   quickscan record                   */
unsigned fid,                      /*   forum to set                       */
BOOL turnon)                       /*   TRUE = make forum "in" quickscan   */
{
     int i;
     long tmphi;

     ASSERT(qsc != NULL);
     i=qsidx(qsc,fid);
     if (turnon) {
          if (i == NOIDX) {
               i=absadqs(qsc,fid);
               if (i == NOIDX) {
                    return(FALSE);
               }
               isethi(qsc,i,1L);
          }
          else {
               tmphi=igethi(qsc,i);
               if (tmphi == 0L) {
                    isethi(qsc,i,1L);
               }
               else if (tmphi < 0L) {
                    isethi(qsc,i,-tmphi);
               }
          }
     }
     else if (i != NOIDX) {
          tmphi=igethi(qsc,i);
          if (tmphi > 0L) {
               isethi(qsc,i,-tmphi);
          }
     }
     return(TRUE);
}

struct otscan *                    /*   returns copy of pointer to dest    */
qsc2ots(                           /* copy quickscan to a one-time scan buf*/
struct qscfg *qsc,                 /*   quickscan to copy                  */
struct otscan *ots)                /*   one-time scan buffer               */
{
     int i;
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     ASSERT(ots != NULL);
     stlcpy(ots->keywds,qsc->kwds,MAXSKWD);
     ots->stmsgid=qsc->stmsg;
     ots->flags=(qsc->flags&NEWMSG) ? SCNEW : 0;
     if (qsc->flags&WATONL) {
          ots->flags|=SCATT;
     }
     if (qsc->flags&TOMEONL) {
          ots->flags|=SCTOU;
     }
     if (qsc->flags&FRMEONL) {
          ots->flags|=SCFRU;
     }
     ots->nforums=0;
     fm=(struct fmidky *)qsc->accmsg;
     for (i=0 ; i < qsc->nforums ; ++i) {
          if (fidxst(fm[i].forum) && igethi(qsc,i) > 0L) {
               addf2ots(ots,fm[i].forum);
          }
     }
     return(ots);
}

void
addf2ots(                          /* add forum to scan list in seq order  */
struct otscan *ots,                /*   one-time scan structure to update  */
unsigned forum)                    /*   forum to add                       */
{
     int i,seq;

     ASSERT(ots != NULL);
     ASSERT(fidxst(forum));
     seq=getdef(forum)->seqid;
     for (i=0 ; i < ots->nforums ; ++i) {
          if (getdef(ots->forlst[i])->seqid > seq) {
               break;
          }
     }
     movmem(&ots->forlst[i],&ots->forlst[i+1],
            sizeof(unsigned)*(ots->nforums-i));
     ots->forlst[i]=forum;
     ++ots->nforums;
}

int                                /*   index of new entry (NOIDX if error)*/
absadqs(                           /* add forum to qs, del others for room */
struct qscfg *qsc,                 /*   quickscan record                   */
unsigned fid)                      /*   forum ID to add                    */
{
     int i;

     ASSERT(qsc != NULL);
     if ((i=add2qs(qsc,fid)) == NOIDX) {
          if ((i=qsdelf(qsc)) == NOIDX) {
               if ((i=qsoldm(qsc)) == NOIDX) {
                    return(NOIDX);
               }
          }
          idelqs(qsc,i);
          i=add2qs(qsc,fid);
     }
     return(i);
}

int                                /*   index in qs arrays (NOIDX if err)  */
qsidx(                             /* get index of forum in quickscan      */
struct qscfg *qsc,                 /*   quickscan record                   */
unsigned fid)                      /*   forum ID to get                    */
{
     int i;
     struct fmidky *fm;

     fm=(struct fmidky *)qsc->accmsg;
     for (i=0 ; i < qsc->nforums ; ++i) {
          if (fm[i].forum == fid) {
               return(i);
          }
     }
     return(NOIDX);
}

int                                /*   index in qs arrays (NOIDX if err)  */
add2qs(                            /* add slot for forum to quickscan      */
struct qscfg *qsc,                 /*   quickscan record                   */
unsigned fid)                      /*   forum ID to add                    */
{
     int i,j;
     char *acc;
     unsigned lastfid;
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     lastfid=EMLID;
     fm=(struct fmidky *)qsc->accmsg;
     for (i=0 ; i < qsc->nforums ; ++i) {
          if (fid == fm[i].forum) {
               return(i);
          }
          else if (fid > lastfid && fid < fm[i].forum) {
               break;
          }
          lastfid=fm[i].forum;
     }
     if (qsc->nforums < MAXQSF) {
          acc=(char *)&fm[qsc->nforums];
          movmem(acc,acc+sizeof(struct fmidky),(qsc->nforums+1)/2);
          acc=(char *)&fm[qsc->nforums+1];
          if (i < qsc->nforums) {
               movmem(&fm[i],&fm[i+1],sizeof(struct fmidky)*(qsc->nforums-i));
               for (j=qsc->nforums ; j > i ; --j) {
                    wracc(acc,acclvl(acc,j-1),j);
               }
          }
          fm[i].forum=fid;
          fm[i].msgid=0L;
          wracc(acc,NOTSET,i);
          ++qsc->nforums;
          return(i);
     }
     return(NOIDX);
}

void
delqs(                             /* delete an entry from the quickscan   */
struct qscfg *qsc,                 /*   pointer to quickscan               */
unsigned forum)                    /*   forum ID to remove from quickscan  */
{
     int i;

     ASSERT(qsc != NULL);
     i=qsidx(qsc,forum);
     if (i != NOIDX) {
          idelqs(qsc,i);
     }
}

BOOL                               /*   returns TRUE if able to set        */
sethi(                             /* set hi message in quickscan          */
struct qscfg *qsc,                 /*   pointer to quickscan               */
unsigned forum,                    /*   forum ID to set for                */
long msgid)                        /*   message ID to set as high message  */
{
     int i;

     ASSERT(qsc != NULL);
     i=qsidx(qsc,forum);
     if (i == NOIDX) {
          i=add2qs(qsc,forum);
          if (i != NOIDX) {
               isethi(qsc,i,msgid);
               return(TRUE);
          }
     }
     else {
          isethi(qsc,i,msgid);
          return(TRUE);
     }
     return(FALSE);
}

BOOL                               /*   returns TRUE if able to set        */
setac(                             /* set forum acc in quickscan           */
struct qscfg *qsc,                 /*   pointer to quickscan               */
unsigned forum,                    /*   forum ID to set for                */
int acc)                           /*   access level to set                */
{
     int i;

     ASSERT(qsc != NULL);
     ASSERT(acc >= 0 && acc <= NOTSET);
     i=qsidx(qsc,forum);
     if (i == NOIDX) {
          i=add2qs(qsc,forum);
          if (i != NOIDX) {
               isetac(qsc,i,acc);
               return(TRUE);
          }
     }
     else {
          isetac(qsc,i,acc);
          return(TRUE);
     }
     return(FALSE);
}

long
gethi(                             /* get hi message in quickscan          */
struct qscfg *qsc,                 /*   pointer to quickscan               */
unsigned forum)                    /*   forum ID to get for                */
{
     int i;

     ASSERT(qsc != NULL);
     i=qsidx(qsc,forum);
     if (i != NOIDX) {
          return(igethi(qsc,i));
     }
     return(FIRSTM);
}

void
idelqs(                            /* delete an entry from the quickscan   */
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx)                           /*   index of entry to delete           */
{
     int i;
     char *acc;
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     fm=(struct fmidky *)qsc->accmsg;
     movmem(&fm[idx+1],&fm[idx],
            sizeof(struct fmidky)*(qsc->nforums-idx-1)+(qsc->nforums+1)/2);
     --qsc->nforums;
     acc=(char *)&fm[qsc->nforums];
     for (i=idx ; i < qsc->nforums ; ++i) {
          wracc(acc,acclvl(acc,i+1),i);
     }
}

void
isethi(                            /* set hi message in quickscan (w/index)*/
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx,                           /*   index of forum to set              */
long msgid)                        /*   message ID to set as high message  */
{
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     fm=(struct fmidky *)qsc->accmsg;
     fm[idx].msgid=msgid;
}

void
isetac(                            /* set forum acc in quickscan (w/index) */
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx,                           /*   index of forum to set              */
int acc)                           /*   access level to set                */
{
     char *acar;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     ASSERT(acc >= 0 && acc <= NOTSET);
     ASSERT(acc != OPAXES && acc != SYAXES);
     acar=qsc->accmsg+sizeof(struct fmidky)*qsc->nforums;
     wracc(acar,acc,idx);
}

unsigned
igetfid(                           /* get forum ID in quickscan (w/index)  */
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx)                           /*   index of forum to get              */
{
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     fm=(struct fmidky *)qsc->accmsg;
     return(fm[idx].forum);
}

long
igethi(                            /* get hi message in quickscan (w/index)*/
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx)                           /*   index of forum to get              */
{
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     fm=(struct fmidky *)qsc->accmsg;
     return(fm[idx].msgid);
}

int
igetac(                            /* get forum acc in quickscan (w/index) */
struct qscfg *qsc,                 /*   pointer to quickscan               */
int idx)                           /*   index of forum to get              */
{
     char *acar;

     ASSERT(qsc != NULL);
     ASSERT(idx >= 0 && idx < qsc->nforums);
     acar=qsc->accmsg+sizeof(struct fmidky)*qsc->nforums;
     return(acclvl(acar,idx));
}

int                                /*   returns index or NOIDX if none     */
qsdelf(                            /* get first non-existent forum         */
struct qscfg *qsc)                 /*   pointer to quickscan               */
{
     int i;
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     fm=(struct fmidky *)qsc->accmsg;
     for (i=0 ; i < qsc->nforums ; ++i) {
          if (!fidxst(fm[i].forum)) {
               return(i);
          }
     }
     return(NOIDX);
}

int                                /*   returns index or NOIDX if none     */
qsoldm(                            /* get oldest "marker" entry            */
struct qscfg *qsc)                 /*   pointer to quickscan               */
{
     int i,lasti;
     long curhi,lowhi;
     struct fmidky *fm;
     char *acc;

     ASSERT(qsc != NULL);
     fm=(struct fmidky *)qsc->accmsg;
     acc=(char *)&fm[qsc->nforums];
     lowhi=-LASTM;
     lasti=NOIDX;
     for (i=0 ; i < qsc->nforums ; ++i) {
          curhi=fm[i].msgid;
          if (curhi <= 0L && -curhi < lowhi && acclvl(acc,i) == NOTSET) {
               lowhi=curhi;
               lasti=i;
          }
     }
     return(lasti);
}

int                                /*   returns index or NOIDX if none     */
qsoldr(                            /* get oldest "real" entry              */
struct qscfg *qsc)                 /*   pointer to quickscan               */
{
     int i,lasti;
     long curhi,lowhi;
     struct fmidky *fm;
     char *acc;

     ASSERT(qsc != NULL);
     fm=(struct fmidky *)qsc->accmsg;
     acc=(char *)&fm[qsc->nforums];
     lowhi=LASTM;
     lasti=NOIDX;
     for (i=0 ; i < qsc->nforums ; ++i) {
          curhi=fm[i].msgid;
          if (curhi > 0L && curhi < lowhi && acclvl(acc,i) == NOTSET) {
               lowhi=curhi;
               lasti=i;
          }
     }
     return(lasti);
}

int                                /*   returns index or NOIDX if none     */
qsoldest(                          /* get oldest real entry w/ or w/out acc*/
struct qscfg *qsc)                 /*   pointer to quickscan               */
{
     int i,lasti;
     long curhi,lowhi;
     struct fmidky *fm;

     ASSERT(qsc != NULL);
     fm=(struct fmidky *)qsc->accmsg;
     lowhi=LASTM;
     lasti=NOIDX;
     for (i=0 ; i < qsc->nforums ; ++i) {
          curhi=fm[i].msgid;
          if (curhi > 0L && curhi < lowhi) {
               lowhi=curhi;
               lasti=i;
          }
     }
     return(lasti);
}

BOOL
valfornm(                          /* is this a valid forum name?          */
char *name)                        /*   name to check                      */
{
     unsigned len;

     for (len=0 ; *name != '\0' ; ++len,++name) {
          if (len == FORNSZ || !isascii(*name)
           || strchr(" @;",*name) != NULL) {
               return(FALSE);
          }
     }
     return(len > 0);
}

void
wracc(                             /* write-access-level in compressd array*/
char *acarpt,                      /*   pointer to access-level array      */
int value,                         /*   value to write                     */
int idx)                           /*   index in array                     */
{
     char *cp;

     ASSERT(acarpt != NULL);
     ASSERT(value <= COAXES || value == NOTSET);
     cp=&acarpt[idx>>1];
     if (idx&1) {
          *cp&=0x0F;
          *cp|=value<<4;
     }
     else {
          *cp&=0xF0;
          *cp|=value;
     }
}

int
acclvl(                            /* get acc level from compressed array  */
char *acarpt,                      /*   pointer to array                   */
int idx)                           /*   index of access level              */
{
     ASSERT(acarpt != NULL);
     if (idx&1) {
          return(acarpt[idx>>1]>>4);
     }
     else {
          return(acarpt[idx>>1]&0x0F);
     }
}

char *
accstr(                            /* string describing forum access level */
int acclvl)                        /*   access level to describe           */
{
     static char *accdsc[]={"Zero","","Read","","Download","","Write","",
                            "Upload","","Co-Op","","Forum-Op","","Sysop",
                            "<unassigned>"};

     ASSERT(acclvl <= NOTSET && (!(acclvl&1) || acclvl == NOTSET));
     return(accdsc[acclvl]);
}

long
cmptid(                            /* compute thread ID                    */
struct message *msg)               /*   message header                     */
{
     if (msg->forum == EMLID) {
          return(emltid(msg));
     }
     else {
          return(fortid(msg));
     }
}

long
emltid(                            /* generate an E-mail thread ID         */
struct message *msg)               /*   message header (to,from,topic)     */
{
     char *adr1,*adr2,*cp1,*cp2;

     ASSERT(msg != NULL);
     adr1=(char *)tmpbuf;
     adr2=adr1+UIDSIZ;
     if (isexpa(msg->from)) {
          cp1=strchr(msg->from,':');
          ASSERT(cp1 != NULL);
          for (cp2=cp1+1 ; *cp2 == ' ' ; ++cp2) {
          }
          ASSERT(UIDSIZ >= (int)(cp1-msg->from+2));
          stlcpy(adr1,msg->from,(int)(cp1-msg->from+2));
          stlcat(adr1,cp2,UIDSIZ);
     }
     else {
          stlcpy(adr1,msg->from,UIDSIZ);
     }
     if (isexpa(msg->to)) {
          cp1=strchr(msg->to,':');
          ASSERT(cp1 != NULL);
          for (cp2=cp1+1 ; *cp2 == ' ' ; ++cp2) {
          }
          ASSERT(UIDSIZ >= (int)(cp1-msg->to+2));
          stlcpy(adr2,msg->to,(int)(cp1-msg->to+2));
          stlcat(adr2,cp2,UIDSIZ);
     }
     else {
          stlcpy(adr2,msg->to,UIDSIZ);
     }
     if (strcmp(adr1,adr2) < 0) {
          ASSERT(strlen(adr2) < UIDSIZ);
          movmem(adr2,adr1+strlen(adr1),strlen(adr2)+1);
     }
     else {
          stlcat(adr2,adr1,2*UIDSIZ-1);
          ASSERT(strlen(adr2) < 2*UIDSIZ-1);
          movmem(adr2,adr1,strlen(adr2)+1);
     }
     stlcat(adr1,msg->topic,2*UIDSIZ+TPCSIZ-2);
     strupr(adr1);
     return(crc32(adr1,strlen(adr1)));
}

long
fortid(                            /* generate a forum thread ID           */
struct message *msg)               /*   message header (topic)             */
{
     char *s;

     ASSERT(msg != NULL);
     s=(char *)tmpbuf;
     stlcpy(s,msg->topic,TPCSIZ);
     strupr(s);
     return(crc32(s,strlen(s)));
}

BOOL
isexpa(                            /* is this an exporter-style address    */
char *adr)                         /*   address to check                   */
{
     char *cp;

     cp=strchr(adr,':');
     if (cp != NULL) {
          for ( ; adr < cp ; ++adr) {
               if (!isalnum(*adr)) {
                    return(FALSE);
               }
          }
          return(TRUE);
     }
     return(FALSE);
}

BOOL                               /*   returns TRUE if a valid file name  */
valfnm(                            /* check for valid file name            */
char *name)                        /*   string to check                    */
{
     char *cp;
     int i;

     for (i=1,cp=name ; *cp != '\0' && *cp != '.' ; ++i,++cp) {
          if (i > 8 || !isfnmc(*cp)) {
               return(FALSE);
          }
     }
     if (*cp++ == '.') {
          for (i=1 ; *cp != '\0' ; ++i,++cp) {
               if (i > 3 || *cp == '.' || !isfnmc(*cp)) {
                    return(FALSE);
               }
          }
     }
     if (rsvnam(name)) {
          return(FALSE);
     }
     return(TRUE);
}

BOOL
valslnm(                           /* is this a valid sysop dist list name?*/
char *name)                        /*   complete name (including @)        */
{
     int i;

     ASSERT(name != NULL);
     if (*name != '@') {
          return(FALSE);
     }
     for (i=1 ; name[i] != '\0' ; i++) {
          if (!isalnum(name[i]) || i >= DLNMSZ-1) {
               return(FALSE);
          }
     }
     return(i > 1);
}

int                                /*   returns new number of echoes       */
addecho(                           /* add an echo to list                  */
char *echoes,                      /*   echo list buffer                   */
char *newadr,                      /*   echo address to add                */
int necho)                         /*   current number of echoes           */
{
     adr_t *tmpecho;

     ASSERT(echoes != NULL);
     ASSERT(newadr != NULL);
     ASSERT(necho < MAXECHO);
     tmpecho=(adr_t *)echoes;
     stlcpy(tmpecho[necho],newadr,MAXADR);
     return(necho+1);
}

int                                /*   returns new number of echoes       */
delecho(                           /* delete an echo from list             */
char *echoes,                      /*   echo list buffer                   */
int echonum,                       /*   index of echo address to delete    */
int necho)                         /*   current number of echoes           */
{
     adr_t *tmpecho;

     ASSERT(echoes != NULL);
     ASSERT(necho < MAXECHO);
     ASSERT(echonum < necho);
     tmpecho=(adr_t *)echoes;
     if (echonum < necho-1) {
          movmem(&tmpecho[echonum+1],&tmpecho[echonum],
                 MAXADR*(necho-echonum-1));
     }
     return(necho-1);
}

void
clrfrom(                           /* "clear" from field of message        */
struct message *msg)               /*   message to clear                   */
{
     msg->flags|=FRCLR;
     msg->from[0]=lwclch(msg->from[0]);
}

void
uclfrom(                           /* un-"clear" from field of message     */
struct message *msg)               /*   message to unclear                 */
{
     if (msg->flags&FRCLR) {
          msg->from[0]=lwclch(msg->from[0]);
     }
}

void
clrto(                             /* "clear" to field of message          */
struct message *msg)               /*   message to clear                   */
{
     msg->flags|=TOCLR;
     msg->to[0]=lwclch(msg->to[0]);
}

void
uclto(                             /* un-"clear" to field of message       */
struct message *msg)               /*   message to unclear                 */
{
     if (msg->flags&TOCLR) {
          msg->to[0]=lwclch(msg->to[0]);
     }
}

char
lwclch(                            /* reversable clobber/unclobber of char */
char c)                            /*   character to clobber/unclobber     */
{
     static char xform[256]={
            0, 32, 34, 39, 40, 41, 44, 45, 46, 48, 49, 50, 51, 52, 53, 54,
           55, 56, 57, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
            1, 78,  2, 79, 80, 81, 82,  3,  4,  5, 42, 83,  6,  7,  8, 47,
            9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 84, 85, 86, 87, 88, 89,
           90, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 33, 35,
           36, 37, 38, 43, 58, 59, 60, 61, 62, 63, 64, 95, 97, 98, 99, 91,
          100, 92, 93, 94, 96,123,124,125,126,127,166,167,168,169,170,171,
          172,173,174,175,176,177,178,179,180,181,182,101,102,103,104,105,
          183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,
          199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,
          215,216,217,218,219,220,106,107,108,109,110,111,112,113,114,115,
          116,117,118,119,120,121,122,128,129,130,131,132,133,134,135,136,
          137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,
          153,154,155,156,157,158,159,160,161,162,163,164,165,224,225,226,
          221,222,223,240,241,242,243,244,245,246,247,248,249,250,251,252,
          227,228,229,230,231,232,233,234,235,236,237,238,239,  0,  0,  0
     };

     return(xform[c]);
}

BOOL                               /*   returns TRUE if directory exists   */
fmdir(                             /* find or make directory               */
char *dirname)                     /*   directory name                     */
{
     ASSERT(dirname != NULL);
     ASSERT(strlen(dirname) < MAXDIR);
     if (!dexist(dirname)) {
          return(makedir(dirname));
     }
     return(TRUE);
}

BOOL                               /*   returns TRUE if successful         */
makedir(                           /* make nested directories              */
char *dirname)                     /*   directory name(s)                  */
{
     char *tmpptr,tmpdir[MAXPATH];

     ASSERT(dirname != NULL);
     ASSERT(strlen(dirname) < MAXPATH);
     if (*dirname == '\0') {
          return(FALSE);
     }
     tmpptr=strchr(dirname,':');
     if (tmpptr == NULL) {
          tmpptr=dirname;
     }
     else {
          ++tmpptr;
          if (*tmpptr == '\0') {
               return(FALSE);
          }
     }
     if (*tmpptr == '\\') {
          ++tmpptr;
     }
     do {
          tmpptr=strchr(tmpptr+1,'\\');
          if (tmpptr == NULL) {
               strcpy(tmpdir,dirname);
          }
          else {
               stlcpy(tmpdir,dirname,(int)(tmpptr-dirname)+1);
          }
          if (!dexist(tmpdir)) {
               if (mkdir(tmpdir) != 0) {
                    return(FALSE);
               }
          }
     } while (tmpptr != NULL);
     return(TRUE);
}

BOOL                               /*   returns TRUE if directory exists   */
dexist(                            /* check if directory exists            */
char *dirname)                     /*   name of directory (no trailing \)  */
{
     char fulpth[MAXPATH];

     ASSERT(dirname != NULL);
     ASSERT(strlen(dirname) < MAXDIR);
     if ((strchr(dirname,'*') == NULL) && (strchr(dirname,'?') == NULL)) {
          normspec(fulpth,dirname);
          if (fnd1st(&gmefb,fulpth,FAMDIR)) {
               return((gmefb.attr&FAMDIR) == FAMDIR);
          }
     }
     return(FALSE);
}

BOOL                               /*   returns TRUE if file exists        */
fexist(                            /* check if file exists                 */
char *name)                        /*   path+file name to find             */
{
     ASSERT(name != NULL);
     ASSERT(strlen(name) < MAXPATH);
     if ((strchr(name,'*') == NULL) && (strchr(name,'?') == NULL)) {
          return(fnd1st(&gmefb,name,0));
     }
     return(FALSE);
}

BOOL
isfnmc(                            /* is this a valid file name character? */
char c)                            /*   character to check                 */
{
     return(c > ' ' && strchr("*?/\\:;,><|[]\"+=",c) == NULL);
}

char *                             /*   returns pointer to file name       */
tmpanam(char *dest)                /* create temp attachment file name     */
{
     int i;
     char *src;
     static char tmpdest[FLNSIZ];
     static unsigned long counter=0;

     if (dest == NULL) {
          dest=tmpdest;
     }
     src=(char *)&counter;
     for (i=0 ; i < 8 ; ++i) {
          if (i&1) {
               dest[i]='A'+((*src)>>4);
               ++src;
          }
          else {
               dest[i]='A'+((*src)&0x0F);
          }
     }
     ++counter;
     strcpy(&dest[8],".TMP");
     return(dest);
}

BOOL
istmp(                             /* is this a temporary attachment file  */
char *fname)                       /*   path+file name to check            */
{
     char *cp;

     cp=filnpart(fname);
     return(cp != NULL && (cp=strchr(cp,'.')) != NULL && sameas(cp,".TMP"));
}

BOOL                               /*   was able to allocate?              */
alcarr(                            /* alloc another item in array in blocks*/
void **ppar,                       /*   pointer to pointer to array        */
int itmsiz,                        /*   size of individual array items     */
int nitems,                        /*   # items currently in array         */
int blksiz)                        /*   # items in each block              */
{                                  /*   (assumes ++nitems after each call) */
     long newlen;
     void *tmp;

     ASSERT(ppar != NULL);
     if (nitems >= MAXBLK/itmsiz) {
          return(FALSE);
     }
     if (nitems%blksiz == 0) {
          newlen=itmsiz*blksiz*(nitems/blksiz+1);
          if (newlen > MAXBLK) {
               newlen=MAXBLK;
          }
          tmp=(void *)malloc((unsigned)newlen);
          if (tmp == NULL) {
               return(FALSE);
          }
          if (*ppar != NULL) {
               movmem(*ppar,tmp,itmsiz*nitems);
               free(*ppar);
          }
          *ppar=tmp;
          #ifdef DEBUG
               setmem(&((char *)(*ppar))[itmsiz*nitems],itmsiz*blksiz,0);
          #endif
     }
     return(TRUE);
}

char *                             /*   returns pointer to string          */
zpad(                              /* zero pad a string                    */
char *s,                           /*   pointer to string                  */
int len)                           /*   length of string buffer            */
{
     int i;

     i=strlen(s);
     setmem(s+i,len-i,0);
     return(s);
}

char *                             /*   copy of pointer to string          */
new2ret(                           /* replace '\n' with '\r' in string     */
char *str)                         /*   string to replace                  */
{
     return(strrpl(str,'\n','\r'));
}

BOOL                               /*   returns TRUE if string was found   */
strsrep(                           /* replace a stg in a stg w/ a stg      */
char *stg,                         /*   the string to search               */
char *from,                        /*   the string to be replaced          */
char *to)                          /*   the string to replace with         */
{
     static char rs[400];
     int loop,len;
     char *o;

     setmem(rs,400,0);
     len=strlen(o=stg);
     if (len >= 200) {
          return(FALSE);
     }
     for (loop=0 ; loop < len ; loop++) {
          if (sameto(from,o)) {
               strcat(rs,to);
               strcat(rs,&o[strlen(from)]);
               strcpy(stg,rs);
               return(TRUE);
          }
          rs[loop]=*o++;
     }
     return(FALSE);
}

#ifdef DEBUG
BOOL
seqok(                             /* check for valid sequence code        */
int sequence,                      /*   sequence code                      */
int forum)                         /*   forum ID                           */
{
     switch (sequence) {
     case ESQTOU:
     case ESQFRU:
     case ESQTHR:
          return(forum == EMLID);
     case FSQFOR:
     case FSQTHR:
     case FSQSCN:
          return(forum != EMLID);
     default:
          return(FALSE);
     }
}

BOOL
ctxok(                             /* check for valid context setup        */
struct rdctx *ctx)                 /*   context structure                  */
{
     ASSERT(ctx != NULL);
     if (seqok(ctx->seq,ctx->fid)
      && (ctx->fid == EMLID || fidxst(ctx->fid))) {
          return(TRUE);
     }
     return(FALSE);
}
#endif

