/***************************************************************************
 *                                                                         *
 *   DOSFACE.C                                                             *
 *                                                                         *
 *   Copyright (C) 1988 GALACTICOMM, Inc.      All Rights Reserved.        *
 *                                                                         *
 *   Here are some garden-variety routines that handle date and time       *
 *   functions.                                                            *
 *                                                                         *
 *                                                                         *
 *   Dates are represented in four ways:  (resol 1 day, range 1980-2069)   *
 *        Packed 16-bit format:  YYYYYYYMMMMDDDDD (as in DOS directories)  *
 *        Count of days since 1/1/1980                                     *
 *        ASCII:  "MM/DD/YY" as in "01/09/89"                              *
 *        ASCII european:  "DD-MMM-YY" as in "09-JAN-89"                   *
 *                                                                         *
 *        today() returns current day (accdg to DOS) in Packed form        *
 *        ncdate() converts Packed to ASCII                                *
 *        ncedat() converts Packed to ASCII european                       *
 *        dcdate() converts ASCII to Packed (year defaults to this year)   *
 *        cofdat() converts Packed to Count                                *
 *        datofc() converts Count to Packed                                *
 *        (see also DOSFACE.H)                                             *
 *                                                                         *
 *   Time-of-day is represented in two ways:  (resol 2 sec, range 1 day)   *
 *        Packed 16-bit format:  HHHHHMMMMMMSSSSS[S] (as in directories)   *
 *        ASCII:  "HH:MM:SS" as in "23:59:58" (2 sec before midnite)       *
 *                                                                         *
 *        now() returns the current time (accdg to DOS) in Packed form     *
 *        nctime() converts Packed to ASCII                                *
 *        dctime() converts ASCII to Packed (seconds default to 00)        *
 *        (see also DOSFACE.H)                                             *
 *                                                                         *
 *                                                                         *
 *                            - RNStein  B03F  2952  01/31/88  31 JAN 88   *
 *                                                                         *
 ***************************************************************************/
 
#include "stdio.h"
#include "dos.h"
#include "dosface.h"
#include "portable.h"
 
 
int today()               /* returns current day in YYYYYYYMMMMDDDDD format */
{
     union REGS regs;
 
     regs.h.ah=0x2A;
     intdos(&regs,&regs);
     return(dddate(regs.h.dh,regs.h.dl,regs.x.cx));
}
 
int now()             /* returns current time in HHHHHMMMMMMSSSSS[S] format */
{
     union REGS regs;
 
     regs.h.ah=0x2C;
     intdos(&regs,&regs);
     return(dttime(regs.h.ch,regs.h.cl,regs.h.dh));
}
 
char *ncdate(date)                          /* Encodes date into "MM/DD/YY" */
int date;
{
     static char buff[9];
 
     sprintf(buff,"%02d/%02d/%02d",
             ddmon(date),ddday(date),ddyear(date)%100);
     return(buff);
}
 
char *nctime(time)                          /* Encodes time into "HH:MM:SS" */
int time;
{
     static char buff[9];
 
     sprintf(buff,"%02d:%02d:%02d",dthour(time),dtmin(time),dtsec(time));
     return(buff);
}
 
char *ncedat(date)        /* Encodes date into "DD-MMM-YY" (european style) */
int date;
{
     static char moname[][4]={"JAN","FEB","MAR","APR","MAY","JUN",
                              "JUL","AUG","SEP","OCT","NOV","DEC"};
     static char buff[10];
 
     sprintf(buff,"%02d-%s-%02d",
             ddday(date),moname[ddmon(date)-1],ddyear(date)%100);
     return(buff);
}
 
#if CNTLIT
#define AABBCC 5                           /* in this compiler, sscanf()    */
#define AABB   3                           /* counts control string matches */
#else
#define AABBCC 3                           /* but in this one, sscanf()     */
#define AABB   2                           /* counts assignments            */
#endif
 
int dcdate(datstr)                         /* Decode date from "MM/DD[/YY]" */
char *datstr;                             /* returns -1 (0xFFFF) if invalid */
{
     int mon,day,year;
 
     if (sscanf(datstr,"%d/%d/%d",&mon,&day,&year) == AABBCC ||
         (year=ddyear(today())%100,sscanf(datstr,"%d/%d",&mon,&day) == AABB)) {
         return(dddate(mon,day,year+(year < 80 ? 2000 : 1900)));
     }
     return(-1);
}
 
int dctime(timstr)                         /* Decode time from "HH:MM[:SS]" */
char *timstr;                             /* returns -1 (0xFFFF) if invalid */
{
     int hour,mint,sec;
 
     if (sscanf(timstr,"%d:%d:%d",&hour,&mint,&sec) == AABBCC ||
         (sec=0,sscanf(timstr,"%d:%d",&hour,&mint) == AABB)) {
         return(dttime(hour,mint,sec));
     }
     return(-1);
}
 
static int daymon[]={0,31,59,90,120,151,181,212,243,273,304,334};
 
int cofdat(date)                     /* Compute number of days since 1/1/80 */
int date;                          /* date in DOS format:  YYYYYYYMMMMDDDDD */
{
     int year,month,days;
 
     year=ddyear(date)-1980;
     month=ddmon(date);
     days=365*year+(year+3)/4;
     days+=daymon[month-1]+(month > 2 && (year&3) == 0);
     return(days+ddday(date)-1);
}
 
int datofc(count)               /* Compute DOS date format YYYYYYYMMMMDDDDD */
int count;                               /* from count of days since 1/1/80 */
{
     int year,month,ydays,mdays,thisyr,ytm;
 
     for (year=1980,ydays=0,thisyr=366 ; ydays+thisyr <= count ; ) {
          ydays+=thisyr;
          thisyr=365+((++year&3) == 0);
     }
     for (month=1,mdays=0 ; month < 12 ; month++) {
          ytm=daymon[month]+((year&3) == 0 && month >= 2);
          if (ytm > count-ydays) {
               break;
          }
          mdays=ytm;
     }
     return(dddate(month,count-ydays-mdays+1,year));
}
 
