/***************************************************************************
 *                                                                         *
 *   GALBJ.C                                                               *
 *                                                                         *
 *   Copyright (C) 1990-1994 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is the Tele-Games Blackjack Game code.                           *
 *                                                                         *
 *                                            - J. Kobal 6/19/90           *
 *                                            - M. Donnelly 11/8/93        *
 *                                                  Stand alone DLL        *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "flash.h"
#include "bbsutils.h"
#include "etlapi.h"
#include "galbj.h"

/* GALBJ.C 26/03/92 19.34.12 */
void delayed_init(void);
void initbjv(void);
void initbj(void);
void bjack(void);
void bprfall(void);
void bprfbt(int num);
void bdeal(void);
void bshuffle(void);
void plybjk(void);
void bfindwin(void);
void bshowin(int num);
void bshwhnd(void);
void bshowh(int num);
int bnxtrn(int num);
void bnext(void);
void bjon(void);
void bjfold(void);
void bjoff(void);
void bjexit(void);
void bjshtdwn(void);
void bjlow(void);
void bjhigh(void);
void bjcall(void);
void bjbet(void);
void bjrais(void);
void bdspcrd(int num);
void bjhit(void);
void bjstay(void);
void bjhlp(void);
void bjscan(void);
void bjpot(void);
void bjturn(void);
int bjhook(int code);
void bjinp(void);

STATIC void btotal(struct bjplyr *player);
STATIC void bdealhnd(struct bjplyr *player);
STATIC void bpsttc(int num,long amtcrd);
STATIC void bjinf(void);

FILE *bjmb;

static
struct bjplyr {               /* Blackjack player structure                */
     int cards[12];           /*   The players hand (12 cards maximum)     */
     int numcards;            /*   Number of cards in player's hand        */
     int total;               /*   Total of player's hand                  */
     long bet;                /*   How much player has bet on current hand */
     int flags;               /*   Various bit flags                       */
} *bjarr,                     /* dynamically allocated player array        */
  *bjptr;                     /* handy bjplyr structure pointer            */

                              /* Bit flag definitions for each player      */
#define REQUEST     0x0001    /*   Player has requested to play            */
#define PLAYING     0x0002    /*   Player is currently playing the game    */
#define STAYEDB     0x0004    /*   Player has discarded already            */
#define BUSTEDB     0x0008    /*   Player has busted                       */

static
int bdck[52],                 /* Randomly-ordered deck of playing cards    */
    bcrdnum,                  /* Index to bdck[] for next card to be dealt */
    bminbt,                   /* Minimum bet for the game                  */
    bmaxbt,                   /* Maximum bet for the game                  */
    bmaxraise,                /* Maximum number of raises per bet round    */
    bjwt,                     /* Number of time cycles game has waited     */
    braise,                   /* Number of times the bet has been raised   */
    brequests,                /* Number of people requesting to play       */
    btrn,                     /* Whose turn it is                          */
    bfrst,                    /* First player in betting cycle             */
    bplayers,                 /* Number of people currently playing        */
    bflags;                   /* Various bit flags for the game            */

extern
int gbldbt;                   /* can users gamble with their debt?         */

                              /* Bit flag definitions for bflags variable  */
#define HIGHSTAKES  0x0001    /*   Game is for high stakes                 */
#define HITTING     0x0002    /*   Players are discarding cards            */
#define ROUNDOVER   0x0004    /*   Players are finished discarding         */
#define WINNER      0x0008    /*   Game has been won                       */

static
long bplybt,                  /* Bet amount needed to "stay in the game"   */
     bpt;                     /* Current "pot" for the game                */


/*--- OPTIONS FROM GALBJ.MSG ---*/
static
int bjkply,                   /* is Blackjack available to users now       */
    minply,                   /* minimum number of people to play game     */
    maxply,                   /* maximum number of people to play game     */
    btwngm,                   /* time to wait between games before checking*/
    btwnst,                   /* time to wait between in game checks       */
    minbtl,                   /* minimum bet in a low stakes bjack game    */
    maxbtl,                   /* maximum bet in a low stakes bjack game    */
    minbth,                   /* minimum bet in a high stakes bjack game   */
    maxbth,                   /* maximum bet in a high stakes bjack game   */
    maxrsl,                   /* maximum number of raises in low stakes    */
    maxrsh,                   /* maximum number of raises in high stakes   */
    resetl,                   /* after each game reset to low stakes? 1=YES*/
    housct,                   /* percent of pot taken by the house         */
    dlygam,                   /* number of checks before prodding a user   */
    kckout,                   /* number of checks before tossing a user    */
    logwin;                   /* log winnings to audit trail?              */

int gbldbt;

void EXPORT
init__bj(void)                /* initialize blackjack when system comes up */
{
     bjmb=opnmsg("GALBJ.MCV");
     setmbk(bjmb);
     bjarr=(struct bjplyr *)alczer(nterms*sizeof(struct bjplyr));
     initbjv();
     bjkply=ynopt(PLYBJCK);
     gbldbt=ynopt(GBLDBT);
     minply=numopt(BMINPLY,1,11);
     maxply=numopt(BMAXPLY,1,11);
     btwngm=numopt(BBTWNGM,15,600);
     btwnst=numopt(BBTWNST,5,60);
     minbtl=numopt(BMINBTL,0,32766);
     maxbtl=numopt(BMAXBTL,0,32766);
     minbth=numopt(BMINBTH,0,32766);
     maxbth=numopt(BMAXBTH,0,32766);
     maxrsl=numopt(BMAXRSL,0,100);
     maxrsh=numopt(BMAXRSH,0,100);
     resetl=ynopt(BRESETL);
     housct=numopt(BHOUSCT,0,100);
     dlygam=numopt(BDLYGAM,1,10000);
     kckout=numopt(BKCKOUT,1,10000);
     logwin=ynopt(LOGWIN);
     bflags=0;
     rtkick(1,delayed_init);
}

void
delayed_init(void)            /* delayed continuation of initialization    */
{
     if (addhook("bj",bjhook) != -1) {
          rtkick(btwngm,bjack);
     }
}

int
bjhook(int code)              /* status 3 entry point                      */
{
     switch (code) {
     case TH_INFO:
          bjinf();
          return(1);
     case TH_INPUT:
          bjinp();
          return(1);
     case TH_EXIT:
          bjexit();
          return(1);
     case TH_SHTDWN:
          bjshtdwn();
          return(1);
     }
     return(0);
}

void
initbjv(void)                 /* initialize things for a new game          */
{
     bflags&=~(HITTING|ROUNDOVER|WINNER);
     bpt=0;
     braise=0;
     bfrst=-1;
     btrn=-1;
     if (resetl && (bflags&HIGHSTAKES)) {
          bflags&=~HIGHSTAKES;
          prfmlt(BLWSTKS);
          bprfall();
     }
}

void
bjinp(void)                   /* normal input handler                      */
{
     setmbk(bjmb);
     if (margc == 1) {
          bjhlp();
          rstmbk();
          return;
     }
     if (sameas(margv[1],"off")) {
          bjoff();
     }
     else if (sameas(margv[1],"on")) {
          bjon();
     }
     else if (sameas(margv[1],"hand")) {
          bshwhnd();
     }
     else if (sameas(margv[1],"scan")) {
          bjscan();
     }
     else if (sameas(margv[1],"turn")) {
          bjturn();
     }
     else if (sameas(margv[1],"fold")) {
          bjfold();
     }
     else if (sameas(margv[1],"pot")) {
          bjpot();
     }
     else if (sameas(margv[1],"call")) {
          bjcall();
     }
     else if (sameas(margv[1],"bet")) {
          bjbet();
     }
     else if (sameas(margv[1],"raise")) {
          bjrais();
     }
     else if (sameas(margv[1],"hit")) {
          bjhit();
     }
     else if (sameas(margv[1],"stay")) {
          bjstay();
     }
     else if (sameas(margv[1],"low")) {
          bjlow();
     }
     else if (sameas(margv[1],"high")) {
          bjhigh();
     }
     else {
          bjhlp();
     }
}

void
bjack(void)                   /* handle initiating a game of blackjack     */
{
     setmbk(bjmb);
     if (brequests < minply) {
          prfmlt(NEDMORB);
          bprfall();
          rtkick(btwngm,bjack);
     }
     else {
          prfmlt(PLAYBJK);
          bprfall();
          bshuffle();
          bdeal();
          if (bplayers > 0) {
               bnext();
               rtkick(btwnst,plybjk);
          }
          else {
               initbjv();
               rtkick(btwnst,bjack);
          }
     }
}

void
bprfall(void)                 /* send a bjack message to requesting users  */
{
     int i;

     tlcpmt();
     for (i=0 ; i < nterms ; i++) {
          if (bjarr[i].flags&REQUEST) {
               outmlt(i);
          }
     }
     clrmlt();
}

void
bprfbt(num)                   /* send bjack message to all but 'num'       */
int num;
{
     int i;

     tlcpmt();
     for (i=0 ; i < nterms ; i++) {
          if ((bjarr[i].flags&REQUEST) && (i != num)) {
               outmlt(i);
          }
     }
     clrmlt();
}

void
bdeal(void)                   /* set users up for playing a game of bjack  */
{
     int num;

     bpt=0;
     bplayers=0;
     bminbt=((bflags&HIGHSTAKES) ? minbth : minbtl);
     bmaxbt=((bflags&HIGHSTAKES) ? maxbth : maxbtl);
     bmaxraise=((bflags&HIGHSTAKES) ? maxrsh : maxrsl);
     bplybt=bminbt;
     for (bjptr=bjarr,num=0 ; num < nterms ; num++,bjptr++) {
          if (bjptr->flags&REQUEST) {
               if (!odedcrd(num,bminbt,!gbldbt,0)) {
                    prfmlt(NOTENFB);
               }
               else {
                    bjptr->flags&=~(STAYEDB|BUSTEDB);
                    bjptr->flags|=PLAYING;
                    bplayers++;
                    bpt+=bminbt;
                    bjptr->bet=bminbt;
                    bdealhnd(bjptr);
                    prfmlt(ANTEBJK,bminbt);
               }
               tlcpmt();
               outmlt(num);
          }
     }
}

STATIC void
btotal(struct bjplyr *player) /* calculate a players card totals           */
{
     int v,t,n,ace;

     t=ace=0;
     for (n=0 ; n < player->numcards ; n++) {
          v=((player->cards[n])%13)+2;
          if (v > 10) {
               if (v == 14) {
                    ace++;
                    v=11;
               }
               else {
                    v=10;
               }
          }
          t=t+v;
     }
     while ((t > 21) && (ace > 0)) {
          ace--;
          t-=10;
     }
     player->total=t;
}

STATIC void
bdealhnd(struct bjplyr *player)    /* deal two cards to a player           */
{
     int a;

     for (a=0 ; a < 2 ; a++) {
          player->cards[a]=bdck[bcrdnum++];
     }
     player->numcards=2;
     btotal(player);
}

void
bshuffle(void)                /* shuffle the deck of cards                 */
{
     int i,temp,swap;

     for (i=0 ; i < 52 ; i++) {
          bdck[i]=i;
     }
     for (i=0 ; i < 51 ; i++) {
          temp=bdck[i];
          swap=genrdn(i,52);
          bdck[i]=bdck[swap];
          bdck[swap]=temp;
     }
     bcrdnum=0;
}

void
plybjk(void)                  /* handle the current blackjack game         */
{
     int num;

     setmbk(bjmb);
     if (bplayers == 0) {
          initbjv();
          prfmlt(NXTGAMB);
          bprfall();
          for (num=0 ; num < nterms ; num++) {
               bjarr[num].flags&=~PLAYING;
          }
          rtkick(btwngm,bjack);
     }
     else if (bplayers <= 1) {
          for (num=0 ; num < nterms ; num++) {
               if (bjarr[num].flags&PLAYING) {
                    bshowin(num);
                    initbjv();
                    prfmlt(NXTGAMB);
                    bprfall();
                    for (num=0 ; num < nterms ; num++) {
                         bjarr[num].flags&=~PLAYING;
                    }
                    break;
               }
          }
          initbjv();
          rtkick(btwngm,bjack);
     }
     else if (btrn == -1) {
          initbjv();
          rtkick(btwngm,bjack);
     }
     else {
          if (!(bjarr[btrn].flags&PLAYING)) {
               bnext();
          }
          else if (++bjwt > dlygam) {
               if (!otstcrd(btrn,(bplybt-bjarr[btrn].bet),!gbldbt)) {
                    prfmlt(BYUROUT);
                    tlcpmt();
                    outmlt(btrn);
                    bjarr[btrn].flags&=~PLAYING;
                    bplayers--;
                    prfmlt(BNOCRED,uacoff(btrn)->userid);
                    prfmlt(BFOLDIN,uacoff(btrn)->userid);
                    bprfbt(btrn);
                    prfmlt(BYUFOLD);
                    outmlt(btrn);
                    num=btrn;
                    bnext();
                    if (bfrst == num) {
                         bfrst=bnxtrn(num);
                    }
                    pmlt("");
                    tlcpmt();
                    outmlt(num);
               }
               else if (bjwt > kckout) {
                    prfmlt(BBYEBYE);
                    tlcpmt();
                    outmlt(btrn);
                    bjarr[btrn].flags&=~PLAYING;
                    bplayers--;
                    prfmlt(BFOLDIN,uacoff(btrn)->userid);
                    bprfbt(btrn);
                    prfmlt(BYUFOLD);
                    tlcpmt();
                    outmlt(btrn);
                    num=btrn;
                    bnext();
                    if (bfrst == num) {
                         bfrst=bnxtrn(num);
                    }
                    pmlt("");
                    tlcpmt();
                    outmlt(num);
               }
               else {
                    prfmlt(WAITINB);
                    tlcpmt();
                    outmlt(btrn);
               }
          }
          if (bflags&WINNER) {
               initbjv();
               rtkick(btwngm,bjack);
          }
          else {
               rtkick(btwnst,plybjk);
          }
     }
}

void
bfindwin(void)                /* find the winner of this game of bjack     */
{
     int num1,num2,t,n;

     t=num1=-1;
     for (num2=0 ; num2 < nterms ; num2++) {
          if ((bjptr=&bjarr[num2])->flags&PLAYING) {
               if (bjptr->total > t) {
                    t=bjptr->total;
                    num1=num2;
                    n=1;
               }
               else if (bjptr->total == t) {
                    n++;
               }
          }
     }
     if ((num1 >= 0) && (n > 0)) {
          if (n == 1) {
               bshowin(num1);
               initbjv();
               prfmlt(NXTGAMB);
               bprfall();
               for (num2=0 ; num2 < nterms ; num2++) {
                    bjarr[num2].flags&=~PLAYING;
               }
          }
          else {
               bpt=bpt/n;
               for (num2=0 ; num2 < nterms ; num2++) {
                    if ((bjptr=&bjarr[num2])->flags&PLAYING) {
                         if (bjptr->total == t) {
                              bshowin(num2);
                         }
                    }
               }
               initbjv();
               prfmlt(NXTGAMB);
               bprfall();
               for (num2=0 ; num2 < nterms ; num2++) {
                    bjarr[num2].flags&=~PLAYING;
               }
          }
     }
}

void
bshowin(int num)              /* show the winner of the bjack game         */
{
     int unum;
     long amt;

     bflags|=WINNER;
     unum=usrnum;
     amt=(bpt*(100-housct))/100;
     bpsttc(num,amt);
     for (usrnum=0 ; usrnum < nterms ; usrnum++) {
          if ((bjarr[usrnum].flags&REQUEST) && (usrnum != num)) {
               usaptr=uacoff(usrnum);
               prfmlt(WNBJACK,uacoff(num)->userid);
               bshowh(num);
               tlcpmt();
               outmlt(usrnum);
          }
     }
     usrnum=unum;
     usaptr=uacoff(usrnum);
     prfmlt(YOUWONB,l2as(amt));
     tlcpmt();
     outmlt(num);
}

void
bshwhnd(void)                 /* show a users hand                         */
{
     bshowh(usrnum);
}

void
bshowh(int num)               /* show hand for a given usrnum              */
{
     int a;

     bjptr=&bjarr[num];
     if (bjptr->flags&PLAYING) {
          dspmlt(INSERT6);
          for (a=0 ; a < bjptr->numcards ; a++) {
               dspmlt(PCARD0+2*(bjptr->cards[a]));
          }
          prfmlt(TOTALM,bjptr->total);
     }
     else {
          prfmlt(WHTHNDB);
     }
}

int
bnxtrn(int num)               /* find out whose turn it is next            */
{
     int n;

     if (bplayers <= 0) {
          n=-1;
     }
     else {
          n=num;
          do {
               n=((n+1)%nterms);
          } while (!(bjarr[n].flags&PLAYING) && (n != num));
          if (!(bjarr[n].flags&PLAYING)) {
               n=-1;
          }
     }
     return(n);
}

void
bnext(void)                   /* determine the next event to happen        */
{
     int num;

     bjwt=0;
     btrn=bnxtrn(btrn);
     if (btrn == -1) {
          initbjv();
          prfmlt(NXTGAMB);
          bprfall();
          for (num=0 ; num < nterms ; num++) {
               bjarr[num].flags&=~PLAYING;
          }
     }
     else if (bfrst == btrn) {
          if (bflags&HITTING) {
               bflags&=~HITTING;
               bflags|=ROUNDOVER;
               if (bplayers > 1) {
                    bfindwin();
               }
               else if (bplayers == 1) {
                    bshowin(btrn);
                    initbjv();
                    prfmlt(NXTGAMB);
                    bprfall();
                    for (num=0 ; num < nterms ; num++) {
                         bjarr[num].flags&=~PLAYING;
                    }
               }
          }
          else {
               bflags|=HITTING;
               bfrst=-1;
               btrn=-1;
               bnext();
          }
     }
     else {
          if (bflags&HITTING) {
               prfmlt(BYURDSC);
          }
          else {
               prfmlt(BYURBET,l2as(bplybt-bjarr[btrn].bet));
          }
          tlcpmt();
          outmlt(btrn);
     }
     if (bfrst == -1) {
          bfrst=btrn;
     }
     clrmlt();
}

void
bjon(void)                    /* respond to user request to play game      */
{
     if (otstcrd(usrnum,1L,!gbldbt)) {
          if (!(bjarr[usrnum].flags&REQUEST)) {
               if (usrptr->flags&INVISB) {
                    prfmlt(YOUINV);
               }
               else if (bjkply && brequests < maxply) {
                    prfmlt(DEALINB,usaptr->userid);
                    bprfbt(usrnum);
                    prfmlt(BJACKON,(bflags&HIGHSTAKES) ? ", for high stakes"
                                                       : "");
                    bjarr[usrnum].flags|=REQUEST;
                    brequests++;
               }
               else {
                    prfmlt(TBLFULL);
               }
          }
          else {
               prfmlt(BJACKON,bflags&HIGHSTAKES ? ", for high stakes" : "");
          }
     }
     else {
          prfmlt(LIVEPLY);
     }
}

void
bjfold(void)                  /* user requested to fold from game          */
{
     if (bjarr[usrnum].flags&PLAYING) {
          bjarr[usrnum].flags&=~PLAYING;
          bplayers--;
          prfmlt(BFOLDIN,usaptr->userid);
          bprfbt(usrnum);
          prfmlt(BYUFOLD);
          tlcpmt();
          outmlt(usrnum);
          if (btrn == usrnum) {
               bnext();
          }
          if (bfrst == usrnum) {
               bfrst=bnxtrn(usrnum);
          }
          pmlt("");
     }
     else {
          prfmlt(CNTFLDB);
     }
}

void
bjoff(void)                   /* user wants out of current status of bjack */
{
     if (bjarr[usrnum].flags&PLAYING) {
          prfmlt(DELOUTB,usaptr->userid);
          bprfbt(usrnum);
          bjfold();
          prfmlt(BJACKOF);
          bjexit();
     }
     else if (bjarr[usrnum].flags&REQUEST) {
          prfmlt(RQSTOFB);
          bjexit();
     }
     else {
          prfmlt(NTPLYNG);
     }
}

void
bjexit(void)                  /* user forced to exit game completely       */
{
     setmbk(bjmb);
     if (bjarr[usrnum].flags&REQUEST) {
          bjarr[usrnum].flags&=~REQUEST;
          brequests--;
     }
     if (bjarr[usrnum].flags&PLAYING) {
          bjarr[usrnum].flags&=~PLAYING;
          if (--bplayers == 0) {
               btrn=-1;
               bfrst=-1;
          }
          else {
               if (btrn == usrnum) {
                    bnext();
               }
               if (bfrst == usrnum) {
                    bfrst=bnxtrn(usrnum);
               }
          }
          pmlt("");
     }
     rstmbk();
}

void
bjshtdwn(void)                /* shutdown routine for Blackjack            */
{
     clsmsg(bjmb);
}

void
bjlow(void)                   /* set game to low stakes only               */
{
     if (hasmkey(STKKEY)) {
          bflags&=~HIGHSTAKES;
          prfmlt(BLWSTKS);
          bprfbt(usrnum);
          prfmlt(BLOWSTK);
     }
     else {
          prfmlt(BJKHELP);
     }
}

void
bjhigh(void)                  /* set game to high stakes                   */
{
     if (hasmkey(STKKEY)) {
          bflags|=HIGHSTAKES;
          prfmlt(BHISTKS);
          bprfbt(usrnum);
          prfmlt(BHGHSTK);
     }
     else {
          prfmlt(BJKHELP);
     }
}

void
bjcall(void)                  /* user is "calling"                         */
{
     long betamt;

     if ((bjarr[usrnum].flags&PLAYING) && (btrn == usrnum)
       && !(bflags&HITTING)) {
          betamt=bplybt-bjarr[usrnum].bet;
          if (!odedcrd(usrnum,betamt,!gbldbt,0)) {
               prfmlt(BCNTBET);
          }
          else {
               bpt+=betamt;
               bjarr[usrnum].bet=bplybt;
               prfmlt(BETCALB, usaptr->userid);
               bprfbt(usrnum);
               prfmlt(YOUCALB);
               tlcpmt();
               outmlt(usrnum);
               bnext();
               pmlt("");
          }
     }
     else {
          prfmlt(NOTYURB);
     }
}

void
bjbet(void)                   /* user is "betting"                         */
{
     long r,b;

     if ((bjarr[usrnum].flags&PLAYING) && (btrn == usrnum)
       && !(bflags&HITTING)) {
          if (margc == 2) {
               bjcall();
          }
          else {
               b=atol(margv[2]);
               if (b < bminbt) {
                    prfmlt(BBTOOLW, bminbt);
               }
               else if (b > bmaxbt) {
                    prfmlt(BBTOOHI, bmaxbt);
               }
               else if (!odedcrd(usrnum,b,!gbldbt,0)) {
                    prfmlt(BCNTBET);
               }
               else {
                    bjarr[usrnum].bet+=b;
                    bpt+=b;
                    r=bjarr[usrnum].bet-bplybt;
                    if (r < 0) {
                         prfmlt(BETSOMB, usaptr->userid, b);
                         bprfbt(usrnum);
                         prfmlt(BETMORB);
                         tlcpmt();
                         outmlt(usrnum);
                    }
                    else {
                         if (r == 0) {
                              prfmlt(BETCALB, usaptr->userid);
                              bprfbt(usrnum);
                              prfmlt(YOUCALB);
                         }
                         else if (braise < bmaxraise) {
                              bplybt+=r;
                              braise++;
                              bfrst=btrn;
                              prfmlt(BETRASB,usaptr->userid,l2as(r));
                              bprfbt(usrnum);
                              prfmlt(YOURASB,l2as(r));
                         }
                         else {
                              bjarr[usrnum].bet-=r;
                              crdusr(usaptr->userid,l2as(r),0,0);
                              bpt-=r;
                              prfmlt(BETCALB, usaptr->userid);
                              bprfbt(usrnum);
                              prfmlt(NORAISB);
                         }
                         tlcpmt();
                         outmlt(usrnum);
                         bnext();
                    }
               }
          }
          pmlt("");
     }
     else {
          prfmlt(NOTYURB);
     }
}

void
bjrais(void)                  /* user is "raising"                         */
{
     int b=0;
     long betamt;

     if ((bjarr[usrnum].flags&PLAYING) && (btrn == usrnum)
       && !(bflags&HITTING)) {
          if (margc == 2) {
               bjcall();
          }
          else {
               b=atoi(margv[2]);
               betamt=(bplybt-bjarr[usrnum].bet)+b;
               if (b < bminbt) {
                    prfmlt(BBTOOLW,bminbt);
               }
               else if (b > bmaxbt) {
                    prfmlt(BBTOOHI,bmaxbt);
               }
               else if (!odedcrd(usrnum,betamt,!gbldbt,0)) {
                    prfmlt(BCNTBET);
               }
               else {
                    bpt+=betamt;
                    bjarr[usrnum].bet=bplybt+b;
                    if (b == 0) {
                         prfmlt(BETCALB,usaptr->userid);
                         bprfbt(usrnum);
                         prfmlt(YOUCALB);
                    }
                    else if (braise < bmaxraise) {
                         bplybt+=b;
                         braise++;
                         bfrst=btrn;
                         prfmlt(BETRASB,usaptr->userid,spr("%d",b));
                         bprfbt(usrnum);
                         prfmlt(YOURASB,spr("%d",b));
                    }
                    else {
                         bjarr[usrnum].bet-=b;
                         crdusr(usaptr->userid,l2as(b),0,0);
                         bpt-=b;
                         prfmlt(BETCALB,usaptr->userid);
                         bprfbt(usrnum);
                         prfmlt(NORAISB);
                    }
                    tlcpmt();
                    outmlt(usrnum);
                    bnext();
               }
          }
          pmlt("");
     }
     else {
          prfmlt(NOTYURB);
     }
}

void
bdspcrd(int num)              /* display a card                            */
{
     static char *crds[13]={
          " Two",
          " Three",
          " Four",
          " Five",
          " Six",
          " Seven",
          "n Eight",
          " Nine",
          " Ten",
          " Jack",
          " Queen",
          " King",
          "n Ace"
     };
     static char *suits[4]={
          "Hearts",
          "Clubs",
          "Diamonds",
          "Spades"
     };

     prfmlt(BDLCARD,crds[num%13],suits[num/13]);
}

void
bjhit(void)                   /* user wants to "hit"                       */
{
     bjptr=&bjarr[usrnum];
     if ((bjptr->flags&PLAYING) && (btrn == usrnum) && (bflags&HITTING)) {
          bdspcrd(bjptr->cards[bjptr->numcards++]=bdck[bcrdnum++]);
          tlcpmt();
          outmlt(usrnum);
          prfmlt(BTAKCRD,usaptr->userid);
          bprfbt(usrnum);
          pmlt("");
          btotal(bjptr);
          if (bjptr->total > 21) {
               bjptr->flags|=BUSTEDB;
               bjptr->flags&=~PLAYING;
               bplayers--;
               prfmlt(BBUSTED, usaptr->userid);
               bprfbt(usrnum);
               prfmlt(YOUBUST);
               tlcpmt();
               outmlt(usrnum);
               bnext();
               if (bfrst == usrnum) {
                    bfrst=bnxtrn(usrnum);
               }
               pmlt("");
          }
     }
     else {
          prfmlt(NOTYURB);
     }
}

void
bjstay(void)                  /* user wants to "stay"                      */
{
     bjptr=&bjarr[usrnum];
     if ((bjptr->flags&PLAYING) && (btrn == usrnum) && (bflags&HITTING)) {
          bjptr->flags|=STAYEDB;
          prfmlt(BSTAYIN,usaptr->userid);
          bprfbt(usrnum);
          prfmlt(YOUSTAY);
          tlcpmt();
          outmlt(usrnum);
          bnext();
          pmlt("");
     }
     else {
          prfmlt(NOTYURB);
     }
}

void
bjhlp(void)                   /* display blackjack help message            */
{
     prfmlt(BJKHELP);
}

void
bjscan(void)                  /* display those involved in blackjack       */
{
     int num;

     prfmlt(BSCNHDR);
     for (num=0 ; num < nterms ; num++) {
          if (bjarr[num].flags&REQUEST) {
               prfmlt(BSCNLIN,uacoff(num)->userid,
                      bjarr[num].flags&PLAYING ? "*" : " ");
          }
     }
}

void
bjpot(void)                   /* show the current blackjack pot            */
{
     prfmlt(BJACKPT,l2as(bpt));
}

void
bjturn(void)                  /* show whose turn it is                     */
{
     if (btrn == usrnum) {
          if (bflags&HITTING) {
               prfmlt(BYURDSC);
          }
          else {
               prfmlt(BYURBET,l2as(bplybt-bjarr[usrnum].bet));
          }
     }
     else {
          if (btrn == -1) {
               prfmlt(NOGMYTB);
          }
          else {
               prfmlt(WHOTRNB,uacoff(btrn)->userid,
                      (bflags&HITTING) ? "hit or stay" : "bet");
          }
     }
}

STATIC void
bpsttc(int num,long amtcrd)   /* post credits to an account                */
{
     crdusr(uacoff(num)->userid,l2as(amtcrd),0,1);
     usrnum=-1;
     if (logwin) {
          shocst("TELE-BJACK WINNER","%s won %s credits",uacoff(num)->userid,
                 l2as(amtcrd));
     }
}

STATIC void
bjinf(void)                   /* display bj help                           */
{
     setmbk(bjmb);
     prfmsg(BJKINF);
}
