{$A+,B+,F+,I+,L+,O+,R-,S-,V-}
UNIT tmpcom;

INTERFACE

USES
  Dos;


TYPE
  com_parity = (com_none, com_even, com_odd, com_zero, com_one);

{This variable is TRUE if the interrupt driver has been installed, or FALSE
if it hasn't.  It's used to prevent installing twice or deinstalling when not
installed.}

CONST
  com_installed: Boolean = FALSE;
  usefossil:boolean = FALSE;

var
  mpcoder:boolean;
  mpcode:array[1..6] of byte;
  fosport:word;
  regs:registers;

function com_find_uart:string;
procedure com_flush_rx;
procedure com_flush_tx;
procedure com_purge_tx;
function com_carrier:boolean;
function com_rx:char;
function com_tx_ready:boolean;
function com_tx_empty:boolean;
function com_rx_empty:boolean;
procedure com_tx (ch: Char);
procedure com_tx_string (st: String);
procedure com_lower_dtr;
procedure com_raise_dtr;
procedure com_set_speed(speed:word);
procedure com_set_parity(parity:com_parity; stop_bits:byte);
procedure com_install(portnum:word; var error:word; dofossil:boolean);
procedure com_deinstall;


implementation

uses common;
{Maximum port number (minimum is 1) }

const
  max_port = 8;


{Base i/o address for each COM port}

const
  uart_base: ARRAY [1..max_port] OF Integer = ($3F8, $2F8, $3E8, $2E8,
                                               4220, 4228, 5220, 5228);


{Interrupt numbers for each COM port}

const
  intnums: ARRAY [1..max_port] OF Byte = ($0C, $0B, $0C, $0B,
                                          $0B, $0B, $0B, $0B);


{irq interrupt levels for each port}

const
  i8259levels: ARRAY [1..max_port] OF Byte = (4, 3, 4, 3, 3, 3, 3, 3);

{UART i/o addresses.  Values depend upon which COMM port is selected.}

var
  uart_data:word;             {Data register}
  uart_ier:word;             {Interrupt enable register}
  uart_iir:word;             {Interrupt identification register}
  uart_lcr:word;             {Line control register}
  uart_mcr:word;             {Modem control register}
  uart_lsr:word;             {Line status register}
  uart_msr:word;             {Modem status register}
  uart_spr:word;             {Scratch pad register}


{Original contents of IER and MCR registers.  Used to restore UART
to whatever state it was in before this driver was loaded.}

var
  old_ier:byte;
  old_mcr:byte;


{Original contents of interrupt vector.  Used to restore the vector when
the interrupt driver is deinstalled.}

var
  old_vector:pointer;


{Original contents of interrupt controller mask.  Used to restore the
bit pertaining to the comm controller we're using.}

var
  old_i8259_mask:byte;


{Bit mask for i8259 interrupt controller}

var
  i8259bit:byte;


{Interrupt vector number}

var
  intnum:byte;


{ Receive queue.  Received characters are held here
  until retrieved by com_rx. }

const rx_queue_size=16; {16 normal, Change to suit}

var
  rx_queue:array[1..rx_queue_size] of byte;
  rx_in:word;           {Index of where to store next character}
  rx_out:word;          {Index of where to retrieve next character}
  rx_chars:word;        {Number of chars in queue}


{ Transmit queue.  Characters to be transmitted are held here
  until the UART is ready to transmit them. }

const tx_queue_size=16;    {16 normal, Change to suit}

var
  tx_queue:array[1..tx_queue_size] of byte;
  tx_in:integer;        {Index of where to store next character}
  tx_out:integer;       {Index of where to retrieve next character}
  tx_chars:integer;     {Number of chars in queue}


{This variable is used to save the next link in the "exit procedure" chain.}

var
  exit_save:pointer;

Function com_find_uart:String;
Type Uart_Registers=Array[0..9] OF Byte;  { Uart Registers              }

Var  URegs: Uart_Registers;      { Uart Register Array                  }
     PA   : Word;                { Port Address Com1=$3F8  Com2=$2F8..  }

     RBR,THR,IER,IIR,FCR,LCR,MCR,LSR,MSR,SCR,DLL,DLM,AFR: Word;

{ Save Uart Registers }
Procedure Save_Uart_Registers(BaseAdd: Word; Var URegs: Uart_Registers);
Var I: Byte;
Begin
  ASM CLI; END;
  For I:=1 to 6 Do URegs[I]:=Port[BaseAdd+I];
  Port[BaseAdd+3]:=Port[BaseAdd+3] or $80;
  URegs[7]:=Port[BaseAdd+0];
  URegs[8]:=Port[BaseAdd+1];
  Port[BaseAdd+3]:=Port[BaseAdd+3] and $7F;
  ASM STI; END;
End; { End Procedure }

{------ Restore Uart Registers --------}
Procedure Restore_Uart_Registers(BaseAdd: Word; URegs: Uart_Registers);
Var I: Byte;
Begin
  ASM CLI; END;
  Port[BaseAdd+3]:=Port[BaseAdd+3] or $80;
  Port[BaseAdd+0]:=URegs[7];
  Port[BaseAdd+1]:=URegs[8];
  Port[BaseAdd+3]:=Port[BaseAdd+3] and $7F;
  For I:=1 to 6 Do Port[BaseAdd+I]:=URegs[I];
  ASM STI; END;
End; { End Procedure }

Procedure Return_Code(C: Byte);
Begin

  Case C of
   0:com_find_uart:='No Uart at Port Address';
   1:com_find_uart:='INS8250, INS8250-B';
   2:com_find_uart:='INS8250A, INS82C50A, NS16450, NS16C450';
   3:com_find_uart:='NS16550A';
   4:com_find_uart:='NS16C552';
   End;

   Restore_Uart_Registers(PA,URegs);

End; { End Procedure }

Procedure Set_Uart_Register_Values(PA: Word);
Begin

RBR:=PA+0;         { Receive Buffer Registers          (R  ) (DLAB=0)     }
THR:=PA+0;         { Transmitter Holding Register      (  W) (DLAB=0)     }
IER:=PA+1;         { Interrupt Enable Register         (R/W) (DLAB=0)     }
IIR:=PA+2;         { Interrupt Ident. Register         (R  )              }
FCR:=PA+2;         { FIFO Control Register             (  W)              }
LCR:=PA+3;         { Line Control Register             (R/W)              }
MCR:=PA+4;         { MODEM Control Register            (R/W)              }
LSR:=PA+5;         { Line Status Register              (R  )              }
MSR:=PA+6;         { MODEM Status Register             (R/W)              }
SCR:=PA+7;         { Scratch Register                  (R/W)              }
DLL:=PA+0;         { Divisor Latch (LSB)               (R/W) (DLAB=1)     }
DLM:=PA+1;         { Divisor Latch (MSB)               (R/W) (DLAB=1)     }
AFR:=PA+2;         { Alternate Function Register       (R/W)              }

End; { End Procedure }

Begin  { Main Section of Program }

PA:=Uart_Base[FosPort+1]; { This can be changed to any port address you want }

Save_Uart_Registers(PA,URegs);  { Saves State of Current Uart Registers    }
Set_Uart_Register_Values(PA);   { Return_Code() Restores Uart Registers    }

Port[LCR]:=$AA;                         { Test LCR Registers               }
If $AA<>Port[LCR] Then Return_Code(0);

Port[DLM]:=$55;                         { Test DLM Present 8-bits          }
If $55<>Port[DLM] Then Return_Code(0);

Port[LCR]:=$55;                         { LCR/ DLAB=0                      }
If $55<>Port[LCR] Then Return_Code(0);

Port[IER]:=$55;                         { Test IER Present 4-bits          }
If $05<>Port[IER] Then Return_Code(0);

Port[FCR]:=$0;                          { FIFO's Off, If Present           }
Port[IER]:=$0;                          { Interrupts Off, IIR Should be 01 }
If $1<>Port[IIR] Then Return_Code(0);

{----- Test Modem Control Register Address. Should be 5-bits Wide -----}
Port[MCR]:=$F5;                         { 8-bit Write                      }
If $15<>Port[MCR] Then Return_Code(0);

{------ Test MCR/MSR Loopback Functions ------}

Port[MCR]:=$10;                         { Set Loop Mode                    }
Port[MSR]:=$0;                          { Clear out Delta Bits             }
If ($F0 and Port[MSR])<>0 Then Return_Code(0); { Check State Bits          }

Port[MCR]:=$1F;                         { Toggle Modem Control Lines       }
If ($F0 and Port[MSR])<>$F0 Then Return_Code(0); { Check State Bits        }

Port[MCR]:=$03;                         { Exit Loop Mode, DTR, RTS Active  }

{---- Port Id Successful at this point. determine port type ----}

Port[SCR]:=$55;                         { Is There a Scratch Register?    }
If $55<>Port[SCR] Then Return_Code(1);  { No SCR, Type = INS8250          }

Port[FCR]:=$CF;                         { Enable FIFO's, If Present       }
If ($C0 and Port[IIR])<>$C0 Then Return_Code(2); { Check FIFO ID bits     }
Port[FCR]:=$0;                          { Turn Off FIFO's                 }

Port[LCR]:=$80;                         { Set DLAB                        }
Port[AFR]:=$07;                         { Write to AFR                    }
If $07<>Port[AFR] Then                  { Read AFR                        }
  Begin
    Port[LCR]:=$0;                      { Reset DLAB                      }
    Return_Code(3);                     { If Not Type=NS16550A            }
  End;

Port[AFR]:=$0;                          { Clear AFR                       }
Port[LCR]:=$0;                          { Reset DLAB                      }
Return_Code(4);

End;


{Macro to disable interrupts.}

Procedure disable_interrupts;
Begin
  Inline($FA);  {CLI}
End;

{Macro to enable interrupts.}

Procedure enable_interrupts;
Begin
  Inline($FB);  {STI}
End;

{Interrupt driver.  The UART is programmed to cause an interrupt whenever
a character has been received or when the UART is ready to transmit another
character.}

procedure com_interrupt_driver; interrupt;
var ch:char;
    dummy,iir:byte;
begin
  if (usefossil) then exit;
    { while bit 0 of the interrupt identification register is 0,
      there is an interrupt to process }
  iir:=port[uart_iir];

  while (not odd(iir)) do begin
    case (iir shr 1) of
        { IIR = 100b: Received data available.  Get the character,
          and if the buffer isn't full, then save it.
          If the buffer is full, then ignore it. }
      2:begin
          ch:=char(port[uart_data]);
          if (rx_chars<=rx_queue_size) then begin
            rx_queue[rx_in]:=ord(ch);
            inc(rx_in);
            if (rx_in>rx_queue_size) then rx_in:=1;
            rx_chars:=succ(rx_chars);
          end;
        end;

        { IIR = 010b: Transmit register empty.  If the transmit buffer
          is empty, then disable the transmitter to prevent any more
          transmit interrupts.  Otherwise, send the character.

          The test of the line-status-register is to see if the transmit
          holding register is truly empty.  Some UARTS seem to cause
          transmit interrupts when the holding register isn't empty,
          causing transmitted characters to be lost. }
      1:if (tx_chars<=0) then
          port[uart_ier]:=port[uart_ier] and not 2
        else
          if (odd(port[uart_lsr] shr 5)) then begin
            port[uart_data]:=tx_queue[tx_out];
            inc(tx_out);
            if (tx_out>tx_queue_size) then tx_out:=1;
            dec(tx_chars);
          end;

        { IIR = 001b: Change in modem status.  We don't expect
          this interrupt, but if one ever occurs we need to read
          the line status to reset it and prevent an endless loop. }
      0:dummy:=Port [uart_msr];

        { IIR = 111b: Change in line status.  We don't expect
          this interrupt, but if one ever occurs we need to read the
          line status to reset it and prevent an endless loop. }
      3:dummy:=port[uart_lsr];

    end;

    iir:=port[uart_iir];
  end;

    { tell the interrupt controller that we're done with this interrupt }
  port[$20]:=$20;
end;

  { flush (empty) the receive buffer. }
procedure com_flush_rx;
var ch:char;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.ah:=$0A;
    intr($14,regs);
    exit;
  end;
  disable_interrupts;
  rx_chars:=0; rx_in:=1; rx_out:=1;
  enable_interrupts;
end;

  { flush (empty) transmit buffer. }
procedure com_flush_tx;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.ah:=$08;
    intr($14,regs);
    exit;
  end;
  disable_interrupts;
  tx_chars:=0; tx_in:=1; tx_out:=1;
  enable_interrupts;
end;

  { purge (empty) transmit buffer. }
procedure com_purge_tx;
begin
  if (not usefossil) then com_flush_tx else begin
    regs.dx:=fosport;
    regs.ah:=$09;
    intr($14,regs);
  end;
end;

  { this function returns TRUE if a carrier is present. }
function com_carrier:boolean;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.ah:=$03;
    intr($14,regs);
    if (regs.ax and $0080) = 0 then
      com_carrier:=FALSE
    else
      com_carrier:=TRUE;
    exit;
  end;
  com_carrier:=((com_installed) and (odd (port[uart_msr] shr 7)));
end;

  { get a character from the receive buffer.
    If the buffer is empty, return NULL (#0). }
function com_rx:char;
begin
  if (usefossil) then begin
    if (com_rx_empty) then com_rx:=#0
    else begin
      regs.dx:=fosport;
      regs.ah:=$02;
      intr($14,regs);
      com_rx:=chr(regs.al);
    end;
    exit;
  end;
  if ((not com_installed) or (rx_chars=0)) then
    com_rx:=#0
  else begin
    disable_interrupts;
    com_rx:=chr(rx_queue[rx_out]);
    inc(rx_out);
    if (rx_out>rx_queue_size) then rx_out:=1;
    dec(rx_chars);
    enable_interrupts;
  end;
end;

  { this function returns TRUE if com_tx can accept a character. }
function com_tx_ready: Boolean;
begin
  if (usefossil) then begin
    com_tx_ready:=TRUE;
    exit;
  end;
  com_tx_ready:=((tx_chars<tx_queue_size) or (not com_installed));
end;

  { this function returns TRUE if the transmit buffer is empty. }
function com_tx_empty:boolean;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.ah:=$03;
    intr($14,regs);
    com_tx_empty:=((regs.ax and $4000) <> 0);
    exit;
  end;
  com_tx_empty:=((tx_chars=0) or (not com_installed));
end;

  { this function returns TRUE if the receive buffer is empty. }
function com_rx_empty:boolean;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.ah:=$0C;
    intr($14,regs);
    com_rx_empty:=(regs.ax = $FFFF);
    exit;
  end;
  com_rx_empty:=((rx_chars=0) or (not com_installed));
end;

  { send a character.  Waits until the transmit buffer isn't full,
    then puts the character into it.  The interrupt driver will
    send the character once the character is at the head of the
    transmit queue and a transmit interrupt occurs. }
procedure com_tx(ch:char);
var result:word;
begin
  if (usefossil) then
  begin
    regs.dx:=fosport;
    regs.al:=ord(ch);
    regs.ah:=$01;
    intr($14,regs);
    exit;
  end;
  if (com_installed) then begin
    repeat until (com_tx_ready);
    disable_interrupts;
    tx_queue[tx_in]:=ord(ch);
    if (tx_in<tx_queue_size) then inc(tx_in) else tx_in:=1;
    inc(tx_chars);
    port[uart_ier]:=port[uart_ier] or 2;
    enable_interrupts;
  end;
end;

  { send a whole string }
procedure com_tx_string(st:string);
var i:byte;
    result:word;
begin
  for i:=1 to length(st) do com_tx(st[i]);
end;

  { lower (deactivate) the DTR line.  Causes most modems to hang up. }
procedure com_lower_dtr;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.al:=$00;
    regs.ah:=$06;
    intr($14,regs);
    exit;
  end;
  if (com_installed) then begin
    disable_interrupts;
    port[uart_mcr]:=port[uart_mcr] and not 1;
    enable_interrupts;
  end;
end;

  { raise (activate) the DTR line. }
procedure com_raise_dtr;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    regs.al:=$01;
    regs.ah:=$06;
    intr($14,regs);
    exit;
  end;
  if (com_installed) then begin
    disable_interrupts;
    port[uart_mcr]:=port[uart_mcr] or 1;
    enable_interrupts;
  end;
end;

  { set the baud rate.  Accepts any speed between 2 and 65535.  However,
    I am not sure that extremely high speeds (those above 19200) will
    always work, since the baud rate divisor will be six or less, where a
    difference of one can represent a difference in baud rate of
    3840 bits per second or more. }
procedure com_set_speed (speed: Word);
var divisor:word;
    baudhigh:boolean;
begin
  if (usefossil) then begin
    regs.dx:=fosport;
    case speed of
    300:regs.al:=(2 shl 5)+3;
    600:regs.al:=(3 shl 5)+3;
    1200:regs.al:=$83;
    2400:regs.al:=$A3;
    4800:regs.al:=$C3;
    9600:regs.al:=$E3;
    12000:regs.al:=$03;
    14400:regs.al:=$03;
    16800:regs.al:=$23;
    19200:regs.al:=$03;
    21600:regs.al:=$23;
    24000:regs.al:=$23;
    26400:regs.al:=$23;
    28800:regs.al:=$23;
    38400:regs.al:=$23;
    57600:begin
           baudhigh:=speed=57600;
           asm
           MOV AH, 1Bh
           INT 14h
           CMP CX, '0X'
           JNE @EXIT
           MOV AH, 1Eh
           MOV BX, 0000h
           MOV CH, 03h
           MOV DX, FOSPORT
           CMP BaudHigh, True
           JE @1
           MOV CL, 84h
           JMP @INT
          @1:
           MOV CL, 82h
          @INT:
           INT 14h
          @EXIT:
          end;
         exit;
         end;
   11520:begin
          baudhigh:=speed=11520;
           asm
           MOV AH, 1Bh
           INT 14h
           CMP CX, '0X'
           JNE @EXIT
           MOV AH, 1Eh
           MOV BX, 0000h
           MOV CH, 03h
           MOV DX, FOSPORT
           CMP BaudHigh, True
           JE @1
           MOV CL, 84h
           JMP @INT
          @1:
           MOV CL, 82h
          @INT:
           INT 14h
          @EXIT:
          end;
         exit;
        end;
    end;
    regs.ah:=$00;
    intr($14,regs);
    exit;
  end;
  if (com_installed) then begin
    if (speed<2) then speed:=2;
    divisor:=115200 div speed;
    disable_interrupts;
    port[uart_lcr]:=port[uart_lcr] or $80;
    portw[uart_data]:=divisor;
    port[uart_lcr]:=port[uart_lcr] and not $80;
    enable_interrupts;
  end;
end;

  { Set the parity and stop bits as follows:

     com_none    8 data bits, no parity
     com_even    7 data bits, even parity
     com_odd     7 data bits, odd parity
     com_zero    7 data bits, parity always zero
     com_one     7 data bits, parity always one }
procedure com_set_parity(parity:com_parity; stop_bits:byte);
var lcr:byte;
begin
  if (usefossil) then exit;
  case parity of
    com_none:lcr:=$00 or $03;
    com_even:lcr:=$18 or $02;
    com_odd:lcr:=$08 or $02;
    com_zero:lcr:=$38 or $02;
    com_one:lcr:=$28 or $02;
  end;
  if (stop_bits=2) then lcr:=lcr or $04;
  disable_interrupts;
  port[uart_lcr]:=port[uart_lcr] and $40 or lcr;
  enable_interrupts;
end;

  { Install the communications driver.  Portnum should be 1..max_port.
    Error codes returned are:

      0 - No error
      1 - Invalid port number
      2 - UART for that port is not present
      3 - Already installed, new installation ignored }

procedure com_install(portnum:word; var error:word; dofossil:boolean);
var ier:byte;
begin
  usefossil:=false;
  fosport:=portnum-1;
  if ((dofossil) and (not usefossil)) then begin
    usefossil:=FALSE;
    fosport:=portnum-1;
    regs.dx:=fosport;
    regs.ah:=$04;
    intr($14,regs);
    if (regs.ax = $1954) then begin
      usefossil:=TRUE;
      regs.dx:=fosport;
      regs.al:=$F0;
      regs.ah:=$0F;
      intr($14,regs);
    end;
  end;
  if (usefossil) then exit;
  if (com_installed) then error:=3
  else
    if ((portnum<1) or (portnum>max_port)) then error:=1
    else begin
        { set i/o addresses and other hardware specifics for selected port}
      uart_data:=uart_base [portnum];
      uart_ier:=uart_data+1;
      uart_iir:=uart_data+2;
      uart_lcr:=uart_data+3;
      uart_mcr:=uart_data+4;
      uart_lsr:=uart_data+5;
      uart_msr:=uart_data+6;
      uart_spr:=uart_data+7;
      intnum:=intnums[portnum];
      if modemr.irqnum=0 then
        i8259bit:=1 shl i8259levels[portnum] else
         i8259bit:=1 shl modemr.irqnum;
        { return error if hardware not installed }

      old_ier:=port[uart_ier];
      port[uart_ier]:=0;
      if (port[uart_ier]<>0) then error:=2
      else begin
        error:=0;

          { save original interrupt controller mask, then disable the
            interrupt controller for this interrupt. }
        disable_interrupts;
        old_i8259_mask:=port[$21];
        port[$21]:=old_i8259_mask or i8259bit;
        enable_interrupts;

          { clear the transmit and receive queues}
        com_flush_tx;
        com_flush_rx;

          { Save current interrupt vector, then set the interrupt
            vector to the address of our interrupt driver. }

        getintvec(intnum,old_vector);
        setintvec(intnum,@com_interrupt_driver);
        com_installed:=TRUE;

          { set parity to none, turn off BREAK signal, and make sure
            we're not addressing the baud rate registers. }
        port[uart_lcr]:=3;

          { save original contents of modem control register, then enable
            interrupts to system bus and activate RTS.  Leave DTR the way
            it was. }
        disable_interrupts;
        old_mcr:=port[uart_mcr];
        port[uart_mcr]:=$A or (old_mcr and 1);
        enable_interrupts;

          { enable interrupt on data-available.  The interrupt for
            transmit-ready is enabled when a character is put into the
            transmit queue, and disabled when the transmit queue is empty. }
        port[uart_ier]:=1;

          { enable the interrupt controller for this interrupt. }
        disable_interrupts;
        port[$21]:=port[$21] and not i8259bit;
        enable_interrupts;
      end;
    end;
end;


  { Deinstall the interrupt driver completely.  It doesn't change
    the baud rate or mess with DTR; it tries to leave the interrupt
    vectors and enables and everything else as it was when the driver
    was installed.

    This procedure MUST be called by the exit procedure of this
    module before the program exits to DOS, or the interrupt driver
    will still be attached to its vector -- the next communications
    interrupt that came along would jump to the interrupt driver which
    is no longer protected and may have been written over. }
procedure com_deinstall;
begin
  if (usefossil) then begin
    usefossil:=FALSE;
    regs.dx:=fosport;
    regs.ah:=$05;
    intr($14,regs);
    exit;
  end;
  if (com_installed) then begin
    com_installed:=FALSE;

      { restore Modem-Control-Register and Interrupt-Enable-Register. }
    port[uart_mcr]:=old_mcr;
    port[uart_ier]:=old_ier;

      { restore appropriate bit of interrupt controller's mask }
    disable_interrupts;
    port[$21]:=port[$21] and not i8259bit or old_i8259_mask and i8259bit;
    enable_interrupts;

      { reset the interrupt vector }
    setintvec(intnum,old_vector);
  end;
end;

  { This procedure is called when the program exits for any reason.  It
    deinstalls the interrupt driver.}
{$F+} procedure exit_procedure; {$F-}
begin
  com_deinstall;
  exitproc:=exit_save;
end;

  { This installs the exit procedure. }
begin
  exit_save:=exitproc;
  exitproc:=@exit_procedure;
end.

