/*
 * traceroute host  - trace the route ip packets follow going to "host".
 *
 * Attempt to trace the route an ip packet would follow to some
 * internet host.  We find out intermediate hops by launching probe
 * packets with a small ttl (time to live) then listening for an
 * icmp "time exceeded" reply from a gateway.  We start our probes
 * with a ttl of one and increase by one until we get an icmp "port
 * unreachable" (which means we got to "host") or hit a max (which
 * defaults to 30 hops & can be changed with the -m flag).  Three
 * probes (change with -q flag) are sent at each ttl setting and a
 * line is printed showing the ttl, address of the gateway and
 * round trip time of each probe.  If the probe answers come from
 * different gateways, the address of each responding system will
 * be printed.  If there is no response within a 5 sec. timeout
 * interval (changed with the -w flag), a "*" is printed for that
 * probe.
 *
 * Probe packets are UDP format.  We don't want the destination
 * host to process them so the destination port is set to an
 * unlikely value (if some clod on the destination is using that
 * value, it can be changed with the -p flag).
 *
 * A sample use might be:
 *
 *     [yak 71]% traceroute nis.nsf.net.
 *     traceroute to nis.nsf.net (35.1.1.48), 30 hops max, 56 byte packet
 *      1  helios.ee.lbl.gov (128.3.112.1)  19 ms  19 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  39 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  40 ms  59 ms  59 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  59 ms
 *      8  129.140.70.13 (129.140.70.13)  99 ms  99 ms  80 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  239 ms  319 ms
 *     10  129.140.81.7 (129.140.81.7)  220 ms  199 ms  199 ms
 *     11  nic.merit.edu (35.1.1.48)  239 ms  239 ms  239 ms
 *
 * Note that lines 2 & 3 are the same.  This is due to a buggy
 * kernel on the 2nd hop system -- lbl-csam.arpa -- that forwards
 * packets with a zero ttl.
 *
 * A more interesting example is:
 *
 *     [yak 72]% traceroute allspice.lcs.mit.edu.
 *     traceroute to allspice.lcs.mit.edu (18.26.0.115), 30 hops max
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  19 ms  19 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  19 ms  39 ms  39 ms
 *      5  ccn-nerif22.Berkeley.EDU (128.32.168.22)  20 ms  39 ms  39 ms
 *      6  128.32.197.4 (128.32.197.4)  59 ms  119 ms  39 ms
 *      7  131.119.2.5 (131.119.2.5)  59 ms  59 ms  39 ms
 *      8  129.140.70.13 (129.140.70.13)  80 ms  79 ms  99 ms
 *      9  129.140.71.6 (129.140.71.6)  139 ms  139 ms  159 ms
 *     10  129.140.81.7 (129.140.81.7)  199 ms  180 ms  300 ms
 *     11  129.140.72.17 (129.140.72.17)  300 ms  239 ms  239 ms
 *     12  * * *
 *     13  128.121.54.72 (128.121.54.72)  259 ms  499 ms  279 ms
 *     14  * * *
 *     15  * * *
 *     16  * * *
 *     17  * * *
 *     18  ALLSPICE.LCS.MIT.EDU (18.26.0.115)  339 ms  279 ms  279 ms
 *
 * (I start to see why I'm having so much trouble with mail to
 * MIT.)  Note that the gateways 12, 14, 15, 16 & 17 hops away
 * either don't send ICMP "time exceeded" messages or send them
 * with a ttl too small to reach us.  14 - 17 are running the
 * MIT C Gateway code that doesn't send "time exceeded"s.  God
 * only knows what's going on with 12.
 *
 * The silent gateway 12 in the above may be the result of a bug in
 * the 4.[23]BSD network code (and its derivatives):  4.x (x <= 3)
 * sends an unreachable message using whatever ttl remains in the
 * original datagram.  Since, for gateways, the remaining ttl is
 * zero, the icmp "time exceeded" is guaranteed to not make it back
 * to us.  The behavior of this bug is slightly more interesting
 * when it appears on the destination system:
 *
 *      1  helios.ee.lbl.gov (128.3.112.1)  0 ms  0 ms  0 ms
 *      2  lilac-dmc.Berkeley.EDU (128.32.216.1)  39 ms  19 ms  39 ms
 *      3  lilac-dmc.Berkeley.EDU (128.32.216.1)  19 ms  39 ms  19 ms
 *      4  ccngw-ner-cc.Berkeley.EDU (128.32.136.23)  39 ms  40 ms  19 ms
 *      5  ccn-nerif35.Berkeley.EDU (128.32.168.35)  39 ms  39 ms  39 ms
 *      6  csgw.Berkeley.EDU (128.32.133.254)  39 ms  59 ms  39 ms
 *      7  * * *
 *      8  * * *
 *      9  * * *
 *     10  * * *
 *     11  * * *
 *     12  * * *
 *     13  rip.Berkeley.EDU (128.32.131.22)  59 ms !  39 ms !  39 ms !
 *
 * Notice that there are 12 "gateways" (13 is the final
 * destination) and exactly the last half of them are "missing".
 * What's really happening is that rip (a Sun-3 running Sun OS3.5)
 * is using the ttl from our arriving datagram as the ttl in its
 * icmp reply.  So, the reply will time out on the return path
 * (with no notice sent to anyone since icmp's aren't sent for
 * icmp's) until we probe with a ttl that's at least twice the path
 * length.  I.e., rip is really only 7 hops away.  A reply that
 * returns with a ttl of 1 is a clue this problem exists.
 * Traceroute prints a "!" after the time if the ttl is <= 1.
 * Since vendors ship a lot of obsolete (DEC's Ultrix, Sun 3.x) or
 * non-standard (HPUX) software, expect to see this problem
 * frequently and/or take care picking the target host of your
 * probes.
 *
 * Other possible annotations after the time are !H, !N, !P (got a host,
 * network or protocol unreachable, respectively), !S or !F (source
 * route failed or fragmentation needed -- neither of these should
 * ever occur and the associated gateway is busted if you see one).  If
 * almost all the probes result in some kind of unreachable, traceroute
 * will give up and exit.
 *
 * Notes
 * -----
 * The udp port usage may appear bizarre (well, ok, it is bizarre).
 * The problem is that an icmp message only contains 8 bytes of
 * data from the original datagram.  8 bytes is the size of a udp
 * header so, if we want to associate replies with the original
 * datagram, the necessary information must be encoded into the
 * udp header (the ip id could be used but there's no way to
 * interlock with the kernel's assignment of ip id's and, anyway,
 * it would have taken a lot more kernel hacking to allow this
 * code to set the ip id).  So, to allow two or more users to
 * use traceroute simultaneously, we use this task's pid as the
 * source port (the high bit is set to move the port number out
 * of the "likely" range).  To keep track of which probe is being
 * replied to (so times and/or hop counts don't get confused by a
 * reply that was delayed in transit), we increment the destination
 * port number before each probe.
 *
 * Don't use this as a coding example.  I was trying to find a
 * routing problem and this code sort-of popped out after 48 hours
 * without sleep.  I was amazed it ever compiled, much less ran.
 *
 * I stole the idea for this program from Steve Deering.  Since
 * the first release, I've learned that had I attended the right
 * IETF working group meetings, I also could have stolen it from Guy
 * Almes or Matt Mathis.  I don't know (or care) who came up with
 * the idea first.  I envy the originators' perspicacity and I'm
 * glad they didn't keep the idea a secret.
 *
 * Tim Seaver, Ken Adelman and C. Philip Wood provided bug fixes and/or
 * enhancements to the original distribution.
 *
 * I've hacked up a round-trip-route version of this that works by
 * sending a loose-source-routed udp datagram through the destination
 * back to yourself.  Unfortunately, SO many gateways botch source
 * routing, the thing is almost worthless.  Maybe one day...
 *
 *  -- Van Jacobson (van@helios.ee.lbl.gov)
 *     Tue Dec 20 03:50:13 PST 1988
 *
 *
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


/*
 * PSC Changes Copyright (c) 1992 Pittsburgh Supercomputing Center.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the modifications to this
 * software were developed by the Pittsburgh Supercomputing Center.
 * The name of the Center may not be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE
 * IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. 
 *
 */

/* DNS lookup portions of this code also subject to the following:  */
/*
 * Use the domain system to resolve a name.
 *
 * Copyright (C) 1988,1990,1992 Dan Nydick, Carnegie-Mellon University
 * Anyone may use this code for non-commercial purposes as long
 * as my name and copyright remain attached.
 */

/*
 *  1/7/92:  Modified to include a new option for MTU discovery.
 *           This option will send out packets with the don't
 *           fragment bit set in order to determine the MTU
 *           of the path being traceroute'd.  Only decreases
 *           in MTU will be detected, and the MTU will initially
 *           be set to the interface MTU which is used for 
 *           routing.  In the event of an MTU decrease in the 
 *           path, the output will include a message of the 
 *           form MTU=#### with the new MTU for the latest 
 *           hop.  This option is invoked with "-M".
 *                            Jamshid Mahdavi,   PSC.
 */

/*
 * 4/12/93:  Modified to include new option (-Q) that will report on
 *           percentage packet loss* in addition to the usual
 *           min/avg/max delay.  When -Q is invoked, delay per packet
 *           reporting is turned off.  Also ^C aborts further packets
 *           sent on that ttl and goes on to the next -- two
 *           consecutive ^Cs will terminate the traceroute entirely.
 *           [modified code rom Matt Mathis' uping.c code] 
 *                          Jon Boone,  PSC.
 */

/*
 * 4/23/93:  Added support for a "-a" switch which will support
 *           automatic cutoff after a certain number of dropped
 *           packets in a row
 *                          Jon Boone, PSC.
 */

/*
 * 10/21/93: (JM) Fixed SGI version, changed the packet sizing scheme
 *           to a saner system based on total packet size (which
 *           also fixed several bugs in the old code w.r.t. 
 *           size of packets).  Added fast timers.  Added network 
 *           owner lookup.  Plan to add AS path lookup eventually...
 */

/*
 * 09/19/96: Rewritten to work with Waterloo TCP/IP
 *           Gisle Vanem <giva@bgnett.no>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <conio.h>
#include <string.h>
#include <signal.h>
#include <math.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <tcp.h>

#define VERSION     "Waterloo TCP/IP traceroute 1.0"
#define PING_TRACE  1

#if PING_TRACE
#define PROBE_PROTOCOL  IPPROTO_ICMP
#else
#define PROBE_PROTOCOL  IPPROTO_UDP
#endif

#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK  (u_long)0x7F000001
#endif

#ifdef __BORLANDC__
#pragma warn -stu-
#endif

#define MAXPACKET       1500    /* max ip packet size */
#define MINPACKET       sizeof(struct opacket) - sizeof(struct ether_header)
#define MY_PORT         1030
#define MAX_IPOPTLEN    40      /* 15*4 - 20 */
#define MAX_GW         ((MAX_IPOPTLEN-IPOPT_MINOFF)/sizeof(DWORD)-1)
#define MAX_SKIP        255

#define ETHPROTO_IP     8
#define IPPROTO_ICMP    1
#define IPPROTO_UDP     17
#define IP_VERSION      4


#include <sys/packon.h>

struct data {
       BYTE    seq;             /* sequence number of this packet */
       BYTE    ttl;             /* ttl packet left with */
       time_t  tv;              /* time packet left     */
     };

struct ping {
       BYTE   type;
       BYTE   code;
       WORD   chksum;
       WORD   ident;
       WORD   seq;
       DWORD  index;
     };

struct opacket {              /* format of a probe packet */
       struct  ether_header  eth;
       struct  ip            ip;
#if PING_TRACE
       struct  ping          echo;
#else
       struct  udphdr        udp;
#endif
       struct  data          data;
     };

struct pseudoHeader {
       DWORD  src;
       DWORD  dst;
       BYTE   mbz;
       BYTE   protocol;
       WORD   length;
       WORD   checksum;
    };

#include <sys/packoff.h>

#ifndef DJGPP
#define UCLOCKS_PER_SEC 1193180
typedef unsigned long uclock_t;
#endif

#ifdef __HIGHC__
#pragma Offwarn (572)
#endif

#define NET_ADDR(ip)  inet_ntoa (*(struct in_addr*)&(ip))

/*
 *  Prototypes
 */
typedef  int (*DebugProc)(void *sock, struct ip *ip, char *fname, unsigned line);

extern   DebugProc   _dbugrecv;

void     SendProbe    (int ttl, int tos);
void     CheckReply   (int seq, int ttl, int probe, uclock_t start, DWORD timeout);
double   DeltaT_ms    (uclock_t dt);
size_t   Reduce_mtu   (size_t value);
void     SigIntHandler(int sig);
int      Check_ICMP   (struct ip *ip, int seq, int *code);
char    *DoResolve    (char *query);
char    *LookupOwner  (struct in_addr adr);
char    *NetName      (struct in_addr adr);
uclock_t uclock       (void);

/*
 *  Local data
 */

char   outbuf [sizeof(struct opacket) + MAX_IPOPTLEN];
struct opacket  *outpacket;      /* last output (udp) packet  */
DWORD  whereto;                  /* Who to reach (net-order)  */
DWORD  from_ip;                  /* Our ip address            */
BYTE   optlist [MAX_IPOPTLEN];   /* IP options list           */
int    optlen      = 0;          /* length of IP options      */
size_t data_len    = 0;          /* packetlength - eth header */

char  *source      = NULL;
int    nprobes     = 3;
int    min_ttl     = 1;
int    max_ttl     = 30;         /* default max hop count */
WORD   port        = 32768+666;  /* start udp dest port # for probe packets */
double waittime    = 3.0;        /* time to wait for response (in seconds) */
int    mtu_disc    = 0;          /* do MTU discovery in path */
int    nflag       = 0;          /* print addresses numerically */
int    Verbose     = 0;          /* print more details */

int    halt_flag   = 0;          /* signal happened */
int    ppdelay     = 1;          /* we normally want per-packet delay */
int    pploss      = 0;          /* we normally don't want packet loss */
int    lost        = 0;          /* how many packets did we not get back */
double throughput  = 0;          /* percentage packets not lost */
int    consecutive = 0;          /* the number of consecutive lost packets */
int    automagic   = 0;          /* automatically quit after 10 lost packets? */

int    usec_timers = 0;          /* Print timings in microseconds */
int    dns_lookup  = 0;          /* Look up owner email in DNS */
int    as_lookup   = 0;          /* Look up AS path in routing registries */
size_t new_mtu     = 0;                   /* reported mtu changed */
size_t min_mtu     = sizeof(struct ping); /* smallest mtu we will handle */

char  *gw_name   [MAX_GW+1];
DWORD  gw_list   [MAX_GW+1];
int    skip_list [MAX_SKIP+1];
struct in_addr lastaddr;
int    got_there;
int    unreachable;
int    indent;
float  min, max, sum = 0, sumsq = 0;


static void Exit (const char *fmt, ...)
{
  va_list args;
  va_start (args, fmt);
  vfprintf (stdout, fmt, args);
  va_end (args);
  exit (1);
}

void usage (int details)
{
  if (details)
     printf ("%s\nWattcp kernel: %s\nCapabilities: %s\n",
             VERSION, wattcpVersion(), wattcpCapabilities());

  printf ("Usage: tracert [-?adDnVvQuOMA] [-f min] [-g lsrr] [-h#] [-m max] [-p port]\n"
          "               [-q num] [-s addr] [-w wait] [-t tos] host [datasize]\n\n");
  if (details)
     printf("Options: -a  Loss Detection\n"
            "         -d  Debug level 1\n"
            "         -D  Debug level 2\n"
            "         -n  Print IP-addresses numerically\n"
            "         -V  Verbose printout mode\n"
            "         -v  Print version information\n"
            "         -Q  Statistics Collection\n"
            "         -f  Min # of hops/TTL    (default %u)\n"
            "         -m  Max # of hops/TTL    (default %u)\n"
            "         -p  Port number to use   (default %u)\n"
            "         -q  # of queries per hop (default %u)\n"
            "         -s  Source IP to use     (default my-IP)\n"
            "         -w  Waittime per hop [s] (default %.1f)\n\n"
            "Advanced options:\n"
            "         -u  Microsecond resolution timing\n"
            "         -O  Report email address for every hop\n"
            "         -M  MTU Discovery as per RFC-1191\n"
            "         -A  AS Path Lookup\n"
            "         -g  Loose Source Route\n"
            "         -h  Skip hop number #\n"
            "         -t  Type-Of-Service: 16 - Lowdelay\n"
            "                               8 - Throughput\n"
            "                               4 - Reliability\n",
            min_ttl, max_ttl, port, nprobes, waittime);
  exit(1);
}

int main (int argc, char **argv)
{
  char *hostname;
  int   ch, probe, ttl, arg;
  int   seq  = 0;
  int   lsrr = 0;
  int   skip = 0;
  int   tos  = 0;

  while ((ch = getopt(argc,argv,"?aAdDnVvQuOMf:g:m:p:q:s:w:t:h:")) != EOF)
      switch (ch)
      {
        case '?': usage (1);
                  break;

        case 'a': automagic = 1;
                  break;

        case 'A': as_lookup = 1;
                  break;

        case 'd': dbug_init();
                  tcp_set_debug_state (1);
                  break;

        case 'D': dbug_init();
                  tcp_set_debug_state (2);
                  break;

        case 'n': nflag = 1;
                  break;

        case 'V': Verbose = 1;
                  break;

        case 'v': puts (VERSION);
                  puts (wattcpVersion());
                  break;

        case 'Q': pploss  = 1;
                  ppdelay = 0;
                  break;

        case 'u': usec_timers = 1;
                  break;

        case 'O': dns_lookup = 1;
                  break;

        case 'M': mtu_disc = 1;
                  break;

        case 'f': if ((min_ttl = atoi(optarg)) <= 1)
                     Exit ("min # of hops/ttl must be >1.");
                  break;

        case 'g': if (lsrr >= (int)MAX_GW)
                     Exit ("No more than %d gateways\n",MAX_GW);
                  gw_name [lsrr++] = optarg;
                  break;

        case 'm': if ((max_ttl = atoi(optarg)) <= 1)
                     Exit ("max # of hops/ttl must be >1.");
                  break;

        case 'p': if ((port = atoi(optarg)) < 1)
                     Exit ("port must be >0.");
                  break;

        case 'q': if ((nprobes = atoi(optarg)) < 1)
                     Exit ("nprobes must be >0.");
                  break;

        case 's': source = optarg;
                  if (inet_addr(source) == INADDR_NONE)
                     Exit ("Illegal source address %s", source);
                  break;

        case 'w': if ((waittime = atof(optarg)) <= 0.055)
                     Exit ("wait must be >0.055 sec.");
                  break;

        case 't': if ((tos = atoi(optarg)) < 0 || tos > 255)
                     Exit ("TOS must be 0 - 255.");
                  break;

        case 'h': arg = atoi (optarg);
                  if (skip >= MAX_SKIP)
                     Exit ("No more than %d hop skips\n",MAX_SKIP);

                  if (arg >= max_ttl || arg <= min_ttl)
                     Exit ("Cannot skip that hop/ttl");
                  skip_list [arg] = 1;
                  skip++;
                  break;

        default : usage (0);
      }

  argc -= optind;
  argv += optind;

  if (argc < 1)
     usage (0);

  sock_init();
  block_icmp = 1;  /* make Watt-32's icmp_handler() drop handling ICMP */

  hostname = *argv++;

  if (*argv)
  {
    data_len = atoi (*argv);
    if (data_len < MINPACKET || data_len > MAXPACKET)
       Exit ("packet size range is %d - %d.", MINPACKET, MAXPACKET);
  }
  else
  {
    if (mtu_disc)
         data_len = MAXPACKET;
    else data_len = MINPACKET;
  }

  whereto = htonl (lookup_host(hostname,NULL));
  if (!whereto)
      Exit (dom_strerror(dom_errno));

  outpacket = (struct opacket*) &outbuf;

  if (!_arp_resolve(ntohl(whereto),&outpacket->eth.ether_dhost,0))
     Exit ("No route to host");

  if (source)
       from_ip = inet_addr (source);
  else from_ip = htonl (my_ip_addr);

 /*
  * ^C puts you to the next hop. Twice will exit.
  */
  signal (SIGINT, SigIntHandler);

  if (lsrr > 0)
  {
    int i;
    for (i = 0; i < lsrr; i++)
    {
      gw_list[i] = htonl (lookup_host(gw_name[i],NULL));
      if (!gw_list[i])
         Exit (dom_strerror(dom_errno));
    }

    optlen = lsrr * sizeof(gw_list[0]) + 4;

    gw_list[lsrr] = whereto;             /* final hop */
    whereto       = gw_list[0];
    optlist [0]   = IPOPT_NOP;           /* force 4 byte alignment */
    optlist [1]   = IPOPT_LSRR;          /* loose source route option */
    optlist [2]   = optlen - 1;          /* LSRR option-list length */
    optlist [3]   = IPOPT_MINOFF;        /* Pointer to LSRR addresses */
    memcpy (&optlist[4], &gw_list[1], optlen-4);
  }

  if (data_len < MINPACKET + optlen)
  {
    /* The chosen size is too small to fit everything...make it bigger: */
    data_len = MINPACKET + optlen;
  }

  if (Verbose && inet_addr(hostname) != INADDR_NONE)
  {
    struct hostent *hp;
    char   name [MAXHOSTNAMELEN];

    printf ("Reverse lookup (%s)..",hostname);
    fflush (stdout);
    hp = gethostbyaddr ((char*)&whereto,sizeof(whereto),AF_INET);
    if (!hp)
       printf ("<unknown>\n");
    else
    {
      hostname = strcpy (name,hp->h_name);
      printf ("`%s'\n", hostname);
    }
  }

  /*
   *  Finished initialising stuff. Enter main traceroute loop
   */

  printf ("\rtraceroute to %s (%s)", hostname, NET_ADDR(whereto));

  if (source)
     printf (" from %s", source);
  printf (", %d hops max, %u byte packets\n", max_ttl, (unsigned)data_len);

  uclock();

  outpacket->ip.ip_src.s_addr = from_ip;
  outpacket->ip.ip_dst.s_addr = whereto;
                             
  for (ttl = min_ttl; ttl <= max_ttl; ++ttl)
  {
    got_there = unreachable = 0;
    lastaddr.s_addr = 0L;
    lost = indent = consecutive = 0;
    throughput = 0.0;

    printf ("%2d ", ttl);

    if (skip_list[ttl])
    {
      puts (" Skipping this hop");
      continue;
    }

    if (new_mtu)
    {
      printf ("MTU=%u ", (unsigned)new_mtu);
      new_mtu = 0;
    }
    fflush (stdout);

    for (probe = 0; probe < nprobes; ++probe)
    {
      DWORD timeout;

      SendProbe (ttl, tos);
      timeout = set_timeout ((DWORD)(1000.0*waittime));
      CheckReply (++seq, ttl, probe, uclock(), timeout);

      /* Reset the ^C action from exit to skip TTL  */

      if (halt_flag == 0 && lost == 1)
         signal (SIGINT, SigIntHandler);

      /* We've missed at least one packet, so let's check for
       * the signal to go to the next ttl
       */
      if (halt_flag > 0)
      {
        halt_flag = 0;
        consecutive = 0;
      }

      if (automagic && consecutive > 9)
         break;
    }

    if (pploss && lost < probe)
    {
      throughput = 100.0 - lost * 100.0 / probe;
      printf ("  (%1.1f ms/%1.1f ms(+-%1.1f ms)/%1.1f ms)",
               min, sum / (probe - lost),
               (float)sqrt((double)sumsq)/(probe-lost), max);
      printf (" %d/%d (%#3.2f%%)", probe - lost, probe, throughput);
    }

    puts ("");
    fflush (stdout);
    if (got_there || unreachable > nprobes)
       return (1);
  
    if (new_mtu)
    {
      ttl--;               /* Redo the same TTL */
      data_len = new_mtu;  /* Set the new data length */
    }
  }
  return (0);
}

/*---------------------------------------------------------------------*/

void SendProbe (int ttl, int tos)
{
  struct ip   *ip    = &outpacket->ip;
  struct data *data  = &outpacket->data;
  static WORD seqnum = 0x10;
  size_t len         = data_len - sizeof(*ip);
  size_t hlen        = sizeof (*ip) + optlen;
  BYTE  *buf         = _eth_formatpacket (&outpacket->eth.ether_dhost,
                                          ETHPROTO_IP);
  if (optlen)
  {
    memcpy (ip+1, &optlist, optlen);
    data += optlen;
  }

  ip->ip_off = mtu_disc ? IP_DF : 0; /* don't fragment bit */
  ip->ip_hl  = hlen >> 2;            /* head + options length */
  ip->ip_len = htons (data_len);
  ip->ip_tos = tos;
  ip->ip_v   = IP_VERSION;
  ip->ip_ttl = ttl;
  ip->ip_id  = htons (seqnum);
  ip->ip_p   = PROBE_PROTOCOL;
  ip->ip_sum = 0;
  ip->ip_sum = ~inchksum (ip,hlen);

  data->seq = seqnum;
  data->ttl = ttl;
  data->tv  = time (NULL);

#if PING_TRACE
  {
    struct ping *echo = (struct ping*) ((BYTE*)&outpacket->echo + optlen);
    echo->type    = ICMP_ECHO;
    echo->code    = 0;
    echo->index   = seqnum;
    echo->seq     = seqnum;
    echo->ident   = (WORD) set_timeout (0);
    echo->chksum  = 0;
    echo->chksum  = ~inchksum (echo,len);
  }
#else
  {
    struct udphdr *udp = (struct udphdr *) ((BYTE*)&outpacket->udp + optlen);
    struct pseudoHeader ph;

    udp->uh_sport = htons (MY_PORT);
    udp->uh_dport = htons (port+seqnum);
    udp->uh_ulen  = htons (len);
    udp->uh_sum   = 0;

    ph.src      = ip->ip_src.s_addr;
    ph.dst      = ip->ip_dst.s_addr;
    ph.mbz      = 0;
    ph.protocol = PROBE_PROTOCOL;
    ph.length   = udp->uh_ulen;
    ph.checksum = inchksum (udp, len);
    udp->uh_sum = ~inchksum (&ph, sizeof(ph));
  }
#endif

  memcpy ((void*)buf, ip, data_len);
  _eth_send (data_len + sizeof(struct ether_header), NULL, __FILE__, __LINE__);
  seqnum += 5;
}

/*---------------------------------------------------------------------*/

void CheckReply (int seq, int ttl, int probe, uclock_t start, DWORD timeout)
{
  while (1)
  {
    int        icmp_code, bcast;
    WORD       rc, packettype;
    BYTE      *packet = _eth_arrived (&packettype, &bcast);
    BYTE       buf [MAXPACKET];
    struct ip *ip = (struct ip*)buf;

    watt_kbhit();

    if (!packet)
    {
      if (chk_timeout(timeout))
      {
        indent += printf (" *");
        fflush (stdout);
        lost++;
        consecutive++;
        if (ttl == 1 && probe == nprobes-1)
           Exit ("\n No reply on 1st hop. Giving up");
        return;
      }
      continue;
    }
    memcpy (buf, packet, sizeof(buf));
    _eth_free (packet, packettype);

    if (_dbugrecv)
      (*_dbugrecv) (NULL, ip, __FILE__, __LINE__);

    if (bcast ||                              /* link-layer broadcast */
        packettype        != ETHPROTO_IP  ||  /* not IP-packet */
        ip->ip_p          != IPPROTO_ICMP ||  /* not ICMP-packet */
        ip->ip_dst.s_addr != from_ip)         /* not from our peer */
      continue;

    rc = Check_ICMP (ip, seq, &icmp_code);
    if (rc != 0)
    {
      uclock_t       dt  = uclock() - start;
      struct in_addr src = ip->ip_src;

      if (src.s_addr == whereto)
         got_there = 1;

      if (sum == 0)
      {
        sum = min = max = dt;
        sumsq = dt*dt;
      }
      else
      {
        if (dt < min) min = dt;
        if (dt > max) max = dt;
        sum += dt;
        sumsq += dt*dt;
      }

      if (src.s_addr != lastaddr.s_addr)
      {
        char buf[256];

        if (nflag)
           printf (" %-*s",15-indent,inet_ntoa(src));
        else
        {
          sprintf (buf," %-*s (%s)", 30-indent,NetName(src),inet_ntoa(src));
          printf ("%-50s",buf);
        }
        if (dns_lookup)
           printf (" [%s]",LookupOwner(src));

        lastaddr = src;
      }

      if (ppdelay)
      {
        if (usec_timers)
             printf (" %3.3f ms", DeltaT_ms(dt));
        else printf (" %5u ms", (WORD)DeltaT_ms(dt));
        fflush (stdout);
      }

      switch (rc - 1)
      {
        case ICMP_UNREACH_PORT:
             if (ip->ip_ttl <= 1)
                printf (" !");
             ++got_there;
             break;
        case ICMP_UNREACH_NET:
             ++unreachable;
             printf (" !N");
             break;
        case ICMP_UNREACH_HOST:
             ++unreachable;
             printf (" !H");
             break;
        case ICMP_UNREACH_PROTOCOL:
             ++got_there;
             printf (" !P");
             break;
        case ICMP_UNREACH_NEEDFRAG:
             if (mtu_disc)
             {
               int    hlen       = ip->ip_hl << 2;
               struct icmp *icmp = (struct icmp *)((BYTE*)ip + hlen);
               size_t mtu        = ntohs (icmp->icmp_nextmtu);
             
               if (mtu >= data_len) printf (" !M>"); /* serious bug somewhere */
               if (mtu <= min_mtu)  printf (" !M<");
               else if (!mtu)       printf (" !M"),  new_mtu = Reduce_mtu (data_len);
               else                 printf (" !M=%u", (unsigned)(new_mtu = mtu));
             }
             else
             {
               ++unreachable;
               printf (" !F");
             }
             break;
        case ICMP_UNREACH_SRCFAIL:
             ++unreachable;
             printf (" !S");
             break;
        case ICMP_UNREACH_FILTER_PROHIB:
             ++unreachable;
             printf (" !X");
             break;
      }
      break;
    }
  }
  fflush (stdout);
}

/*---------------------------------------------------------------------*/

double DeltaT_ms (uclock_t dt)
{
  return (1000.0 * ((double)dt / (double) UCLOCKS_PER_SEC));
}

/*---------------------------------------------------------------------*/

size_t Reduce_mtu (size_t value)
{
  /* The following heuristic taken from RFC1191
   */
  static int mtu[] = { 1492, 1006, 508,
                        296,   68, -1
                     };
  int i = 0;

  while (value <= (size_t)mtu[i])
        i++;
  if (mtu[i] <= 0)
     Exit (" Invalid MTU !!\n");
  return (size_t)mtu[i];
}

/*---------------------------------------------------------------------*/

char * ICMP_type (BYTE type)
{
  static char buf[30];
  switch (type)
  {
    case ICMP_ECHOREPLY   : return ("Echoreply");
    case ICMP_UNREACH     : return ("Unreach");
    case ICMP_SOURCEQUENCH: return ("SourceQuench");
    case ICMP_REDIRECT    : return ("Redirect");
    case ICMP_ECHO        : return ("Echo");
    case ICMP_TIMXCEED    : return ("Time exceed");
    case ICMP_PARAMPROB   : return ("ParamProblem");
    case ICMP_TSTAMP      : return ("Tstamp");
    case ICMP_TSTAMPREPLY : return ("TstampReply");
    case ICMP_IREQ        : return ("Ireq");
    case ICMP_IREQREPLY   : return ("IreqReply");
    case ICMP_MASKREQ     : return ("MaskReq");
    case ICMP_MASKREPLY   : return ("MaskReply");
    default               : sprintf (buf,"unknown type %u",type);
                            return (buf);
  }
}

/*---------------------------------------------------------------------*/

int Check_ICMP (struct ip *ip, int seq, int *ret_code)
{
  int    hlen       = ip->ip_hl << 2;
  struct icmp *icmp = (struct icmp *) ((BYTE*)ip + hlen);
  BYTE   type       = icmp->icmp_type;
  BYTE   code       = icmp->icmp_code;
  int    ret        = 0;

  if (ntohl(ip->ip_src.s_addr) == INADDR_LOOPBACK)
     return (-1);

  if ((type == ICMP_TIMXCEED && code == ICMP_TIMXCEED_INTRANS) ||
       type == ICMP_UNREACH)
  {
    struct ip     *orig_ip;
    struct udphdr *orig_udp;

    orig_ip   = (struct ip*) &icmp->icmp_ip;
    hlen      = orig_ip->ip_hl << 2;
    orig_udp  = (struct udphdr*) ((BYTE*)orig_ip + hlen);

    if (orig_ip->ip_p      == PROBE_PROTOCOL &&
#if PROBE_PROTOCOL == ICMP_PROTO
      /* todo !! */
#else
        orig_udp->uh_sport == htons (MY_PORT) &&
        orig_udp->uh_dport == htons (port+seq))
#endif
    {
      ret = (type == ICMP_TIMXCEED ? -1 : code+1);
      if (Verbose)
         printf ("IP encap (%s/%u) from %s\n", ICMP_type(type), code,
                 NET_ADDR(orig_ip->ip_src.s_addr));
    }
    else
      ret = -1;
  }
  *ret_code = code;
  return (ret);
}

/*---------------------------------------------------------------------*/

void SigIntHandler (int sig)
{
  halt_flag++;
  signal (sig, SIG_DFL);
}

/*
 * Construct an Internet address representation.
 * If the nflag has been supplied, give numeric value, otherwise
 * try for symbolic name.
 * An address in our domain is printed without domain-name
 */
char * NetName (struct in_addr adr)
{
  char   *cp;
  static char line[80];
  static char domain[MAXHOSTNAMELEN + 1];
  static int  first = 1;

  if (first && !nflag)
  {
    first = 0;
    if (gethostname(domain, MAXHOSTNAMELEN) == 0 &&
        (cp = strchr(domain,'.')) != NULL)
         strcpy(domain, cp+1);
    else domain[0] = 0;
  }
  cp = NULL;

  if (!nflag && adr.s_addr)
  {
    struct hostent *hp = gethostbyaddr ((char*)&adr,sizeof(adr),AF_INET);
    if (hp)
    {
      char name [MAXHOSTNAMELEN+1];
      strncpy (name,hp->h_name,MAXHOSTNAMELEN);
      name [MAXHOSTNAMELEN] = '\0';
      cp = strchr (name,'.');

      if (cp && !strcmp(cp+1,domain))
         *cp = 0;
      cp = name;
    }
  }
  if (cp)
       return strcpy (line,cp);
  else return inet_ntoa (adr);
}


/*
 *  Lookup owner of the net using SOA records
 */
char *LookupOwner (struct in_addr addr)
{
  char  dns_query[100];
  char *owner, *dot_ptr;
  BYTE *a = (BYTE*) &addr;

  if (a[0] >= 192)
       sprintf (dns_query, "%d.%d.%d.in-addr.arpa", a[2], a[1], a[0]);
  else if (a[0] >= 128)
       sprintf (dns_query, "%d.%d.in-addr.arpa", a[1], a[0]);
  else sprintf (dns_query, "%d.in-addr.arpa", a[0]);

  owner = DoResolve (dns_query);
  
  /* reformat slightly */
  if (owner)
  {
    if (strchr(owner,'@') == NULL)
    {
      dot_ptr = strchr (owner, '.');
      if (dot_ptr)
         *dot_ptr = '@';
    }
    if (strlen(owner) > 0)
    {
      dot_ptr = owner + strlen (owner) - 1;
      while (*dot_ptr == ' ' || *dot_ptr == '.' )
             *dot_ptr-- = 0;
    }
  }
  return (owner);
}


char *DoResolve (char *query)
{
  (void)query;
  return (NULL); /* !! to-do */
}

#ifndef DJGPP
DWORD uclock (void)
{
  return clock() * UCLOCKS_PER_SEC / CLOCKS_PER_SEC;  // !! to-do
}
#endif

/*
 * this function by default is not included in the
 * small-model library. Include it here if you want
 */
#if defined(__SMALL__) && !defined(DOS386) 
int resolve_ip (DWORD ip, char *result)
{
  ip = ip;
  result = result;
  return (0);
}
#endif
