#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>      
#include <netdb.h>
#include <time.h>
#include <stdio.h>
#include <string.h>

#include "portable.h"

/* difference between Unix time and net time */
#define BASE1970        2208988800UL

#ifdef HAVE_ALARM
jmp_buf jmp;

void alarm_handler(int signum)
{
  signal(signum, alarm_handler);
  alarm(5);
  longjmp(jmp, 1);
}
#endif

time_t
RemoteDate_UDP(char *host)
{
  struct  hostent *him;       /* host table entry */
  struct  servent *timeServ;  /* sevice file entry */
  struct  sockaddr_in sin;    /* socket address */
  int     fd;                 /* network file descriptor */
  time_t  unixTime;           /* time in Unix format */
  u_char  netTime[4];         /* time in network format */
  int     i;                  /* loop variable */
  int     n;

  if ((him = gethostbyname(host)) == NULL) {
     fprintf(stderr, "rdate: Unknown host %s\n", host);
     return(-1);
  }

  if ((timeServ = getservbyname("time","udp")) == NULL) {
     fprintf(stderr, "rdate: time/udp: unknown service\n");
     return(-1);
  }

  if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
     perror("rdate");
     return(-1);
  }

  sin.sin_family = him->h_addrtype;
  memcpy(&sin.sin_addr, him->h_addr, him->h_length);
  sin.sin_port = timeServ->s_port;

  /* what are we supposed to send ?
   */
  memset (&netTime, 0, sizeof(netTime));

  i = sendto(fd, &netTime, sizeof(netTime), 0,
             (struct sockaddr*)&sin, sizeof(sin));
  if (i < 0) {
    perror("sendto");
    close(fd);
    return (-1);
  }

  i = sizeof(sin);

#ifdef HAVE_ALARM
  /* read in the response, with an alarm */

  signal(SIGALRM, alarm_handler);
  alarm(5);
  if (setjmp(jmp)) {
     fprintf(stderr, "rdate: timed out waiting for data from %s\n", host);
     close(fd);
     return (-1);
  }
#endif
  n = recvfrom(fd, netTime, sizeof(netTime), 0, (struct sockaddr*)&sin, &i);
  if (n < 0) {
    perror("recvfrom");
    close(fd);
    return (-1);
  }

  if (n < sizeof(netTime)) {
    fprintf(stderr, "rdate: partial data read from %s\n", host);
    close(fd);
    return (-1);
  }

  close(fd);

  unixTime = ((time_t)netTime[0] << 24 |
              (time_t)netTime[1] << 16 |
              (time_t)netTime[2] << 8  |
              (time_t)netTime[3] << 0  ) - BASE1970;

  return (unixTime);
}

time_t
RemoteDate(char *host)
{
  struct  hostent *him;       /* host table entry */
  struct  servent *timeServ;  /* sevice file entry */
  struct  sockaddr_in sin;    /* socket address */
  int     fd;                 /* network file descriptor */
  time_t  unixTime;           /* time in Unix format */
  u_char  netTime[4];         /* time in network format */
  int     i;                  /* loop variable */

  if ((him = gethostbyname(host)) == NULL) {
     fprintf(stderr, "rdate: Unknown host %s\n", host);
     return(-1);
  }

  if ((timeServ = getservbyname("time","tcp")) == NULL) {
     fprintf(stderr, "rdate: time/tcp: unknown service\n");
     return(-1);
  }

  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("rdate");
    return(-1);
  }

  sin.sin_family = him->h_addrtype;
  memcpy(&sin.sin_addr, him->h_addr, him->h_length);
  sin.sin_port = timeServ->s_port;

  if (connect(fd, (const struct sockaddr*)&sin, sizeof(sin)) < 0) {
     perror("rdate");
     close(fd);
     return(-1);
  }

  /* read in the response */
  for (i = 0; i < 4; ) {
    int l = read(fd, (char*)&netTime[i], 4-i);
    if (l <= 0) {
      perror("rdate");
      close(fd);
      return(-1);
    }
    i += l;
  }

  close(fd);

  unixTime = ((time_t)netTime[0] << 24 |
              (time_t)netTime[1] << 16 |
              (time_t)netTime[2] << 8  |
              (time_t)netTime[3] << 0  ) - BASE1970;

  return (unixTime);
}

int 
main(int argc, char *argv[])
{
  int o, mode = 0;
  int debug = 0;
  time_t t;

  while ((o = getopt(argc, argv, "?dspu")) != EOF)
    switch (o) {
      case 'd':
           debug++;
           break;
      case 'p':
           mode |= 1;
           break;
      case 's':
           mode |= 2;
           break;
      case 'u':
           mode |= 4;
           break;
      default:
      usage:
           fprintf(stderr, "Usage: rdate [-d] [-s] [-p] [-u] <host> ...\n");
      return 1;
    }

  if (optind >= argc)
     goto usage;
  if ((mode & 3) == 0)
     mode |= 1;

#ifdef WATT32
  if (debug)
     dbug_init();
  sock_init();
#endif

  for (o = optind; o < argc; ++o) {
    t = (mode & 4) ? RemoteDate_UDP(argv[o]) : RemoteDate(argv[o]);
    if (t == (time_t)-1)
      continue;
    if (mode & 1)
      printf("[%s]\t%s", argv[o], ctime(&t));
    if (mode & 2)
      if (stime(&t) < 0) {
        perror("rdate");
        return 1;
      }
  }
  return 0;
}


#if defined (__DJGPP__)
int stime (time_t *tp)
{
  struct timeval tv;

  tv.tv_sec  = *tp;
  tv.tv_usec = 0;
  if (settimeofday(&tv) < 0) {
    errno = EINVAL;
    return (-1);
  }
  return (0);
}

#elif defined (WATT32) && !defined(__BORLANDC__)
int stime (time_t *tp)
{
  struct dosdate_t date;
  struct dostime_t time;
  struct tm        tm = *localtime (tp);

  date.year  = tm.tm_year;
  date.month = tm.tm_mon;
  date.day   = tm.tm_mday;

  time.hour    = tm.tm_hour;
  time.minute  = tm.tm_min;
  time.second  = tm.tm_sec;
  time.hsecond = 0;

  if (_dos_setdate(&date) || _dos_settime(&time)) {
    errno = EINVAL;
    return (-1);
  }
  return (0);
}
#endif

