
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "ntp.h"

/*
windump.exe: listening on \Device\NPF_{93380695-0E31-456C-9EB0-8802E111C09D}
reuest:
02:04:16.504784 IP (tos 0x0, ttl  64, id 4476, offset 0, flags [none],
                length: 76) 10.0.0.6.123 > 192.5.41.40.123: [udp sum ok]
                NTPv3 sym_act, strat 0, poll 10, prec -6 dist 0.281219,
                disp 9.144195, ref (unspec)@3267136116.234453305
                orig 0.000000000 rec -0.000000000 xmt -1027830239.504249989
reply:
02:04:16.786654 IP (tos 0x10, ttl  48, id 8770, offset 0, flags [DF],
                length: 76) 192.5.41.40.123 > 10.0.0.6.123: [udp sum ok]
                NTPv3 sym_pas, strat 1, poll 10, prec -17 dist 0.000000,
                disp 0.000366, ref USNO@3267137050.127041995
                orig 3267137056.495750010 rec -2.045492000 xmt -2.045458998
*/
static void p_sfix      (const struct s_fixedpt *);
static void p_ntp_time  (const struct l_fixedpt *);
static void p_ntp_delta (const struct l_fixedpt *, const struct l_fixedpt *);

/* 
 * Print ntp requests
 */
void ntp_print (const u_char *cp, u_int length)
{
  const  struct ntpdata *bp = (struct ntpdata *) cp;
  int    mode, version, leapind;
  static char rclock[5];

  if (length != sizeof(struct ntpdata))
     PRINTF (" [len=%d]", length);

  TCHECK (bp->status);

  version = (bp->status & VERSIONMASK) >> 3;
  PRINTF (" v%d", version);

  leapind = bp->status & LEAPMASK;
  switch (leapind)
  {
    case NO_WARNING:
         break;

    case PLUS_SEC:
         PUTS (" +1s");
         break;

    case MINUS_SEC:
         PUTS (" -1s");
         break;
  }

  mode = bp->status & MODEMASK;
  switch (mode)
  {
    case MODE_UNSPEC:        /* unspecified */
         PUTS (" unspec");
         break;

    case MODE_SYM_ACT:       /* symmetric active */
         PUTS (" sym_act");
         break;

    case MODE_SYM_PAS:       /* symmetric passive */
         PUTS (" sym_pas");
         break;

    case MODE_CLIENT:        /* client */
         PUTS (" client");
         break;

    case MODE_SERVER:        /* server */
         PUTS (" server");
         break;

    case MODE_BROADCAST:     /* broadcast */
         PUTS (" bcast");
         break;

    case MODE_RES1:          /* reserved */
         PUTS (" res1");
         break;

    case MODE_RES2:          /* reserved */
         PUTS (" res2");
         break;

  }

  TCHECK (bp->stratum);
  PRINTF (" strat %d", bp->stratum);

  TCHECK (bp->ppoll);
  PRINTF (" poll %d", bp->ppoll);

  /* Can't TCHECK bp->precision bitfield so bp->distance + 0 instead */
  TCHECK2 (bp->distance, 0);
  PRINTF (" prec %d", bp->precision);

  if (!vflag)
     return;

  TCHECK (bp->distance);
  PUTS (" dist ");
  p_sfix (&bp->distance);

  TCHECK (bp->dispersion);
  PUTS (" disp ");
  p_sfix (&bp->dispersion);

  TCHECK (bp->refid);
  PUTS (" ref ");
  /* Interpretation depends on stratum */

  switch (bp->stratum)
  {
    case UNSPECIFIED:
         PUTS ("(unspec)");
         break;

    case PRIM_REF:
         strncpy (rclock, (char*)&bp->refid, 4);
         rclock[4] = '\0';
         PUTS (rclock);
         break;

    case INFO_QUERY:
         PRINTF ("%s INFO_QUERY", ipaddr_string (&bp->refid));
         /* this doesn't have more content */
         return;

    case INFO_REPLY:
         PRINTF ("%s INFO_REPLY", ipaddr_string (&bp->refid));
         /* this is too complex to be worth printing */
         return;

    default:
         PUTS (ipaddr_string(&bp->refid));
         break;
  }

  TCHECK (bp->reftime);
  PUTCHAR ('@');
  p_ntp_time (&bp->reftime);

  TCHECK (bp->org);
  PUTS (" orig ");
  p_ntp_time (&bp->org);

  TCHECK (bp->rec);
  PUTS (" rec ");
  p_ntp_delta (&bp->org, &bp->rec);

  TCHECK (bp->xmt);
  PUTS (" xmt ");
  p_ntp_delta (&bp->org, &bp->xmt);

  return;

trunc:
  PUTS (" [|ntp]");
}

static void p_sfix (const struct s_fixedpt *sfp)
{
  int   i, f;
  float ff;

  i  = ntohs (sfp->int_part);
  f  = ntohs (sfp->fraction);
  ff = f / 65536.0;                /* shift radix point by 16 bits */
  f  = ff * 1000000.0;          /* Treat fraction as parts per million */
  PRINTF ("%d.%06d", i, f);
}

#define        FMAXINT        (4294967296.0)        /* floating point rep. of MAXINT */

static void p_ntp_time (const struct l_fixedpt *lfp)
{
  u_int i;
  u_int uf;
  u_int f;
  float ff;

  i  = ntohl (lfp->int_part);
  uf = ntohl (lfp->fraction);
  ff = uf;
  if (ff < 0.0)                 /* some compilers are buggy */
      ff += FMAXINT;
  ff = ff / FMAXINT;                /* shift radix point by 32 bits */
  f  = ff * 1000000000.0;       /* treat fraction as parts per billion */
  PRINTF ("%u.%09d", i, f);
}

/*
 * Prints time difference between *lfp and *olfp
 */
static void p_ntp_delta (const struct l_fixedpt *olfp,
                         const struct l_fixedpt *lfp)
{
  u_int uf, ouf, f;
  float ff;
  int   i, signbit;

  i   = ntohl (lfp->int_part) - ntohl (olfp->int_part);
  uf  = ntohl (lfp->fraction);
  ouf = ntohl (olfp->fraction);

  if (i > 0)
  {                                /* new is definitely greater than old */
    signbit = 0;
    f = uf - ouf;
    if (ouf > uf)                /* must borrow from high-order bits */
       i -= 1;
  }
  else if (i < 0)
  {                                /* new is definitely less than old */
    signbit = 1;
    f = ouf - uf;
    if (uf > ouf)                /* must carry into the high-order bits */
       i += 1;
    i = -i;
  }
  else
  {                                /* int_part is zero */
    if (uf > ouf)
    {
      signbit = 0;
      f = uf - ouf;
    }
    else
    {
      signbit = 1;
      f = ouf - uf;
    }
  }

  ff = f;
  if (ff < 0.0)                        /* some compilers are buggy */
      ff += FMAXINT;
  ff = ff / FMAXINT;                /* shift radix point by 32 bits */
  f  = ff * 1000000000.0;       /* treat fraction as parts per billion */
  if (signbit)
       PUTCHAR ('-');
  else PUTCHAR ('+');

  PRINTF ("%d.%09d", i, f);
}
