// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (C) 1992-1994 Microsoft Corporation
// All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and the
// Books Online documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

#include "stdafx.h"

#ifdef AFXCTL_CORE2_SEG
#pragma code_seg(AFXCTL_CORE2_SEG)
#endif

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define new DEBUG_NEW


extern void _AfxDrawBorder(CDC* pDC, CRect& rc, int cxBorder, int cyBorder,
	COLORREF cr);

/////////////////////////////////////////////////////////////////////////////
// COleControl::CreateTracker - Creates tracker for UIActive control

void COleControl::CreateTracker(BOOL bHandles, BOOL bHatching)
{
	ASSERT(bHandles || bHatching);
	ASSERT(! m_bOpen);
	ASSERT(m_bUIActive);
	ASSERT(m_pRectTracker == NULL);

	UINT nStyle = 0;
	if (bHandles)
		nStyle |= CRectTracker::resizeOutside;
	if (bHatching)
		nStyle |= CRectTracker::hatchedBorder;

	ASSERT(nStyle != 0);

	TRY
	{
		// Create the tracker.
		CRect rectTmp = m_rcPos;
		rectTmp.OffsetRect(-rectTmp.left, -rectTmp.top);
		m_pRectTracker = new CRectTracker(rectTmp, nStyle);
		UINT nHandleSize = m_pRectTracker->m_nHandleSize++;

		// Enlarge window to expose non-client area.
		CWnd* pWndOuter = GetOuterWindow();
		CWnd* pWndParent = pWndOuter->GetParent();
		CRect rectWindow;
		CRect rectParent;
		pWndOuter->GetWindowRect(rectWindow);
		pWndParent->GetClientRect(rectParent);
		pWndParent->ClientToScreen(rectParent);
		rectWindow.OffsetRect(-rectParent.left, -rectParent.top);
		rectWindow.InflateRect(nHandleSize, nHandleSize);
		pWndOuter->MoveWindow(rectWindow);
	}
	CATCH (CException, e)
	{
		// If anything went wrong, just continue without the tracker.
		if (m_pRectTracker != NULL)
		{
			delete m_pRectTracker;
			m_pRectTracker = NULL;
		}
	}
	END_CATCH
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::DestroyTracker - destroys tracker when control UIDeactivates

void COleControl::DestroyTracker()
{
	ASSERT(! m_bOpen);
	ASSERT(m_bUIActive);

	if (m_pRectTracker == NULL)
		return;

	UINT nHandleSize = m_pRectTracker->m_nHandleSize - 1;

	// Destroy the tracker.
	delete m_pRectTracker;
	m_pRectTracker = NULL;

	// Restore window to its original (pre-UIActive) size.
	CWnd* pWndOuter = GetOuterWindow();
	CWnd* pWndParent = pWndOuter->GetParent();
	CRect rectWindow;
	CRect rectParent;
	pWndOuter->GetWindowRect(rectWindow);
	pWndParent->GetClientRect(rectParent);
	pWndParent->ClientToScreen(rectParent);
	rectWindow.OffsetRect(-rectParent.left, -rectParent.top);
	rectWindow.InflateRect(-(int)nHandleSize, -(int)nHandleSize);
	pWndOuter->MoveWindow(rectWindow);
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::OnNcPaint - handler for WM_NCPAINT message

void COleControl::OnNcPaint()
{
	if (m_bOpen || m_pRectTracker == NULL)
	{
		Default();
		return;
	}

	UINT nHandleSize = m_pRectTracker->m_nHandleSize - 1;
	CWindowDC dc(this);

	// Convert client coords to window coords and draw tracker.
	m_pRectTracker->m_rect.OffsetRect(nHandleSize, nHandleSize);
	m_pRectTracker->Draw(&dc);

	// Draw border, if any.
	if (m_sBorderStyle == 1)
	{
		::_AfxDrawBorder(&dc, m_pRectTracker->m_rect,
			GetSystemMetrics(SM_CXBORDER), GetSystemMetrics(SM_CYBORDER),
			GetSysColor(COLOR_WINDOWFRAME));
	}

	// Convert window coords back to client coords.
	m_pRectTracker->m_rect.OffsetRect(-(int)nHandleSize, -(int)nHandleSize);
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::OnNcCalcSize - handler for WM_NCCALCSIZE message

void COleControl::OnNcCalcSize(BOOL bCalcValidRects,
	NCCALCSIZE_PARAMS* lpParams)
{
	UNUSED bCalcValidRects;

	if (m_bOpen || m_pRectTracker == NULL)
	{
		Default();
		return;
	}

	UINT nHandleSize = m_pRectTracker->m_nHandleSize - 1;

	// Adjust client rect to make room for tracker.
	::InflateRect(lpParams->rgrc, -(int)nHandleSize, -(int)nHandleSize);

	// Ajust client rect to make room for border, if any.
	if (m_sBorderStyle == 1)
		::InflateRect(lpParams->rgrc,
			-GetSystemMetrics(SM_CXBORDER), -GetSystemMetrics(SM_CYBORDER));
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::OnNcHitTest - handler for WM_NCHITTEST message

UINT COleControl::OnNcHitTest(CPoint point)
{
	if (m_bOpen || m_pRectTracker == NULL)
		return (UINT)Default();

	// Check for client area or border hit.
	ScreenToClient(&point);
	CRect rectClient;
	GetClientRect(rectClient);
	if (m_sBorderStyle == 1)
		rectClient.InflateRect(GetSystemMetrics(SM_CXBORDER), 
			GetSystemMetrics(SM_CYBORDER));

	if (rectClient.PtInRect(point))
		return HTCLIENT;

	// If tracker detects a hit, return HTSIZE; otherwise HTNOWHERE.
	int nHitCode = m_pRectTracker->HitTest(point);
	return (nHitCode == CRectTracker::hitNothing) ? HTNOWHERE : HTSIZE;
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::OnNcLButtonDown - handler for WM_NCLBUTTONDOWN message

void COleControl::OnNcLButtonDown(UINT nHitTest, CPoint point)
 {
	UNUSED nHitTest;

	if (m_bOpen || m_pRectTracker == NULL)
	{
		Default();
		return;
	}

	ScreenToClient(&point);

	// Setup a (semi-)permanent CWnd for the control's parent window
	CRect rectBefore = m_pRectTracker->m_rect;
	CWnd wndClip;
	wndClip.Attach(::GetParent(GetOuterWindow()->m_hWnd));

	// Move or resize the tracker.
	AfxLockTempMaps();
	BOOL bTrack = m_pRectTracker->Track(this, point, FALSE, &wndClip);
	AfxUnlockTempMaps();

	if (bTrack)
	{
		ASSERT(m_pInPlaceSite);

		CRect rectAfter = m_pRectTracker->m_rect;
		if (rectBefore != rectAfter)
		{
			// If rectangle changed, adjust the tracker's rectangle and move
			// the control.
			m_pRectTracker->m_rect.OffsetRect(-m_pRectTracker->m_rect.left,
				-m_pRectTracker->m_rect.top);
			CWnd* pWndOuter = GetOuterWindow();
			CWnd* pWndParent = pWndOuter->GetParent();
			CRect rectWindow;
			CRect rectParent;
			pWndOuter->GetWindowRect(rectWindow);
			pWndParent->GetClientRect(rectParent);
			pWndParent->ClientToScreen(rectParent);
			UINT nHandleSize = m_pRectTracker->m_nHandleSize - 1;
			rectAfter.OffsetRect(rectWindow.left - rectParent.left + nHandleSize,
				rectWindow.top - rectParent.top + nHandleSize);

			// Update the control's extents.
			SIZEL szlPixels;
			SIZEL szlHimetric;
			szlPixels.cx = (long)rectAfter.Width();
			szlPixels.cy = (long)rectAfter.Height();
			_XformSizeInPixelsToHimetric(NULL, &szlPixels, &szlHimetric);
			if ((m_cxExtent != szlHimetric.cx) ||
				(m_cyExtent != szlHimetric.cy))
			{
				m_cxExtent = szlHimetric.cx;
				m_cyExtent = szlHimetric.cy;
				SetModifiedFlag();
			}

			// Move/resize the control's window.
			m_pInPlaceSite->OnPosRectChange(rectAfter);
		}
	}

	wndClip.Detach();
}

/////////////////////////////////////////////////////////////////////////////
// COleControl::OnSetCursor - handler for WM_SETCURSOR message

BOOL COleControl::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message)
{
	UNUSED nHitTest;
	UNUSED message;

	if (m_bOpen || m_pRectTracker == NULL)
		return (BOOL)Default();

	if (nHitTest == HTCLIENT)
	{
		// In client area: use default cursor or arrow.
		if (!Default())
			::SetCursor(::LoadCursor(NULL, IDC_ARROW));
	}
	else
	{
		// In non-client area: use tracker-supplied cursor.
		m_pRectTracker->SetCursor(pWnd, HTCLIENT);
	}

	return TRUE;
}
