///////////////////////////////////////////////////////////////////////////////
//
//  File Name 
//      COMWINDS.H
//
//  Description
//      Common remote call and client-side utility functions for provider
//      communicating with the WINDS messaging host.
//
//  Author
//      Irving De la Cruz
//
//  Revision: 1.7
//
// Written for Microsoft Windows Developer Support
// Copyright (c) 1995-1996 Microsoft Corporation. All rights reserved.
//
#ifndef _COMWINDS_H
#define _COMWINDS_H

#define STRICT
#include <WINDOWS.H>
#include <WINDOWSX.H>
#include <CTL3D.H>
#include <MAPIWIN.H>
#include <MAPISPI.H>
#include <MAPIUTIL.H>
#include <MAPICODE.H>
#include <MAPIVAL.H>
#include <MAPIWZ.H>

// Constants for communicating with the WINDS Server Messaging Host
#include "WINDSGBL.H"

// Header file for the trace and logging functions.
#include "TRACES.H"

// This header file is generated by the MIDL compiler
#include "WINDS.H"

#include "UISRVCS.H"

#ifdef UNICODE
#define RemoteChangePassword            RemoteChangePasswordW
#define RemoteLogonMailBox              RemoteLogonMailBoxW
#define RemoteLogonMailBoxAndSetNotif   RemoteLogonMailBoxAndSetNotifW
#define RemoteTerminateNotif            RemoteTerminateNotifW
#define RemoteValidateNotif             RemoteValidateNotifW
#define RemoteOpenMsgUploadPipe         RemoteOpenMsgUploadPipeW
#define RemoteSendMsgToAccount          RemoteSendMsgToAccountW
#define RemoteOpenMsgDownloadPipe       RemoteOpenMsgDownloadPipeW
#define RemoteOpenHeaderDownloadPipe    RemoteOpenHeaderDownloadPipeW
#define RemoteCheckNewMail              RemoteCheckNewMailW
#define RemoteOpenOneMsgDownloadPipe    RemoteOpenOneMsgDownloadPipeW
#define RemoteGetRecipientProps         RemoteGetRecipientPropsW
#define RemoteGetDistListProps          RemoteGetDistListPropsW
#else
#define RemoteChangePassword            RemoteChangePasswordA
#define RemoteLogonMailBox              RemoteLogonMailBoxA
#define RemoteLogonMailBoxAndSetNotif   RemoteLogonMailBoxAndSetNotifA
#define RemoteTerminateNotif            RemoteTerminateNotifA
#define RemoteValidateNotif             RemoteValidateNotifA
#define RemoteOpenMsgUploadPipe         RemoteOpenMsgUploadPipeA
#define RemoteSendMsgToAccount          RemoteSendMsgToAccountA
#define RemoteOpenMsgDownloadPipe       RemoteOpenMsgDownloadPipeA
#define RemoteOpenHeaderDownloadPipe    RemoteOpenHeaderDownloadPipeA
#define RemoteCheckNewMail              RemoteCheckNewMailA
#define RemoteOpenOneMsgDownloadPipe    RemoteOpenOneMsgDownloadPipeA
#define RemoteGetRecipientProps         RemoteGetRecipientPropsA
#define RemoteGetDistListProps          RemoteGetDistListPropsA
#endif // UNICODE

// Custom properties used by this service provider. These properties are
// non-transmittable and only used in the configuration and private
// profile section of this provider/service.
#define BASE_PROVIDER_ID                    0x6600  // Look at the comments in MAPITAGS.H

// Common Custom Properties for all WINDS service providers
#define PR_SMP_REMOTE_SERVER            PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0000))
#define PR_SMP_USER_NAME                PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0001))
#define PR_SMP_BROWSE_MB                PROP_TAG (PT_OBJECT,     (BASE_PROVIDER_ID + 0x0002))
#define PR_SMP_CHANGE_PASSWORD          PROP_TAG (PT_OBJECT,     (BASE_PROVIDER_ID + 0x0003))
#define PR_SMP_SESSION_UID              PROP_TAG (PT_BINARY,     (BASE_PROVIDER_ID + 0x0004))
#define PR_SMP_CONNECTION_TYPE          PROP_TAG (PT_LONG,       (BASE_PROVIDER_ID + 0x0005))

// Secured properties that users should not see by looking at the registry.
// On Windows NT the registry is secured by the system. On Windows 95, this
// properties are stored encrypted, along with the system's list of passwords
#define PR_SMP_MAILBOX_NAME         PROP_TAG (PT_TSTRING,    (PROP_ID_SECURE_MIN + 0x0001))
#define PR_SMP_MAILBOX_ID           PROP_TAG (PT_LONG,       (PROP_ID_SECURE_MIN + 0x0002))
#define PR_SMP_MAILBOX_PASSWORD     PROP_TAG (PT_TSTRING,    (PROP_ID_SECURE_MIN + 0x0003))
                                                             
// Address books                                             
#define PR_SMP_LUL_LAST_UPDATE      PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0020))
#define PR_SMP_LUL_FILENAME         PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0021))
#define PR_SMP_LUL_DOWNLOAD         PROP_TAG (PT_OBJECT,     (BASE_PROVIDER_ID + 0x0022))
#define PR_SMP_MANAGER_NAME_TBL     PROP_TAG (PT_OBJECT,     (BASE_PROVIDER_ID + 0x0023))
// Custom properties for IMailUser and IDistList objects. This properties are in the named
// property range so that PAB address books and other writeable address books may copy them.
// Address book service providers should not use the 0x6600-0x67FF property ID range for
// custom properties in the recipient objects. Instead user NAMED properties
#define PR_WINDS_AB_MANAGER_NAME        PROP_TAG (PT_TSTRING, 0x8000)
#define PR_WINDS_AB_MANAGER_ENTRYID     PROP_TAG (PT_BINARY,  0x8001)
#define PR_WINDS_AB_DL_OWNER_ACCT       PROP_TAG (PT_TSTRING, 0x8002)
#define PR_WINDS_AB_DL_OWNER_NAME       PROP_TAG (PT_TSTRING, 0x8003)
// Address entry named property names. They have to be UNICODE strings
#define WINDS_AB_MANAGER_NAME       L"WINDS AB - Manager Name"
#define WINDS_AB_MANAGER_EID        L"WINDS AB - Manager Entry ID"
#define WINDS_AB_DL_OWNER_ACCT      L"WINDS AB - DL Owner Account"
#define WINDS_AB_DL_OWNER_NAME      L"WINDS AB - DL Owner Name"

// Transports                                                
#define PR_SMP_UPLOAD_TIME          PROP_TAG (PT_SYSTIME,    (BASE_PROVIDER_ID + 0x0040))
#define PR_SMP_HEADERS_FILENAME     PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0041))
#define PR_TMP_UPLOAD_TIME          PROP_TAG (PT_TSTRING,    (BASE_PROVIDER_ID + 0x0042))
#define PR_SMP_GET_HEADERS          PROP_TAG (PT_BOOLEAN,    (BASE_PROVIDER_ID + 0x0043))

// Values available for PR_SMP_CONNECTION_TYPE
typedef enum _CONNECTION_TYPE
{
    // Don't change the order of these unless the string resources ID's are modified accordingly
    LAN_CONNECT         = 1,    
    OFFLINE_CONNECT     = 2,
    CONNECTION_BROKEN   = 3,
    NOT_AVAIL_NO_ABD    = 4
} CONNECTION_TYPE;

// Index properties into the logon property array of ALL providers using the WINDS server
enum
{
    SERVER_NAME             = 0,
    MAILBOX_NAME            = 1,
    MAILBOX_ID              = 2,
    USER_NAME               = 3, 
    PASSWORD                = 4,
    NET_CON                 = 5,
    // Number of properties the WIZARD setup pages collect for the configuration of the WINDS providers
    WIZARD_PROPERTIES       = 5
};

// Number of controls in the configuration page of ALL WINDS Providers
#define NUM_CFG_USERINFO_PAGE_CTLS     13

// Structure passed to the configuration dialog.
typedef struct _CFGDLG
{
    HINSTANCE      hInst;      // Instance of provider's DLL for loading resources for the UI
    HWND           hWnd;       // Handle of parent window or NULL is not UI should be used
    LPSPropValue * ppProps;    // Property array with the configuration properties
    LPMAPISUP      pSupObj;    // IMAPISupport object for creating the UI property sheets
    ULONG          ulFlags;    // 0 or UI_READONLY for access control to the configuration values
    HANDLE         hUIMutex;   // Handle to the MUTEX to control exclusive access to the configuration UI
} CFGDLG, *PCFGDLG;

// This code is to protect on reentrancy in the function
// that display the configuration UI for the WINDS services
#define CONFIG_UI_MUTEX     TEXT("WINDS_SERVICE_CONFIG_UI_MUTEX")

// PR_AB_PROVIDER_ID is the GUID that represents this Address Book Provider.
// This GUID is available as a property in the hierarchy. The UID is hard coded.
// This number is generated only once per provider during the developement
// of it using the standard tools (GUIDGEN or UUIDGEN) in VC++ or the Win32 SDK.
// With this number you could identify your provider across multiple platforms.
// GUIDGEN: 0xe33c67a0, 0xc81f, 0x11ce, 0xb2, 0xe4, 0x0, 0xaa, 0x0, 0x51, 0xe, 0x3b
// This is the UID of the entry ID of objects in the WINDS address book provider.
#define AB_UID_PROVIDER { 0xe3, 0x3c, 0x67, 0xa0, \
                          0xc8, 0x1f, 0x11, 0xce, \
                          0xb2, 0xe4, 0x0,  0xaa, \
                          0x0,  0x51, 0xe,  0x3b }


// This is the UID in the entry ID's the a WINDS transport uses to open entries in
// the WINDS address book, through the support object.
// This UID is hard coded because we only allow one instance of our provider
// to be installed in a single profile at any time.
// GUIDGEN: 0x8B43A160, 0xDFAC, 0x11ce, 0xB2, 0xEC, 0x00, 0xAA, 0x00, 0x51, 0x0E, 0x3B
#define XP_WINDS_AB_ENTRIES { 0x8b, 0x43, 0xa1, 0x60, \
                              0xdf, 0xac, 0x11, 0xce, \
                              0xb2, 0xec, 0x0,  0xaa, \
                              0x0,  0x51, 0xe,  0x3b }


// This is the private entry ID structure for entries that a WINDS transport
// would like to open in the WINDS address book
typedef struct _PRIVATE_XP_ENTRYID
{
    // MAPI-required fields
    BYTE        abFlags[4];         // 4 bytes      // MAPI Flags
    MAPIUID     uidGlobal;          // 16 bytes     // UID Unique for XP entries
    // Provider-defined fields
    MAPIUID     uidWINDSEntries;    // 16 bytes     // UID Unique for WINDS entries in the WINDS address book
    char        szObjectAlias[34];  // 34 bytes
                                    // 70 Bytes Total
} PRIVATE_XP_ENTRYID;
#define     CB_PRIVATE_XP_EID       70      // Size of the structure above


// This is the subdirectory in the TEMP directoy where WINDS service
// providers create temporary data files
#define WINDS_DATA_DIRECTORY                TEXT("WINDS Messaging Services Data Files")
// This is the directory where the temporary files for downloaded messages are stored
#define WINDS_DOWNLOAD_DIR_NAME_FORMAT      TEXT("%s\\Pending Downloads - %s")
#define XP_MSG_FILE_PREFIX                  TEXT("XRM")
#define XP_MSG_FILE_SUFFIX                  TEXT("TMP")

// Variables and static data structure used by all WINDS providers 
extern "C"
{
    extern HINSTANCE    ghInstance;
    extern TCHAR        szBlank[];
    extern TCHAR        szFilter[];
    extern TCHAR        szPhoneFilter[];
    extern TCHAR        szTimeFilter[];
    extern DTBLLABEL    DtLabel;
    extern DTBLGROUPBOX DtGroupBox;
    extern DTBLPAGE     DtPage;
    extern DTBLEDIT     DtObjAlias;
    extern DTBLEDIT     DtObjFullName;
    extern DTBLEDIT     DtUserName;
    extern DTBLEDIT     DtServer;
    extern DTBLEDIT     DtComment;
    extern DTCTL        UserConfigPage[];
};

int WINAPI PrivateMessageBox
                    (UINT                       ids, 
                     HWND                       hOwnerWnd = NULL,
                     UINT                       uFlags = MB_OK | MB_ICONSTOP,
                     HINSTANCE                  hInst = NULL);
extern "C"
{
    HRESULT WINAPI BindToServer
                    (LPTSTR                     szServer);
    BOOL WINAPI IsWINDSServerAvailable
                    (LPTSTR                     szServerName);
    BOOL WINAPI PingRemoteServer
                    (HWND                       hOwnerWnd,
                     LPSPropValue               pProps);
    BOOL WINAPI IsValidServerName
                    (LPTSTR                     szServerName);
    HRESULT WINAPI DoServerLogon
                    (PMAILBOX_INFO              pMBInfo,
                     LPSPropValue               pProps,
                     LPPROFSECT                 pProfileObj,
                     BOOL                       fUIAllowed,
                     HWND                       hOwnerWnd,
                     BOOL                       fSetNotification,
                     DWORD                      dwNotifMask,
                     DWORD *                    pdwConnectID,
                     BOOL                       fOffLineOK);
    HRESULT WINAPI ChangeMBServerPassword
                    (LPTSTR                     szServerName,
                     LPTSTR                     szMailboxName,
                     LPTSTR                     szOldPassword,
                     LPTSTR                     szNewPassword);
    HRESULT WINAPI TerminateServerNotifications
                    (LPTSTR                     szServerName,
                     DWORD                      dwConnectionID);
    HRESULT WINAPI ResetServerNotifications
                    (LPTSTR                     szServerName,
                     LPTSTR                     szMailboxName,
                     DWORD                      dwNotifMask,
                     DWORD *                    pdwConnectionID);
    void WINAPI TerminateRemoteConnections
                    ();
    HRESULT WINAPI MergeConfigProps
                    (ULONG                      ulCfgProps,
                     LPSPropValue               pCfgProps,
                     LPSPropValue               pLogonProps);
    HRESULT WINAPI OpenServiceProfileSection
                    (LPMAPISUP                  pSupObj,
                     LPPROFSECT *               ppProfSectObj,
                     LPFREEBUFFER               pfnFreeBuffer);
    // Transport Providers
    HRESULT WINAPI OpenServerUploadPipe
                    (LPTSTR                     szServerName,
                     LPTSTR                     szSenderMailbox,
                     HANDLE                     hMsgFile,
                     LPTSTR                     szConnectionInfo,
                     BOOL *                     pfErrorInTheServer);
    HRESULT WINAPI SendMsgToAccount
                    (LPTSTR                     szServerName,
                     LPTSTR                     szRecipMailbox,
                     LPTSTR                     szHeader,
                     LPTSTR                     szConnectionInfo,
                     BOOL *                     pfErrorInTheServer);
    HRESULT WINAPI FinishUploadConnection
                    (LPTSTR                     szServerName,
                     LPTSTR                     szConnectionInfo);
    HRESULT WINAPI GetHeadersFromServer
                    (LPTSTR                     szServerName,
                     LPTSTR                     szMailbox,
                     LPTSTR                     szHeaderFile);
    HRESULT WINAPI OpenRemoteServerDownLoadPipe
                    (LPTSTR                     szServer,
                     LPTSTR                     szMBox,
                     HANDLE *                   phPipe);
    HRESULT WINAPI GetNextMailboxMsg
                    (LPTSTR                     szServer,
                     LPTSTR                     szMailbox,
                     LPTSTR                     szLocalMsgFile);
    HRESULT WINAPI CheckForPendingMessages
                    (LPTSTR                     szServerName,
                     LPTSTR                     szMailboxName,
                     ULONG *                    pulMsgWaiting);
};

// CTL3D Dynamic Loading Support
void WINAPI PrivInitialize3DCtl
                    (HINSTANCE                  hInstance);
void WINAPI PrivUninitialize3DCtl
                    (HINSTANCE                  hInstance);

// Forward declarations
class CUIMAPIProp;

// Custom callback function prototype for the implementation
// and activation of IMAPIControl on display tables on property sheets.
typedef void (STDAPICALLTYPE BUTTONCALLBACK) (CUIMAPIProp * pObj, HWND hOwnerWnd, HINSTANCE hInstance);
typedef BUTTONCALLBACK *LPBUTTONCALLBACK;

class CUIMAPIProp : public IMAPIProp
{
public:
    inline STDMETHODIMP QueryInterface
                    (REFIID                     riid,
                     LPVOID *                   ppvObj)
                    { *ppvObj = NULL;
                      HRESULT hResult = S_OK;
                      EnterCriticalSection (&m_csObj);
                      if (riid == IID_IMAPIProp || riid == IID_IUnknown)
                      {
                          *ppvObj = (LPVOID)this;
                          AddRef();
                      }
                      else
                      {
                          hResult = E_NOINTERFACE;
                      }
                      LeaveCriticalSection (&m_csObj);
                      return hResult; }
    inline STDMETHODIMP_(ULONG) AddRef
                    () { EnterCriticalSection (&m_csObj);
                         ++m_cRef; ULONG ulCount = m_cRef;
                         LeaveCriticalSection (&m_csObj);
                         return ulCount; }
    inline STDMETHODIMP_(ULONG) Release
                    () { EnterCriticalSection (&m_csObj);
                         ULONG ulCount = --m_cRef;
                         LeaveCriticalSection (&m_csObj);
                         if (!ulCount) { delete this; }
                         return ulCount;}
    inline STDMETHODIMP GetLastError
                    (HRESULT                    hResult,
                     ULONG                      ulFlags,
                     LPMAPIERROR *              ppMAPIError)
                    { return m_pImpObj->GetLastError (hResult, ulFlags, ppMAPIError); }
    inline STDMETHODIMP SaveChanges
                    (ULONG                      ulFlags)
                    { return S_OK; }
    inline STDMETHODIMP GetProps
                    (LPSPropTagArray            pPropTagArray,
                     ULONG                      ulFlags,
                     ULONG *                    pcValues,
                     LPSPropValue *             ppPropArray)
                    { return m_pImpObj->GetProps (pPropTagArray, ulFlags, pcValues, ppPropArray); }
    inline STDMETHODIMP GetPropList
                    (ULONG                      ulFlags,
                     LPSPropTagArray *          ppAllTags)
                    { return m_pImpObj->GetPropList (ulFlags, ppAllTags); }
    STDMETHODIMP OpenProperty
                    (ULONG                      ulPropTag,
                     LPCIID                     piid,
                     ULONG                      ulInterfaceOptions,
                     ULONG                      ulFlags,
                     LPUNKNOWN *                ppUnk);
    inline STDMETHODIMP SetProps
                    (ULONG                      cValues,
                     LPSPropValue               pPropArray,
                     LPSPropProblemArray *      ppProblems)
                    { return m_pImpObj->SetProps (cValues, pPropArray, ppProblems); }
    inline STDMETHODIMP DeleteProps
                    (LPSPropTagArray            pPropTagArray,
                     LPSPropProblemArray *      ppProblems)
                    { return m_pImpObj->DeleteProps (pPropTagArray, ppProblems); }
    inline STDMETHODIMP CopyTo
                    (ULONG                      ciidExclude,
                     LPCIID                     rgiidExclude,
                     LPSPropTagArray            pExcludeProps,
                     ULONG                      ulUIParam,
                     LPMAPIPROGRESS             pProgress,
                     LPCIID                     pInterface,
                     LPVOID                     pDestObj,
                     ULONG                      ulFlags,
                     LPSPropProblemArray *      ppProblems)
                    { return m_pImpObj->CopyTo (ciidExclude, rgiidExclude, pExcludeProps, ulUIParam, pProgress, pInterface, pDestObj, ulFlags, ppProblems); }
    inline STDMETHODIMP CopyProps
                    (LPSPropTagArray            pIncludeProps,
                     ULONG                      ulUIParam,
                     LPMAPIPROGRESS             pProgress,
                     LPCIID                     pInterface,
                     LPVOID                     pDestObj,
                     ULONG                      ulFlags,
                     LPSPropProblemArray *      ppProblems)
                    { return m_pImpObj->CopyProps (pIncludeProps, ulUIParam, pProgress, pInterface, pDestObj, ulFlags, ppProblems); }
    inline STDMETHODIMP GetNamesFromIDs
                    (LPSPropTagArray *          ppPropTags,
                     LPGUID                     pPropSetGuid,
                     ULONG                      ulFlags,
                     ULONG *                    pcPropNames,
                     LPMAPINAMEID **            pppPropNames)
                    { return MAPI_E_NO_SUPPORT; }
    inline STDMETHODIMP GetIDsFromNames
                    (ULONG                      cPropNames,
                     LPMAPINAMEID *             ppPropNames,
                     ULONG                      ulFlags,
                     LPSPropTagArray *          ppPropTags)
                    { return MAPI_E_NO_SUPPORT; }

///////////////////////////////////////////////////////////////////////////////
// Other methods private to this class
//
    inline void WINAPI SetTableData
                    (LPTABLEDATA                pTable)
                    { m_pTableData = pTable; m_pTableData->AddRef(); }
    inline void WINAPI SetPropCallBack
                    (ULONG                      ulPropTag,
                     LPBUTTONCALLBACK           pfnCallBack)
                    { m_ulCustomPropTag = ulPropTag; m_pfnCallBack = pfnCallBack; }

///////////////////////////////////////////////////////////////////////////////
// Constructors and destructors
//
public :
    CUIMAPIProp     (HINSTANCE                  hInstance,
                     LPALLOCATEBUFFER           pfnAllocateBuffer,
                     LPALLOCATEMORE             pfnAllocateMore,
                     LPFREEBUFFER               pfnFreeBuffer,
                     BOOL                       fReadOnly);
    ~CUIMAPIProp    () { m_pImpObj->Release(); m_pImpObj = NULL;
                         if (m_pTableData) { m_pTableData->Release(); }
                         DeleteCriticalSection (&m_csObj);}

///////////////////////////////////////////////////////////////////////////////
// Data members
//
private :
    ULONG               m_cRef;
    BOOL                m_fReadOnly;
    LPPROPDATA          m_pImpObj;
    ULONG               m_ulCustomPropTag;
    LPBUTTONCALLBACK    m_pfnCallBack;
    CRITICAL_SECTION    m_csObj;
public :
    LPTABLEDATA         m_pTableData;
    HINSTANCE           m_hInstance;
};

class CMAPIControl : public IMAPIControl
{
    inline STDMETHODIMP QueryInterface
                    (REFIID                     riid,
                     LPVOID *                   ppvObj)
                    { *ppvObj = NULL;
                      HRESULT hResult = S_OK;
                      EnterCriticalSection (&m_csObj);
                      if (riid == IID_IMAPIControl || riid == IID_IUnknown)
                      {
                          *ppvObj = (LPVOID)this;
                          // Increase usage count of this object
                          AddRef();
                      }
                      else
                      {
                          hResult = E_NOINTERFACE;
                      }
                      LeaveCriticalSection (&m_csObj);
                      return hResult; }
    inline STDMETHODIMP_(ULONG) AddRef
                    () { EnterCriticalSection (&m_csObj);
                         ++m_cRef; ULONG ulCount = m_cRef;
                         LeaveCriticalSection (&m_csObj);
                         return ulCount; }
    inline STDMETHODIMP_(ULONG) Release
                    () { EnterCriticalSection (&m_csObj);
                         ULONG ulCount = --m_cRef;
                         LeaveCriticalSection (&m_csObj);
                         if (!ulCount) { delete this; }
                         return ulCount;}
    inline STDMETHODIMP GetLastError
                    (HRESULT                    hResult,
                     ULONG                      ulFlags,
                     LPMAPIERROR *              ppMAPIError)
                    { return MAPI_E_NO_SUPPORT; }
    inline STDMETHODIMP GetState
                    (ULONG                      ulFlags,
                     ULONG *                    pulState)
                    { *pulState = (m_fReadOnly ? MAPI_DISABLED : MAPI_ENABLED); return S_OK; }
    inline STDMETHODIMP Activate
                    (ULONG                      ulFlags,
                     ULONG                      ulUIParam)
                    { (*m_pfnCallBack)(m_pProp, (HWND)ulUIParam, m_pProp->m_hInstance); return S_OK; }

///////////////////////////////////////////////////////////////////////////////
// Constructors and destructors
//
public :
    CMAPIControl    (CUIMAPIProp *              pProp,
                     LPBUTTONCALLBACK           pfnCallBack,
                     BOOL                       fReadOnly)
                    { m_cRef = 1; m_pfnCallBack = pfnCallBack; m_fReadOnly = fReadOnly;
                      m_pProp = pProp; m_pProp->AddRef();
                      InitializeCriticalSection (&m_csObj); }
    ~CMAPIControl   ()
                    { m_pProp->Release();
                      DeleteCriticalSection (&m_csObj);}
///////////////////////////////////////////////////////////////////////////////
// Data members
//
private :
    ULONG               m_cRef;
    LPBUTTONCALLBACK    m_pfnCallBack;
    BOOL                m_fReadOnly;
    CUIMAPIProp *       m_pProp;
    CRITICAL_SECTION    m_csObj;
};

#endif // _COMWINDS_H

// End of file for COMWINDS.H
