// varmap16.h: 16-bit version of CMapVariantToVariant
//
// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (C) 1992-1993 Microsoft Corporation
// All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and Microsoft
// QuickHelp and/or WinHelp documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

/////////////////////////////////////////////////////////////////////////////

class CMapVariantToVariant : public CObject
{

	DECLARE_SERIAL(CMapVariantToVariant)
protected:
	// Association
	struct CAssoc
	{
		CAssoc* pNext;
		UINT nHashValue;  // needed for efficient iteration
		CVariant key;
		CVariant value;
	};
public:

// Construction
	CMapVariantToVariant(int nBlockSize=10);

// Attributes
	// number of elements
	int GetCount() const;
	BOOL IsEmpty() const;

	// Lookup
	BOOL Lookup(LPCVARIANT key, CVariant& rValue) const;

// Operations
	// Lookup and add if not there
	CVariant& operator[](LPCVARIANT key);

	// add a new (key, value) pair
	void SetAt(LPCVARIANT key, LPCVARIANT newValue);

	// removing existing (key, ?) pair
	BOOL RemoveKey(LPCVARIANT key);
	void RemoveAll();

	// iterating all (key, value) pairs
	POSITION GetStartPosition() const;
	void GetNextAssoc(POSITION& rNextPosition, CVariant& rKey, CVariant& rValue) const;

	// advanced features for derived classes
	UINT GetHashTableSize() const;
	void InitHashTable(UINT hashSize);

// Implementation
protected:
	CAssoc** m_pHashTable;
	UINT m_nHashTableSize;
	int m_nCount;
	CAssoc* m_pFreeList;
	struct CPlex* m_pBlocks;
	int m_nBlockSize;

	CAssoc* NewAssoc();
	void FreeAssoc(CAssoc*);
	CAssoc* GetAssocAt(LPCVARIANT, UINT&) const;

public:
	~CMapVariantToVariant();

	void Serialize(CArchive&);
#ifdef _DEBUG
	void Dump(CDumpContext&) const;
	void AssertValid() const;
#endif
};

inline int CMapVariantToVariant::GetCount() const
	{ return m_nCount; }
inline BOOL CMapVariantToVariant::IsEmpty() const
	{ return m_nCount == 0; }
inline void CMapVariantToVariant::SetAt(LPCVARIANT key, LPCVARIANT newValue)
	{ (*this)[key] = newValue; }
inline POSITION CMapVariantToVariant::GetStartPosition() const
	{ return (m_nCount == 0) ? NULL : BEFORE_START_POSITION; }
inline UINT CMapVariantToVariant::GetHashTableSize() const
	{ return m_nHashTableSize; }
