/* Intel 386(TM)/486(TM) C Code Builder(TM) Kit
 * Copyright 1991 Intel Corporation.  All Rights Reserved.
 * project.c - 3-D projection calculations.
 * $Version: 1.3 $
 */

/*
 *  This file contains the functions that compute the 3-D to 2-D perspective
 *  retaining projections.  Given the size of the object to be viewed and the
 *  position of the eye, a set of constants are computed.  From then on the
 *  perspective transformation can be fed 3-D coordinates and it will produce
 *  2-D coordinates.
 *
 *  There are several functions in this file.  They are placed here so they
 *  can share some global data without any other program being able to
 *  manipulate the data.
 *
 *  These functions were written from the book
 *       Programming Principles in Computer Graphics, by Leendert Ammeraal,
 *       printed by John Wiley & Sons, Ltd., 1986, ppg 60-75.
 *  However, these equations all do a translation.  The graphics library
 *  being used will do this computation with proper definition of the
 *  viewport and graphics port.
 */

#include <stdio.h>
#include <math.h>
#include "debug.h"
#include "points.h"
#include "bool.h"

/*
 *  Global data consists of the rotation/translation transformation matrix.
 *  This is the rotation/translaion matrix.  It is represented as several
 *  discreet elements rather than one matrix because only part of the
 *  matrix is needed, and to allow faster execution of the compiled code.
 */
static double v11, v12, v13, v21, v22, v23, v32, v33, v43;

/*
 *  Save the eye coordinates.  These are used in hidden line removal to
 *  construct a plane perpendicular to the "Z" axis of the eye and
 *  passing through the objects center (0, 0, 0).
 */
static double eye_x, eye_y, eye_z;

/*
 *  Also needed is a scaling factor computed from the size of the screen
 *  and the size of the object being viewed.
 */
static double screen_dist;


/*
 *  For hidden line removal, the caller needs to know the distance from
 *  the eye to the origin of the object.
 */
static double rho;            /* Distance of eye from viewpoint (0,0,0) */



/*
 *  This function allows the caller to set the viewing coordinates.  It
 *  assumes that the screen is blank, that the object has yet to be
 *  drawn.  Given the eye coordinate, assuming the eye is looking toward
 *  (0, 0, 0) of the object, compute the rotation/translation matrix.
 */

void InitProject (x, y, z, scrn_size, obj_size)
    double x, y, z;        /* The new eye coordinates */
    double scrn_size;      /* The size of the screen */
    double obj_size;       /* The size of the object */
{
    double theta;          /* Angle of eye from the X axis */
    double phi;            /* Angle of eye from the Z axis */
    double sinth;          /* Sine of theta */
    double costh;          /* Cosine of theta */
    double sinph;          /* Sine of phi */
    double cosph;          /* Cosine of phi */

    /*
     *  Save the eye coordinates in global memory.
     */
    eye_x = x;
    eye_y = y;
    eye_z = z;

    /*
     *  Compute the distances.
     */
    rho  = sqrt (x*x + y*y + z*z);

    /*
     *  Compute the angles.
     */
    phi   = acos (z/rho);
    theta = acos (x/(rho*sin(phi)));
    if (y < 0) theta = -theta;

    /*
     *  Compute the sines and cosines.  These are constant for the matrix
     *  computations so only compute them once.
     */
    costh = cos(theta);
    sinth = sin(theta);
    cosph = cos(phi);
    sinph = sin(phi);

    /*
     *  Compute the elements of the matrix.
     */
    v11 = -sinth;  v12 = -cosph * costh;  v13 = -sinph * costh;
    v21 =  costh;  v22 = -cosph * sinth;  v23 = -sinph * sinth;
                   v32 =  sinph;          v33 = -cosph;
                                          v43 =  rho;

    /*
     *  Compute the distance of the eye to the screen.
     */
    screen_dist = rho * scrn_size / obj_size;

#ifdef DEBUG
    fprintf (fd, "***************** Init *****************\n");
    fprintf (fd, "\n\nMatrix:\n");
    fprintf (fd, "    v11 = %f, v12 = %f, v13 = %f\n", v11, v12, v13);
    fprintf (fd, "    v21 = %f, v22 = %f, v23 = %f\n", v21, v22, v23);
    fprintf (fd, "              v32 = %f, v33 = %f\n", v32, v33);
    fprintf (fd, "                        v43 = %f\n", v43);
    fprintf (fd, "    Dist = %f\n", screen_dist);
    fflush (fd);
#endif
}




/*
 *  This next function does the actual mapping for a point.  It takes
 *  a point in three space and translates it to a point in two space.
 */
void Project (x, y, z, nx, ny)
    double x, y, z;            /* The point in three space */
    double *nx, *ny;           /* The project point in two space */
{
    double xe, ye, ze;         /* The point in eye coordinates */

#ifdef DEBUG
    fprintf (fd, "************* Project **************\n");
    fprintf (fd, "Matrix:\n");
    fprintf (fd, "    v11 = %f, v12 = %f, v13 = %f\n", v11, v12, v13);
    fprintf (fd, "    v21 = %f, v22 = %f, v23 = %f\n", v21, v22, v23);
    fprintf (fd, "              v32 = %f, v33 = %f\n", v32, v33);
    fprintf (fd, "                        v43 = %f\n", v43);
    fprintf (fd, "    Dist = %f\n", screen_dist);
    fprintf (fd, "Points are (%f, %f, %f)\n", x, y, z);
#endif
    /*
     *  Translate from world coordinates to eye coordinates.
     */
    xe = v11*x + v21*y;
    ye = v12*x + v22*y + v32*z;
    ze = v13*x + v23*y + v33*z + v43;


    /*
     *  Translate to screen coordinates.  Don't do the extra scaling
     *  and translation.
     */
    *nx = screen_dist * xe / ze;
    *ny = screen_dist * ye / ze;

#ifdef DEBUG
    fprintf (fd, "Translated to eye coords are (%f, %f, %f)\n", xe, ye, ze);
    fprintf (fd, "Translated to screen coordinates are (%f, %f)\n", *nx, *ny);
    fflush (fd);
#endif
}




/**************************************************************************
 *
 *  Project points and record hidden line information.
 *
 *  Hidden line removal is somewhat simple, but only works on "nicely
 *  behaved" objects: objects that don't have any "bumps" or discontinuities.
 *  I.e. it only works for single squares, elipses, triangles, etc.
 *
 *  Each point of the object is projected.  Then a simple test is performed
 *  to tell if the point can be seen or not: does the point lie further
 *  away from the eye than the center of the object or closer.  If the
 *  point is closer then it can be seen, but if it is futher away then
 *  it cannot be seen.  This is equivalent to slicing the object with a
 *  plane perpendicular to the "Z" axis of the eye coordinate system, but
 *  passing through the center of the object.  Only half of the points can
 *  be seen at any one time.
 *
 *  This works well, except that the edges are a little jagged.  Extra
 *  processing is needed to find the exact intersection with the
 *  perpendicular plane.
 *
 *  This next function maps an entire set of points at once, and records
 *  whether the points can be seen (they are less than or equal to the
 *  distance to the origin) or cannot be seen (greater than the distance
 *  to the origin).
 *
 *  Note, speed is critical for this function.
 */

void HiddenProject (points, proj, seen, count)
    THREE_D register *points;   /* The set of 3-D points */
    TWO_D   register *proj;     /* The resulting set of 2-D points */
    int     register *seen;     /* The resulting vector of seen/not seen */
    int               count;    /* The number of points to project */
{
    double register xe, ye, ze;    /* The point in eye coordinates */
    THREE_D register *last_addr;   /* The last points address */
    double register x, y, z;       /* Coordinates of one 3-D point */
    double register eye_to_scrn;   /* Distance from eye to screen */
    double register eye_to_orig;   /* Distance from eye to origin */
    double register factor;        /* Factor to be applied to Xe and Ye */

#ifdef DEBUG
    fprintf (fd, "************* Hidden Project **************\n");
    fprintf (fd, "Matrix:\n");
    fprintf (fd, "    v11 = %f, v12 = %f, v13 = %f\n", v11, v12, v13);
    fprintf (fd, "    v21 = %f, v22 = %f, v23 = %f\n", v21, v22, v23);
    fprintf (fd, "              v32 = %f, v33 = %f\n", v32, v33);
    fprintf (fd, "                        v43 = %f\n", v43);
    fprintf (fd, "    Dist = %f\n", screen_dist);
    fprintf (fd, "Points are (%f, %f, %f)\n", x, y, z);
#endif

    /*
     *  Go through every point in the points array.  Use pointer
     *  arithmetic for all addressing (to speed up this code).
     */
    eye_to_scrn = screen_dist;
    eye_to_orig = rho;
    last_addr = &(points [count]);
    while (points < last_addr) {

        /*
         *  Get the coordinates of the 3-D point.
         */
        x = points->x;
        y = points->y;
        z = points->z;
        points++;

        /*
         *  Translate from world coordinates to eye coordinates.
         */
        xe = v11*x + v21*y;
        ye = v12*x + v22*y + v32*z;
        ze = v13*x + v23*y + v33*z + v43;

        /*
         *  Translate to screen coordinates.  Don't do the extra
         *  coordinate translation.
         */
        factor  = eye_to_scrn / ze;
        proj->x = factor * xe;
        proj->y = factor * ye;
        proj++;

        /*
         *  Record if the point can be seen or not.
         */
        *seen = (ze <= eye_to_orig);
        seen++;
    };
}



/******************************************************************************
 *
 *  Compute the last point of a line segment that can be seen.  This requires
 *  a bit of analytic geometry and relies heavily on the hidden line removal
 *  constraints (about the shape of the object).
 *
 *  Envision the object sliced by a plane that goes through the object origin
 *  and is perpendicular to the "Z" axis of the eye coordinate system.  Such
 *  a plane serves to seperate the points that can be seen (those closer to
 *  the eye than the plane) from those that cannot be seen (those further
 *  from the eye than the plane).  The purpose of this hidden line stuff is
 *  to only plot points that can be seen, and to not display points that
 *  can be seen.
 *
 *  This function is called with a line segment that crosses the plane.
 *  One point of the segment (it doesn't matter which) can be seen and the
 *  other cannot be seen.  We need to find out how much of the segment can
 *  be seen; or, in other words, where does the segment intersect the plane.
 *
 *  The solution to this problem has three steps:
 *       1) Describe the plane by an equation
 *       2) Describe the line by an equation
 *       3) Solve the resuting system of linear equations to find the
 *          intersection.
 *  All of the equation derivation stuff is in this header.  The resulting
 *  formulas (pretty messy) are implemented in the "C" code.
 *
 *  Step 1, describing the plane, is simple.  The equation of a 3-D plane
 *  is Ax + By + Cz = D.  Since the plane must go through the origin, the
 *  D value becomes zero.  So how do we get the A, B and C values?  Well,
 *  if you express the eye coordinate as a vector, it is simply
 *            eye_x*i + eye_y*j + eye_z*k
 *  where i, j and k are unit vectors.  Then (by some magic of vector
 *  properities) the plane perpendicular to this vector has the same
 *  coefficients as the vector, or
 *            eye_x*x + eye_y*y + eye_z+z = 0.
 *  As a solution, then, the equation of the plane is easily derived from
 *  the eye coordinates.
 *
 *  Step 2, describing the intersecting line, is almost as simple.  It
 *  is just a set of linear equations:
 *
 *             (x - x1)   (y - y1)   (z - z1)
 *             -------- = -------- = --------
 *                A          B          C
 *
 *  Where x1, y1 and z1 are the coordinates of one of the points the line
 *  passes through, and A, B and C are the coefficients of the vector
 *  parellel to the line (given by the difference between the vectors
 *  defined by two points on the line):
 *
 *              A = x2 - x1
 *              B = y2 - y1
 *              Z = z2 - z1
 *
 *  So, since the two endpoints of the segment are known, they can be
 *  used to derive the equation of the line passing through them.
 *
 *  The hard part (requiring lots of algebraic scribbling) is to find
 *  a closed form of the solution between the line and the plane.  I
 *  worked it all out (and even checked my results), and I get:
 *
 *  C0 = eye_z * C / B
 *  C1 = eye_y + C0
 *  C2 = B / A
 *  C3 = C1 * C2
 *
 *      C3*x1 + (C0-C1)*y1 - eye_z*z1
 *  xi = -----------------------------
 *              eye_x + C3
 *
 *  yi = C2 * (x-x1) + y1
 *
 *      - eye_x*x - eye_y*y
 *  zi = -------------------
 *             eye_z
 *
 *  Note that 1.0/eye_z can be computed when the eye coordinates are
 *  set.  Unfortunately, none of the other computations can be pre-
 *  computed since they depend on the equation of the line.
 */

#define EPSILON 1.0e-10           /* For checking for zero */

ComputeCrossing (p1, p2, result)
    THREE_D *p1;                 /* Start of segment */
    THREE_D *p2;                 /* End of segment */
    THREE_D *result;             /* Point of intersection */
{

    double A, B, C;              /* Coefficients of the segment */
    double C0, C1, C2, C3;       /* Intermediate values */
    bool   A_zero, B_zero, C_zero; /* TRUE if A, B or C is close to zero */

    /*
     *  Compute the coefficients of the line passing through the two
     *  points.
     */
    A_zero = FALSE;
    A = p2->x - p1->x;
    if (fabs(A) < EPSILON)
        A_zero = TRUE;

    B_zero = FALSE;
    B = p2->y - p1->y;
    if (fabs(B) < EPSILON)
        B_zero = TRUE;

    C_zero = FALSE;
    C = p2->z - p1->z;
    if (fabs(C) < EPSILON)
        C_zero = TRUE;

    /*
     *  For the general equation these coefficients are used as
     *  denominators.  Since they cannot be zero for the division we
     *  must have special code to handle the zero cases.  Note that
     *  if A_zero then p2->x == p1->x, so the "X" intersection point
     *  is known.  So, cases where A, B or C are zero are degenerate
     *  cases.  Handle each of the degenerate cases.  Note that the
     *  choosen order of processing minimizes the number of tests.
     */
    if ((A_zero) && (B_zero) && (C_zero)) {

        /*
         *  This only happens when the two endpoints are equal.
         *  Return either endpoint.
         */
        result->x = p1->x;
        result->y = p1->y;
        result->z = p1->z;

    } else if ((A_zero) && (B_zero)) {

        /*
         *  p1->x == p2->x and p1->y == p2->y.  Set up the result, then
         *  solve for z from the plane equation.
         */
        result->x = p1->x;
        result->y = p1->y;

        if (fabs(eye_z) < EPSILON) {
            result->z == 0;
        } else {
            result->z = (- (eye_x * result->x) - (eye_y * result->y)) /
                                         eye_z;
        };

    } else if ((A_zero) && (C_zero)) {

        /*
         *  p1->x == p2->x and p1->z == p2->z.  Set up the result, then
         *  solve for y from the plane equation.
         */
        result->x = p1->x;
        result->z = p1->z;

        if (fabs(eye_y) < EPSILON) {
            result->y = 0;
        } else {
            result->y = (- (eye_x * result->x) - (eye_z * result->z)) /
                                        eye_y;
        };

    } else if ((B_zero) && (C_zero)) {

        /*
         *  p1->y == p2->y and p1->z == p2->z.  Set up the result, then
         *  solve for x from the plane equation.
         */
        result->y = p1->y;
        result->z = p1->z;

        if (fabs(eye_x) < EPSILON) {
            result->x = 0;
        } else {
            result->x = (- (eye_y * result->y) - (eye_z * result->z)) /
                                            eye_x;
        };

    } else if (A_zero) {

        /*
         *  p1->x == p2->x.  Use this to simplify solving for y and z.
         */
        result->x = p1->x;

        C0 = eye_z * C / B;
        result->y = ((- eye_x*p1->x) + C0*p1->y - eye_z*p1->z) /
                                (eye_y + C0);

        if (fabs(eye_z) < EPSILON) {
            result->z = 0;
        } else {
            result->z = (- (eye_x * result->x) - (eye_y * result->y)) /
                                            eye_z;
        };

    } else if (B_zero) {

        /*
         *  p1->y == p2->y.  Use this to simplify solving for x and z.
         */
        result->y = p1->y;

        C0 = eye_z * C / A;
        result->x = ((- eye_y*p1->y) + C0*p1->x - eye_z*p1->z) /
                                (eye_x + C0);


        if (fabs(eye_z) < EPSILON) {
            result->z = 0;
        } else {
            result->z = (- (eye_x * result->x) - (eye_y * result->y)) /
                                            eye_z;
        };

    } else if (C_zero) {

        /*
         *  p1->z == p2->z.  Use this to simplify solving for x and y.
         */
        result->z = p1->z;

        C0 = eye_y * B / A;
        result->x = ((- eye_z*p1->z) + C0*p1->x - eye_y*p1->y) /
                                (eye_x + C0);

        if (fabs(eye_y) < EPSILON) {
            result->y = 0;
        } else {
            result->y = (- (eye_x * result->x) - (eye_z * result->z)) /
                                             eye_y;
        };

    } else {

        /*
         *  The coefficients of the plane are already computed (the eye
         *  coordinates).  Set up the intermediate values that will be
         *  used to find the intersection.
         */
        C0 = eye_z * C / B;
        C1 = eye_y + C0;
        C2 = B / A;
        C3 = C1 * C2;

        /*
         *  Find the intersection.  The justification for these equations
         *  is found in the header for this routine.
         */
        result->x = (C3*p1->x + (C0-C1)*p1->y - eye_z*p1->z) /
                              (eye_x + C3);

        result->y = C2*(result->x - p1->x) + p1->y;

        if (fabs(eye_z) < EPSILON) {
            result->z = 0;
        } else {
            result->z = (- (eye_x * result->x) - (eye_y * result->y)) /
                                             eye_z;
        };

    };
    /*
     *  OK, that's it.
     */
}
