/* Intel 386(TM)/486(TM) C Code Builder(TM) Kit
 * Copyright 1991 Intel Corporation.  All Rights Reserved.
 * memsize.c - Graph memory size functions.
 * $Version: 1.6 $
 */

/*
 *  These functions graph the size of memory using a 3-D coordinate system.
 *  Since the memory size is a single variable function, two of the dimensions
 *  are wasted.  It is just a fancy way to graph.
 */

#include <stdio.h>
#include <graph.h>
#include <stdlib.h>
#include <sys\types.h>
#include <malloc.h>
#include <math.h>
#include "srnchr.h"
#include "const.h"
#include "bool.h"
#include "poly.h"
#include "debug.h"



/*
 *  The graph is divided into three regions:
 *    REAL    : 0 -> extended (at 640k)
 *    EXTEND  : extended -> virtual (the rest of physical ram)
 *    VIRTUAL : everything else up to a maximum
 *
 *  The ranges are needed for both memory values (long integers) and
 *  their place on the "Y" axis of the graph.  The graph is NOT
 *  linear, and these ranges define the scale values.
 *
 *  Some of the memory values are computed (such as the amount of
 *  physical memory).  These are static variables.  Non-computed values
 *  are defined.  All of the "Y" values are defined.
 *
 *  In addition, the maximum graph value on the "Y" axis must be
 *  specified.
 */

#define MAX_Y    6000000.0                     /* Maximum "Y" value to graph */
#define MAX_MEM  (((long) 1024) * (long) 1024 * (long) 6)

#define REAL_MEM ((640 * ((long) 1024)))                /* Real memory limit */
#define REAL_Y   (MAX_Y * 0.30)                     /* Real memory "Y" value */

static  long extend_mem = 0;                        /* Extended memory limit */
#define EXTEND_Y (MAX_Y * 0.70)                 /* Extended memory "Y" value */

#define VIRTUAL_MEM (((long) 1024) * (long) 1024 * (long) 100)

#if LARGE_TEST
#define REAL_MEM ((long) 1024 * 64)
#define MAX_MEM  (((long) 1024) * (long) 256)
#define VIRTUAL_MEM (((long) 1024) * (long) 352)
#endif


/*
 *  Also needed is the eye coordinate viewpoint.  These should be
 *  expressed in terms of the maximum Y value plotted, so the view
 *  is the same if the graph gets bigger.
 */

#define VIEW_X (6   * MAX_Y)                            /* View X coordinate */
#define VIEW_Y (0.5 * MAX_Y)                            /* View Y coordinate */
#define VIEW_Z (3.0 * MAX_Y)                            /* View Z coordinate */

#define SCREEN_SIZE 1.0                       /* Size of the graphics window */

#define LOW_CHUNK (size_t) (0x8000)        /* Maximum size of one allocation */

#if LARGE_TEST
#define LOW_CHUNK (size_t) (0x0100)
#endif

#define TOTAL_WAIT     5000                         /* Milliseconds for demo */
#define ALLOC_WAIT     500               /* Milliseconds between allocations */

extern long GetMilli ();


/************************************************************************
 *  This is the function that controls everything.  It starts up the
 *  screen, then allocates memory and plots the allocation in a graph.
 *  There is a "barrier", a plane, at the 640k barrier.  If the bytes
 *  of memory exceed 640k the barrier flashes.
 *
 *  Memory is allocated in large chunks (of size LOW_CHUNK) until the
 *  barrier is reached, then the allocation size doubles each time
 *  memory is allocated.  The first word of a chunk of allocated memory
 *  is set to point to the last allocated chunk.  In this way the
 *  allocated memory can be freed at the end of the demo.  When the
 *  demo runs again all the memory is again available.
 */

MemorySize (scrn)
SCREEN_CHAR *scrn;                             /* The screen characteristics */
   {
   long mem_size;                     /* The total number of bytes allocated */
   size_t chunk_size;                           /* Size of the current chunk */
   char **last_chunk;                    /* The first memory chunk allocated */
   char **chunk;                           /* Current memory chunk allocated */
   long demo_start;                            /* Start time in milliseconds */
   long demo_end;                                /* End time in milliseconds */
   bool crossed;                          /* TRUE if already crossed barrier */
   size_t req_size;                                  /* Requested chunk size */
   long alloc_start;                         /* Start of a memory allocation */
   long alloc_end;                             /* End of a memory allocation */
   double mill_per_elem;               /* Seconds per element of computation */

   /*
    *  Get the current time.  This will be used to make sure that
    *  graphing takes a particular amount of time.
    */
   demo_start = GetMilli ();

   /*
    *  Get the screen ready to draw things for the memory plot.  The
    *  window and viewport must be defined.
    */
   InitMemScreen (scrn);

   /*
    *  Set the eye coordinates for viewing and tell the screen and object
    *  size.
    */
   InitProject (VIEW_X, VIEW_Y, VIEW_Z, SCREEN_SIZE*0.90, MAX_Y*0.33);

   #ifdef LARGE_TEST
      extend_mem = (long) 1024 * (long) 128;
   #else
      extend_mem = (long) 1024 * (long) 1024 * 4;
   #endif

   /*
    *  Put up the initial screen: the axis and barrier plane.
    *  Note that all of these "constant" objects are fully defined
    *  by constant later in the file.
    */
   DrawAxis (TRUE);
   DrawBarrier (TRUE);

   #ifdef DEBUG
      fprintf (fd, "Memory MAX = %ld\n", VIRTUAL_MEM);
      fprintf (fd, "Allocation size = %u\n", LOW_CHUNK);
      fflush (fd);
   #endif

   /*
    *  This is the interesting loop.  Allocate memory in large chunks
    *  until one of two conditions:
    *     1) We run out of memory
    *     2) We hit the "large" limit
    *  Always allocate a "big" chunk, but take the amount returned by
    *  the operating system.  If we get passed the 640k barrier,
    *  start allocating larger and larger chunks.
    *
    *  Save pointers to the allocated memory in the "pieces" array.
    */
   req_size = LOW_CHUNK;
   crossed  = FALSE;
   alloc_start = GetMilli ();
   chunk    = (char **) malloc (req_size);
   if (chunk == NULL) {
      ShutDownDemo ( scrn );
      printf ("Could not allocate initial chunk size %d\n", req_size);
      exit (0);
      }
   chunk_size = _msize (chunk);

   last_chunk = NULL;
   mem_size   = 0;
   while ((mem_size < VIRTUAL_MEM) && (chunk_size > 0)) {
      #ifdef DEBUG
         fprintf (fd, "\n\n");
         fprintf (fd, "Memory size = %ld, allocated %u\n",
            mem_size, chunk_size);
         fflush (fd);
      #endif

      /*
       *  Compute over the entire memory that was allocated.  Time
       *  the computations for display.
       */
      ComputeMem (chunk, chunk_size);
      alloc_end   = GetMilli ();
      mill_per_elem = (double) (((double) (alloc_end - alloc_start))
                                     / (double) chunk_size);

      /*
       *  Update the graph.  Draw the plot for the new point, then
       *  re-draw the barrier.  If we have hit the memory barrier,
       *  flash it, then redraw the important parts of the screen.
       *  Redraw the entire plot if the barrier flashes, and only
       *  the new part if not.
       */
      mem_size += chunk_size;
      if ((mem_size > REAL_MEM) && (!crossed)) {
         DrawPlot (REAL_MEM, chunk_size, FALSE);
         DrawBarrier (FALSE);
         FlashBarrier ();
         DrawBarrier (TRUE);
         DrawAxis (FALSE);
         DrawPlot (mem_size, chunk_size, TRUE);
         DrawBarrier (FALSE);
         alloc_start = GetMilli ();
         crossed = TRUE;
         }
      else {
         DrawPlot (mem_size, chunk_size, FALSE);
         DrawBarrier (FALSE);
         }

      WaitUntil (alloc_start, ALLOC_WAIT);
      alloc_start += ALLOC_WAIT;

      /*
       *  Update loop parameters for the next loop iteration.  If the
       *  memory has exceeded the barrier, then start doubling the
       *  allocation size.
       */
      chunk[0]   = (char *) last_chunk;
      last_chunk = chunk;
      chunk      = (char **) malloc (req_size);

      if (chunk != NULL) {
         chunk_size = _msize (chunk);
         }
      else {
         chunk_size = 0;
         }

      /*
       *  When we get passed the memory barrier increment the
       *  requested memory size.
       */
      if (mem_size > REAL_MEM)
         if ((size_t) req_size + req_size > 0)
            req_size += req_size;
      }

   /*
    *  Free the memory that was allocated.  The last memory allocation
    *  may have been zero size but not NULL.  First free the last
    *  allocated chunk if it is not NULL, then go on to free the list
    *  of allocated memory.
    */
   if (chunk != NULL)
      free (chunk);

   for (chunk = last_chunk; chunk != NULL; chunk = (char **) chunk[0])
      free (chunk);

   /*
    *  That's the end of this demo.  Wait for the required amount of
    *  time.
    */
   WaitFor (TOTAL_WAIT);
   }


/*
 *  These globals divide the screen into several viewports, and are
 *  used for displaying information in various places on the screen.
 *  The SCREEN_CHAR structure is used for the four points it contains.
 *
 *  The plot_view is a rectangle describing the plotting area.  All
 *  plotting information will be clipped to this rectangle.
 *
 *  The message_view is an area where textual information is displayed.
 *  Some of this message viewport is constant, so is drawn once.
 *  Other parts change with the plot and so their areas are described
 *  by additional viewports (for ease of erasing):
 *     1) mem_from_view - where the memory was allocated: REAL,
 *                        EXTENDED, and VIRTUAL
 *     2) sizes_view    - for putting the memory size information.
 *                        This contains two more viewports:
 *          a) req_size_view - amount of memory requested in the last
 *                             memory allocation (the size of a chunk)
 *          b) mem_size_view - total amount of memory allocated
 */

static SCREEN_CHAR plot_view;     /* Viewport for the plot */
static SCREEN_CHAR mess_view;     /* Viewport for message */
static SCREEN_CHAR mem_from_view; /* Message about where memory allocated */
static SCREEN_CHAR sizes_view;    /* For the memory sizes */
static SCREEN_CHAR req_size_view; /* Requested allocation size */
static SCREEN_CHAR mem_size_view; /* Memory allocation size */

static short plot_x_orig;         /* Origin translation for X */
static short plot_y_orig;         /* Origin translation for Y */

#define X_ORIG 50                 /* Viewport coords for object center */
#define Y_ORIG 60
#define TITLE_MARGIN 5            /* Margin for title text */


/*********************************************************************
 *
 *  Initialize the screen attributes for the memory allocation demo.
 *  Set the viewport and the graphics window.  Take whatever background
 *  has been put in the graphics area already.
 *
 *  The screen is divided into several sections, two of which are used
 *  by other functions, and two of which are written once by this
 *  function:
 *
 *         +------------------------------------------+
 *         |              Title                       |
 *         +-----------------------------+------------+
 *         |                             |            |
 *         |                             |   Message  |
 *         |                             |            |
 *         |                             +------------+
 *         |            Plot                          |
 *         |                                          |
 *         |                                          |
 *         |                                          |
 *         +------------------------------------------+
 *
 *  The Title is written once, by this function, and never touched
 *  again.
 *
 *  The Plot and the Message are written by other functions.  The
 *  Plot is the graph.  The Message box is for text strings such
 *  as the number of bytes of memory allocated.  Parts of the message
 *  box are constant and written here.  Other parts must be updated
 *  with each plot of the graph.
 */

InitMemScreen (scrn)
SCREEN_CHAR *scrn;
   {
   double bx, by;                       /* Starting points of a line segment */
   double ex, ey;                         /* Ending points of a line segment */
   double factor;                        /* Screen shape factor for squaring */
   double trns_x, trns_y;            /* Translate (0,0) to these coordinates */
   char *text;                                         /* Text to be written */
   short x, y;                          /* x and y coordinates for viewports */
   struct _fontinfo font;                       /* Characteristics of a font */

   #ifdef DEBUG
      fprintf (fd, "\nInitializing the screen\n");
      fflush (fd);
   #endif

   /*
    *  Write out the title.  Center it in the available screen, along
    *  the top rows.  Keep track of the width of the title, and compute
    *  the next row that can be written.
    */
   if (sizeof (size_t) == 2) {
      text = "Memory Availability With Leading 16-bit Compiler";
      }
   else {
      text = "Extending Horizons with Unbounded Memory Support";
      }

   #ifndef NOFONTS
      _setfont ((const unsigned char *)"t'tms rmn'h20b");
      _setviewport (scrn->left_col,  scrn->first_row,
         scrn->right_col, scrn->last_row);
   #endif

   #ifdef DEBUG
      fprintf (fd, "Screen viewport dimensions are (%d, %d) to (%d, %d)\n",
         scrn->left_col,  scrn->first_row,
         scrn->right_col, scrn->last_row);
      fflush (fd);
   #endif

   #ifndef NOFONTS
      x = _getgtextextent ((const unsigned char *)text);
      x = ((scrn->right_col - scrn->left_col) - x) >> 1;
      y = TITLE_MARGIN;
      _setcolor (MEMTITLE);
      _moveto (x, y);
      _outgtext ((const unsigned char *)text);
      _getfontinfo (&font);
      y  += scrn->first_row + font.pixheight + TITLE_MARGIN;
   #else
      y = TITLE_MARGIN + 20;
   #endif

   /*
    *  Compute the boundaries of the graphics area.  The first row is
    *  the last row of the title.  The last row and first column are
    *  stored in the screen.  Compute the last column.
    *
    *  The last column is computed in a way that makes the plot area
    *  fairly square.  It should be the same distance horizontally
    *  (in the x direction) as vertically (in the y direction).  Of
    *  course, the pixels are not square, so apply the fudge factor
    *  stored in the "scrn" structure.
    */
   plot_view.first_row = y;
   plot_view.last_row  = scrn->last_row;
   plot_view.left_col  = scrn->left_col;
   plot_view.right_col = scrn->right_col;
   plot_view.width     = plot_view.right_col - plot_view.left_col;
   plot_view.height    = plot_view.last_row  - plot_view.first_row;

   #ifdef DEBUG
      fprintf (fd, "Plot viewport dimensions are (%d, %d) to (%d, %d)\n",
         plot_view.left_col,  plot_view.first_row,
         plot_view.right_col, plot_view.last_row);
      fflush (fd);
   #endif

   /*
    *  Store the boundaries of the message viewport.  The message
    *  viewport is to appear in the plot viewport, so compute everything
    *  relative to the plot viewport.  Note that constants are used
    *  here for the width and stuff, but they are only used once.
    *
    *  Come in a little from the right and down some from the top of
    *  the plot viewport.  Then compute the other sides from the
    *  needed width and height.
    */
   mess_view.right_col = plot_view.right_col - 10;
   mess_view.first_row = plot_view.first_row + 10;
   mess_view.width     = 200;
   mess_view.height    = 100;
   mess_view.left_col  = mess_view.right_col - mess_view.width;
   mess_view.last_row  = mess_view.first_row + mess_view.height;

   #ifdef DEBUG
      fprintf (fd, "Message viewport dimensions are (%d, %d) to (%d, %d)\n",
         mess_view.left_col,  mess_view.first_row,
         mess_view.right_col, mess_view.last_row);
      fflush (fd);
   #endif

   /*
    *  Write out the text into the message viewport that is constant.
    *  Set up the interior message viewports.  Write out the constant
    *  part of a line, then use the resulting cursor coordinate to
    *  compute the starting position of an interior viewport.  Since
    *  all viewports are interior, compute positions from the message
    *  viewport.
    */

   _setviewport (mess_view.left_col, mess_view.first_row,
      mess_view.right_col, mess_view.last_row);
      x = 0;
      y = 0;
   #ifndef NOFONTS
      _setfont ((const unsigned char *)"t'tms rmn'h16b");
      _setcolor (BLACK);
      _moveto (x, y);
      _outgtext ((const unsigned char *)"Memory allocation in");
   #endif

   /*  The next row is the beginning of the memory from text. */
   #ifndef NOFONTS
      _getfontinfo (&font);
      y += font.pixheight;
      mem_from_view.width     =
         _getgtextextent ((const unsigned char *)"EXTENDED");
      mem_from_view.height    = font.pixheight;
   #else
      y += 16;
      mem_from_view.width     = 10 * strlen("EXTENDEN");
      mem_from_view.height    = 16;
   #endif

   mem_from_view.first_row = mess_view.first_row + y;
   mem_from_view.left_col  = mess_view.left_col + 40;
   mem_from_view.last_row  = mem_from_view.first_row + mem_from_view.height;
   mem_from_view.right_col = mem_from_view.left_col  + mem_from_view.width;

   /*
    *  The next row is the beginning of the sizes viewport.  Put a boarder
    *  around the sizes viewport.  Leave some vertical space from the
    *  previous row for the boarder.
    */
   #ifndef NOFONTS
      y += font.pixheight + 5;
   #else
      y += 16 + 5;
   #endif

   sizes_view.first_row    = mess_view.first_row + y;
   sizes_view.left_col     = mess_view.left_col;
   sizes_view.right_col    = mess_view.right_col;
   sizes_view.last_row     = mess_view.last_row;
   sizes_view.width        = sizes_view.right_col - sizes_view.left_col;
   sizes_view.height       = sizes_view.last_row  - sizes_view.first_row;
   _setviewport (sizes_view.left_col, sizes_view.first_row,
      sizes_view.right_col, sizes_view.last_row);
   _setcolor (MEMBULBORD);
   _rectangle (_GBORDER, 0, 0, sizes_view.width, sizes_view.height);

   /*
    *  Set up the interior viewports.  Since the sizes viewport will be
    *  boardered, leave a little margin around the entire viewport.
    *
    *  Requested memory allocation size:
    */
   _setviewport (sizes_view.left_col, sizes_view.first_row,
      sizes_view.right_col, sizes_view.last_row);
   y = 5;
   x = 5;
   text = "Req Size: ";
   _moveto (x, y);

   #ifndef NOFONTS
      _outgtext ((const unsigned char *)text);
      x = x + _getgtextextent ((unsigned const char *)text);
      req_size_view.height     = font.pixheight;
   #else
      x = x + 10 * strlen (text);
      req_size_view.height     = 16;
   #endif

   req_size_view.first_row  = sizes_view.first_row + y;
   req_size_view.last_row   = req_size_view.first_row + req_size_view.height;
   req_size_view.left_col   = sizes_view.left_col + x;
   req_size_view.right_col  = sizes_view.right_col - 5;
   req_size_view.width      = req_size_view.right_col - req_size_view.left_col;

   x  = 5;             /* Total memory size.  Leave some margin for clarity. */
   text = "Size Alloc: ";

   #ifndef NOFONTS
      y += font.pixheight + 5;
      _moveto (x, y);
      _outgtext ((const unsigned char *)text);
      x = x + _getgtextextent ((const unsigned char *)text);
   #else
      y += 16 + 5;
      x += 10 * strlen (text);
   #endif

   mem_size_view.first_row  = sizes_view.first_row + y;
   mem_size_view.last_row   = sizes_view.last_row - 1;
   mem_size_view.height     = mem_size_view.last_row - mem_size_view.first_row;
   mem_size_view.left_col   = sizes_view.left_col + x;
   mem_size_view.right_col  = sizes_view.right_col - 5;
   mem_size_view.width      = mem_size_view.right_col - mem_size_view.left_col;

   /*
    *  Set up for plotting.  Set the viewport to the plot viewport and
    *  compute a world coordinate window.  Use 1.0 to 0.0 in the vertical
    *  direction, and whatever is needed in the horizontal direction to
    *  retain squareness.  Move the viewport origin so that the origin
    *  of the object will be in the lower left portion of the viewport.
    *  The way the projection functions work the center of the object
    *  will lie in the center of the window.  So adjust the viewport
    *  origin so that the world coordinate center is the lower left.
    */
   _setviewport (plot_view.left_col, plot_view.first_row,
      plot_view.right_col,  plot_view.last_row);

   plot_x_orig = (plot_view.width >> 1) - X_ORIG;
   plot_y_orig = (plot_view.height >> 1) - (plot_view.last_row - Y_ORIG);
   _setvieworg (-plot_x_orig, -plot_y_orig);

   factor = (plot_view.right_col - plot_view.left_col) /
      ((plot_view.last_row - plot_view.first_row) *
      scrn->width_over_height);

   by = -SCREEN_SIZE;
   ey = -by;
   bx = ey * factor;
   ex = -bx;
   _setwindow (TRUE, bx, by, ex, ey);
   }



/**************************************************************************
 *
 *  Fill in the changing text in the message viewport.  This consists of
 *  computing the text for each of the interior viewports.  Do each
 *  viewport one by one.
 *
 *  One viewport has a text font that changes depending on the memory size.
 *  The mapping from memory size to text font is stored in a table.  The
 *  table works by searching through the memory sizes for the slot that the
 *  memory value falls bewteen.  It is important to begin the table with
 *  memory size 0, and end the table with a very large memory size (too
 *  large to reach).  When the correct slot is found, that font size is
 *  used.
 *
 *  Leave the viewport set to the plot viewport.
 */

typedef struct font_table_struct FONT_TAB;                 /* The font table */
struct font_table_struct {
   long mem;                                                  /* Memory size */
   int  font;                                               /* The font size */
   };

void MemAlloc (value, req)
long value;                                     /* Bytes of memory allocated */
size_t req;                                     /* Requested allocation size */
   {
   char buf [100];                        /* Buffer for constructing message */
   short x, y;                               /* X and Y viewport coordinates */
   struct _fontinfo font;                       /* Characteristics of a font */
   static FONT_TAB font_tab[] = {       /* Memory size to font mapping table */
      (long) 0,          16,                                /* 0 memory size */
      REAL_MEM,          20,
      MAX_MEM,           26,
      VIRTUAL_MEM + 100000000, 0                        /* Large memory size */
      };
   FONT_TAB *cur_entry;                  /* Used to search through the table */

   #ifdef DEBUG
      fprintf (fd, "Writing memory size = %ld, request = %hu\n", value, req);
      fflush (fd);
   #endif

   #ifndef NOFONTS
      /*
       *  Start off with the memory from viewport.  Figure out, from the
       *  memory size, where it was allocated from and display it.  The
       *  memory may be from REAL memory (less than the memory barrier),
       *  EXTENDed memory (greater than the memory barrier but still in
       *  ram) or VIRTUAL (greater than ram).  Clear the boarder before
       *  writing the text.
       */
      _setviewport (mem_from_view.left_col, mem_from_view.first_row,
         mem_from_view.right_col, mem_from_view.last_row);
      _setcolor (MEMSIZEBK);
      _rectangle (_GFILLINTERIOR, 0, 0,
         mem_from_view.width, mem_from_view.height);
      _moveto (0, 0);
      _setfont ((const unsigned char *)"t'tms rmn'h16b");
      if (value <= REAL_MEM) {
         _setcolor (RED);
         _outgtext ((const unsigned char *)"REAL");
         }
      else if (value <= extend_mem) {
         _setcolor (GREEN);
         _outgtext ((const unsigned char *)"EXTENDED");
         }
      else {
         _setcolor (BLUE);
         _outgtext ((const unsigned char *)"VIRTUAL");
         }

      /*
       *  Write the memory size out to its viewport.  Compute the memory
       *  size as a three digit number (bytes, k bytes, or M bytes).
       *  Clear the viewport before writing the value.  Search through the
       *  font size table for the correct font size.
       */
      _setcolor (MEMSIZEBK);
      _setviewport (mem_size_view.left_col, mem_size_view.first_row,
         mem_size_view.right_col, mem_size_view.last_row);
      _rectangle (_GFILLINTERIOR, 0, 0,
         mem_size_view.width, mem_size_view.height);
      _setcolor (MEMSZTEXT);
      _moveto (0, 0);

      cur_entry = font_tab;
      while (cur_entry->mem < value)
         cur_entry++;
      cur_entry--;
      sprintf (buf, "t'tms rmn'h%db", cur_entry->font);
      _setfont ((const unsigned char *)buf);

      if (value < 1024) {
         sprintf (buf, "%ld", value);
         }
      else if (value < (long) 1024 * 1024) {
         sprintf (buf, "%ld k", value >> 10);
         }
      else {
         sprintf (buf, "%ld M", value >> 20);
         }
      _setcolor (MEMSZVAL);
      _outgtext ((const unsigned char *)buf);

      /*
       *  Write the requested size out to its viewport.  Compute the request
       *  size as a three digit number (bytes, k bytes, or M bytes).
       *  Clear the viewport before writing the value.
       */
      _setfont ((const unsigned char *)"t'tms rmn'h16b");
      _setcolor (MEMSIZEBK);
      _setviewport (req_size_view.left_col, req_size_view.first_row,
         req_size_view.right_col, req_size_view.last_row);
      _rectangle (_GFILLINTERIOR, 0, 0,
         req_size_view.width, req_size_view.height);
      _setcolor (MEMSZTEXT);
      _moveto (0, 0);
      if (req < 1024) {
         sprintf (buf, "%d", req);
         }
      else if (req < (long) 1024 * 1024) {
         sprintf (buf, "%d k", req >> 10);
         }
      else {
         sprintf (buf, "%d M", req >> 20);
         }
      _setcolor (MEMSZVAL);
      _outgtext ((const unsigned char *)buf);
   #endif

                            /* Set the viewport back to the "plot" viewport. */
   _setviewport (plot_view.left_col, plot_view.first_row,
      plot_view.right_col,  plot_view.last_row);
   _setvieworg (-plot_x_orig, -plot_y_orig);
   }


/***********************************************************************
 *
 *  Draw the barrier.  If the "entire" flag is set then draw the entire
 *  thing.  Otherwise, draw only the front and top edges.  All of the
 *  rest of the parameters come from the following constants describing
 *  the barrier position, height and width:
 */

#define BARRIER_SIZE_X MAX_Y * 0.25
#define BARRIER_SIZE_Z MAX_Y * 0.5
#define BARRIER_SIZE_Y MAX_Y * 0.01

DrawBarrier (entire)
bool entire;                     /* Flag to say whether to draw entire thing */
   {
   double x, y, z;                              /* A 3-D point to be plotted */
   double bx, by;                         /* Beginning coordinates of a line */
   double ex, ey;                            /* Ending coordinates of a line */

   #ifdef DEBUG
      fprintf (fd, "Drawing the barrier (%d)\n", entire);
      fflush (fd);
   #endif

   _setcolor (MEMBARRIER);                         /* Set the barrier color. */

   x = BARRIER_SIZE_X;
   y = REAL_Y;
   z = BARRIER_SIZE_Z;

   if (entire) {        /*  Only draw this face if the "entire" flag is set. */
      NewPoly ();                                    /* Display barrier face */
      Add3DVertex ( x, y,   z);
      Add3DVertex ( x, y, 0.0);
      Add3DVertex (-x, y, 0.0);
      Add3DVertex (-x, y,   z);
      DisplayPoly (_GBORDER);
      }

   NewPoly ();                                      /* Now do the front side */
   Add3DVertex ( x, y,   z);
   Add3DVertex ( x, y, 0.0);
   y -= BARRIER_SIZE_Y;
   Add3DVertex ( x, y, 0.0);
   Add3DVertex ( x, y,   z);
   DisplayPoly (_GBORDER);

   NewPoly ();                                        /* Now do the top side */
   y = REAL_Y;
   Add3DVertex ( x, y, z);
   Add3DVertex (-x, y, z);
   y -= BARRIER_SIZE_Y;
   Add3DVertex (-x, y, z);
   Add3DVertex ( x, y, z);
   DisplayPoly (_GBORDER);
   }

/**************************************************************************
 *
 *  Flash the barrier.  Compute a polygon that outlines the barrier, then
 *  cycle through all the available colors, as fast as possible, for some
 *  amount of time.  Leave the screen unchanged.  Copy the image then
 *  do the flashing, then restore the image.
 */

#define COLOR_CYCLES   2

FlashBarrier ()
   {
   int color_cycle;                  /* Number of times through the spectrum */
   int a_color;                          /* Current color in the color_cycle */
   double x, y, z;                                            /* A 3-D point */

   #ifdef DEBUG
      fprintf (fd, "Flashing the barrier\n");
      fflush (fd);
   #endif

            /* Compute the barrier perimeter.  Leave out the interior edges. */
   x = BARRIER_SIZE_X;
   y = REAL_Y;
   z = BARRIER_SIZE_Z;
   NewPoly ();
   Add3DVertex ( x, y,-z);
   y -= BARRIER_SIZE_Y;
   Add3DVertex ( x, y,-z);
   Add3DVertex ( x, y, z);
   Add3DVertex (-x, y, z);
   y += BARRIER_SIZE_Y;
   Add3DVertex (-x, y, z);
   Add3DVertex (-x, y,-z);

                                     /* Flash the barrier through all colors */
   for (color_cycle = 0; color_cycle < COLOR_CYCLES; color_cycle++) {
      for (a_color = 0; a_color < 15; a_color++) {
         _setcolor (a_color);
         DisplayPoly (_GFILLINTERIOR);
         }
      }

                               /* Leave the barrier in the background color. */
   _setcolor (MEMSIZEBK);
   DisplayPoly (_GFILLINTERIOR);
   }


/***********************************************************************
 *
 *  Draw the axies on the screen.  If the "entire" flag is set then
 *  draw everything.  Otherwise just draw the "Y" axis.  Don't draw the
 *  X axis, only the Z and Y.  The "X" axis doesn't display anything
 *  worthwhile, we're in 3-D space just for effect.  The size of
 *  the axies segments are defined by the following constants:
 */

#define X_AXIS    MAX_Y                       /* Size of the positive X_AXIS */
#define Y_AXIS    MAX_Y                       /* Size of the positive Y_AXIS */
#define Z_AXIS    MAX_Y                       /* Size of the positive Z_AXIS */

#define INTER_SIZE  (MAX_Y*0.025)                /* Intersection line length */

DrawAxis (entire)
bool entire;                                       /* TRUE to draw all axies */
   {
   double bx, by;                       /* Starting points of a line segment */
   double ex, ey;                         /* Ending points of a line segment */
   char buf[100];                     /* Buffer for writing out intersection */
   short  x, y;                              /* Position to write axis title */
   char *title_text;                                       /* The axis title */
   struct _fontinfo font;                 /* Characteristics of current font */

   #ifdef DEBUG
      fprintf (fd, "Drawing the axis (%d)\n", entire);
      fflush (fd);
   #endif

   /*
    *  Draw the axis.  Only draw the Z or X axis if the "entire"
    *  flag is set.  Label the Z and Y axis.
    */
   _setcolor (MEMAXIS);
   if (entire) {
      Project (-X_AXIS, 0.0, 0.0, &bx, &by);
      Project ( X_AXIS, 0.0, 0.0, &ex, &ey);
      _moveto_w (bx, by);
      _lineto_w (ex, ey);

      Project (0.0, 0.0,-Z_AXIS, &bx, &by);
      Project (0.0, 0.0, Z_AXIS, &ex, &ey);
      _moveto_w (bx, by);
      _lineto_w (ex, ey);

      #ifndef NOFONTS
         /*
          *  Write out the axis title.  Center it in the window, and bring it
          *  up from the bottom.  Note that this does not use world coordinates.
          *  It is independent of the axis position.
          */
         _setfont ((const unsigned char *)"t'tms rmn'h20b");
         title_text = "Memory Allocation";
         _getfontinfo (&font);
         x = _getgtextextent ((const unsigned char *)title_text);
         x = ((plot_view.right_col - x) >> 1);
         y = plot_view.last_row - font.pixheight;
         _setvieworg (0, 0);
         _moveto (x, y);
         _outgtext ((const unsigned char *)title_text);
         _setvieworg (-plot_x_orig, -plot_y_orig);
      #endif

      /*
       *  Mark the point of intersection of the EXTEND memory limit
       *  with the "Y" axis.
       */
      Project ( 0.0, EXTEND_Y, INTER_SIZE, &bx, &by);
      Project ( 0.0, EXTEND_Y,-INTER_SIZE, &ex, &ey);
      _moveto_w (bx, by);
      _lineto_w (ex, ey);
      #ifndef NOFONTS
         _setfont ((const unsigned char *)"t'tms rmn'h15b");
         if (extend_mem < 1024) {
            sprintf (buf, "%ld", extend_mem);
            }
         else if (extend_mem < (long) 1024 * 1024) {
            sprintf (buf, "%ld k", extend_mem >> 10);
            }
         else {
            sprintf (buf, "%ld M", extend_mem >> 20);
            }
         _outgtext ((const unsigned char *)buf);
      #endif
      }
                                                           /* Draw "Y" axis. */
   Project (0.0,-Y_AXIS, 0.0, &bx, &by);
   Project (0.0, Y_AXIS, 0.0, &ex, &ey);
   _moveto_w (bx, by);
   _lineto_w (ex, ey);

                      /* Mark intersection of the REAL memory with "Y" axis. */
   Project ( 0.0, REAL_Y, INTER_SIZE, &bx, &by);
   Project ( 0.0, REAL_Y,-INTER_SIZE, &ex, &ey);
   _moveto_w (bx, by);
   _lineto_w (ex, ey);

   #ifndef NOFONTS
      _setfont ((const unsigned char *)"t'tms rmn'h15b");
      if (REAL_MEM < 1024) {
         sprintf (buf, "%ld", REAL_MEM);
         }
      else if (REAL_MEM < (long) 1024 * 1024) {
         sprintf (buf, "%ld k", REAL_MEM >> 10);
         }
      else {
         sprintf (buf, "%ld M", REAL_MEM >> 20);
         }
      _outgtext ((const unsigned char *)buf);
   #endif
   }


/***************************************************************************
 *
 *  Perform computations on the memory buffer allocated.  Just fill the
 *  memory with integer values, then sum them up mod 4096 (FFF).
 */

ComputeMem (buf, size)
long   buf[];                                               /* Memory buffer */
size_t size;                                    /* Bytes of memory in buffer */
   {
   size_t i;                                  /* An index into the buf array */
   long sum;                                      /* The resulting summation */

   for (i = 0; i < size/sizeof(long); i++)
      buf[i] = i;

   sum = 0;
   for (i = 0; i < size/sizeof(long); i++) {
      sum = (sum + buf[i]) & 0x0FFF;
      }
   }

/**************************************************************************
 *
 *  Draw the plot for some arbitrary point.  The plot is divided into
 *  three sections: Real, Extended, and Virtual memory.  The bounds for
 *  these memory sections are given by constants defined at the beginning
 *  of the file.  Each segment is plotted in a different color.  Since
 *  this function is a one dimensional function, the "Y" coordinate, the
 *  values for the other two dimensions are derived from it.  The "X"
 *  value starts at 0.0, is set to Y/2 up to the memory barrier, and
 *  remains constant at the memory barrier value / 2 for the rest of the
 *  graph.  The "Z" value is always the "Y" value / 2.
 *
 *  If the entire flag is set then draw the entire plot.  If the entire
 *  flag is not set then only draw one portion of the plot (either
 *  the part below the barrier, or the part above the barrier), but not
 *  all.
 */

#define PLOT_WIDTH_Z (MAX_Y * 0.025)
static int too_big = FALSE;        /* True when plot gets too big for screen */


DrawPlot (value, req, entire)
long value;                                   /* The function value to value */
size_t req;                                     /* Requested allocation size */
bool entire;                               /* TRUE if valueting entire graph */
   {
   double y;                                  /* The "Y" value to be plotted */
   long   plot;                            /* The memory value to be plotted */
   double mid_y;                               /* Y location for the barrier */

   #ifdef DEBUG
      fprintf (fd, "Drawing a value for memory size %ld, req size = %d\n",
         value, req);
      fflush (fd);
   #endif

   /*
    *  Display the amount of memory allocated and the requested chunk
    *  size in the text box.
    */
   MemAlloc (value, req);

   /*
    *  Draw the Real memory part of the graph.  This is only drawn if
    *  the value is in the range, or the "entire" flag is set.
    *  Otherwise it is untouched.  Scale the value to plot, then plot
    *  it.
    */
   if ((entire) || (value <= REAL_MEM)) {
      plot = min (value, REAL_MEM);
      y    = plot * (REAL_Y / REAL_MEM);
      _setcolor (RED);
      DrawASeg (0.0, y);
      too_big = FALSE;
      }

   /*
    *  Draw the Extended memory part.  This is only drawn if the value
    *  is in the range, or the "entire" flag is set.  Otherwise it is
    *  left untouched.  Scale the value to plot, then plot the value.
    *  The value to be plotted is the "Y" value of the previous segment
    *  plus however much extended memory scaled to the extended memory
    *  region.
    */
   if ((value > REAL_MEM) && ((entire) || (value <= extend_mem))) {
      plot = min (value, extend_mem);
      y = REAL_Y + ((plot - REAL_MEM) * ((EXTEND_Y - REAL_Y) /
         (extend_mem - REAL_MEM)));
      _setcolor (GREEN);
      DrawASeg (REAL_Y, y);
      too_big = FALSE;
      }

   /*
    *  Draw the Virtual memory part.  This is only drawn if the value
    *  is in the range, or the "entire" flag is set.  Otherwise it is
    *  left untouched.  Scale the value to plot, then plot the value.
    *  The value to be plotted is the "Y" value of the previous segment
    *  plus however much virtual memory, scaled to the virtual memory
    *  region.  Don't plot passed the maximum plot value.  Also, once
    *  the maximum memory has been plotted there is no reason to plot
    *  anything else unless the "entire" flag gets set, or a smaller
    *  value is plotted.
    */
   if ((value > extend_mem) && ((entire) || (value <= VIRTUAL_MEM))) {
      plot = min (value, MAX_MEM);
      y = EXTEND_Y + ((plot - extend_mem) * ((MAX_Y - EXTEND_Y) /
         (MAX_MEM - extend_mem)));
      if ((!too_big) || (entire)) {
         _setcolor (BLUE);
         DrawASeg (EXTEND_Y, y);
         if (value >= MAX_MEM)
            too_big = TRUE;
         }
      }
   }

/**************************************************************************
 *
 *  Draw a segment of the graph.  This takes two end-points of the segment
 *  and makes them into a 3-D polygon (actually three faces, a top, front,
 *  and head faces).  The end-points are taken to be values for the "Y"
 *  axis.  Width is given by assigning Y/2 to the "X" values and then using
 *  X and -X.  Depth is given by adding the plot depth constant to "Y"/2.
 */

#define Z_SCALE 2.5                  /* Scale "Y" by this to get the Z value */
#define X_SCALE 2.0                  /* Scale "Y" by this to get the X value */

DrawASeg ( min_y, max_y )
double min_y;                         /* Beginning "Y" value for the segment */
double max_y;                            /* Ending "Y" value for the segment */
   {
   double half_y;         /* Value of "Y"/2 */
   double min_x, max_x;   /* Minimum and maximum "X" values */
   double min_z, max_z;   /* Minimum and maximum "Z" values */
   short  color;          /* Current color */

   /*
    *  The current color has been set for the color of the faces.  However,
    *  each face will be outlined in another color, so it will be necessary
    *  to switch colors around.  Get the current color so we can switch
    *  back and forth.
    */
   color = _getcolor ();

   /*
    *  Find the minium "X" and "Z" values.  The "Z" values are just "Y"/3,
    *  but the "X" values are clipped.
    */
   min_z = min_y / Z_SCALE;
   max_z = max_y / Z_SCALE;
   min_x = min_y / X_SCALE;
   if (min_x > REAL_Y / X_SCALE)
       min_x = REAL_Y / X_SCALE;
   max_x = max_y / X_SCALE;
   if (max_x > REAL_Y / X_SCALE)
       max_x = REAL_Y / X_SCALE;

   /*
    *  Draw the top face.  Give the coordinates in the clockwise
    *  direction.
    */
   NewPoly ();
   Add3DVertex ( min_x, min_y, min_z + PLOT_WIDTH_Z);
   if (min_x != -min_x)
       Add3DVertex (-min_x, min_y, min_z + PLOT_WIDTH_Z);
   Add3DVertex (-max_x, max_y, max_z + PLOT_WIDTH_Z);
   Add3DVertex ( max_x, max_y, max_z + PLOT_WIDTH_Z);
   _setcolor (color);
   DisplayPoly (_GFILLINTERIOR);
   _setcolor (MEMSMBD);
   DisplayPoly (_GBORDER);

   /*
    *  Draw the front face.  Give the coordinates in the clockwise
    *  direction.
    */
   NewPoly ();
   Add3DVertex ( min_x, min_y, min_z);
   Add3DVertex ( min_x, min_y, min_z + PLOT_WIDTH_Z);
   Add3DVertex ( max_x, max_y, max_z + PLOT_WIDTH_Z);
   Add3DVertex ( max_x, max_y, max_z);
   _setcolor (color);
   DisplayPoly (_GFILLINTERIOR);
   _setcolor (MEMSMBD);
   DisplayPoly (_GBORDER);

   /*
    *  Draw the head face.  Give the coordinates in the clockwise
    *  direction (looking toward the object down the "Y" axis).
    */
   NewPoly ();
   Add3DVertex ( max_x, max_y, max_z);
   Add3DVertex ( max_x, max_y, max_z + PLOT_WIDTH_Z);
   Add3DVertex (-max_x, max_y, max_z + PLOT_WIDTH_Z);
   Add3DVertex (-max_x, max_y, max_z);
   _setcolor (color);
   DisplayPoly (_GFILLINTERIOR);
   _setcolor (MEMSMBD);
   DisplayPoly (_GBORDER);
   }
