/* Intel 386(TM)/486(TM) C Code Builder(TM) Kit
 * Copyright 1991 Intel Corporation.  All Rights Reserved.
 * mainblt.c - Clear to new background color.
 * $Version: 1.5 $
 */

/*
 *  This demo does the clear to a new background color using random
 *  squares.  It does this change twice, and in between displays some
 *  bullets about this demo.
 */

#include <stdio.h>
#include <graph.h>
#include <stdlib.h>
#include <sys\types.h>
#include "const.h"
#include "srnchr.h"
#include "bool.h"
#include "debug.h"

extern SCREEN_CHAR *ShutDownDemo ();
extern long         GetMilli ();
static void ScreenClear ();

/*
 *  Wait times for this function.  Times are in milliseconds.
 */
#define BLT_TIME          1000  /* Time between bullets */
#define TOTAL_TIME    5 * 1000  /* Time after last bullet */


static char *blt_text[] = {
    "32-bit C Compiler, Libraries, Linker, and Librarian",
    "  DPMI-compliant DOS Extender and Virtual Memory Manager",
    "    Source Level Symbolic Debugger",
    "      MAKE Utility",
    "        User's Guide and Reference Manual",
    "          Unlimited access to toll-free customer support",
    NULL
};

static char *emph_text[] = {
    "          $695, Royalty-free distribution for",
    "            Code Builder derivative products",
    NULL
};


/********************************************************************
 *
 *  This is the function that controls the screen background color
 *  change.  It sets up the screen, does one screen fade, displays
 *  bullets, then does a second screen fade.
 */

void MainBullets (scrn, bkclr)
    SCREEN_CHAR *scrn;     /* Screen size data */
    int          bkclr;    /* Ending backgroud color */
{
    int x;                 /* Horizontal distance to place x */
    long start_time;       /* Starting time in seconds */
    int y;                 /* Vertical pixel position for text */
    int vert_space;        /* Spacing between lines of text */
    struct _fontinfo font; /* Font information for display text */
    int     bullet_line;   /* Line number of the bullet */

    /*
     *  Time this routine so the text will stay up on the screen.
     */
    start_time = GetMilli ();

    /*
     *  Set the viewport to the graphics area.  Set the window to be
     *  (0,0) to (1.0, 1.0).  Then do all computations using window
     *  coordinates.
     */
    _setviewport (scrn->left_col,  scrn->first_row,
                  scrn->right_col, scrn->last_row);
    _setwindow (FALSE, 0.0, 0.0, 1.0, 1.0);

    /*
     *  Clear the screen the first time to the background color for
     *  the main text.
     */
    ScreenClear (scrn, MAINBK);
    _setcolor (MAINTEXT);

#ifndef NOFONTS
    /*
     *  Now display the main bullets.  Select a big font and the color,
     *  then just print out the bullets.
     */
    if (!_setfont ((const unsigned char *)"t'tms rmn'h20b")) {
        ShutDownDemo ( scrn );
        printf ("Cannot set font at file %s, line %d\n", __FILE__, __LINE__);
        exit (0);
    };
    _getfontinfo (&font);

    /*
     *  Compute the starting row and column for the text, and also the
     *  space between lines.
     */
    x = font.avgwidth;
    y = 2 * font.pixheight;
    vert_space = 2.0 * font.pixheight;

    /*
     *  Write out the lines, one at time.  Pause after each so the lines
     *  can be read easily.
     */
    bullet_line = 0;
    while (blt_text[bullet_line] != NULL) {
        _moveto (x, y);
        _outgtext ((const unsigned char *)blt_text[bullet_line]);
        WaitFor (BLT_TIME);
        y += vert_space;
        bullet_line++;
    };
    x = font.avgwidth;

    /*
     *  Now display the emphasized bullets.  Select a big font and the color,
     *  then just print out the bullets.
     */
    if (!_setfont ((const unsigned char *)"t'tms rmn'h26b")) {
        ShutDownDemo ( scrn );
        printf ("Cannot set font at file %s, line %d\n", __FILE__, __LINE__);
        exit (0);
    };
    _setcolor (EMPHTEXT);
    _getfontinfo (&font);

    /*
     *  Compute the starting row and column for the text, and also the
     *  space between lines.
     */
    vert_space = 1.5 * font.pixheight;

    /*
     *  Write out the lines, one at time.  Pause after each so the lines
     *  can be read easily.
     */
    bullet_line = 0;
    while (emph_text[bullet_line] != NULL) {
        y += vert_space;
        _moveto (x, y);
        _outgtext ((const unsigned char *)emph_text[bullet_line]);
        bullet_line++;
    };

    /*
     *  Wait the requested number of seconds.
     */
    WaitFor (TOTAL_TIME);
#endif

    /*
     *  Clear the screen to the resulting background color.
     */
    ScreenClear (scrn, bkclr);
}





/***********************************************************************
 *
 *  This is the screen fade routine.  It changes the background of the
 *  screen by clearing random squares.
 *
 *  The graphics area is divided up into small squares and then each
 *  square is semi-randomly changed to the new background color.  The
 *  semi-random is important.  There must be a higher distribution of
 *  changed at the bottom of the screen than at the top.
 *
 *  What happens is this.  The screen is not only divided into (almost)
 *  equally sized squares, but there is also a "window" band that moves
 *  up the graphics area.  A few squares are changed at the top of the
 *  window, and almost all are changed at the bottom.  After the bottom
 *  of the window has moved passed a row of squares, they are guarenteed
 *  to be all changed.  The squares and window look like:
 *
 *            +---+---+---+---+---+---+---+---+
 *            |   |   |   |   |   |   |   |   |
 *            +---+---+---+---+---+---+---+---+
 *            |   |   |   |   | X |   |   |   |   --> top of window
 *            +---+---+---+---+---+---+---+---+
 *            |   |   | X |   |   |   |   | X |
 *            +---+---+---+---+---+---+---+---+
 *            | X |   |   | X |   |   | X |   |
 *            +---+---+---+---+---+---+---+---+
 *            |   | X | X | X | X |   | X | X |   --> bottom of window
 *            +---+---+---+---+---+---+---+---+
 *            | X | X | X | X | X | X | X | X |
 *            +---+---+---+---+---+---+---+---+
 *
 *  Where "X" means that the square has been cleared, a blank means
 *  it has not been cleared.
 *
 *  The window starts off with the top of the window on the lowest line
 *  of squares.  A few squares are randomly choosen and changed to the
 *  background.  Then the window is moved up by one row of squares.
 *  A few squares from the first two squares are changed.  This continues
 *  until the bottom of the window gets on the screen.  The row where the
 *  bottom of the window resides always gets all of its squares changed.
 *  When the bottom of the window has passed across a row it is guarenteed
 *  to be entirely changed.
 *
 *  Note that the number of squares cleared in each row is not random,
 *  but a function of the number of squares cleared per row.  Thus,
 *  for any row in the window, the number of squares cleared in the
 *  row can be computed.  However, the squares that are cleared are
 *  not known since they are random.  So a list of cleared and uncleared
 *  squares are kept up.  When a square is to be cleared, a random
 *  number between zero and the number of uncleared squares (minus
 *  one) is generated.  Count over to that square skipping the
 *  cleared squares.
 */

#define ROWS      16       /* Number of rows of squares */
#define WIND_SIZE ROWS     /* Number of squares in the window */

#define CHANGE_TIME      5000  /* Milliseconds for complete screen change */
#define SQUARE_TIME      10    /* Milliseconds to change a square */

static void ScreenClear (scrn, bkclr)
    SCREEN_CHAR *scrn;     /* Screen size data */
    int          bkclr;    /* Ending backgroud color */
{
    double height, width;  /* The size of a square */
    int    rows, cols;     /* Number of square rows and columns */
    bool  *cleared_sqs;    /* Boolean array of cleared squares */
    int    top_row;        /* Square row of top of window */
    int    row, col;       /* Current square row and column */
    int    top_in_wind;    /* First square row residing in window */
    int    bot_in_wind;    /* Last square row residing in window */
    int    stall_cntr;     /* Counter for waiting */
    int    max_clr_cnt;    /* Maximum number of squares to clear in a row */
    int    sqs_to_clr;     /* Number of squares to clear in a row */
    int    sqs_uncleared;  /* Number of uncleared squares in a row */
    int    clr_cnt;        /* Number of squares cleared */
    int    to_clr;         /* The next square to clear */
    long   start_time;     /* The starting time */
    long   start_square;   /* Time that the square started */
    double temp;           /* A temporary location */

    /*
     *  Get the time way up here so all of the other functions will
     *  take the same amount of time.
     */
    start_time = GetMilli ();

    /*
     *  Assuming the screen is a square screen, compute the width and
     *  height of a square.  The number of rows of squares is given,
     *  so use it to compute the height.  Then, from the ratio of the
     *  number of pixels across the screen to the number of pixels
     *  down the screen, and the height of a square, compute the width
     *  of a square.  Compute the width and height in window
     *  coordinates.
     */
    height = 1.0 / (double) ROWS;
    width  = ((height * (scrn->last_row  - scrn->first_row)) /
             (scrn->right_col - scrn->left_col)) /
             scrn->width_over_height;

    /*
     *  Compute the number of columns of squares.  This is needed to
     *  allocate a structure that will keep track of the squares that
     *  have been cleared.
     */
    rows = ROWS;
    cols = (int) (1.0 / width);
    if (cols * width < 1.0)
        cols++;

    /*
     *  Compute the maximum number of squares to be cleared in one row
     *  of the window.  This number must be set up so that the last
     *  row of the window has at most this number of squares left to
     *  clear.  So make the computations truncate up instead of down.
     */
    max_clr_cnt = cols / WIND_SIZE;

    /*
     *  Allocate a two dimensional array of booleans and set every
     *  element of the array to FALSE (not yet cleared).
     */
    cleared_sqs = (bool *) malloc (sizeof (bool) * rows * cols);
    if (cleared_sqs == NULL) {
        ShutDownDemo ( scrn );
        printf ("Cannot allocate memory for cleared squares array\n");
        exit (0);
    };

    /*
     *  Clear out the array that keeps track of the squares cleared.
     */
    for (row = 0; row < rows; row++)
        for (col = 0; col < cols; col++)
            cleared_sqs [row*cols + col] = FALSE;

    /*
     *  Time the total processing time for each square so that they
     *  will appear on the screen with the same amount of time between
     *  each.  The wait time should exceed any processing time.
     */
    start_square = GetMilli ();

    /*
     *  Move the window up the squares one row at a time.  Each iteration
     *  clear random squares that fall inside the window.
     */
    _setcolor (bkclr);
    for (top_row = ROWS;
          top_row >= -WIND_SIZE;
           top_row--) {

        /*
         *  Clear squares that fall inside the window.  Figure out the
         *  first and last row of squares that need to be cleared, then
         *  loop through all of them clearing random squares within each
         *  row.
         */
        top_in_wind = max (top_row, 0);
        bot_in_wind = min (top_row + WIND_SIZE, ROWS-1);
        for (row = top_in_wind; row <= bot_in_wind; row++) {

            /*
             *  Clear random squares from the row "row".  Figure out
             *  the number of squares left to clear in the row (based
             *  on the row of the window!) then choose a random number
             *  within that range.  Clear the required number of
             *  squares.
             */
            sqs_uncleared = 0;
            for (col = 0; col < cols; col++) {
                if (!cleared_sqs [row * cols + col])
                    sqs_uncleared++;
            };
            sqs_to_clr = min (max_clr_cnt, sqs_uncleared);
            if (row == top_row + WIND_SIZE)
                sqs_to_clr = sqs_uncleared;
            for (clr_cnt = 0; clr_cnt < sqs_to_clr; clr_cnt++) {

                /*
                 *  Clear a square.  Get a random number for one of the
                 *  uncleared squares.  Find the column number of the
                 *  uncleared square by going through the cleared array.
                 *  Clear the square and record the fact in the cleared
                 *  array and decrement the uncleared square count so
                 *  the next random number will have a smaller range.
                 *  Wait for a while before continuing the clearing.
                 */
                to_clr = rand()-1;
                temp   = (to_clr / (double) RAND_MAX);
                temp   = temp * (double) (sqs_uncleared-1);
                to_clr = (int) temp;
                col = 0;
                while (to_clr >= 0) {
                    if (!cleared_sqs [row * cols + col])
                        to_clr--;
                    col++;
                };
                col--;
                _rectangle_w (_GFILLINTERIOR,
                              col*width,     row*height,
                              (col+1)*width, (row+1)*height);
                if ((row < 0) || (row >= rows) ||
                    (col < 0) || (col >= cols))
                    FatalError ("Bad row/col in SquareClear\n");
                cleared_sqs [row * cols + col] = TRUE;
                sqs_uncleared--;

                /*
                 *  Wait for the appropriate amount of time, then
                 *  start timing for the next square.
                 */
                WaitUntil (start_square, SQUARE_TIME);
                start_square += SQUARE_TIME;
            };
        };
    };

    /*
     *  Return the space allocated by this function
     */
    free (cleared_sqs);

    /*
     *  Wait for the required amount of time.
     */
    WaitUntil (start_time, CHANGE_TIME);
}
