/* Intel 386(TM)/486(TM) C Code Builder(TM) Kit
 * Copyright 1991 Intel Corporation.  All Rights Reserved.
 * fungraph.c - Draw an ellipse.
 * $Version: 1.4 $
 */

/*
 *  This demo draws a function (a 3-D ellipse) on the screen.  All the
 *  plotting is done in spherical coordinates, and on the screen in
 *  world coordinates.  The points are projected as they are generated
 *  to preserve perspective.
 */

#include <stdio.h>
#include <graph.h>
#include <sys\types.h>
#include <math.h>
#include "const.h"
#include "debug.h"
#include "srnchr.h"
#include "bool.h"
#include "points.h"

extern long GetMilli ();

/*
 *  When doing spherical coordinates, one needs the value of PI
 *  Give enough digits to ensure an acurate double precision value.
 */
#define PI 3.14159265358979

/*
 *  The function displayed is a sphere, constructed using the equation
 *
 *   r = RADIUS
 *
 *  Of course, this equation is in spherical coordinates.  A couple
 *  of important properties of this equation:
 *
 *       1) The origin is the center of the figure.
 *       2) The size of the figure is RADIUS
 *
 *  The RADIUS is for the sphere in the X/Y plane.  This figure is then
 *  rotated around the X axis.
 */
#define RADIUS 2.0       /* Radius of the sphere */

/*
 *  Define the world coordinate size for the screen.
 */
#define SCREEN_SIZE (RADIUS)

/*
 *  Define the size of the object.
 */
#define OBJ_SIZE 4.0    /*(RADIUS+0.1)*/

/*
 *  Define the maximum Z coordinate and the radius for the circle the
 *  eye is to describe.  Also define the number of snap shots to take
 *  while rotating ellipse.
 */
#define EYE_RAD 200.0
#define Z_RISE  200.0
#define EYE_DIV  40
/*
 *  The length of time (in milliseconds) this function is to take.
 */
#define FUNC_MILLI 5 * 1000



FunctionGraph (scrn)
    SCREEN_CHAR *scrn;
{
    double view_x, view_y, view_z;  /* Eye coordinates */
    double theta;                   /* Circle */
    double theta_delta;             /* Delta for going around circle */
    int    num_theta;               /* Number of theta divisions */
    long   start_time;              /* Starting and ending time */

    /*
     *  Start the timer.
     */
    start_time = GetMilli ();

    /*
     *  Initialize the screen for the function graph.
     */
    InitFunScreen (scrn);

    /*
     *  Compute 3-D points in the object, and compute the polygons for
     *  the faces.  This function allocates enough memory space to
     *  hold the two lists.
     */
    ComputeFunction ();

    /*
     *  Make the eye go around its circle.  For each point in the
     *  circle display the graph.
     */
    theta_delta = (2.0 * PI) / (EYE_DIV - 1);
    for (num_theta = 0; num_theta < EYE_DIV; num_theta++) {
        theta = (num_theta+5) * theta_delta;
        view_y = cos(theta) * EYE_RAD;
        view_x = sin(theta) * EYE_RAD;
        view_z = sin(theta) * Z_RISE;

        /*
         *  Set the perspective eye point.
         */
        InitProject (view_x, view_y, view_z, SCREEN_SIZE, OBJ_SIZE);

        /*
         *  Project the points and display them.
         */
        DrawFunction ();
    };

    /*
     *  Wait for the requested amount of time.
     */
    WaitFor (FUNC_MILLI);
}



/*
 *  These globals divide the screen into several viewports, and are
 *  used for displaying information in various places on the screen.
 *  The SCREEN_CHAR structure is used for the four points it contains.
 */
static SCREEN_CHAR plot_view;     /* Viewport for the plot */

#define TITLE_MARGIN 5            /* Margin for title text */


/*********************************************************************
 *
 *  Initialize the screen attributes for the 3-D function demo.
 *  Set the viewport and the graphics window.  Take whatever background
 *  has been put in the graphics area already.
 *
 *  The screen is divided into several sections, one of which is used
 *  by other functions.
 *
 *         +------------------------------------------+
 *         |              Title                       |
 *         +-----------------------------+------------+
 *         |                             |            |
 *         |                             |            |
 *         |                             |  Bullets   |
 *         |         Plot                |            |
 *         |                             |            |
 *         |                             +------------+
 *         |                             |            |
 *         |                             |            |
 *         +-----------------------------+------------+
 *
 *  The Title and Bullets are written once, by this function, then
 *  not touched again.  The Title is the title of this demo.  The
 *  Bullets desribe the characteristics that this demo demonstrates.
 *
 *  The Plot is written by other functions.  The Plot is the 3-D
 *  function display.
 */

InitFunScreen (scrn)
    SCREEN_CHAR *scrn;
{
    double bx, by;             /* Starting points of a line segment */
    double ex, ey;             /* Ending points of a line segment */
    double factor;             /* Screen shape factor for squaring */
    char *text;                /* Text to be written */
    short x, y;                /* x and y coordinates for viewports */
    struct _fontinfo font;     /* Characteristics of a font */

#if DEBUG
    fprintf (fd, "\nInitializing Function screen\n");
    fflush (fd);
#endif

    /*
     *  Write out the title.  Center it in the available screen, along
     *  the top rows.  Keep track of the width of the title, and compute
     *  the next row that can be written.
     */
    _setviewport (scrn->left_col,  scrn->first_row,
                  scrn->right_col, scrn->last_row);

_setcolor (BRIGHTWHITE);
_rectangle (_GFILLINTERIOR, 0, 0, scrn->right_col - scrn->left_col,
                                  scrn->last_row - scrn->first_row);

#if DEBUG
    fprintf (fd, "Screen viewport dimensions are (%d, %d) to (%d, %d)\n",
                  scrn->left_col,  scrn->first_row,
                  scrn->right_col, scrn->last_row);
    fflush (fd);
#endif

#ifndef NOFONTS
    text = "New Dimensions with True 32-bit Processing";
    _setfont ((const unsigned char *)"t'tms rmn'h20b");
    x = _getgtextextent ((const unsigned char *)text);
    x = ((scrn->right_col - scrn->left_col) - x) >> 1;
    y = TITLE_MARGIN;
    _moveto (x, y);
    _setcolor (BLACK);
    _outgtext ((const unsigned char *)text);
    _getfontinfo (&font);
    y  += scrn->first_row + font.pixheight + TITLE_MARGIN;
#else
    y  = TITLE_MARGIN + scrn->first_row + 20 + TITLE_MARGIN;
#endif

    /*
     *  Compute the boundaries of the graphics area.  The first row is
     *  the last row of the title.  The last row and first column are
     *  stored in the screen.  Compute the last column.
     *
     *  The last column is computed in a way that makes the plot area
     *  fairly square.  It should be the same distance horizontally
     *  (in the x direction) as vertically (in the y direction).  Of
     *  course, the pixels are not square, so apply the fudge factor
     *  stored in the "scrn" structure.
     */
    plot_view.first_row = y;
    plot_view.last_row  = scrn->last_row;
    plot_view.left_col  = scrn->left_col;
    plot_view.right_col = scrn->right_col;

#ifdef DEBUG
    fprintf (fd, "Plot viewport dimensions are (%d, %d) to (%d, %d)\n",
                  plot_view.left_col,  plot_view.first_row,
                  plot_view.right_col, plot_view.last_row);
    fflush (fd);
#endif

    /*
     *  Set up for plotting.  Set the viewport to the plot viewport and
     *  compute a world coordinate window.  Use 1.0 to 0.0 in the vertical
     *  direction, and whatever is needed in the horizontal direction to
     *  retain squareness.
     */
    _setviewport (plot_view.left_col, plot_view.first_row,
                  plot_view.right_col,  plot_view.last_row);

    factor = (plot_view.right_col - plot_view.left_col) /
             ((plot_view.last_row - plot_view.first_row) *
             scrn->width_over_height);

    by = -SCREEN_SIZE;
    ey = -by;
    bx =  SCREEN_SIZE * factor;
    ex = -bx;
    _setwindow (TRUE, bx, by, ex, ey);
}



/*
 *  Only a few points in the figure are computed, and lines are drawn
 *  between these points.  The points are evenly spaced around the
 *  figure in the X/Y plane, and evenly spaced when turned around the
 *  X axis.  So give the number of spaces around the X/Y plane (the
 *  number of times to divide "a"), and the spaces when rotating in
 *  the Z (using "b" as the angle).
 *
 *  Notice that rotating the figure around an entire circle means
 *  that only half of the figure needs to be computed.  The other
 *  points will be computed when the figure is rotated.  So A_DIV
 *  divides the half circle from 0 to PI, and B_DIV divides a
 *  full circle from 0 to 2*PI.
 */
#define A_DIV 10
#define B_DIV 15

/*
 *  The function values of the figure are computed and stored in an
 *  array of 3-D points.  There are a variable number of entries
 *  in the points array, and each entry has three fields.
 */
static THREE_D  points [A_DIV * B_DIV];
static int      point_rows = 0;

/*
 *  Display the object with a set of faces.  Each face has four
 *  points (indexes into the "points" vector.  A face is some
 *  distorted version of
 *
 *       V2          V3
 *         +---------+
 *         |         |
 *         |         |
 *         |         |
 *         +---------+
 *       V1          V4
 *
 *  Where two of the points may be the same value, and hence
 *  coincide.  There is one such face for every square in the
 *  figure.
 *
 *  To aid in hidden line removal the first vertex (V1) is
 *  duplicated at the end of the vertex list.  This allows
 *  simple scanning from V1->V2, V2->V3, V3->V4 and V4->V1
 *  without funny looping.
 */
#define MAX_SURF_VERT 5        /* Number of verticies */
typedef struct surface_struct SURFACE;
struct surface_struct {
    int vertex[MAX_SURF_VERT]; /* The verticies */
};
static SURFACE surfaces [A_DIV * B_DIV];



/****************************************************************
 *
 *  Compute the function points and the polygon faces.  The function
 *  points are first computed using spherical coordinates, then
 *  translated to rectangular coordinates.  The polygon faces are
 *  filled with indicies to the computed points.  This method
 *  allows the 3-D points to be projected to 2-D points, and
 *  then display the polygon faces.
 *
 *  Note, the polygon computations "knows" the order that the
 *  function points are generated.  If the function points change
 *  order, the polygon face computation must also change order.
 */

ComputeFunction ()
{
    double theta;                 /* Radians around the X/Y plane */
    double theta_delta;           /* Angle change around the X/Y plane */
    int    num_theta;             /* Number of theta deltas */
    double phi;                   /* Radians around the Y/Z plane */
    double phi_delta;             /* Angle change around the Y/Z plane */
    int    num_phi;               /* Number of phi deltas */
    double x, y, z;               /* Rectangular coordinates */
    double rYZ;                   /* Radius in the Y/Z plane */
    int    v1;                    /* V1 index */

#if DEBUG
    fprintf (fd, "Computing function values\n");
    fprintf (fd, "    A_DIV = %d, B_DIV = %d\n", A_DIV, B_DIV);
    fprintf (fd, "    RADIUS = %f, PI = %f\n", RADIUS, PI);
    fflush  (fd);
#endif

    /*
     *  Compute the points for the function.  Go through the circle in
     *  the X/Y plane and compute the function value "r".  Convert
     *  that spherical coordinate to rectangular coordinates.  Rotate
     *  the rectangular coordinates around the "X" axis.  I.e.  keeping
     *  the "x" value the same, and using the "y" value as the radix of
     *  a circle, compute the "y" and "z" values in spherical, then
     *  rectangular coordinates.  Adjust the X coordinate to translate
     *  the object origin from a foci to the "real" center.
     */
    theta_delta = PI / (A_DIV-1);
    phi_delta   = (2.0 * PI) / (B_DIV-1);
    point_rows  = 0;
    for (num_theta = 0; num_theta < A_DIV; num_theta++) {
        theta = num_theta * theta_delta;
        x     = cos (theta) * RADIUS;
        rYZ   = sin (theta) * RADIUS;
        for (num_phi = 0; num_phi < B_DIV; num_phi++) {
            phi = num_phi * phi_delta;
            y = cos (phi) * rYZ;      /* Rectangular coordinate "Y" */
            z = sin (phi) * rYZ;      /* Rectangular coordinate "Z" */

            /*
             *  Store the value of the point in the point array.
             */
#ifdef DEBUG
            fprintf (fd, "    Phi = %f, y = %f, z = %f\n", phi, y, z);
            fflush  (fd);
#endif
            points[point_rows].x = x;
            points[point_rows].y = y;
            points[point_rows].z = z;
            point_rows++;
        };
    };

    /*
     *  Set up the surfaces.  Loop through the V1 points and use the
     *  V1 index to construct the V2, V3 and V4 index.
     */
    for (num_theta = 0; num_theta < A_DIV-1; num_theta++) {
        for (num_phi = 0; num_phi < B_DIV-1; num_phi++) {
            v1 = num_theta * B_DIV + num_phi;
            surfaces [v1].vertex[0] = v1;
            surfaces [v1].vertex[1] = v1+1;
            surfaces [v1].vertex[2] = v1 + B_DIV + 1;
            surfaces [v1].vertex[3] = v1 + B_DIV;
            surfaces [v1].vertex[4] = v1;
        };
    };
}





/*
 *  A couple of extra structures are needed to hold the list of projected
 *  points and whether a point is seen or hidden.
 */
static TWO_D proj [A_DIV * B_DIV];
static int   seen [A_DIV * B_DIV];




/************************************************************************
 *
 *  Draw the function.  Go through the points and project them.  Store
 *  the results in a memory buffer.  Then go through the surfaces and
 *  draw the polygon.  Use each vertex index as an index into both the
 *  projected point array and the seen array.
 *
 *  Use a state machine to draw the surfaces.  As each vertex is examined
 *  the state machine keeps track of when to draw a line, compute a
 *  crossing from seen to hidden, etc.  So define the states, then
 *  statically set up the state machine.  Actions taken in each state
 *  are explained in the code.
 */
typedef enum {
    FIRST,                      /* First point */
    HIDDEN,                     /* All points hidden so far */
    FIRSTSEEN,                  /* First visible point */
    HIDCROSSMV,                 /* Hidden to seen, start drawing */
    HIDCROSSDR,                 /* Hidden to seen, continue drawing */
    DRAW,                       /* Draw points that are seen */
    SEENCROSSDR,                /* Seen to hidden, continue drawing later */
    HIDDENDR,                   /* Hidden points, continue drawing later */
    MAX_STATES                  /* Maximum number of states */
} DRAW_STATES;                  /* The different states */

static DRAW_STATES machine [MAX_STATES] [2] = {
                       /* HIDDEN         SEEN */
/* FIRST */               HIDDEN,        FIRSTSEEN,
/* HIDDEN */              HIDDEN,        HIDCROSSMV,
/* FIRSTSEEN */           SEENCROSSDR,   DRAW,
/* HIDCROSSMV */          SEENCROSSDR,   DRAW,
/* HIDCROSSDR */          SEENCROSSDR,   DRAW,
/* DRAW */                SEENCROSSDR,   DRAW,
/* SEENCROSSDR */         HIDDENDR,      HIDCROSSDR,
/* HIDDENDR */            HIDDENDR,      HIDCROSSDR
};

DrawFunction ()
{
    int     row;         /* A point to be projected */
    int     line;        /* Current line being drawn */
    int     point;       /* Current point being drawn */
    double  z;           /* Z value of projected point */
    THREE_D seen_pnt;    /* Last point where line segment can be seen */
    TWO_D   seen_proj;   /* Projected values of seen point */
    TWO_D   first_proj;  /* First projected crossing point */
    int     prev_row;    /* Previous point being projected */
    SURFACE *surf;       /* Current surface being plotted */
    DRAW_STATES state;   /* Current state of drawing */
    int     *vert;       /* Current vertex of surface being looked at */

#ifdef DEBUG
    fprintf (fd, "Drawing the function\n");
    fflush (fd);
#endif

    /*
     *  Go through the points array, project each point and record
     *  whether the point can be seen.
     */
    HiddenProject (points, proj, seen, point_rows);

    /*
     *  Clear the screen to the appropriate background color.  Use the
     *  rectangle function to do the clearing.
     */
    _setcolor (BRIGHTWHITE);
    _rectangle (_GFILLINTERIOR, 0, 0,
                plot_view.right_col - plot_view.left_col,
                plot_view.last_row  - plot_view.first_row);

    /*
     *  Go through the surfaces and draw them.  Construct a polygon
     *  then display it.
     */
    surf = surfaces;
    for (line = 0; line < A_DIV-1; line++) {
        for (point = 0; point < B_DIV-1; point++) {
            if (point < B_DIV >> 1) {
                _setcolor (GREEN);
            } else {
                _setcolor (LIGHTBLUE);
            };

            /*
             *  Draw the surface pointed to by "surf".  Look at each vertex
             *  to see if it can be seen or not, and draw the appropriate
             *  lines.  Use a state machine to determine what actions to
             *  take at what time.  This is "in-lined" here for speed, even
             *  though it makes the code a bit difficult to read.  The state
             *  machine takes the current state and whether the current
             *  vertex can be seen or not, and computes the next state.
             */
            state = FIRST;
            prev_row = 0;
            for (vert = surf->vertex;
                  vert < surf->vertex + MAX_SURF_VERT;
                   vert++) {
                row   = *vert;
                state = machine [state][seen[row]];
                switch (state) {

                  case HIDDEN:
                    /*
                     *  So far all verticies are hidden.  Don't do anything,
                     *  just look for a crossing.
                     */
                    break;

                  case FIRSTSEEN:
                    /*
                     *  The first vertex looked at is visible.  Move to
                     *  that point without drawing anything.  This can
                     *  only happen on the first point.
                     */
                    _moveto_w (proj[row].x, proj[row].y);
                    break;

                  case HIDCROSSMV:
                    /*
                     *  All previous points have been hidden, but this point
                     *  can be seen.  Figure out where the crossing is and
                     *  start drawing from there.  Draw a partial line to
                     *  the current vertex since it can be seen.
                     */
                    ComputeCrossing (points + prev_row, points + row,
                                     &seen_pnt);
                    Project (seen_pnt.x, seen_pnt.y, seen_pnt.z,
                             &first_proj.x, &first_proj.y);
                    _moveto_w (first_proj.x, first_proj.y);
                    _lineto_w (proj[row].x, proj[row].y);
                    break;

                  case HIDCROSSDR:
                    /*
                     *  Some previous verticies have been seen, but at
                     *  least the last vertex was hidden.  Figure out
                     *  where the crossing is and draw a line from
                     *  where-ever we left off previously to this cross.
                     *  Draw another line from the crossing to the current
                     *  vertex.
                     */
                    ComputeCrossing (points + prev_row, points + row,
                                     &seen_pnt);
                    Project (seen_pnt.x, seen_pnt.y, seen_pnt.z,
                             &seen_proj.x, &seen_proj.y);
                    _lineto_w (seen_proj.x, seen_proj.y);
                    _lineto_w (proj[row].x, proj[row].y);
                    break;

                  case DRAW:
                    /*
                     *  The last vertex was seen, and this vertex can be
                     *  seen, so draw a line between them.
                     */
                    _lineto_w (proj[row].x, proj[row].y);
                    break;

                  case SEENCROSSDR:
                    /*
                     *  The last vertex was seen but this point cannot be
                     *  seen.  Draw a line to the crossing, then leave
                     *  off drawing until HIDCROSSDR is hit.
                     */
                    ComputeCrossing (points + prev_row, points + row,
                                     &seen_pnt);
                    Project (seen_pnt.x, seen_pnt.y, seen_pnt.z,
                             &seen_proj.x, &seen_proj.y);
                    _lineto_w (seen_proj.x, seen_proj.y);
                    break;

                  case HIDDENDR:
                    /*
                     *  Some drawing has been done, but the last vertex
                     *  could not be seen and this vertex cannot be seen.
                     *  Ignore the vertex and look for a crossing.
                     */
                    break;

                  default:
                    /*
                     *  Sanity check the state incase something has been
                     *  trashed.
                     */
                    FatalError ("Unrecognized state in DrawFunction");
                };

                prev_row = row;
            };

            /*
             *  If the first point (last point processed) was hidden
             *  but there has been some drawing (the ending state
             *  is either HIDDENDR or SEENCROSSDR), draw a line to
             *  the first crossover point.
             */
            if ((state == HIDDENDR) || (state == SEENCROSSDR)) {
                _lineto_w (first_proj.x, first_proj.y);
            };

            surf++;
        };
        surf++;
    };
}
