!======================================================================
!
! Module Cards			(Playing Cards routines)
!
! Copyright (C) 1995 Microsoft Corporation, All Rights Reserved.
!

!----------------------------------------------------------------------
!
! Module CardBasic	(Basic properties and derived type for cards)
!
module CardBasic
implicit none

integer, parameter :: CardSizeX = 71
integer, parameter :: CardSizeY = 96

integer, parameter :: CardClub = 1
integer, parameter :: CardSpade = 2
integer, parameter :: CardHeart = 3
integer, parameter :: CardDiamond = 4

integer, parameter :: CardJack = 11
integer, parameter :: CardQueen = 12
integer, parameter :: CardKing = 13
integer, parameter :: CardAce = 1

type card 
	integer*1 suit
	integer*1 number
end type card

end module cardbasic

!----------------------------------------------------------------------
!
! Module CardHand		(5 card hand module)
!
module CardHand

use cardbasic
implicit none

private
public hand, cardhandsame
public HandJacksOrBetter, HandTensOrBetter, HandThreeOfAKind
public HandFourOfAKind, HandFullHouse, HandTwoPair, HandFlush
public HandRoyalFlush, HandStraightFlush, HandStraight

integer(1), parameter :: LowSpecialCode = 20

integer(1), parameter :: AnySuit = 20
integer(1), parameter :: MatchLastSuit = 21

integer(1), parameter :: AnyNumber = 30
integer(1), parameter :: GreaterThan10 = 31
integer(1), parameter :: GreaterThan9 = 32
integer(1), parameter :: MatchLastNumber = 33
integer(1), parameter :: NextNumber = 34

type hand
	type(card) :: cards(5)
end type

!!! Scoring hands in 5 card poker
type(hand), parameter :: HandJacksOrBetter = &
	HAND( (/ CARD(AnySuit,GreaterThan10), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,AnyNumber) /) )

type(hand), parameter :: HandTensOrBetter = &
	HAND( (/ CARD(AnySuit,GreaterThan9), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,AnyNumber) /) )

type(hand), parameter :: HandThreeOfAKind = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,AnyNumber) /) )

type(hand), parameter :: HandFourOfAKind = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber) /) )

type(hand), parameter :: HandFullHouse = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber) /) )

type(hand), parameter :: HandTwoPair = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,MatchLastNumber), &
	         CARD(AnySuit,AnyNumber) /) )

type(hand), parameter :: HandFlush = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(MatchLastSuit,AnyNumber), &
	         CARD(MatchLastSuit,AnyNumber), &
	         CARD(MatchLastSuit,AnyNumber), &
	         CARD(MatchLastSuit,AnyNumber) /) )

type(hand), parameter :: HandRoyalFlush = &
	HAND( (/ CARD(AnySuit,10), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber) /) )

type(hand), parameter :: HandStraightFlush = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber), &
	         CARD(MatchLastSuit,NextNumber) /) )

type(hand), parameter :: HandStraight = &
	HAND( (/ CARD(AnySuit,AnyNumber), &
	         CARD(AnySuit,NextNumber), &
	         CARD(AnySuit,NextNumber), &
	         CARD(AnySuit,NextNumber), &
	         CARD(AnySuit,NextNumber) /) )

contains

logical function CardHandSame( InA, InB )
type(hand) :: A, B, InA, InB
logical :: Used(5) 
integer i

A = InA
B = InB
if( B%cards(1)%suit >= LowSpecialCode ) then
    if( A%cards(1)%suit >= LowSpecialCode ) then
	do i = 1, 5
		if( A%cards(i)%suit /= B%cards(i)%suit .or.  &
		    A%cards(i)%number /= B%cards(i)%number ) then
			CardHandSame = .false.
			return
		endif
	end do
	CardHandSame = .true.
	return
    else
	B = InA
	A = InB
    endif
endif

Used = .false.

CardHandSame = FindSame( 1, A%cards(1), Used )
contains

recursive function FindSame( Aindex, Last, Used ) result(FindSameResult)
logical FindSameResult
type(card) Last
integer Aindex, i
logical matched
logical used(5)
if( Aindex > 5 ) then
	FindSameResult = .true.
	return
endif
do i = 1, 5
   if( (.not. Used(i)) .and. &
       ((A%cards(Aindex)%suit == AnySuit) .or. &
        (B%cards(i)%suit == Last%suit)) ) then
       select case( A%cards(Aindex)%number )
	    case( GreaterThan10 )
		matched = B%cards(i)%number > 10 .or. B%cards(i)%number == 1
	    case( GreaterThan9 )
		matched = B%cards(i)%number > 9 .or. B%cards(i)%number == 1
	    case( MatchLastNumber )
	        matched = B%cards(i)%number == Last%number
	    case( NextNumber )
		matched = (Aindex == 5 .and. &
			   B%cards(i)%number==1 .and. Last%number==13) .or. &
			  (B%cards(i)%number == Last%number+1)
	    case( AnyNumber )
		matched = .true.
	    case default
		matched = B%cards(i)%number == A%cards(Aindex)%number
	end select 
	if( matched ) then
		Used(i) = .true.
		if( FindSame( Aindex+1, B%cards(i), Used ) ) then
			FindSameResult = .true.
			return
		endif
		Used(i) = .false.
	endif
   endif
end do
FindSameResult = .false.
return
end function
end function 

end module cardhand

!----------------------------------------------------------------------
!
! Module CardDeck		(Deck of cards module)
!
module CardDeck

use CardBasic
implicit none

private
public deck, carddeckshuffle, carddeckdraw

type deck
	private
	type(card) :: cards(52)
	integer :: top
end type

integer(1), parameter :: CardDeckEmpty = 0

contains

subroutine CardDeckShuffle( DeckToShuffle )
type(deck) DeckToShuffle
type(card) temp
real r
integer i, j, isuit, inumber
DeckToShuffle%Cards = (/ ((CARD(isuit,inumber),isuit=1,4),inumber=1,13) /)
do i = 1, 52
	call Random_Number(r)
	j = int(r*(52-i+1))+i
	temp = DeckToShuffle%Cards(i)
	DeckToShuffle%Cards(i) = DeckToShuffle%Cards(j)
	DeckToShuffle%Cards(j) = temp
end do
DeckToShuffle%top = 1
end subroutine CardDeckShuffle

function CardDeckDraw( DeckToDraw ) result(CardToDraw)
type(deck) DeckToDraw
type(card) CardToDraw
if( DeckToDraw%top > 52 ) then
	CardToDraw%Suit = CardDeckEmpty
	CardToDraw%Number = CardDeckEmpty
else
	CardToDraw = DeckToDraw%cards(DeckToDraw%top)
	DeckToDraw%top = DeckToDraw%top + 1
endif
end function CardDeckDraw

end module CardDeck

!----------------------------------------------------------------------
!
! Module CardDrawMod		(Card Drawing)
!
module carddrawmod

use cardbasic
use BitmapDraw
implicit none

private
public CardDraw, CardDrawInit, CardBlank

integer, parameter :: PictSizeY = 74
integer, parameter :: PictSizeX2 = 24

! These definitions match the colors setup in the palette created by
! CardDrawInit().
integer, parameter :: ColorIndexBlack = 0
integer, parameter :: ColorIndexBlue = 7
integer, parameter :: ColorIndexGreen = 6
integer, parameter :: ColorIndexRed = 2
integer, parameter :: ColorIndexWhite = 1
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pjc = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 16,  2, 32, 34, 32, 32, 34,  2,  2, 34,  2,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  0, 17, 17, 17, 17,  0, 34,  2, 32, 32, 34,  2,  2, 32, 32, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  2,  1, 17, 17, 17,  0,  2, 32,  0, 32,  0,  2,  0,  2,  1, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  2, 32, 17, 17, 16, 85,  2, 34, 34, 34, 34, 34, 34, 32,  1, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  2, 34,  1, 17, 16, 80,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  0,  0,  0, 17, 21, 80,  1, 17, 17, 17, 16, 48, 48, 48, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  3, 51,  1, 17,  5, 80,  1, 17, 17, 17, 16, 48, 48, 48, 17, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  0, 51,  1, 17,  0,  3,  0,  0, 16,  0, 16, 48, 48, 48, 17, 17,  0,  1,  0,  1,  0,  1, 17,  1, &
  3, 51,  1, 17, 17,  3,  0,  1,  0,  1,  0, 48, 48, 48, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  0, 51,  1, 17, 17,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  3, 51,  1, 17, 16,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 51,  1, 17,  3,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  3, 51,  1, 17,  3,  3,  1, 16,  0, 17, 16, 48, 48, 48,  1, 16,  0,  0, 16, 16,  0,  0, 17,  1, &
  0,  3,  1, 17,  0,  3,  1, 17, 17, 17, 16, 48, 48,  3, 48, 17,  0,  1, 16, 17,  0,  1, 17,  1, &
  3, 51,  1, 16, 51,  3, 16, 16,  0, 17, 16, 48,  0, 51,  0, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  0,  3,  1, 16, 48, 51,  0, 17, 17, 17,  0, 48, 51,  0,  0,  0, 17, 16,  0,  0, 17, 17, 17,  1, &
  3, 51,  1, 16, 51, 48,  2,  1, 17, 16, 32, 48,  3,  0,  1,  1,  1, 17, 17, 17, 17, 17, 17,  1, &
  3, 51,  0, 17,  0,  2,  2, 32,  0,  2, 34,  3, 51,  0, 16,  1, 32,  1, 17, 17, 17, 17, 16,  1, &
  0, 48, 17,  0, 32, 32,  2, 34, 32,  2, 34,  0,  0,  0,  3,  1, 34, 32,  1, 17, 17, 17,  3,  1, &
  1,  1, 17, 16,  2,  2,  2, 51, 32,  0, 32,  1, 16,  1,  3,  1, 34, 50, 16,  1, 17, 16, 51,  1, &
  0,  0,  0, 16, 32, 32,  2, 34, 32, 16,  0, 17,  0, 16, 51,  1, 35, 51, 33,  0,  0,  3, 51,  1, &
  1, 16,  1, 16,  2,  2,  2, 51, 32, 17,  1, 16,  1,  3, 51,  1, 51, 35, 49,  0,  0,  3, 51,  1, &
  0,  3,  1, 16, 32, 32,  2, 34, 32,  1, 17,  0, 16, 51,  0,  1, 35, 51, 33,  0,  0,  3, 51,  1, &
  1,  3,  1, 16,  2,  2,  2, 51, 32,  0, 16,  1, 32, 51, 51,  1, 34, 50, 33,  0,  0,  3, 51,  1, &
  0, 51,  1, 16, 32, 32,  2, 34, 32, 16,  0, 18,  3, 48,  0,  1, 34, 34, 33,  0,  0,  3, 51,  1, &
  1,  3,  1,  2,  2,  0,  2, 51, 32,  1,  1,  0, 51, 51, 51,  1, 34, 50, 33,  0,  0,  3, 51,  1, &
  0, 51,  0,  0, 32,  0,  2, 34, 32, 32, 16, 32, 48, 48,  0,  1, 35, 51, 33,  0,  0,  3, 51,  1, &
  3, 51,  0,  0,  0,  0, 35, 50,  3,  2,  2,  3, 51, 51, 51,  1, 51, 35, 49,  0,  0,  3, 51,  1, &
  3, 51,  0, 16,  0, 16, 34, 34,  0, 48, 32, 48, 48,  0,  0,  1, 35, 51, 32,  0, 34,  3, 51,  1, &
  3, 51,  1, 17,  1, 16, 35, 50,  3, 51,  3, 51, 51, 51, 51,  1, 34, 50, 32, 34, 51,  3, 51,  1, &
  3, 51,  1,  1, 17,  0, 34, 34,  0,  0, 51,  3,  0,  0,  0,  1,  0,  0, 34, 34, 34,  3, 51,  1, &
  3, 51,  0, 16, 16, 16, 35, 50,  3, 51, 51, 51, 51, 51, 51,  0, 17,  2, 35, 50, 34,  3, 51,  1, &
  3, 51,  1,  1,  1,  0, 34, 34,  0, 50,  3,  0,  3,  2, 48, 17, 16, 34, 34, 34,  0,  3, 51,  1, &
  3, 51,  0, 16, 16, 18, 51, 32, 51, 48, 32, 17, 16, 32,  1, 17, 16, 35, 50, 32,  1,  3, 51,  1, &
  3, 51,  1,  1,  1,  2, 34, 32,  3, 51,  1,  0,  1,  1, 17, 17,  2, 34, 34,  1,  1,  3, 51,  1, &
  3, 51,  0, 16, 16,  2, 51, 32, 48,  0, 16, 34, 32, 16, 17, 16,  2, 51, 32, 16, 16,  3, 51,  1, &
  3, 51,  0,  0,  0, 35, 50,  0,  1, 16, 16, 34, 32, 16,  3, 48, 35, 50,  0,  0,  0,  3, 51,  1, &
  3, 51,  1,  1,  2, 34, 34,  1, 17, 16,  1,  0,  1,  3, 51,  0, 34, 34,  1,  1,  1,  3, 51,  1, &
  3, 51,  1,  0, 34, 51, 32, 17, 17,  0, 32, 17, 16, 32, 51, 48, 35, 50, 16, 16, 16,  3, 51,  1, &
  3, 51,  0,  2, 34, 34, 32, 17, 16, 50,  3,  0,  3,  2, 48,  2, 34, 32,  1,  1,  1,  3, 51,  1, &
  3, 51,  2, 34, 51, 34,  1, 17,  3, 51, 51, 51, 51, 51, 51,  2, 51, 32, 16, 16, 16,  3, 51,  1, &
  3, 51,  2, 34, 34, 32,  0,  0,  0,  0,  0,  3,  3, 48,  0,  2, 34, 32,  1, 17,  1,  3, 51,  1, &
  3, 51,  3, 50, 32, 34, 50, 33,  3, 51, 51, 51, 51,  3, 51,  2, 51, 32, 17,  1, 17,  3, 51,  1, &
  3, 51,  2, 32,  0, 35, 51, 33,  0,  0,  0, 48, 48, 32, 48,  2, 34, 32, 16,  0, 16,  3, 51,  1, &
  3, 51,  0,  0,  1, 51, 35, 49,  3, 51, 51, 51,  2,  2,  3,  2, 51, 32,  0,  0,  0,  3, 51,  1, &
  3, 51,  0,  0,  1, 35, 51, 33,  0,  0, 48, 48, 32, 16, 32, 34, 34,  0,  0, 32,  0,  3, 48,  1, &
  3, 51,  0,  0,  1, 34, 50, 33,  3, 51, 51, 48,  1,  1,  0, 35, 50,  0,  2,  2,  1,  3,  1,  1, &
  3, 51,  0,  0,  1, 34, 34, 33,  0,  0, 51,  2, 16,  0, 16, 34, 34,  0, 32, 32, 17,  3, 48,  1, &
  3, 51,  0,  0,  1, 34, 50, 33,  3, 51, 48, 33,  0, 16,  0, 35, 50,  2,  2,  0, 17,  3,  1,  1, &
  3, 51,  0,  0,  1, 35, 51, 33,  0,  3, 48, 16,  1, 17,  0, 34, 34,  0, 32, 32, 17,  3,  0,  1, &
  3, 51,  0,  0,  1, 51, 35, 49,  3, 51,  1,  0, 17,  1, 16, 35, 50,  2,  2,  0, 17,  0, 17,  1, &
  3, 51,  0,  0,  1, 35, 51, 33,  3, 48, 16,  1, 16,  0, 16, 34, 34,  0, 32, 32, 16,  0,  0,  1, &
  3, 48, 17, 17,  0, 18, 50, 33,  3,  0,  0, 17,  0, 32,  0, 35, 50,  2,  2,  0, 17, 17,  1,  1, &
  3,  1, 17, 17, 17,  0, 34, 33,  3,  0,  0,  0,  2, 34,  0, 34, 34,  0, 32, 32,  1, 16, 48,  1, &
  0, 17, 17, 17, 17, 17,  0, 33,  0, 16,  3, 51,  2, 34,  0,  0, 34,  2,  0,  1, 16,  3, 51,  1, &
  1, 17, 17, 17, 17, 17, 17,  1,  1,  0,  3,  0, 48, 32, 17, 17,  2,  0, 51, 48, 17,  3, 51,  1, &
  1, 17, 17, 16,  0,  0, 17, 16,  0,  0,  3, 48, 48,  1, 17, 17, 16,  3, 48, 48, 17,  3,  0,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16,  3, 48,  0, 48, 17, 16,  0, 16, 19,  3, 48, 17,  3, 51,  1, &
  1, 17,  0,  1, 16, 17,  0,  1, 16, 51,  0, 48, 48, 17, 17, 17, 17,  3,  0,  1, 17,  3,  0,  1, &
  1, 16,  0,  0, 16, 16,  0,  0, 17,  0, 48, 48, 48, 17, 16,  0, 17,  3,  3,  1, 17,  3, 51,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 17, 17,  1, 17,  3,  3,  1, 17,  3, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 17, 17,  1, 17,  3,  0, 17, 17,  3, 51,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 17, 17,  1, 17,  3,  1, 17, 17,  3, 48,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 16, 48, 48, 48,  1,  0,  1,  0,  3,  1, 17, 17,  3, 51,  1, &
  1, 17,  0,  1,  0,  1,  0,  1, 17, 16, 48, 48, 48, 16,  0, 16,  0,  3,  0,  1, 17,  3, 48,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 16, 48, 48, 48, 17, 17, 17, 17,  0, 85,  1, 17,  3, 51,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 16, 48, 48, 48, 17, 17, 17, 17,  0, 85, 17, 16,  0,  0,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 16,  0,  0,  0,  0,  0,  0,  0,  0, 80, 17, 17,  2, 34,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17,  0, 34, 34, 34, 34, 34, 34, 34,  5, 80, 17, 17, 16, 34,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17,  2,  0,  2,  0,  0, 32,  0, 34,  0,  1, 17, 17, 17,  2,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16, 32, 34,  2,  2, 32, 32, 34,  2, 32,  1, 17, 17, 17, 16,  1, &
  1, 17, 17, 17, 17, 17, 17, 17,  2,  2, 34,  2,  2, 32, 32, 34, 32, 34,  0, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pjd = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17,  2, 18, 34, 18, 33, 34, 18, 33, 34, 18,  0, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 16, 33, 34, 18, 33, 34, 18, 33, 33, 32,  1, 17, 17, 17, 33, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 17,  2, 17, 18, 33, 17, 18, 33, 18,  0, 17, 17, 17, 18, 34, 17, 17, 17,  1, &
  0, 17, 17, 17, 17, 17, 16, 34, 34, 34, 34, 34, 34, 34,  1, 17, 17, 17, 34, 34, 33, 17, 17,  1, &
  0,  1, 17, 17, 17, 17, 16,  0,  0,  0,  0,  0,  0,  0,  1, 17, 17, 20, 34, 34, 36, 17, 17,  1, &
  0, 48, 17, 17, 17, 17, 16,  0,  0, 48, 17, 17, 17, 17,  0, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  0, 48, 17, 17, 17, 17, 16,  0, 48, 48, 17, 17, 17, 17,  0, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  0, 48, 17, 17, 17, 17, 16, 48, 48, 48, 16,  0, 16,  0,  3,  1, 18, 34, 34, 34, 34, 34, 17,  1, &
  0, 48, 17, 17,  1, 17, 16, 48, 48, 48,  1,  0,  1,  0,  3,  1, 34, 34, 34, 34, 34, 34, 33,  1, &
  0, 48, 17, 16,  1, 17, 16, 48, 48, 48, 17, 17,  1, 17,  3,  1, 18, 34, 34, 34, 34, 34, 17,  1, &
  0, 48, 16,  3,  1, 17, 16, 48, 48, 48, 17, 17,  1, 17,  3,  1, 17, 66, 34, 34, 34, 65, 17,  1, &
  0,  0,  3, 48, 17, 17, 16, 48, 48, 48, 17, 17,  1, 17,  3,  1, 17, 66, 34, 34, 34, 65, 17,  1, &
  2, 51, 48,  0, 17, 17, 16, 48, 48, 48, 17, 16,  0, 17,  3,  1, 17, 20, 34, 34, 36, 17, 17,  1, &
  0, 48,  0,  1, 17, 17, 16, 48, 48, 48, 17, 17, 17, 17,  3,  0,  1, 17, 34, 34, 33, 17, 17,  1, &
  2, 48,  1, 17, 17, 16,  3, 48, 48, 48, 17, 16,  0, 16, 19,  3, 48, 17, 18, 34, 17, 17, 17,  1, &
  0, 48, 17, 17, 17,  3, 48, 48, 48, 48,  1, 17, 17, 16,  3, 48, 48, 17, 17, 33, 17, 17, 17,  1, &
  2, 48, 17, 17, 17,  3,  0, 48,  3,  2, 16, 17, 17,  1, 32, 51, 48, 17, 17, 17, 17, 17, 16,  1, &
  0, 48, 17, 17, 17,  3, 51,  3, 48, 34, 34,  0,  0, 34, 34,  0,  0,  1, 17, 17, 17, 17,  2,  1, &
  2, 48, 17, 17, 17,  0,  0,  0,  0, 34, 34, 34, 34, 34, 32,  1,  3,  0,  1, 17, 17, 16, 48,  1, &
  0, 48, 17, 17,  0, 32,  0, 16,  0,  0, 34, 34, 34, 32,  0,  0,  3,  1,  0,  1, 17, 16, 50,  1, &
  2, 48, 17,  0,  3,  2,  0,  0, 16,  0,  0,  0,  0,  0,  0, 16,  3,  0,  3, 48,  1, 16, 48,  1, &
  0, 48,  0,  0,  0, 48, 32,  0,  0, 16,  0,  0,  0,  0, 16,  0,  3,  1,  3, 48, 32,  0, 50,  1, &
  2, 48,  0,  0,  0,  3,  2,  0,  0,  0, 16, 16, 16, 16,  0, 32,  3,  0,  0, 48, 34, 32, 48,  1, &
  0, 48,  0, 32,  0,  0, 48, 32,  0,  0,  0,  0,  0,  0, 32, 32,  3, 48, 16, 51,  0, 32, 50,  1, &
  0,  0, 16,  2,  0,  0,  3,  2,  0, 32, 32, 32, 32, 32, 32, 32, 16, 48,  0,  3, 51,  0, 48,  1, &
  1, 16, 16, 48, 32,  0,  0, 48, 32,  2, 50, 50, 50, 50, 50, 48, 16, 51,  1,  0,  3, 48, 50,  1, &
  0,  0, 16,  3,  2,  0,  0,  3,  2,  0, 32, 32, 32, 32, 32, 32, 16,  3, 48,  1,  0,  0, 48,  1, &
  1, 16,  0, 48, 48, 32,  0,  0, 48, 32,  0, 32, 32, 32, 32, 32, 16,  0, 51, 48,  0, 16, 50,  1, &
  0,  0, 34,  3,  0,  3,  0,  0,  3,  2,  0, 32, 32, 32, 32, 32, 16,  1,  0, 51, 48,  0, 48,  1, &
  1,  0, 18,  3,  3, 48, 32,  0,  0, 48, 32,  2, 50, 50, 50, 48, 16,  1,  1,  0, 51, 48, 50,  1, &
  0, 48, 34, 32, 48,  0, 35,  0,  0,  3,  2,  0, 32, 32, 32, 32, 16,  1,  0, 48,  0,  0, 48,  1, &
  0, 48, 17, 32, 48, 48, 35, 32,  0,  0, 48, 32,  0, 32, 32, 32, 16,  1,  0,  0, 48, 32, 50,  1, &
  2, 48, 34, 32, 48,  0, 35, 35,  0, 16,  3,  2,  0, 32, 32, 32, 16,  1,  3, 48, 48, 16, 48,  1, &
  0, 48, 17, 32, 48, 48, 35, 32,  0,  0,  2, 48, 32,  2, 50, 48,  0,  1,  0,  3,  2, 32, 50,  1, &
  2, 48, 34, 32, 48,  0, 35,  0, 16,  0, 50, 35,  2,  0, 32, 32,  0,  0, 51,  3,  2, 16, 48,  1, &
  0, 48, 17, 32, 48, 48, 32,  0,  0,  2,  3, 34, 48, 32,  2,  0, 16,  0,  0, 48, 34, 32, 50,  1, &
  2, 48, 34, 32, 48,  0,  0, 16,  0,  0, 32, 50, 35,  2,  0,  0,  0, 32, 48, 48, 33, 16, 48,  1, &
  0, 48, 18,  3,  3,  0,  0,  0, 32, 32,  2,  3, 34, 48,  0, 16,  3, 32,  0, 48, 34, 32, 50,  1, &
  2, 48, 34,  3,  0,  1,  0,  0, 50, 50,  0, 32, 50,  0,  0,  0, 35, 32, 48, 48, 33, 16, 48,  1, &
  0, 48, 16, 48, 51,  1,  0, 16, 32, 32, 32,  2,  3,  0, 16,  3, 35, 32,  0, 48, 34, 32, 50,  1, &
  2, 48, 32, 48,  0,  1,  0, 16, 32, 32, 32,  0, 32, 48,  0,  0, 35, 32, 48, 48, 33, 16, 48,  1, &
  0, 48,  0,  0, 48,  1,  0, 16, 32, 32, 32, 32,  2,  3,  0,  0,  3, 32,  0, 48, 34, 32, 48,  1, &
  2, 48, 51, 48,  1,  1,  0, 16, 50, 50, 50, 50,  0, 32, 48,  0,  0, 32, 51,  3,  2, 16,  1,  1, &
  0, 48,  0, 51, 48,  1,  0, 16, 32, 32, 32, 32, 32,  2,  3,  0,  0,  3,  0,  3,  2, 32,  0,  1, &
  2, 48, 16,  0, 51, 48,  0, 16, 32, 32, 32, 32, 32,  0, 32, 48,  0,  0, 32, 48, 48,  0, 17,  1, &
  0, 48,  0,  1,  0, 51,  0, 16, 32, 32, 32, 32, 32, 32,  2,  3,  0,  0,  2,  3,  0, 16,  0,  1, &
  2, 48, 51,  0,  1,  3, 48, 16, 50, 50, 50, 50, 50, 50,  0, 32, 48,  0,  0, 32, 48, 16, 17,  1, &
  0, 48,  3, 51,  0,  0, 48, 16, 32, 32, 32, 32, 32, 32, 32,  2,  3,  0,  0,  2,  0, 16,  0,  1, &
  2, 48, 32,  3, 48, 16, 51,  0, 32, 32,  0,  0,  0,  0,  0,  0, 32, 48,  0,  0, 32,  0, 48,  1, &
  0, 48, 34, 32, 48,  0,  3,  0, 32,  0, 16, 16, 16, 16,  0,  0,  2,  3,  0,  0,  0,  0, 50,  1, &
  2, 48,  0, 32, 51,  1,  3,  0,  0, 16,  0,  0,  0,  0, 16,  0,  0, 32, 48,  0,  0,  0, 48,  1, &
  0, 48, 17,  0, 51,  0,  3,  0, 16,  0,  0,  0,  0,  0,  0, 16,  0,  2,  3,  0,  1, 16, 50,  1, &
  2, 48, 17, 17,  0,  1,  3,  0,  0,  0, 34, 34, 34, 32,  0,  0, 16,  0, 32,  1, 17, 16, 48,  1, &
  0, 48, 17, 17, 17,  0,  3,  1,  0, 34, 34, 34, 34, 34, 32,  0,  0,  0,  1, 17, 17, 16, 50,  1, &
  2,  1, 17, 17, 17, 17,  0,  0,  2, 34, 32,  0,  2, 34, 32, 51,  3, 51,  1, 17, 17, 16, 48,  1, &
  0, 17, 17, 17, 17, 17, 16, 51, 48, 33,  1, 17, 16, 18,  3,  0, 48,  3,  1, 17, 17, 16, 50,  1, &
  1, 17, 17, 17, 33, 17, 16, 48, 51,  0, 17, 17, 17,  0, 48, 48, 48, 51,  1, 17, 17, 16, 48,  1, &
  1, 17, 17, 18, 34, 17, 16, 51,  3, 16, 16,  0, 17, 16, 48, 48, 51,  0, 17, 17, 17,  0, 50,  1, &
  1, 17, 17, 34, 34, 33, 17,  0,  3,  1, 17, 17, 17, 16, 48, 48, 48, 17, 17, 17,  0,  0, 48,  1, &
  1, 17, 20, 34, 34, 36, 17, 17,  3,  1, 16,  0, 17, 16, 48, 48, 48, 17, 17, 16,  0, 51, 50,  1, &
  1, 17, 66, 34, 34, 34, 65, 17,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17, 17, 16, 51,  0,  0,  1, &
  1, 17, 66, 34, 34, 34, 65, 17,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17, 17,  3,  0, 16, 48,  1, &
  1, 18, 34, 34, 34, 34, 34, 17,  3,  1, 17,  1, 17, 16, 48, 48, 48, 17, 17,  0, 17, 16, 48,  1, &
  1, 34, 34, 34, 34, 34, 34, 33,  3,  0,  1,  0,  1,  0, 48, 48, 48, 17, 17,  1, 17, 16, 48,  1, &
  1, 18, 34, 34, 34, 34, 34, 17,  3,  0,  0, 16,  0, 16, 48, 48, 48, 17, 17, 17, 17, 16, 48,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 16,  1, 17, 17, 17, 16, 48, 48,  0, 17, 17, 17, 17, 16, 48,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 16,  1, 17, 17, 17, 16, 48,  0,  0, 17, 17, 17, 17, 16, 48,  1, &
  1, 17, 20, 34, 34, 36, 17, 17, 17,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17, 17, 17,  0,  1, &
  1, 17, 17, 34, 34, 33, 17, 17, 17,  2, 34, 34, 34, 34, 34, 34, 32, 17, 17, 17, 17, 17, 16,  1, &
  1, 17, 17, 18, 34, 17, 17, 17, 16,  2, 17, 34, 17, 17, 34, 17, 18,  1, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 33, 17, 17, 17,  0, 33, 33, 34, 18, 33, 34, 18, 33, 32, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 16,  2, 18, 33, 34, 18, 33, 34, 18, 34, 18,  1, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pjh = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17,  2, 33, 34, 33, 33, 34, 18, 18, 34, 18,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  0, 17,  0,  1, 17, 16, 34, 18, 33, 33, 34, 18, 18, 33, 32, 17, 34, 36, 17, 20, 34, 33, 17,  1, &
  2,  1, 16, 48, 17, 17,  2, 33, 17, 33, 17, 18, 17, 18,  0, 18, 34, 34, 65, 66, 34, 34, 17,  1, &
  2,  1, 16, 51,  1, 17,  2, 34, 34, 34, 34, 34, 34, 32,  1, 34, 34, 34, 33, 34, 34, 34, 33,  1, &
  2,  1, 16, 51,  1, 17,  0,  0,  0,  0,  0,  0,  0,  0,  1, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  0,  0,  1,  3, 48, 17, 16, 48, 48, 48, 48, 17, 17, 16, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 33, 33, 16, 48, 17, 16, 48, 48, 48, 48, 17,  0, 16, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 33, 33, 17,  0, 17,  3,  3,  3,  3,  1,  0, 16, 16, 17, 66, 34, 34, 34, 34, 34, 34, 65,  1, &
  1,  1,  1, 16, 48, 17,  3,  3,  3,  3,  1, 17, 17, 22, 17, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  0,  0,  1,  3, 48, 17,  3,  3,  3,  3,  1, 17, 17, 17,  1, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  2,  1, 16, 51,  1, 17,  3,  3,  3,  3,  1, 17, 17, 17, 16, 17, 34, 34, 34, 34, 34, 65, 17,  1, &
  2,  1, 16, 51,  1, 17,  3,  3,  3,  3,  1, 17,  0, 22,  0, 17, 18, 34, 34, 34, 34, 17, 17,  1, &
  2,  1, 16, 48, 17, 17,  3,  3,  3,  3,  1,  0,  1, 16, 17, 17, 17, 34, 34, 34, 33, 17, 17,  1, &
  2,  1,  0,  1, 17,  0,  3,  3,  3,  3,  1, 17, 17, 16, 17, 17, 17, 18, 34, 34, 17, 17, 17,  1, &
  2,  1, 17, 17, 16, 51, 48,  3,  3,  3,  1, 17, 16,  0, 17, 17, 17, 17, 34, 33, 17,  1, 17,  1, &
  2,  1, 17, 17,  3, 48, 48, 48, 48,  3,  1, 17, 17, 16, 17, 17, 17, 17, 18, 17,  0, 32,  1,  1, &
  2,  1, 17, 17,  3,  3, 48, 48, 48, 48, 34, 33, 17, 16,  0,  0,  1, 17, 17, 17,  2, 34,  1,  1, &
  2,  1, 17, 17,  3, 48,  3, 48, 48,  2, 34, 34,  0,  3, 48, 51, 48, 17, 17, 17, 16, 32, 17,  1, &
  2,  1, 17, 17, 16, 51, 51,  3,  0, 34, 34, 34, 34, 32, 51,  0, 48, 17, 17,  0,  0, 32, 17,  1, &
  2,  1, 17, 17, 16,  0,  0,  0, 51,  0, 34, 34, 34, 32,  3, 51, 48, 17, 16, 17, 16, 32, 17,  1, &
  2,  1, 17, 16,  2, 18,  3, 51, 51, 51,  0,  0,  0, 51, 48, 51,  0, 17,  1, 17, 17,  0,  1,  1, &
  2,  1, 16,  0,  0,  2, 16,  3, 51, 51, 51, 51, 51, 51, 51,  0, 18,  0,  1, 16,  1,  1,  1,  1, &
  2,  0,  1, 18, 32,  0,  2, 16,  0, 51, 51, 51, 51, 48,  0, 18, 34,  0,  0, 17,  0,  0,  1,  1, &
  2,  2, 33, 17, 18, 32,  0,  2, 18,  0,  0,  0,  0,  2, 18, 34,  0,  0, 34,  1, 16, 17,  1,  1, &
  0,  0,  2, 33, 17, 18, 32,  0,  2, 18, 18, 18, 18, 18, 34,  0,  0, 34, 17, 16,  0,  0,  1,  1, &
  3, 51,  0,  2, 33, 17, 18, 32,  0,  2,  0,  0,  0,  2,  0,  0, 34, 17, 17, 32,  0, 16, 17,  1, &
  3, 48, 51, 48,  2, 33, 17, 18, 32,  0, 32, 17, 16, 32,  0, 34, 17, 17, 34,  0, 17,  0, 17,  1, &
  3,  3, 51,  3, 48,  2, 33, 17, 18, 32,  2,  1,  2,  0, 34, 17, 17, 34,  0, 48, 16, 16,  1,  1, &
  0,  0, 48, 51, 32,  0,  2, 33, 17, 18,  0, 32, 32,  2, 17, 17, 34,  0, 51, 48, 17, 17,  0,  1, &
  1, 17,  3, 50,  0, 51, 48,  2, 33, 17, 32,  2,  0, 33, 17, 18,  0, 51, 51, 48, 16, 16,  0,  1, &
  0, 17, 16, 32,  3, 51, 51, 48, 34, 17, 18,  0,  2, 17, 18, 32, 51, 51, 51, 48, 17, 17,  0,  1, &
  1, 16, 16,  0, 48,  0,  0,  0,  0, 33, 17, 32, 33, 17, 32,  0,  0,  0,  0, 48, 16, 16,  0,  1, &
  0, 17, 17,  3, 48, 17, 17, 17, 16, 34, 17,  0,  1, 18, 32, 17, 17, 17, 16, 48, 17, 17,  0,  1, &
  0,  1,  1,  3, 48, 16,  0,  0, 16,  2, 32, 17, 16, 34,  0, 16,  0,  0, 16, 48, 16, 16,  0,  1, &
  0,  1, 17, 16, 48, 16, 51,  0,  0, 16,  1,  0,  1,  0, 16,  0,  3, 48, 16, 48, 17, 17,  0,  1, &
  0,  0, 16, 16, 48, 16, 51,  0, 17,  0, 16, 34, 32, 16,  1, 16,  3, 48, 16, 48, 16, 16,  0,  1, &
  0,  0, 16, 16, 48, 16, 51,  0, 17,  0, 16, 34, 32, 16,  1, 16,  3, 48, 16, 48, 16, 16,  2,  1, &
  0,  1, 17, 16, 48, 16, 51,  0,  0, 16,  1,  0,  1,  0, 16,  0,  3, 48, 16, 48, 17,  0, 35,  1, &
  0,  0, 16, 16, 48, 16,  0,  0, 16,  2, 32, 17, 16, 34,  0, 16,  0,  0, 16, 51,  0,  2, 51,  1, &
  0,  1, 17, 16, 48, 17, 17, 17, 16, 34, 17,  0,  1, 18, 32, 17, 17, 17, 16, 51,  0, 35, 48,  1, &
  0,  0, 16, 16, 48,  0,  0,  0,  0, 33, 17, 32, 33, 17, 32,  0,  0,  0,  0, 48,  2, 51,  0,  1, &
  0,  1, 17, 16, 51, 51, 51, 48, 34, 17, 18,  0,  2, 17, 18, 32, 51, 51, 51,  0, 35, 48,  0,  1, &
  0,  0, 16, 16, 51, 51, 48,  2, 17, 17, 32,  2,  0, 33, 17, 34,  0, 51, 48,  2, 51,  0,  0,  1, &
  0,  1, 17, 16, 51, 48,  2, 33, 17, 18,  0, 32, 32,  2, 17, 17, 34,  0,  0, 35, 48,  0, 51,  1, &
  1,  0, 16, 16, 48,  2, 33, 17, 18, 32,  2,  1,  2,  0, 34, 17, 17, 34,  0, 51,  0,  3, 51,  1, &
  1, 16,  1, 16,  2, 33, 17, 18, 32,  0, 32, 17, 16, 32,  0, 34, 17, 17, 34,  0,  3, 48, 51,  1, &
  1, 16, 16,  0, 33, 17, 18, 32,  0,  2,  0,  0,  0,  2,  0,  0, 34, 17, 17, 34,  0,  0, 48,  1, &
  1,  0,  0,  0, 17, 18, 32,  0,  2, 34, 18, 18, 18, 18, 18,  0,  0, 34, 17, 17, 34,  0,  0,  1, &
  1,  1, 16, 17,  2, 32,  0,  2, 34, 18,  0,  0,  0,  0,  2, 18,  0,  0, 34, 17, 17, 34,  2,  1, &
  1,  0,  0,  1, 16,  0,  2, 34, 16,  0, 51, 51, 51, 51, 48,  0, 18,  0,  0, 34, 17,  0,  2,  1, &
  1,  1,  1,  0, 17,  0,  2, 16,  3, 51, 51, 51, 51, 51, 51, 51,  0, 18,  0,  0,  0, 17,  2,  1, &
  1,  0,  1, 17, 17,  1, 16,  3, 48,  3, 48,  0,  0,  3, 51, 51, 51,  2, 18,  0, 17, 17,  2,  1, &
  1, 16, 32, 17, 16, 17, 16, 51, 51, 48,  2, 34, 34, 32,  3, 48,  0,  0,  0, 17, 17, 17,  2,  1, &
  1, 16, 32,  0,  1, 17, 16, 48,  3, 51,  0,  0, 34, 34, 32,  3,  3, 51, 48, 17, 17, 17,  2,  1, &
  1, 16, 32, 17, 17, 17, 16, 51, 48, 48, 17, 17,  0, 34,  0, 48, 51,  0, 51,  1, 17, 17,  2,  1, &
  1,  2, 34,  1, 17, 17, 17,  0,  1,  0, 17, 17, 17,  0, 48, 48, 48, 51,  3,  1, 17, 17,  2,  1, &
  1,  0, 32,  1, 18, 17, 17, 17, 17, 16, 17, 17, 17, 16, 48, 48, 48, 48, 51,  1, 17, 17,  2,  1, &
  1, 17,  1, 17, 34, 33, 17, 17, 17, 16,  0, 17, 17, 19,  3,  3,  0, 51, 48, 17, 17, 17,  2,  1, &
  1, 17, 17, 18, 34, 34, 17, 17, 17, 16, 17, 17, 17,  3,  3,  3,  3,  0,  1, 17,  0,  1,  2,  1, &
  1, 17, 17, 34, 34, 34, 33, 17, 17, 16, 17,  0,  1,  3,  3,  3,  3,  1, 17, 16, 48, 17,  2,  1, &
  1, 17, 18, 34, 34, 34, 34, 17, 16,  0, 16,  1, 17,  3,  3,  3,  3,  1, 17,  3, 48, 17,  2,  1, &
  1, 17, 66, 34, 34, 34, 34, 33, 16, 17, 17, 17, 17,  3,  3,  3,  3,  1, 17,  3, 48, 17,  2,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 17,  1, 17, 17, 17,  3,  3,  3,  3,  1, 16, 51,  1,  0,  0,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 17, 16, 17, 17, 17,  3,  3,  3,  3,  1, 16, 48, 17,  1,  1,  1, &
  1, 66, 34, 34, 34, 34, 34, 34, 65, 16, 16, 16,  1,  3,  3,  3,  3,  1, 16,  1, 17, 33, 33,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 16,  1, 16, 48, 48, 48, 48, 17, 16, 48, 17, 33, 33,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 17, 17, 16, 48, 48, 48, 48, 17, 16, 51,  1,  0,  0,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33,  0,  0,  0,  0,  0,  0,  0,  0,  1, 17,  3, 48, 17,  2,  1, &
  1, 34, 34, 34, 33, 34, 34, 34, 33,  0, 34, 34, 34, 34, 34, 34, 34,  1, 17,  3, 48, 17,  2,  1, &
  1, 18, 34, 34, 65, 66, 34, 34, 16,  2, 17, 18, 17, 17, 33, 17, 34,  1, 17, 16, 48, 17,  2,  1, &
  1, 17, 34, 36, 17, 20, 34, 33, 16, 33, 34, 18, 18, 33, 33, 34, 18, 32, 17, 17,  0,  1, 16,  1, &
  1, 17, 17, 17, 17, 17, 17, 17,  2, 18, 34, 18, 18, 33, 33, 34, 33, 34,  1, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pjs = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17,  0, 17, 16, 32, 34, 32, 32, 34,  2,  2, 34,  2, 32, 17, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 16, 51,  1, 17,  2,  2, 32, 32, 34,  2,  2, 32, 34,  1, 17, 17, 17, 16, 17, 17, 17, 17,  1, &
  1,  3,  2, 48, 17, 16, 32,  0, 32,  0,  2,  0,  2, 32, 17, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1,  3,  2, 48, 17, 17,  2, 34, 34, 34, 34, 34, 34, 32, 17, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1,  3,  2, 48, 17, 17,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1, 16, 51,  1, 17, 17,  1, 17, 17,  3,  3,  3,  3,  1, 17, 17, 16,  0,  0,  0,  0, 17, 17,  1, &
  1, 17,  0, 17, 17, 17,  1,  0, 17,  3,  3,  3,  3,  1, 17, 17,  0,  0,  0,  0,  0,  1, 17,  1, &
  1, 16, 51,  1, 17, 17,  1,  1,  0, 16, 48, 48, 48, 48, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  1,  3,  2, 48, 17, 17, 97, 17, 17, 16, 48, 48, 48, 48, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  1, 16, 51,  1, 17, 16, 17, 17, 17, 16, 48, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1, 17,  0, 17, 17,  1, 17, 17, 17, 16, 48, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1, 16, 51,  1, 17,  0, 17,  0, 22, 16, 48, 48, 48, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1,  3,  2, 48, 17, 17,  1, 16,  6, 16, 48, 48, 48, 48, 17,  0,  0,  0, 16, 16,  0,  0,  1,  1, &
  1,  3,  2, 48, 17, 17,  1, 17, 17, 16, 48, 48, 48, 51,  0, 16,  0,  1, 16, 17,  0,  0, 17,  1, &
  1,  3,  2, 48, 17, 17,  0,  1, 17,  3, 48, 48, 48, 51, 51,  1, 17, 17,  0,  1, 17, 17, 17,  1, &
  1, 16, 51,  1, 17, 17,  1, 17, 17,  3, 48, 51,  3,  3,  3, 48, 17, 16,  0,  0, 17, 17, 17,  1, &
  1,  0,  0,  1, 17, 16,  1, 17, 17, 32,  3,  3,  3,  3, 48, 48, 17, 17, 17, 17, 17, 17, 17,  1, &
  1,  1, 17,  0, 16,  0,  6, 17, 34, 34,  3,  0, 48, 48,  3, 48, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 16,  1, 16,  0,  0,  0,  2, 34, 34,  0, 51,  3,  3, 51,  0, 17, 17, 17, 17, 17, 17, 17,  1, &
  1,  1, 16,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17, 17, 17, 17,  1, &
  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 16,  0, 17, 17, 17, 17, 17,  1, &
  1,  0,  1,  0,  0, 16, 48, 48, 48, 48, 48, 48, 48, 48, 48, 16,  0, 16,  0, 17, 17, 17, 17,  1, &
  0, 32,  0, 16, 16, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 16, 16,  2,  0, 17, 17, 17,  1, &
  2,  0, 32, 16,  0,  1,  2, 34, 34, 34, 34, 34, 34, 34, 32, 16,  0, 16, 34, 32,  0, 17,  0,  1, &
  0,  2,  0,  1,  1,  1,  0, 35, 34, 50, 35, 34, 50, 35, 32, 16, 16, 16, 32,  0, 34,  0, 48,  1, &
  0, 32,  3,  1,  0,  0, 16, 34, 35,  3, 35, 35,  3, 34, 32, 16,  0, 16,  0, 34,  0, 51, 48,  1, &
  2,  0, 48, 48, 16, 16, 16,  2, 48, 50, 51, 50, 48, 50, 32, 16, 16, 16, 34,  0, 51,  0,  0,  1, &
  0,  3,  3,  0, 16,  0,  0, 48, 35, 34, 48, 50, 35, 34, 32, 16,  0, 16,  0,  3, 51, 51,  0,  1, &
  0, 48, 48, 48,  1,  0, 48,  0,  0,  1,  0,  1,  0,  1,  0, 16, 16, 16,  0, 51,  0,  0,  3,  1, &
  3,  3,  3,  0,  0,  3,  0, 48, 16, 16, 16, 16, 16, 16, 16, 16,  0,  0,  3, 51, 51, 48,  3,  1, &
  0, 48, 48,  2,  0, 48,  3, 48,  1,  0,  1,  0,  1,  0,  0, 16, 16,  0, 51,  3,  0,  0, 48,  1, &
  3, 51,  0, 18,  3,  0,  0, 48,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 51, 51, 51,  0, 48,  1, &
  3,  0,  2, 32, 48,  3, 51, 48, 17, 17, 17, 17, 17, 17, 17, 17, 17,  3, 48, 48,  0,  3,  2,  1, &
  0,  0, 18,  3,  0, 48,  0, 48, 34, 34, 34, 34, 34, 34, 34, 34, 32, 51, 51, 51, 48,  3, 34,  1, &
  0,  2, 32, 48,  3, 51, 51,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 48, 48,  0,  0, 48, 34,  1, &
  0, 50,  3,  0,  0, 48,  3,  0,  3,  0, 16, 17, 16,  1,  0, 16,  3, 51, 51, 51,  3, 34, 34,  1, &
  3, 34, 35,  3, 51, 51, 51,  0, 16,  1,  0, 17, 16, 16,  3,  0,  3,  0, 48,  0,  3,  2, 32,  1, &
  2, 48, 48,  0,  0, 48, 48,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 51, 51,  0, 48, 34,  0,  1, &
  3,  3,  0, 51, 51, 51, 48, 34, 34, 34, 34, 34, 34, 34, 34, 32, 48,  0, 48,  3,  2, 16,  3,  1, &
  2,  3,  0,  0, 48, 51,  1, 17, 17, 17, 17, 17, 17, 17, 17, 16, 51, 51,  0, 48, 34,  0, 48,  1, &
  0, 48,  3, 51, 51, 51,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 48,  0,  3,  2, 16,  3,  3,  1, &
  0, 48,  0,  3,  3, 48,  0, 16, 16,  0,  1,  0,  1,  0,  1,  0, 51,  0, 48,  2,  0, 48, 48,  1, &
  3,  0, 51, 51, 51,  0,  0,  0, 16, 16, 16, 16, 16, 16, 16, 16, 48,  3,  0,  0,  3,  3,  3,  1, &
  3,  0,  0,  3, 48,  0, 16, 16, 16,  1,  0,  1,  0,  1,  0,  0,  0, 48,  1,  0, 48, 48, 48,  1, &
  0,  3, 51, 51,  0,  0, 16,  0, 16, 34, 35, 34, 48, 50, 35, 32, 48,  0,  0, 16,  3,  3,  0,  1, &
  0,  0,  3, 48,  2, 32, 16, 16, 16, 34, 48, 50, 51, 50, 48, 50,  0, 16, 16, 16, 48, 48,  2,  1, &
  0, 51, 48,  2, 32,  0, 16,  0, 16, 34, 35,  3, 35, 35,  3, 34, 32, 16,  0,  1,  3,  0, 32,  1, &
  0, 48,  2, 32,  0, 32, 16, 16, 16, 35, 34, 50, 35, 34, 50, 35, 32,  1,  1,  1,  0,  2,  0,  1, &
  0,  1, 16,  0, 34, 32, 16,  0, 16, 34, 34, 34, 34, 34, 34, 34, 34,  1,  0,  0, 16, 32,  2,  1, &
  1, 17, 17, 16,  2,  0, 16, 16, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 16, 16,  0, 32,  1, &
  1, 17, 17, 17, 16,  0, 16,  0, 16, 48, 48, 48, 48, 48, 48, 48, 48, 48, 16,  0,  1,  0,  1,  1, &
  1, 17, 17, 17, 17, 16,  0, 16, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1, &
  1, 17, 17, 17, 17, 17, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, 17,  1,  1, &
  1, 17, 17, 17, 17, 17, 17, 16,  3, 51,  3,  3, 48,  2, 34, 34,  0,  0,  0,  0, 17,  0, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 51,  0, 48, 48,  3,  2, 34, 33, 22,  0,  0, 16,  1, 17,  1,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 48, 51,  3,  3,  3,  0, 33, 17, 17,  0, 17, 17,  0,  0,  1,  1, &
  1, 17, 17, 16,  0,  0, 17, 16, 51,  3,  3,  3, 48, 51,  1, 17, 17,  1, 17, 17,  3, 48, 17,  1, &
  1, 17, 17, 17,  0,  1, 17, 17,  3, 51, 48, 48, 48, 51,  1, 17,  0,  1, 17, 16, 50,  3,  1,  1, &
  1, 16,  0,  1, 16, 17,  0,  0, 16,  3, 48, 48, 48, 48, 17, 17, 17,  1, 17, 16, 50,  3,  1,  1, &
  1,  0,  0,  0, 16, 16,  0,  0,  1, 16, 48, 48, 48, 48, 22,  0, 17,  1, 17, 16, 50,  3,  1,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 48, 22, 16,  1, 16,  1, 17,  3, 48, 17,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 48, 17, 17, 17, 17,  1, 17, 16,  1, 17,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 48, 48, 48, 17, 17, 17, 16, 17, 17,  3, 48, 17,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 16, 48, 48, 48, 48, 17, 17, 17, 97, 17, 16, 50,  3,  1,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 16, 48, 48, 48, 48, 16,  1,  1,  1, 17, 17,  3, 48, 17,  1, &
  1, 17,  0,  0,  0,  0,  0,  1, 17, 17,  3,  3,  3,  3,  1, 16,  1,  1, 17, 17, 16,  1, 17,  1, &
  1, 17, 16,  0,  0,  0,  0, 17, 17, 17,  3,  3,  3,  3,  1, 17, 17,  1, 17, 17,  3, 48, 17,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 16,  0,  0,  0,  0,  0,  0,  0,  1, 17, 16, 50,  3,  1,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 16, 34, 34, 34, 34, 34, 34, 34,  1, 17, 16, 50,  3,  1,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 17, 16, 34,  0,  2,  0,  0, 32,  0, 32, 17, 16, 50,  3,  1,  1, &
  1, 17, 17, 17, 16, 17, 17, 17, 17,  2, 32, 34,  2,  2, 32, 32, 34,  2,  1, 17,  3, 48, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 17, 16, 34,  2, 34,  2,  2, 32, 32, 34, 32, 32, 17, 16,  1, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pqc = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  3, 51, 51, 51, 51, 50,  0, 33,  1,  1,  0, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,  1, &
  0, 51, 51, 51, 51, 50,  0, 33,  0, 16, 16,  0, 17, 17, 17, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1,  3, 51, 51,  0, 50,  0, 33,  1, 16,  1,  0, 17, 17, 17, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1, 16, 51, 51,  3, 34,  0, 33,  1,  1, 16, 16, 17, 17, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1, 16, 51, 51, 50, 32,  2, 33,  0, 17, 17,  0,  1, 17, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1, 17,  3, 51, 50, 32,  2, 17,  1, 17, 17, 16,  1, 17, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1, 17,  3,  0, 50,  0,  2, 17,  0,  1,  0,  0, 32, 17, 17, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1, 17,  3,  3, 34,  0, 34, 16, 16,  1, 16,  0, 32, 17, 17, 17,  0,  1,  0,  1,  0,  1, 17,  1, &
  1, 16, 51, 50, 32,  0, 33, 16, 17, 16, 17, 16,  2,  1, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  1, 16, 51, 50, 32,  2, 33,  1, 17, 16, 17, 16,  2,  1, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1,  3,  0, 34,  0, 34, 17,  1, 17, 16, 17, 16,  0, 32, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 51,  2, 32,  0, 33, 16, 17, 17,  0,  1, 16, 32, 34,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1,  0, 34,  0,  2, 33, 16, 17, 17, 17, 17, 16, 34,  2, 32, 16,  0,  0, 16, 16,  0,  0, 17,  1, &
  1, 17,  0,  0, 34, 17,  2,  1, 17,  0,  1,  0,  2,  0, 34,  1,  0,  1, 16, 17,  0,  1, 17,  1, &
  1, 17, 16,  0, 33, 17,  0, 32, 17, 16, 17,  0,  0, 32,  2, 32, 17, 17,  0,  1, 17, 17, 17,  1, &
  1, 17,  0,  2, 33, 16,  3,  0,  1, 17, 16, 48,  0,  2,  0, 34,  1, 16,  0,  0, 17, 17, 17,  1, &
  1, 16,  0, 34, 16,  2, 50, 48, 48,  0,  3,  2, 51, 32, 34,  0, 32, 17, 17, 17, 17, 17, 17,  1, &
  1,  0,  2, 33,  1, 16, 35, 34, 48, 48, 50, 35, 50,  1,  2, 32,  2,  1, 17,  0, 17, 17, 17,  1, &
  0,  0, 32,  0,  0,  0,  2, 51, 34, 34, 35, 51, 32,  0,  0, 34,  2,  1, 16, 34,  0, 17, 17,  1, &
  0,  2, 51, 48,  0,  0, 16,  2, 51, 51, 51, 34,  1,  1,  3,  2,  2,  1,  2,  0, 34,  1, 17,  1, &
  0, 51,  3, 51,  1, 16, 17,  0,  2,  0,  2,  0, 17,  0, 48, 48, 32, 16, 34, 34, 32, 17, 17,  1, &
  3, 51, 48, 48, 48, 16, 16,  0, 16, 34, 32, 16,  0,  3, 51,  0,  1, 17,  0,  2,  1,  0,  1,  1, &
  2, 34, 34,  3, 51,  0,  0, 16, 16, 35, 32, 16, 16, 48, 48,  0, 48, 17, 17, 16, 16, 51, 48,  1, &
  2, 34, 34, 32, 51,  3,  0, 16, 16, 34, 32, 16,  3, 51,  2,  0, 48,  1,  0,  1,  3, 48,  1,  1, &
  2, 34, 34, 34,  0, 51, 51,  0,  0,  2,  0,  0, 48, 48, 34,  0, 48,  0, 51, 48,  0,  0, 17,  1, &
  3,  3,  3,  2, 34,  0, 48, 51, 51,  0,  3, 51, 48,  0, 50,  0, 48,  3, 51,  1,  1, 17,  1,  1, &
  0, 48, 48, 48, 48, 34,  0,  3, 51, 51, 51, 48,  0,  0, 50, 32,  3,  0,  0,  1, 16,  0,  1,  1, &
  3,  3,  0,  0,  0,  0, 34, 48,  0,  0,  0,  0,  3,  0, 50, 32,  0, 34,  2,  0, 16, 17,  1,  1, &
  0, 48,  3, 51, 51, 51,  0, 34, 48,  0,  3,  0, 48,  3, 34, 34,  2, 17, 32,  0,  0,  0,  1,  1, &
  3,  3, 51,  0,  0,  3, 51,  2, 32,  0,  0, 48, 48,  3, 34, 32, 48, 33, 34,  2,  1, 17,  1,  1, &
  0, 51,  0,  3,  3,  0,  3, 48, 35,  0,  0,  3,  0,  3, 34,  3, 48,  2, 34, 32,  0,  0, 17,  1, &
  2,  0,  3,  3,  3,  3,  0, 51,  2,  0,  0, 48, 48,  3, 34,  3,  3, 48, 33, 18,  2,  1, 17,  1, &
  2, 32,  3,  0,  0,  0, 51,  3,  2, 48,  3,  0,  3,  3, 32, 51,  0,  0,  2, 18, 32,  0, 17,  1, &
  2, 18,  0, 34, 34, 32,  0,  3, 48, 32, 48,  0,  0,  3, 32, 48, 51,  2, 32, 34, 34,  2,  1,  1, &
  2, 17, 32, 33, 17, 34,  3, 48, 48,  0,  0,  3, 48,  0,  0, 48,  0,  2, 17,  2, 17, 32,  0,  1, &
  0, 34, 34,  1,  0, 18,  0,  0, 48, 34, 32, 34, 34,  3, 48, 48, 51,  2, 16, 16, 33, 34,  2,  1, &
  2,  2, 33, 32, 16, 18,  3, 48, 48, 51, 48, 34, 34,  2, 32, 48,  0,  2, 16,  1,  2, 34, 32,  1, &
  0,  0, 33, 18,  1, 18,  0,  0, 48,  0,  0,  3, 48,  0,  0, 48, 51,  2, 33, 17, 32, 33, 18,  1, &
  1,  2,  2, 34, 32, 34,  3, 48, 48, 35,  3,  0,  0,  0, 32, 51,  0,  0, 34, 34, 32,  2, 18,  1, &
  1, 16,  0, 34, 18,  0,  0,  3, 48, 35,  0, 48,  0, 48, 50,  3,  3, 48,  0,  0,  3,  0, 34,  1, &
  1, 17,  2,  2, 17, 32, 51,  3,  2, 35,  0,  3,  3,  0,  2,  3, 48,  3,  3,  3,  3,  0,  2,  1, &
  1, 16,  0,  0, 34, 34,  0, 51,  2, 35,  0,  0, 48,  0,  3, 32, 51,  0,  3,  3,  0,  3, 48,  1, &
  1,  1, 17,  2,  2, 33, 32, 48, 34, 35,  0,  3,  3,  0,  0, 34,  3, 51,  0,  0,  3, 51,  3,  1, &
  1,  0,  0,  0,  0, 33, 18,  2, 34, 35,  0,  3,  0, 51,  0, 50, 32,  3, 51, 51, 51,  0, 48,  1, &
  1,  1, 16, 16,  2,  2, 32,  0, 34, 48,  0, 48,  0,  0,  0,  0, 50, 32,  0,  0,  0,  3,  3,  1, &
  1,  0,  0, 17,  0,  0,  3,  0, 34, 48,  0,  0, 51, 51, 51, 51,  0,  2, 32, 48, 48, 48, 48,  1, &
  1,  1, 17,  1,  3, 51,  0, 48,  2, 48,  0, 51, 51,  0,  3, 51, 48, 48,  2, 34,  3,  3,  3,  1, &
  1, 16,  0,  0, 51, 48,  0, 48,  2, 32, 51, 48,  0,  2,  0,  0,  3, 51, 48,  2, 34, 34, 34,  1, &
  1,  0, 51,  1,  0,  1,  0, 48,  2,  3, 51,  0, 16, 34, 32, 16, 16,  3,  3, 48, 34, 34, 34,  1, &
  0, 51, 48, 16, 17, 17, 16, 48,  0, 48, 48, 16, 16, 35, 32, 16, 16,  0,  3, 51,  2, 34, 34,  1, &
  1,  0,  1,  2,  0,  1, 17,  0,  3, 51,  0,  0, 16, 34, 32, 16,  0, 16, 16, 48, 48, 51, 51,  1, &
  1, 17, 16, 34, 34, 32, 16, 32, 48, 48,  1, 16,  2,  0,  2,  0,  1, 16, 17,  3, 51,  3, 48,  1, &
  1, 17,  2, 32,  2,  1,  2,  2,  3,  1,  1,  2, 35, 51, 51, 50,  0, 16,  0,  0, 51, 50,  0,  1, &
  1, 17, 16,  2, 32, 17,  2,  2, 32,  0,  0, 35, 51, 34, 34, 35, 50,  0,  0,  0,  0, 32,  0,  1, &
  1, 17, 17, 16,  1, 17,  2,  0, 34,  1,  2, 51, 34, 48, 48, 50, 35, 32, 17,  1, 34,  0,  1,  1, &
  1, 17, 17, 17, 17, 17, 16, 32,  2, 32, 35, 50,  3,  0,  0, 48, 50, 50,  0, 18, 32,  0, 17,  1, &
  1, 17, 17, 16,  0,  0, 17,  2, 32,  2,  0,  0, 48, 17, 17,  0,  3,  0, 17, 34,  0,  1, 17,  1, &
  1, 17, 17, 17,  0,  1, 17, 16, 34,  0, 32,  0,  1, 16, 17, 16, 32,  1, 17, 32,  0, 17, 17,  1, &
  1, 17,  0,  1, 16, 17,  0,  1,  2, 32,  2,  0,  1,  0,  1, 17,  2,  1, 18, 32,  0,  1, 17,  1, &
  1, 16,  0,  0, 16, 16,  0,  0, 16, 34,  2, 32, 17, 17, 17, 17, 16, 17, 34,  0,  2, 32,  1,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1,  2, 32, 32, 17,  0,  1, 17, 16, 17, 32,  0, 34,  3, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 32,  0, 17, 16, 17, 17,  1, 18, 32,  2, 32,  3,  1,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 17,  2,  0, 17, 16, 17, 17,  1, 34,  0, 34, 51, 48, 17,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 17,  2,  0, 17, 16, 17, 16, 17, 32,  0, 34, 51, 48, 17,  1, &
  1, 17,  0,  1,  0,  1,  0,  1, 17, 17, 16, 32,  0, 17,  0, 16, 18, 32,  2, 35,  3,  1, 17,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 17, 16, 32,  0,  1,  0,  1, 18,  0,  2, 48,  3,  1, 17,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 17, 17,  0, 17, 17, 17,  1, 18,  0, 34, 51, 51,  1, 17,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 17, 17,  0,  1, 17, 16,  1, 34,  0, 34, 51, 51, 48, 17,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 17, 17, 16, 16, 17,  1,  1, 32,  2, 35,  3, 51, 48, 17,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 17, 17, 16,  1,  0, 17,  1, 32,  2, 48,  3, 51, 51,  1,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 17, 17, 17, 16,  0, 16, 16,  1, 32,  2, 51, 51, 51, 51, 48,  1, &
  1, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 16,  1,  1,  1, 32,  2, 51, 51, 51, 51, 51,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pqd = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1,  3, 51, 51, 51, 51,  2, 32, 17,  1,  1,  0, 48, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 16, 51, 51, 48,  3,  2, 32, 16, 16, 16, 16, 51,  1, 17, 17, 17, 17, 17, 33, 17, 17, 17,  1, &
  1, 16, 51, 51, 51, 51,  2, 32, 17, 16,  1,  0,  3,  1, 17, 17, 17, 17, 18, 34, 17, 17, 17,  1, &
  1, 17,  3, 51, 48,  3,  2, 32, 17,  1, 16, 16,  0, 48, 17, 17, 17, 17, 34, 34, 33, 17, 17,  1, &
  1, 17,  3, 51, 51, 48,  2,  1,  0, 17, 17,  0,  0,  3,  1, 17, 17, 20, 34, 34, 36, 17, 17,  1, &
  1, 17,  3, 51,  0, 48, 34,  0, 17, 17, 17, 16, 32,  3,  1, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  1, 17,  3, 51, 51,  0, 32, 17,  0,  1,  0,  0, 32,  0, 48, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  1, 17,  3, 48,  3,  2, 32, 16, 16,  0, 16,  0, 34,  0, 48, 17, 18, 34, 34, 34, 34, 34, 17,  1, &
  1, 17,  3, 51, 48,  2,  0, 17, 17, 16, 17, 16, 34,  0,  3,  1, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 16, 51,  0, 48, 34,  1,  1, 17, 16, 17, 16, 34, 32,  3,  1, 18, 34, 34, 34, 34, 34, 17,  1, &
  1, 16, 51, 51,  0, 32, 16, 16, 17, 16, 17, 16, 34, 32,  3,  1, 17, 66, 34, 34, 34, 65, 17,  1, &
  1,  3, 48, 48,  2, 32, 16,  1, 17,  0,  1, 16, 34, 32,  3,  1, 17, 66, 34, 34, 34, 65, 17,  1, &
  1, 16, 51,  0, 34,  1, 16, 17, 17, 17, 17, 16, 34, 34,  0, 48, 17, 20, 34, 34, 36, 17, 17,  1, &
  1, 17,  0,  2, 32, 17,  2,  1, 17,  0,  1,  3,  2, 34,  0, 48, 17, 17, 34, 34, 33, 17, 17,  1, &
  1, 17, 16, 34,  1, 17,  0, 32, 17, 16, 17,  3,  0, 34,  0, 48, 17, 17, 18, 34, 17, 17, 17,  1, &
  1, 17,  2, 32, 17, 16, 51,  0,  1, 17, 16, 51,  1,  2,  0, 48, 17, 17, 17, 33, 17, 17, 17,  1, &
  1, 16, 34,  0,  0,  1,  3, 51,  0,  0,  0, 51,  1,  0,  0, 48, 17, 17, 17, 17, 17, 17, 17,  1, &
  1,  2, 32,  1,  0,  0, 16,  0,  3, 51, 48,  0, 16,  0,  0, 48, 17, 17, 17,  0,  1, 17, 17,  1, &
  0, 34,  0, 16, 16,  0,  1, 17, 16,  0,  1, 17,  0,  1,  0, 48, 17, 17,  0, 51,  0, 17, 17,  1, &
  2, 32, 16,  1,  0,  0,  0,  0, 17, 17, 17,  0,  1,  0,  3,  0, 17, 17,  3,  0, 51,  1,  1,  1, &
  2,  3,  1,  0,  0, 16,  1,  0,  0,  0,  0,  0, 16, 16,  0, 48, 17, 16, 51, 51, 48, 17,  0,  1, &
  0,  0, 48, 16,  0,  0, 16, 16,  0,  1,  0,  0,  1,  0,  1,  3,  1, 17,  0,  3,  1, 16, 48,  1, &
  0,  3,  3,  1,  0,  0, 16, 16, 16, 16, 16, 16,  0,  0, 16, 51,  0, 17, 17, 16,  1,  3, 48,  1, &
  0, 48,  0, 48, 17,  0,  1,  0,  0, 16, 16,  0,  0,  1,  3, 48,  0,  1, 17, 17,  0, 48,  1,  1, &
  3,  0, 32,  3,  0, 16,  0,  0,  0,  1,  0,  0,  1, 16, 48, 48,  1,  0, 17, 16,  0,  0, 17,  1, &
  0,  2,  0, 48, 51,  0,  3,  0,  0,  0,  0,  1, 16,  3,  0, 48, 16,  0,  1,  1, 17,  0,  1,  1, &
  0, 32,  3,  2,  0, 51,  0,  0, 17, 17, 17, 16,  3, 48,  0, 48, 16,  0,  0, 17,  0,  1,  1,  1, &
  2,  0, 48,  0, 33,  0, 51, 48,  0,  0,  0,  3, 48,  0, 32, 48, 16,  0,  3,  1, 16, 16,  1,  1, &
  0,  3,  0, 32,  2, 16,  0,  3, 51, 51, 51, 48,  3,  0, 32, 48,  1,  0, 51,  0, 17,  1,  1,  1, &
  0, 48,  3,  2,  0, 33, 32, 48,  0,  0,  0,  3, 51,  0, 33,  3,  0,  3, 48,  0,  1,  1,  1,  1, &
  3,  0,  3,  0, 32, 18, 16,  3,  0,  0,  3, 48,  3,  0, 33,  3, 48, 51,  1,  1,  0, 16, 17,  1, &
  1,  1,  0, 48, 32, 33, 33,  3,  1, 32,  3, 51, 51,  0, 33, 32, 51, 48,  0,  0,  0,  1, 17,  1, &
  0,  0,  0, 48, 32, 18, 16,  3,  1, 32,  3,  0,  3,  0, 32,  2,  3,  0,  1,  1,  1,  0, 17,  1, &
  1,  1,  1,  3,  0, 33, 33,  3,  1, 32,  3, 51, 51,  0,  2, 32, 48,  3,  0,  0,  0,  0,  1,  1, &
  0,  0,  0,  3,  2, 18, 16,  3,  0,  0,  0,  0,  0, 34, 35, 32, 48, 32, 48,  1,  1,  1,  0,  1, &
  1,  1,  1,  0, 48, 33, 33,  3,  2, 34, 34, 34, 34, 51, 50,  3,  0, 18,  3,  0,  0,  0,  0,  1, &
  0,  0,  0,  0,  3,  2, 16,  3,  2, 51, 50, 34, 34, 34, 34,  3,  1, 33, 32, 48,  1,  1,  1,  1, &
  0,  1,  1,  1,  0, 48, 32, 48, 35, 34, 32,  0,  0,  0,  0,  3,  0, 18, 18,  3,  0,  0,  0,  1, &
  1,  0,  0,  0,  0,  3,  0, 48, 34,  0,  3, 51, 51,  0, 33,  3,  1, 33, 32,  3,  1,  1,  1,  1, &
  1, 16,  1,  1,  1,  0,  3,  2,  0, 32,  3,  0,  3,  0, 33,  3,  0, 18, 16, 32, 48,  0,  0,  1, &
  1, 17,  0,  0,  0,  0, 51, 48, 33, 32,  3, 51, 51,  0, 33,  3,  1, 33, 32, 32, 48,  1,  1,  1, &
  1, 16, 16,  1,  1,  3, 48, 51,  1, 32,  3,  0, 51,  0,  0,  3,  0, 18, 16, 32,  3,  0,  3,  1, &
  1,  1,  1,  0,  0, 51,  0,  3,  0, 32,  3, 51,  0,  0,  0,  0, 48, 33, 32,  2,  3,  0, 48,  1, &
  1,  1,  1, 16,  3, 48,  1,  0, 48, 32,  3,  0, 51, 51, 51, 51,  0,  0, 18,  0, 32,  3,  0,  1, &
  1,  0, 16, 17,  3,  0,  0, 16, 48, 32,  0, 51,  0,  0,  0,  0, 51, 48,  1, 32,  0, 48,  2,  1, &
  1,  1,  0,  1, 16,  0,  0, 16, 48,  0, 51,  0, 17, 17, 17, 16,  0,  3, 48,  2,  3,  0, 32,  1, &
  1,  0,  1, 17,  1,  0,  0, 16, 48,  3,  0, 17,  0,  0,  0,  0,  3,  0,  3, 48, 48,  2,  0,  1, &
  1, 16,  0,  0, 17, 16,  1,  0, 48, 48, 17,  0,  0,  1,  0,  0,  0,  0, 16,  3,  0, 32,  3,  1, &
  1,  0, 48,  1, 17, 17,  0,  0, 51,  1,  0,  0,  0, 16, 16,  0,  1,  0,  1, 16, 48,  0, 48,  1, &
  0, 51,  1,  0, 17, 17, 16,  3, 48, 16,  0,  0, 16, 16, 16, 16, 16, 16,  0,  1,  3,  3,  0,  1, &
  0, 48, 17,  3,  0,  1, 17,  3,  1,  0,  1,  0,  0,  1,  0,  0, 16, 16,  0,  0, 16, 48,  0,  1, &
  0,  1, 16, 51, 51, 48, 17, 16, 48,  0, 16, 16,  0,  0,  0,  0,  1,  0, 16,  0,  1,  3,  2,  1, &
  1,  1,  3, 48,  3,  1, 17, 16,  3,  0,  1,  0,  1, 17, 17, 16,  0,  0,  0,  1,  0, 16, 34,  1, &
  1, 17, 16,  3, 48,  1, 17, 16, 48,  1,  0,  1, 17,  0,  0, 17, 17,  0,  0, 16, 16,  2, 32,  1, &
  1, 17, 17,  0,  1, 17, 17, 16, 48,  0,  0, 16,  0, 51, 51,  0,  0, 16,  0,  1,  0, 34,  1,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 48,  0,  1,  3, 48,  0,  0,  3, 51,  1,  0,  0,  2, 32, 17,  1, &
  1, 17, 17, 17, 33, 17, 17, 16, 48,  2,  1,  3, 48, 17, 17,  0,  3, 48, 17, 16, 34,  1, 17,  1, &
  1, 17, 17, 18, 34, 17, 17, 16, 48,  2, 32,  3,  1, 16, 17, 16, 32,  1, 17,  2, 32, 17, 17,  1, &
  1, 17, 17, 34, 34, 33, 17, 16, 48,  2, 34,  3,  1,  0,  1, 17,  2,  1, 16, 34,  0,  1, 17,  1, &
  1, 17, 20, 34, 34, 36, 17, 16, 48,  2, 34, 32, 17, 17, 17, 17, 16, 17,  2, 32,  3, 48, 17,  1, &
  1, 17, 66, 34, 34, 34, 65, 17,  3,  0, 34, 32, 17,  0,  1, 17,  0, 16, 34,  0, 48, 51,  1,  1, &
  1, 17, 66, 34, 34, 34, 65, 17,  3,  0, 34, 32, 17, 16, 17, 16, 16, 16, 32,  3, 51, 48, 17,  1, &
  1, 18, 34, 34, 34, 34, 34, 17,  3,  0, 34, 32, 17, 16, 17, 17,  1,  2, 32, 48,  3, 48, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 33,  3,  0,  2, 32, 17, 16, 17, 17, 16,  2,  0, 51, 51,  1, 17,  1, &
  1, 18, 34, 34, 34, 34, 34, 17, 16, 48,  2, 32,  0, 16,  0, 16, 16, 34,  3,  0, 51,  1, 17,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 16, 48,  0, 32,  0,  1,  0,  1, 16, 32,  3, 51, 51,  1, 17,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 17,  3,  0, 32, 17, 17, 17, 16,  2, 32, 48,  3, 51,  1, 17,  1, &
  1, 17, 20, 34, 34, 36, 17, 17, 17,  3,  0,  0,  1, 17, 16,  1,  2,  0, 51, 51, 51,  1, 17,  1, &
  1, 17, 17, 34, 34, 33, 17, 17, 17, 16, 48,  0, 16, 17,  1, 16, 34,  3,  0, 51, 51,  1, 17,  1, &
  1, 17, 17, 18, 34, 17, 17, 17, 17, 17,  3,  0,  1,  0, 17, 16, 34,  3, 51, 51, 51, 48, 17,  1, &
  1, 17, 17, 17, 33, 17, 17, 17, 17, 17,  3, 48, 16, 16, 16, 16, 34,  3,  0, 51, 51, 48, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 48,  1,  1,  1, 16, 34,  3, 51, 51, 51, 51,  1,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pqh = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 17,  0, 16, 16, 17,  0,  2, 51, 51, 48, 17, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 16,  1,  1,  1,  1,  0,  2, 48, 51,  1, 17, 34, 36, 17, 20, 34, 33, 17,  1, &
  1, 17, 17, 17, 17,  0,  0, 16,  1, 17,  0,  2, 51, 51,  1, 18, 34, 34, 65, 66, 34, 34, 17,  1, &
  1, 17, 17, 17, 17,  3,  1,  1, 16, 17, 16,  2, 48, 48, 17, 34, 34, 34, 33, 34, 34, 34, 33,  1, &
  1, 17, 17, 17, 16,  3,  0, 17, 17,  0, 16,  2, 35, 48, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 17, 17, 17, 16, 50,  1, 17, 17, 17,  0,  0, 35, 48, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 17, 17, 17,  0, 50,  0,  0, 16,  0, 16,  0, 34, 51,  1, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 17, 17, 17,  3, 34,  0,  1,  0,  1,  0,  0,  2, 35,  1, 66, 34, 34, 34, 34, 34, 34, 65,  1, &
  1, 17, 17, 16,  3, 34,  1, 17,  1, 17, 16, 16,  0, 32, 17, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  1, 17, 17, 16, 50, 35,  1, 17,  1, 17, 16, 16,  0,  1, 17, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  1, 17, 17, 16, 50, 35,  1, 17,  1, 17, 16, 17,  0,  0, 17, 17, 34, 34, 34, 34, 34, 65, 17,  1, &
  1, 17, 17, 16, 50, 35,  1, 16,  0, 17, 17,  1,  0,  0,  1, 17, 18, 34, 34, 34, 34, 17, 17,  1, &
  1, 17, 17,  3, 34, 51,  1, 17, 17, 17, 17,  1, 16,  0,  0, 17, 17, 34, 34, 34, 33, 17, 17,  1, &
  1, 17, 17,  3, 34, 48,  0, 16,  0, 17, 16, 48, 17,  0,  0,  1, 17, 18, 34, 34, 17, 17, 17,  1, &
  1, 17, 17,  3, 34,  0, 48, 17,  1, 17,  3, 48, 17, 16,  0,  0,  1, 17, 34, 33, 17, 17, 17,  1, &
  1, 17, 17,  3, 32, 48,  3,  1, 17, 16, 51,  0,  1, 17,  0,  0,  0,  1, 18, 17, 17, 17, 17,  1, &
  1, 17, 17,  3,  1, 48,  0, 48,  0,  3, 48,  0, 48, 17, 16,  0,  0,  1, 17, 17, 17, 17, 17,  1, &
  1, 17, 17,  0, 49,  3,  0,  3, 51, 48,  0,  3,  1,  1, 17,  0,  0, 17, 17,  0, 16,  1, 17,  1, &
  1, 17, 16,  3,  3, 33, 48,  0,  0,  0,  0, 50, 19,  0,  1, 17,  1, 17, 16, 51,  3, 48, 17,  1, &
  1, 17,  3, 51, 48, 49,  3, 48,  0,  3, 51,  1, 48,  0,  0,  0,  1, 17, 16, 50, 34,  1, 17,  1, &
  1, 16,  0, 48,  3,  3, 33,  3, 51, 48, 18, 51,  0,  0, 17, 17, 16, 17, 17,  3, 35, 48, 17,  1, &
  1,  3, 51, 51, 51, 48, 51, 50, 16, 35, 51,  0, 16,  1, 16, 16, 16,  1,  0,  0, 51,  1,  1,  1, &
  0,  0,  3,  0,  3, 48,  0,  3, 51, 48,  0,  3,  0, 16, 17, 17,  2, 32, 51, 48,  0, 48,  1,  1, &
  3, 51, 51, 51, 51,  0, 17,  0,  0,  3,  0,  3,  0, 17, 17,  0, 34,  3, 51,  1, 16,  3,  1,  1, &
  0,  0, 48,  0,  3,  1,  1,  0,  0,  0, 48, 48,  1,  1,  0, 34,  0,  0,  0, 17, 16, 51,  1,  1, &
  3, 51, 51, 51, 48,  1,  0,  0, 17,  0, 48, 48,  1, 16, 34, 32, 51, 48, 16,  0,  0, 48, 17,  1, &
  0, 48, 48,  0, 48,  0, 16,  1, 16, 16,  3,  0, 16, 16, 34,  3, 34, 32, 16, 32, 16,  1, 17,  1, &
  3, 51, 51, 51, 48, 16, 16, 32,  1, 16,  3,  0, 17,  2, 32, 50, 35, 32,  1,  0, 16,  1, 17,  1, &
  0,  3,  3,  0, 48,  0, 16, 50, 32, 17,  3,  1,  1,  2, 32, 50, 34,  0,  1,  1,  1, 16, 17,  1, &
  3, 51, 51, 51, 48,  1,  0, 51, 34,  1,  0,  1, 16, 34,  3, 34, 50,  0, 16,  1,  1, 16, 17,  1, &
  0, 48,  0,  3,  0, 16,  3,  3, 50,  0,  0,  1, 16, 34,  3, 34, 34,  1,  0,  1, 16,  0, 17,  1, &
  3, 51, 51, 51,  0, 16,  3,  0, 50,  0, 51,  0,  0, 34,  3, 35, 34,  1,  1,  1, 17,  2,  1,  1, &
  0, 48,  3, 48,  0, 16,  3, 32, 51, 32,  0, 51, 48,  2,  3, 34,  0,  1,  0,  1,  0, 34, 32,  1, &
  3, 51, 51,  2,  0,  1,  3, 32, 50, 32, 48,  0,  3, 48,  0,  0, 34,  0, 16,  0, 34, 34, 34,  1, &
  0,  0, 48, 34,  1,  1,  3,  0, 51, 32, 48, 51, 48,  3, 48, 34, 35,  1,  1,  2, 34, 34, 32,  1, &
  3, 51,  2, 34,  0, 16,  3,  3, 50,  3,  3, 34, 35,  0,  2, 35, 51,  0,  1,  2, 34, 32,  3,  1, &
  3,  0, 34, 34,  1,  0,  3, 51, 34,  0,  3, 34, 35,  3,  2, 51,  3,  0, 16,  2, 34,  3, 51,  1, &
  0, 34, 34, 34,  1,  1,  3, 34, 32, 51,  0, 51, 48, 48, 35, 48,  3,  1,  1,  2, 32, 48,  0,  1, &
  2, 34, 34, 32,  0, 16,  2, 32,  0,  0, 51,  0,  0, 48, 34, 48, 35,  1,  0,  2,  3, 51, 51,  1, &
  0, 34, 32,  1,  0,  1,  0,  2, 35,  2,  0, 51, 48,  0, 35, 48, 35,  0, 16,  0, 51,  0, 48,  1, &
  1,  2,  1, 17,  1,  1,  2, 35, 35,  2, 32,  0,  3, 48,  2, 48,  3,  0, 16,  3, 51, 51, 51,  1, &
  1, 16,  0, 17,  0,  1,  2, 34, 35,  2, 32, 17,  0,  0,  2, 51,  3,  0, 16,  3,  0,  0, 48,  1, &
  1, 16, 17,  1,  0, 16,  2, 50, 35,  2, 32, 17,  0,  1,  2, 35, 48,  1,  0, 51, 51, 51, 51,  1, &
  1, 16, 17,  1,  1,  0,  2, 34, 48, 34,  1,  1,  3,  1, 16, 34, 48, 16,  0, 48,  3,  3,  0,  1, &
  1, 17,  0, 16,  1,  0, 35, 34, 48, 34,  1, 16,  3,  0, 17,  0, 32, 16, 16, 51, 51, 51, 51,  1, &
  1, 17,  0, 16, 32, 16, 34, 35,  2, 32, 16, 16,  3,  0, 16, 17,  0, 16,  0, 48,  0, 48, 48,  1, &
  1, 16, 48,  0,  0, 16, 51, 48, 34, 32, 17,  0, 48, 48,  1, 16,  0,  1,  0, 51, 51, 51, 51,  1, &
  1,  3, 48, 17, 16,  0,  0,  2, 32,  1,  1,  0, 48, 48,  0,  0,  1,  1,  3,  0,  0, 48,  0,  1, &
  1,  3,  0, 17,  3, 51,  2, 32,  1, 17, 16,  3,  0,  3,  0,  0,  1, 16,  3, 51, 51, 51, 51,  1, &
  1,  0, 48,  0, 51, 48, 34,  1, 17, 16, 16,  3,  0,  0, 51, 51,  0,  0, 51,  0,  3,  0,  0,  1, &
  1,  1,  3, 48,  0,  1,  0, 16, 16, 17,  0, 16,  3, 51, 32, 18, 51, 48, 51, 51, 51, 51,  1,  1, &
  1, 16, 51, 35,  1, 17, 16, 17, 17, 16,  0,  3, 50, 16, 51, 51,  1, 35,  3,  0, 48,  0, 17,  1, &
  1, 17,  2, 34, 48, 17, 17,  0,  0,  0,  0, 49,  3, 51,  0,  0, 51,  1, 48, 51, 51,  1, 17,  1, &
  1, 16, 51,  3, 48, 17, 17,  1, 17,  0,  3, 18, 48,  0,  0,  0,  0, 49, 35,  3,  0, 17, 17,  1, &
  1, 17,  0, 16,  1, 17, 16,  0,  1, 17,  1,  3,  0,  0, 51, 51,  0,  3,  1, 48,  1, 17, 17,  1, &
  1, 17, 17, 17, 17, 17,  0,  0,  0, 17, 16, 48,  0, 51,  0,  0, 48,  0, 49,  3,  1, 17, 17,  1, &
  1, 17, 17, 17, 18, 17,  0,  0,  0,  1, 17,  0,  3, 48, 17, 17,  3,  0, 48, 35,  1, 17, 17,  1, &
  1, 17, 17, 17, 34, 33, 17,  0,  0,  0, 17, 16, 51,  1, 17,  1, 16, 48,  2, 35,  1, 17, 17,  1, &
  1, 17, 17, 18, 34, 34, 17, 17,  0,  0,  1, 16, 48, 17, 16,  0, 16,  0, 50, 35,  1, 17, 17,  1, &
  1, 17, 17, 34, 34, 34, 33, 17, 16,  0,  0, 17,  1, 17, 17, 17, 17,  3, 50, 35,  1, 17, 17,  1, &
  1, 17, 18, 34, 34, 34, 34, 17, 17,  0,  0,  1,  1, 17, 16,  0, 17,  3, 34, 48, 17, 17, 17,  1, &
  1, 17, 66, 34, 34, 34, 34, 33, 17, 16,  0,  1, 16, 17, 17,  1, 17,  3, 34, 48, 17, 17, 17,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 17, 17,  0,  0, 16, 17, 17,  1, 17,  3, 34, 48, 17, 17, 17,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 17, 16, 32,  0, 16, 17, 17,  1, 17,  2, 35,  0, 17, 17, 17,  1, &
  1, 66, 34, 34, 34, 34, 34, 34, 65,  3, 34,  0,  0,  1,  0,  1,  0,  2, 35,  1, 17, 17, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33,  3, 50, 32,  0, 16,  0, 16,  0,  2, 48,  1, 17, 17, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 51, 32,  0,  1, 17, 17, 17,  2, 48, 17, 17, 17, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 51, 34,  0, 16,  1, 17, 16,  3,  0, 17, 17, 17, 17,  1, &
  1, 34, 34, 34, 33, 34, 34, 34, 33, 16, 48, 50,  0,  1, 16, 17,  1,  3,  1, 17, 17, 17, 17,  1, &
  1, 18, 34, 34, 65, 66, 34, 34, 17,  3, 51, 50,  0,  1, 17,  0, 16,  0,  1, 17, 17, 17, 17,  1, &
  1, 17, 34, 36, 17, 20, 34, 33, 17,  3, 48, 50,  0,  1,  1,  1,  1,  0, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 17, 16, 51, 51, 50,  0,  1, 16, 16, 16,  1, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pqs = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 16,  1,  0, 16, 16, 18,  3,  0, 51, 51,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  1, 16, 17, 17, 17,  0,  0,  1,  1,  0, 18,  3, 48, 51, 48, 17, 17, 17, 16, 17, 17, 17, 17,  1, &
  1, 16, 17, 17, 17,  0, 16, 16,  1, 16, 18,  3, 51, 51, 48, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1, 16, 17, 17, 17,  0,  1,  1, 16, 17,  2,  3,  0, 51,  1, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1,  3,  1, 17, 17,  1,  0, 17, 17,  1,  2,  3, 48, 51,  1, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1,  3,  1, 17, 16,  1,  1, 17, 17, 16,  2, 32, 51, 51, 48, 17, 16,  0,  0,  0,  0, 17, 17,  1, &
  1,  3,  1, 17, 16, 33,  0,  0, 16,  0,  2, 32, 48,  3, 48, 17,  0,  0,  0,  0,  0,  1, 17,  1, &
  1,  3,  1, 17, 16, 33,  0,  1,  0,  1,  2, 34,  3,  3,  1, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  0, 51, 48, 17, 16, 33,  1, 17,  1, 17,  1, 34,  3, 48, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  0, 48, 48, 17, 16, 33,  1, 17,  1, 17, 16, 34, 32, 48, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 51, 48, 17,  2, 33,  1, 17,  1, 17, 16, 18, 34,  0, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1,  3,  1, 17,  2, 16,  1, 16,  0, 17, 17,  2, 34,  0, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  1,  3,  1, 17,  2, 16,  1, 17, 17, 17, 17,  1, 34, 32, 17,  0,  0,  0, 16, 16,  0,  0,  1,  1, &
  0,  0,  0, 17,  1,  0,  0, 16,  0, 17, 16, 32, 18, 34,  1, 16,  0,  1, 16, 17,  0,  0, 17,  1, &
  0, 51, 48, 16, 33,  0,  0, 17,  1, 17,  2,  0, 18, 34,  0, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1,  0,  1, 16, 33,  0,  3,  1, 17, 16,  0, 48,  1, 34, 32,  1, 17, 16,  0,  0, 17, 17, 17,  1, &
  0, 51, 48, 16, 33,  3,  0, 48,  0,  3,  3,  3, 48, 18, 34,  0, 17, 17, 17, 17, 17, 17, 17,  1, &
  0, 51, 48, 16,  0,  3, 48,  3,  3,  3,  0, 51,  0,  1, 17, 16,  1, 17, 17, 17, 34, 17, 17,  1, &
  1,  0,  1, 16,  0,  3, 51, 48,  0,  0, 51, 48,  0,  0,  0,  2,  0, 17, 17, 34,  0, 34, 17,  1, &
  1,  2,  1,  0,  1,  0, 48, 51, 51, 51,  3,  0,  1, 16,  2, 34, 32, 49, 18, 32, 51,  2, 33,  1, &
  1,  3,  0, 48,  1,  0, 51,  3,  3,  3, 48,  1, 16,  0, 34, 33,  3,  0, 17, 34,  0, 34, 16,  1, &
  1,  3,  2, 51,  0,  0,  3, 51, 51, 51,  0, 16,  0, 34, 34, 16, 48,  3,  0, 17, 34, 17,  2,  1, &
  0,  3,  2, 35,  0, 16,  3,  3,  3, 48,  1,  0,  2, 34, 17,  3,  0,  3, 51,  1,  1, 16, 50,  1, &
  0,  3,  2, 35,  0,  0,  0, 51, 51,  0, 16,  2, 34, 33, 16, 48, 32,  0, 51, 48,  1,  0,  3,  1, &
  0,  3,  2, 35, 48,  1,  0, 51, 48,  1,  0, 34, 34, 18, 32, 48,  0,  0,  0,  3,  0,  0,  3,  1, &
  0,  3,  1, 34, 48,  0, 16,  0, 48, 16,  2, 34, 17, 34, 32, 48, 34, 32,  0,  0, 17, 17,  3,  1, &
  0,  3,  0, 34, 51,  0,  0, 51,  1,  0, 34, 33, 34, 32,  0, 48,  0,  0,  0,  1, 17, 17,  3,  1, &
  0,  3,  1,  2, 35,  0,  3, 34, 48,  2, 34, 18, 32,  0,  0, 51,  2, 34, 32, 17, 17, 17,  3,  1, &
  0,  3,  0, 18, 35, 48,  3, 34, 48, 34, 33, 32,  0, 17,  0,  3,  0,  0,  1, 16, 17,  0, 50,  1, &
  0,  3,  1,  0, 34, 51,  3, 34, 48, 34, 18,  0,  1, 17, 16,  0, 48, 34,  1,  0,  0,  0,  2,  1, &
  0,  3,  0, 16, 34, 35, 48, 51,  2, 33,  0,  2, 17,  0, 17,  0,  3,  0,  0,  3,  0,  0,  0,  1, &
  0,  3,  1,  1,  2, 34, 51,  0,  2, 16,  0, 17,  0,  0,  1, 16,  0, 48, 32, 51,  3,  0,  3,  1, &
  0,  3,  0, 16, 16, 34, 34, 34, 34,  0,  1, 17, 17, 17, 17, 17,  0,  3,  3, 51,  3, 51, 48,  1, &
  0,  3,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 51, 51, 48,  3,  1, &
  0,  3,  0, 16, 16, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 32,  0,  0,  3, 48,  1, &
  0,  3,  1,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  3, 51, 48,  0,  1, &
  0,  0, 51, 51,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  1,  0,  0,  0,  3,  0,  1, &
  0, 51,  0,  0,  0, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 32, 16, 16,  3,  0,  1, &
  3,  0, 51, 51, 48,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  3,  0,  1, &
  0, 51, 51,  3, 51,  3,  0,  1, 17, 17, 17, 17, 17,  0,  2, 34, 34, 34, 32, 16, 16,  3,  0,  1, &
  3,  0,  3,  3, 48, 32, 48,  0, 17,  0,  0,  1, 16,  0, 18,  0,  3, 50, 34,  1,  1,  3,  0,  1, &
  0,  0,  0,  3,  0,  0,  3,  0,  1, 16,  1, 18,  0,  1, 34,  3, 48, 51, 34, 32, 16,  3,  0,  1, &
  2,  0,  0,  0,  1,  2, 32, 48,  0, 17, 17,  0,  2, 18, 32, 50, 35,  3, 50, 32,  1,  3,  0,  1, &
  2, 48,  1, 16, 17,  0,  0,  3,  0,  1, 16,  0, 33, 34, 32, 50, 35,  0, 51, 34, 16,  3,  0,  1, &
  3,  1, 17, 17, 16, 34, 34,  3, 48,  0,  0, 34, 18, 34,  0, 50, 35,  0,  3, 34,  1,  3,  0,  1, &
  3,  1, 17, 17,  0,  0,  0,  0, 48,  0, 34, 33, 34, 32,  3,  3, 48,  0,  3, 50, 32,  3,  0,  1, &
  3,  1, 17, 16,  0,  0, 34, 32, 48, 34, 33, 18, 34,  0,  0, 48,  0,  0,  0, 50, 33,  3,  0,  1, &
  3,  0,  0,  3,  0,  0,  0,  0, 48, 34, 18, 34, 32,  1,  0, 51, 48,  0,  0, 51, 34,  3,  0,  1, &
  3,  0,  0,  0, 51, 48,  0, 32, 48, 17, 34, 34,  0, 16,  3, 51, 48,  0,  0,  3, 34,  3,  0,  1, &
  2, 48,  1,  1,  3, 51,  0,  3,  1, 18, 34,  0,  1,  0, 51,  3,  3,  0,  0,  3, 34,  3,  0,  1, &
  2,  0, 18, 33, 16,  3,  0, 48, 18, 34, 32,  0, 16,  3, 51, 51, 51,  0,  0,  3, 50,  3,  1,  1, &
  0,  2, 32,  2, 33, 16,  3,  1, 34, 32,  0, 16,  0, 51,  3,  3,  3, 48,  0,  0, 48,  3,  1,  1, &
  0, 34,  3, 48, 34, 17, 48, 34, 34,  0, 16,  0,  3,  3, 51, 51, 48, 48,  0,  0,  1,  2,  1,  1, &
  1, 18, 32,  2, 33, 17, 16,  2,  0,  0,  0,  0, 51, 48,  0,  0, 51, 51,  0,  0, 17,  0,  1,  1, &
  1, 17, 18, 33, 17, 17, 17,  0, 16,  0,  0,  3, 48,  3,  3,  3,  0, 51,  0,  0, 16, 51, 48,  1, &
  1, 17, 17, 17, 17, 17, 17, 16,  2, 34,  0, 51,  3,  3,  0,  0, 48,  3,  1, 32, 16, 51, 48,  1, &
  1, 17, 17, 16,  0,  0, 17, 17,  0, 34, 33,  0, 48,  0, 17, 17,  3,  0,  1, 32, 17,  0,  1,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16,  2, 34, 16,  2,  1, 17,  1, 16,  0,  1, 32, 16, 51, 48,  1, &
  1, 16,  0,  1, 16, 17,  0,  0, 17,  2, 34, 32, 32, 17, 16,  0, 16,  0,  1,  1, 16,  0,  0,  1, &
  1,  0,  0,  0, 16, 16,  0,  0,  1, 16, 34, 33,  1, 17, 17, 17, 17,  0, 18,  1, 17,  3,  1,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16,  2, 34,  1, 17, 16,  0, 17,  0, 18,  1, 17,  3,  1,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16,  2, 34, 16, 17, 17,  1, 17,  1, 34,  1, 16, 51, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 48, 34, 32, 17, 17,  1, 17,  1, 32, 17, 16, 48, 48,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17,  0, 51,  2, 33,  1, 17,  1, 17,  1, 32, 17, 16, 51, 48,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 16,  3,  3,  2, 34,  1,  0,  1,  0,  1, 32, 17, 17,  3,  1,  1, &
  1, 17,  0,  0,  0,  0,  0,  1, 16, 51,  0, 48, 34,  0,  0, 16,  0,  1, 32, 17, 17,  3,  1,  1, &
  1, 17, 16,  0,  0,  0,  0, 17, 16, 51, 51, 48, 34,  0, 17, 17, 17,  2,  0, 17, 17,  3,  1,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17,  3, 48, 51,  2,  1,  1, 17, 16,  2,  1, 17, 17,  3,  1,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17,  3, 48,  3,  2,  1, 16, 17,  1,  0,  1, 17, 17, 16, 17,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16, 51, 51, 51,  2, 16, 17,  0, 16, 16,  1, 17, 17, 16, 17,  1, &
  1, 17, 17, 17, 16, 17, 17, 17,  0, 51, 48, 51,  2, 16,  1,  1,  0,  0,  1, 17, 17, 16, 17,  1, &
  1, 17, 17, 17, 17, 17, 17, 16,  3, 51, 48,  3,  2, 16, 16, 16,  1,  0, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pkc = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 17, 17, 17, 16, 51, 48, 51, 48, 51, 48, 51, 51, 51, 51,  1, 17, 17, 17,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 17,  3,  0,  3,  0,  3,  0,  3, 48, 51, 48, 17, 17, 17, 17,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 16, 34, 34, 34, 34, 34, 34,  0,  3,  1, 17, 17, 17, 17,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 17,  0,  0,  0,  0,  0,  0, 34, 32, 17, 17, 17, 17,  1,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 17,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17, 17, 16,  0,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 16,  1, 17, 17, 17, 16, 16,  0,  0, 17, 17, 17,  1,  2,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17,  1,  1, 17, 17, 17, 16, 16, 16,  0, 17, 17, 17,  1,  2,  1, &
  1, 17,  0,  1,  0,  1,  0,  1, 17,  1,  0,  0, 16,  0,  0, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17,  0,  0,  1,  0,  1,  0, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1, 16,  0,  0, 16, 16,  0,  0, 17,  1,  1, 16,  0, 17, 16, 16, 16, 16, 17, 17, 17,  1,  2,  1, &
  1, 17,  0,  1, 16, 17,  0,  1, 16, 17,  0,  1, 17,  0, 16, 16, 16, 16,  0, 17, 17,  1,  2,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16,  0,  1, 16,  0, 17, 16, 16, 16, 16, 17,  1, 17,  1,  2,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17,  1,  1, 17, 17, 17, 16, 16, 17,  1,  1,  0, 17,  1,  2,  1, &
  1, 17, 17, 17, 17, 17, 17, 17,  0,  1,  1,  1, 17,  1,  0, 16, 16,  0, 16,  2,  0,  1,  2,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 17,  1,  1,  1,  1,  1,  1,  0, 16, 16, 16, 34, 34,  1,  2,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 16, 17,  1,  1,  1,  1,  0, 16, 17, 16,  2, 50, 34,  1,  2,  1, &
  1, 17, 17, 17, 17, 17,  0,  0, 17, 16, 16, 16, 32, 16, 17,  2,  0,  2, 35, 50, 34,  1,  2,  1, &
  1, 17, 17, 17, 16,  0,  0,  0,  0,  0,  2,  2, 34,  2,  0, 34, 34,  2, 34, 34, 48,  1,  2,  1, &
  1, 17, 17,  0,  2, 32, 51,  0, 16,  3,  0, 34, 34, 34, 34, 34,  0, 35, 34, 34,  1,  1,  2,  1, &
  1, 17, 16, 34, 34, 34,  3, 48,  0,  0, 48,  0, 34, 34, 34,  0,  2, 51, 34, 48, 17,  1,  2,  1, &
  1, 16,  2, 18, 18, 18, 16, 51,  0, 16, 48, 48,  0, 34,  0,  3,  2, 34, 34,  1, 16,  1,  2,  1, &
  1,  0,  2, 34, 34, 34, 34,  3, 48,  0,  3,  3,  0,  0,  0, 48, 34, 34, 48, 17,  0,  1,  2,  1, &
  2,  0,  1, 33, 33, 33, 33, 32, 51,  0,  3,  3, 48,  0,  3, 48, 35, 34, 32, 16,  0,  1,  2,  1, &
  2,  0,  2, 34, 34, 34, 34, 32,  3, 32,  0, 48,  3,  0, 48,  2, 51, 34,  1, 16,  0,  1,  2,  1, &
  2,  0,  2, 18, 18, 18, 18, 16,  0, 50,  0, 48, 51, 51, 51,  2, 34, 35,  1,  0, 32,  1,  2,  1, &
  2,  0,  2, 34, 34,  0,  0, 32,  2,  3,  0,  3,  0,  2,  0,  2, 34, 34,  1,  0,  0,  1,  2,  1, &
  0, 51,  0, 33, 32, 17, 17,  0,  0,  0, 48,  3,  3, 51, 51,  2, 50, 48, 17,  0, 32,  1,  2,  1, &
  3, 51, 48, 34,  1, 16,  1, 16,  2, 32, 48,  3,  0,  2,  0, 35, 50, 32, 16,  0,  0,  1,  2,  1, &
  0, 51,  0,  2,  1,  2, 32, 16,  0,  0,  3,  0, 48, 51, 48, 34, 34, 48, 16,  0,  0,  0,  2,  1, &
  0,  0,  0,  0,  1,  2, 32, 16,  2, 34,  3,  0, 48,  2,  0, 34, 34, 32, 16,  0, 17,  1, 16,  1, &
  0, 34,  0,  0,  1, 16,  1, 16,  0,  0,  0, 48, 48, 51, 48, 34, 50, 48, 16,  1, 17, 16,  0,  1, &
  0,  0,  0,  0,  0, 17, 17,  0,  0,  0,  0,  0,  0,  0,  0, 35, 50, 32, 16,  1,  0, 17, 16,  1, &
  0, 17,  1, 16,  0,  0,  0, 34,  1, 17, 16, 34, 32,  0,  0,  2, 34, 48, 16,  1, 17,  0,  0,  1, &
  0,  0,  1, 17,  0, 16, 50, 34,  0,  0,  0, 34, 32, 17, 17,  2, 32,  0,  0,  0, 17,  1, 16,  1, &
  0, 17, 16,  1,  0, 16, 34, 51, 32,  0,  0,  0,  0,  0,  0,  0,  1, 17, 16,  0,  0,  0,  0,  1, &
  0,  0, 17, 17,  0, 16, 50, 50, 32, 51, 48, 48, 48,  0,  0,  0, 17,  0, 17,  0,  0,  2, 32,  1, &
  0, 17,  1, 16,  0, 16, 34, 34, 32,  2,  0, 48,  3,  2, 34,  0, 16, 34,  1,  0,  0,  0,  0,  1, &
  2,  0,  0,  0,  0, 16, 50, 34, 32, 51, 48, 48,  3,  0,  0,  0, 16, 34,  1,  2,  0,  3, 48,  1, &
  2,  1,  0,  0,  0, 16, 34, 51, 32,  2,  0,  3,  0, 48, 34,  0, 17,  0, 17,  2, 32, 51, 51,  1, &
  2,  1,  0, 32,  1, 16, 50, 50,  3, 51, 51,  3,  0, 48,  0,  0,  1, 17, 16, 33, 32,  3, 48,  1, &
  2,  1,  0,  0,  1,  2, 34, 34,  0,  2,  0,  3,  0,  3,  2,  0, 32,  0,  2, 34, 34,  0,  2,  1, &
  2,  1,  0, 32,  1,  3, 34, 34,  3, 51, 51, 48, 48,  2, 48,  0, 18, 18, 18, 18, 18,  0,  2,  1, &
  2,  1,  0,  0, 17,  2, 35, 50,  0, 48,  3,  0, 48,  0, 35,  0, 34, 34, 34, 34, 34,  0,  2,  1, &
  2,  1,  0,  0, 16, 34, 35, 32, 51,  0,  0, 51,  3,  0,  3, 48, 33, 33, 33, 33, 33,  0,  2,  1, &
  2,  1,  0,  1, 16, 50, 34, 32, 48,  0,  0,  3,  3,  0,  0, 51,  2, 34, 34, 34, 34,  0,  1,  1, &
  2,  1,  0, 17,  2, 34, 34,  3,  0,  2, 32,  0, 48, 48, 16,  3, 48, 18, 18, 18, 18,  0, 17,  1, &
  2,  1,  1, 16, 50, 35, 50,  0,  2, 34, 34, 32,  0, 48,  0,  0, 51,  2, 34, 34, 32, 17, 17,  1, &
  2,  1,  1,  2, 34, 35, 32,  2, 34, 34, 34, 34, 32,  3,  0, 16,  3, 48, 34,  0,  1, 17, 17,  1, &
  2,  1,  0, 50, 34, 34,  2, 34, 32,  2,  2, 34,  2,  0,  0,  0,  0,  0,  0, 17, 17, 17, 17,  1, &
  2,  1,  2, 34, 51, 34,  0,  2,  1, 16, 16, 32, 16, 16, 17, 16,  0,  1, 17, 17, 17, 17, 17,  1, &
  2,  1,  2, 34, 50,  0, 17, 16, 16,  1,  1,  1,  1,  1, 16, 16, 17, 17, 17, 17, 17, 17, 17,  1, &
  2,  1,  2, 34, 32, 16, 16, 16,  1,  1,  1,  1,  1,  1,  1, 16, 17, 17, 17, 17, 17, 17, 17,  1, &
  2,  1,  0,  2,  0, 16,  0, 16, 16,  1,  1, 17,  1,  1,  0,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  2,  1,  1, 16,  1,  1,  1, 16, 16, 17, 17, 17, 17,  1,  1, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  2,  1,  1, 17,  1, 16, 16, 16, 16, 17, 16,  0, 17,  0,  0, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  2,  1,  1, 17, 16,  0, 16, 16, 16, 16,  1, 17,  0,  1, 16, 17,  0,  1, 16, 17,  0,  1, 17,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16, 17, 16,  0, 17,  1,  1, 16,  0,  0, 16, 16,  0,  0, 17,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16,  1,  0,  1,  0,  0,  1, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  2,  1,  1, 17, 17, 16, 16, 16, 16,  0,  0, 16,  0,  1,  1, 17,  0,  1,  0,  1,  0,  1, 17,  1, &
  2,  1,  1, 17, 17, 16,  0, 16, 16, 17, 17, 17, 17,  1,  1, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  2,  1,  1, 17, 17, 16,  0,  0, 16, 17, 17, 17, 17,  0, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  0,  0, 17, 17, 17, 16,  0,  0,  0,  0,  0,  0,  0,  1, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1,  1, 17, 17, 17, 16, 34, 32,  0,  0,  0,  0,  0,  1, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  1, 17, 17, 17, 17,  3,  0,  2, 34, 34, 34, 34, 34, 32, 17, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1, 17, 17, 17, 16, 51, 48, 51,  0,  3,  0,  3,  0,  3,  1, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1, 17, 17, 17,  3, 51, 51, 51, 48, 51, 48, 51, 48, 51, 48, 17, 17, 17, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pkd = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 17, 17,  3, 51, 48, 51, 48, 51, 51, 51, 51, 51, 51, 48, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 33, 17, 17, 16, 51,  0,  3,  0,  3, 48, 51, 51, 51, 51,  1, 17,  0,  1, 16,  1, &
  1, 17, 17, 18, 34, 17, 17, 17,  2, 34, 34, 34, 35,  0,  3, 48, 51, 48, 17, 16, 48, 17,  2,  1, &
  1, 17, 17, 34, 34, 33, 17, 17,  0,  0,  0,  0,  2, 34, 35,  0,  3,  0, 17,  3, 48, 17,  2,  1, &
  1, 17, 20, 34, 34, 36, 17, 17, 16,  0,  0,  0,  0,  0,  2, 34, 34,  1, 17,  3, 48, 17,  2,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 16, 17, 17, 17,  0, 16,  0,  0,  0,  1, 16, 51,  1,  0,  0,  1, &
  1, 17, 66, 34, 34, 34, 65, 17, 16, 16,  1, 17,  1,  1,  0,  0,  0,  1, 16, 48, 17, 33, 33,  1, &
  1, 18, 34, 34, 34, 34, 34, 17, 16,  0, 16, 17,  1,  1,  1,  1,  1,  1, 16,  1, 17, 33, 33,  1, &
  1, 34, 34, 34, 34, 34, 34, 33, 22, 17, 17, 17,  1,  1,  1,  1,  1,  1, 16, 48, 17,  1,  1,  1, &
  1, 18, 34, 34, 34, 34, 34, 17,  1, 17, 17, 17,  1,  1,  1,  1,  1,  1, 16, 51,  1,  0,  0,  1, &
  1, 17, 66, 34, 34, 34, 65, 16, 17, 17, 17, 17,  1,  1,  1,  1,  1,  1, 17,  3, 48, 17,  2,  1, &
  1, 17, 66, 34, 34, 34, 65, 16,  6, 17, 17, 17, 16, 16, 16, 16, 16, 16, 17,  3, 48, 17,  2,  1, &
  1, 17, 20, 34, 34, 36, 17, 17, 16,  0,  1,  1, 16, 16, 16, 16, 16, 16, 17, 16, 48, 17,  2,  1, &
  1, 17, 17, 34, 34, 33, 17, 17, 16, 17,  0,  1, 17,  1,  1,  1,  1,  1,  1, 17,  0,  1,  2,  1, &
  1, 17, 17, 18, 34, 17, 17, 17, 17,  0, 17, 17, 17, 16, 16, 16, 16, 16, 16, 17, 17, 17,  2,  1, &
  1, 17, 17, 17, 33, 17, 17, 17, 16, 17, 17, 17,  0,  1, 16, 16, 16, 16, 16, 17, 17, 17,  2,  1, &
  1, 16,  0, 17, 17, 17, 17, 17, 16,  1, 17, 16, 17, 16, 16, 16, 16, 16, 16, 17, 17, 17,  2,  1, &
  1,  1,  1,  0,  1, 17, 17,  0, 16, 17, 17, 16, 16,  1, 16, 16, 16, 16, 16, 17, 17, 17,  2,  1, &
  1,  0, 16, 16, 16, 17, 16, 16,  1, 17, 17, 16,  1, 17,  1,  1,  1,  1,  1, 17, 17, 17,  2,  1, &
  0,  1,  1,  1,  1,  1, 16, 17, 17, 17, 17, 17,  0,  0,  0,  0,  0,  0,  0,  0, 17, 17,  2,  1, &
  0, 16, 16, 17, 17, 16,  3,  1, 17, 17,  0,  0,  0,  0, 51,  1, 17,  3, 48, 17,  0,  0,  0,  1, &
  1,  1,  1, 17, 16,  0,  0, 48,  0,  0,  0,  0,  0, 16,  3, 48, 16,  0, 51,  1, 16, 51,  1,  1, &
  1, 16, 17,  0,  0,  2, 32,  3,  0,  2, 32,  0,  1,  0, 16, 51,  0,  1,  3, 48, 16,  3, 48,  1, &
  1, 17,  0, 32,  3,  2, 32,  0, 48, 32, 34,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 49,  1, &
  1,  0, 34, 34, 32, 48,  1,  0, 48, 34,  2,  0, 51, 51, 51, 51, 51, 51, 51,  0,  0,  0,  0,  1, &
  0, 34, 34, 32,  3,  3,  0, 34,  0,  2, 32,  1,  1,  1,  1,  1,  1,  1, 16, 48,  0,  0,  0,  1, &
  2, 34, 32,  3, 51, 48, 48, 34,  3,  0,  1, 17, 17, 17, 17, 17, 17, 17,  3,  2, 32,  1, 16,  1, &
  2, 32,  3, 51, 48,  0,  3,  0,  3,  1, 17,  0,  0,  0,  0,  0,  1, 16, 48, 48, 34,  0, 16,  1, &
  2,  3, 51, 48,  0,  0,  3,  1,  3,  1,  0, 51, 51, 51, 51, 51,  1,  3,  0, 35,  2, 32,  0,  1, &
  0, 51, 48,  0,  0, 16,  3,  0,  3,  0, 51,  0,  0,  0,  0, 48, 16, 48, 51,  2, 48, 34,  0,  1, &
  3, 51,  0,  0, 17, 16,  0, 48,  0,  3,  0,  0, 34, 34,  3,  1, 16, 48,  3, 48, 35,  2, 32,  1, &
  3, 48, 34,  0,  1, 17,  0, 48,  0, 48,  0, 34, 34, 34,  3,  1,  3,  0,  0, 48,  0,  0,  0,  1, &
  3,  2,  2, 32,  0, 17,  0, 48,  0, 48,  0,  0,  0,  0, 48, 17,  3,  0,  0,  2, 48, 35,  2,  1, &
  0, 51, 32, 34,  0,  1,  0,  3,  0, 48, 34, 34, 34, 32, 48, 16, 48,  0,  3,  2, 48, 35,  2,  1, &
  0, 35, 50,  2, 32,  0, 16,  3,  0, 48,  0,  0,  0,  3,  1, 16, 48,  2, 35,  2,  0,  0,  0,  1, &
  2, 32, 35, 50,  0,  0,  0,  0,  0,  3,  1, 17, 17,  3,  0,  0,  0,  0,  0,  2, 51, 32, 34,  1, &
  2, 34,  2,  0,  3, 34,  0, 48, 17,  3,  1, 17, 17,  3,  0,  3,  0,  0,  2, 32, 35, 50,  2,  1, &
  0,  0,  0,  2,  3, 34,  0, 48, 17,  3,  0,  0,  0,  3,  0,  3,  0, 16,  0, 34,  2, 51, 32,  1, &
  2,  3, 32, 50,  3,  0,  0, 48, 16,  2, 34, 34, 34, 32, 48,  3,  0,  1,  0,  2, 32, 35, 48,  1, &
  2,  3, 32, 50,  0,  0,  3,  1, 16,  0,  0,  0,  0,  0, 48,  0, 48,  1, 16,  0, 34,  2,  3,  1, &
  0,  0,  0,  0, 48,  0,  3,  1,  3,  2, 34, 34, 32,  0, 48,  0, 48,  1, 17,  0,  2, 32, 51,  1, &
  0, 34,  3, 32, 51,  0, 48, 17,  3,  2, 34, 32,  0,  3,  0,  0, 48,  0, 17, 16,  0,  3, 51,  1, &
  0,  2, 32, 50,  3, 48, 48, 16, 48,  0,  0,  0,  3, 48,  3,  0,  3,  0, 16,  0,  0, 51, 48,  1, &
  0,  0, 34,  3, 32,  3,  1,  3, 51, 51, 51, 51, 48,  0,  3,  0,  3,  0,  0,  0, 51, 51,  2,  1, &
  0, 16,  2, 32, 48, 48, 17,  0,  0,  0,  0,  0,  1, 17,  3,  0,  3,  0,  0, 51, 51,  0, 34,  1, &
  0, 17,  0, 34,  3,  1, 17, 17, 17, 17, 17, 17, 17,  0,  3,  2, 32, 48, 51, 51,  0, 34, 34,  1, &
  0,  0,  0,  0, 48, 17,  1,  1,  1,  1,  1,  1,  0, 34,  0,  2, 32,  3,  3,  0, 34, 34, 32,  1, &
  0,  0,  0,  0,  3, 51, 51, 51, 51, 51, 51, 48,  2,  2, 32, 48,  1,  0, 48, 34, 34, 32,  1,  1, &
  1, 48, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, 32, 32, 48,  0, 34,  3,  0, 32,  1, 17,  1, &
  0, 51,  0, 16, 51,  1,  0,  3, 48, 16,  1,  0,  0, 34,  0,  3,  0, 34,  0,  0,  1,  0, 17,  1, &
  1,  3, 48, 17,  3, 48,  0, 16, 51,  0, 16,  0,  0,  0,  0,  0, 48,  0,  0, 17, 17,  0,  1,  1, &
  0,  0,  0,  1, 16, 51,  1, 17,  3, 48,  0,  0,  0,  1, 17, 17,  3,  0, 17, 17, 16, 16, 16,  1, &
  2,  1, 17, 16,  0,  0,  0,  0,  0,  0,  0,  1, 17, 17, 17, 17, 16, 17,  1,  1,  1,  1,  0,  1, &
  2,  1, 17, 17, 17,  1,  1,  1,  1,  1, 17,  0, 17, 17, 17,  0, 16, 17, 16, 16, 16, 16,  1,  1, &
  2,  1, 17, 17, 16, 16, 16, 16, 16, 17,  0, 16, 17, 17, 16, 16,  1, 17, 17,  0,  1,  1,  1,  1, &
  2,  1, 17, 17, 16, 16, 16, 16, 16, 16, 17, 16, 17, 17,  0, 17, 17, 17, 17, 17, 16,  0, 17,  1, &
  2,  1, 17, 17, 16, 16, 16, 16, 16, 17,  0,  1, 17, 17, 16, 17, 17, 17, 17, 33, 17, 17, 17,  1, &
  2,  1, 17, 17, 16, 16, 16, 16, 16, 16, 17, 17, 17, 16,  1, 17, 17, 17, 18, 34, 17, 17, 17,  1, &
  2,  1,  0,  1, 17,  1,  1,  1,  1,  1,  1, 17,  0,  1, 16, 17, 17, 17, 34, 34, 33, 17, 17,  1, &
  2,  1, 16, 48, 17, 16, 16, 16, 16, 16, 16, 17,  1,  0,  0, 17, 17, 20, 34, 34, 36, 17, 17,  1, &
  2,  1, 16, 51,  1, 16, 16, 16, 16, 16, 16, 17, 17, 17, 22,  0, 17, 66, 34, 34, 34, 65, 17,  1, &
  2,  1, 16, 51,  1, 17,  1,  1,  1,  1,  1,  1, 17, 17, 17, 16, 17, 66, 34, 34, 34, 65, 17,  1, &
  0,  0,  1,  3, 48, 17,  1,  1,  1,  1,  1,  1, 17, 17, 17,  1, 18, 34, 34, 34, 34, 34, 17,  1, &
  1,  1,  1, 16, 48, 17,  1,  1,  1,  1,  1,  1, 17, 17, 22, 17, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 33, 33, 17,  0, 17,  1,  1,  1,  1,  1,  1, 16, 16,  0, 17, 18, 34, 34, 34, 34, 34, 17,  1, &
  1, 33, 33, 16, 48, 17,  0,  0,  0,  1,  1,  1, 17,  0, 16, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  0,  0,  1,  3, 48, 17,  0,  0,  0,  0, 16,  1, 17, 17, 16, 17, 17, 66, 34, 34, 34, 65, 17,  1, &
  2,  1, 16, 51,  1, 17,  2, 34, 34,  0,  0,  0,  0,  0,  0, 17, 17, 20, 34, 34, 36, 17, 17,  1, &
  2,  1, 16, 51,  1, 16,  3,  0,  3, 34, 34,  0,  0,  0,  0,  1, 17, 17, 34, 34, 33, 17, 17,  1, &
  2,  1, 16, 48, 17, 16, 51, 48, 51,  0,  3, 34, 34, 34, 34,  1, 17, 17, 18, 34, 17, 17, 17,  1, &
  0, 17,  0,  1, 17,  3, 51, 51, 51, 48, 51,  0,  3,  0,  3, 48, 17, 17, 17, 33, 17, 17, 17,  1, &
  1, 17, 17, 17, 16, 51, 51, 51, 51, 51, 51, 48, 51, 48, 51, 51,  1, 17, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pkh = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 17, 17, 17,  3,  3, 51,  3, 51,  3, 51,  3, 51, 51, 48, 17, 17, 17, 17,  1, &
  1, 17, 34, 36, 17, 20, 34, 33, 16,  0, 48,  0, 48,  0, 48,  0, 48,  3,  1, 17, 17, 17, 17,  1, &
  1, 18, 34, 34, 65, 66, 34, 34, 17,  3, 51, 51, 51, 51, 51, 48, 51,  0, 17,  0, 17, 17, 17,  1, &
  1, 34, 34, 34, 33, 34, 34, 34, 33, 16, 34, 34, 34, 34, 35, 51, 51,  1, 16, 51,  1, 17, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16,  0,  0,  0,  0,  2, 34, 34,  1, 16, 51,  0,  1, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 17, 17, 17, 17,  0,  0,  0,  0,  0, 48, 17,  0, 17,  1, &
  1, 34, 34, 34, 34, 34, 34, 34, 33, 16, 17, 17, 17, 17,  1,  1,  1,  1, 16, 48, 17,  3,  1,  1, &
  1, 66, 34, 34, 34, 34, 34, 34, 65,  0,  0,  1,  0,  0,  1,  1,  1,  0,  3, 48,  0,  3,  1,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 16, 16,  0, 16,  0, 16,  1,  1,  1,  0,  3, 48,  1, 16, 17,  1, &
  1, 20, 34, 34, 34, 34, 34, 36, 16, 16, 17, 16, 17, 17,  1,  1,  1,  1,  3, 48, 17, 16, 17,  1, &
  1, 17, 66, 34, 34, 34, 34, 33, 16, 16, 17, 16, 17, 17,  1,  1,  1,  1, 16,  0,  0,  0, 17,  1, &
  1, 17, 18, 34, 34, 34, 34, 17, 16, 16, 17, 16, 17, 17,  1,  1,  1, 16, 17,  2, 34, 32, 17,  1, &
  1, 17, 17, 34, 34, 34, 33, 17,  1,  1, 17,  0,  1, 17, 16, 16, 16, 16, 16,  0,  0,  0,  1,  1, &
  1, 17, 17, 18, 34, 34, 17, 17,  1,  1, 17, 17, 17, 17, 16, 17,  1,  1,  0, 32, 32, 32, 32,  1, &
  1, 17, 17, 17, 34, 33, 17, 17, 16,  1,  1,  0,  1, 17, 17,  1,  1,  1,  2, 34, 34, 34,  0,  1, &
  1, 17, 17, 17, 18, 17, 16,  0,  0, 17,  1, 17, 17, 16, 16, 16, 16, 17,  0, 34, 34, 32, 17,  1, &
  1, 17, 17, 17, 17, 16,  0, 32,  1, 16, 16, 17, 16, 16, 16, 16, 17, 16, 17,  0,  2,  1, 34,  1, &
  1, 17, 17, 17,  0,  0,  0,  2,  0,  1, 16, 16, 16, 16,  0, 17, 16,  1,  1, 17,  0,  2, 32,  1, &
  1, 17, 17,  0,  2,  1, 16,  2, 32, 16,  1, 16, 16, 17,  1,  0,  1, 17, 17,  0, 32,  0,  0,  1, &
  1, 17,  0, 17,  2,  1, 17,  0, 34,  0, 17,  0, 16,  1,  3, 48, 17,  1, 16, 32, 32, 48,  0,  1, &
  1,  0,  1,  1,  2,  1,  1, 16,  2, 32,  0, 48,  1,  0, 51, 48, 16, 17,  2, 48, 32,  0, 48,  1, &
  0, 51, 32, 16, 16, 32, 16, 17,  2, 32,  3,  3,  0, 51, 51,  1, 17, 16, 35,  2,  0,  0,  3,  1, &
  3, 34, 32,  1, 16, 32, 17, 17,  0, 34,  0, 48,  3, 51, 51,  0, 17, 16, 50,  2,  3,  0,  0,  1, &
  2, 32,  0,  0, 17,  2,  1,  1, 16, 34, 32,  0, 51, 51, 51,  1, 17,  3, 32, 32,  2, 51,  0,  1, &
  2,  0,  0,  2,  1,  2, 32, 16, 16,  2, 32,  3, 51, 51, 48, 17,  1,  2,  2, 32,  0,  3, 48,  1, &
  0, 48,  0, 34, 32, 16, 34,  1, 17,  2, 34,  3, 51, 51,  0, 16, 17,  0, 34,  0,  0, 34, 51,  1, &
  3, 34, 48,  0, 34,  1,  2, 34,  1,  0, 32, 51, 51, 48, 32, 17, 16,  2, 32,  0,  2, 34, 35,  1, &
  2, 35, 51, 32,  0,  0, 16,  2, 32,  1,  0, 51, 51, 48,  1, 16,  2, 32,  0,  0, 34, 50,  3,  1, &
  2, 51, 48, 35, 48,  0,  1, 16,  0,  0,  0,  0,  0,  0,  0,  2, 32,  0,  0,  2, 34, 34, 51,  1, &
  3, 51,  2,  0,  3,  0,  0,  1, 17, 17, 34, 34, 34, 34, 34, 32,  0,  0,  0, 34, 50,  3, 48,  1, &
  3, 51,  2, 51, 51,  3,  1, 17, 18,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2, 34, 34, 51, 48,  1, &
  3, 51,  2,  0,  3,  3,  1, 17, 17, 17, 17,  2, 34, 32, 17, 17,  0,  0, 34, 50,  3, 51,  0,  1, &
  3, 51,  2, 51, 51,  3,  1, 17, 17, 17,  0,  1, 34, 32, 16, 16,  0,  2, 50, 34, 51, 48,  0,  1, &
  0, 51, 48, 35, 48,  0,  0,  0,  0,  0, 16,  0,  0, 32,  1,  0,  0,  0,  2,  3, 51,  0,  2,  1, &
  0,  3, 51, 48,  0,  0,  0,  0, 17, 16,  3, 51, 51,  0, 17,  0,  0, 34,  3, 51, 48,  0, 32,  1, &
  2,  0, 48,  0,  0, 51, 51, 48, 17, 16,  3,  2,  3,  1, 16, 16, 51, 51, 51, 51,  0,  0, 34,  1, &
  2, 32,  0,  3, 51, 51, 51, 48, 16, 17,  3,  2,  3,  0, 17, 16, 51, 51, 48,  0,  0, 48,  2,  1, &
  0, 32,  0, 51, 51,  2, 32,  0,  1, 16,  3, 51, 51,  0, 17, 16,  0,  0,  0,  0, 51, 51,  0,  1, &
  2,  0,  3, 51,  2,  0,  0,  0,  1,  0, 32,  0,  0, 16,  0,  0,  0,  0,  0, 51, 32, 51, 48,  1, &
  0,  0, 51, 50, 34, 50,  0,  0,  0, 16, 34, 33,  0,  1, 17, 17, 17,  3,  3, 51, 50,  3, 51,  1, &
  0,  3, 51,  2, 50, 32,  0,  0, 17, 16, 34, 34,  1, 17, 17, 17, 17,  3,  3,  0,  2,  3, 51,  1, &
  0, 51, 50, 34, 34,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, 17, 17,  3,  3, 51, 50,  3, 51,  1, &
  0, 51,  2, 50, 32,  0,  0,  0, 34, 34, 34, 34, 34, 33, 17, 17,  0,  0,  3,  0,  2,  3, 51,  1, &
  3, 50, 34, 34,  0,  0,  0, 34,  0,  0,  0,  0,  0,  0,  0,  0, 17,  0,  0, 51, 32, 51, 50,  1, &
  3,  2, 50, 32,  0,  0, 34,  0, 17,  0, 51, 51, 48,  1,  0, 34,  0, 16,  0,  0, 35, 51, 34,  1, &
  3, 34, 34,  0,  0, 34,  0, 17, 16, 32, 51, 51, 48, 32,  1,  2, 34,  1,  2, 32,  0, 50, 35,  1, &
  3, 50, 32,  0,  2, 32,  1, 16, 16,  3, 51, 51,  2, 34,  1, 17,  2, 32, 16, 34, 32,  0, 48,  1, &
  0, 51,  0,  0, 34,  2,  1,  1, 16, 51, 51, 51,  0, 34,  0, 16, 16, 34,  1,  2,  0,  0,  2,  1, &
  0,  3, 50,  0, 32, 35,  1, 17,  3, 51, 51, 48,  0, 34, 32, 17,  1,  2,  1, 16,  0,  0, 34,  1, &
  0,  0,  3,  2,  2, 48, 17, 16,  3, 51, 51,  0, 48,  2, 32,  1, 17, 16, 32, 17,  0, 34, 35,  1, &
  3,  0,  0,  2,  3, 32, 17, 17,  3, 51, 48,  3,  3,  0, 34,  1, 16, 16, 32, 16, 16, 35, 48,  1, &
  0, 48,  0, 32, 50,  1, 16, 16, 51, 48,  1,  0, 48,  0, 34,  0, 17,  1,  2,  1,  1,  0,  1,  1, &
  0,  0, 48, 32, 32, 17,  1, 16, 51,  1,  0, 16,  1, 16,  2, 32,  1, 17,  2,  1, 16,  1, 17,  1, &
  0,  0,  0, 32,  1, 17, 17,  0,  1,  1, 16, 16, 17,  0, 16, 34,  0, 17,  2,  0,  1, 17, 17,  1, &
  0, 34,  0,  1, 17,  1,  0, 17, 16,  0, 16, 16, 16, 17,  0,  2,  0,  0,  0,  1, 17, 17, 17,  1, &
  2, 33,  2,  0,  0, 16, 17, 16, 16, 16, 16, 17, 16, 16, 17,  0, 32,  0, 17, 17, 17, 17, 17,  1, &
  1, 16, 34, 34, 32,  1, 16, 16, 16, 16, 17, 17, 17,  1, 16,  0,  0, 17, 18, 17, 17, 17, 17,  1, &
  0,  2, 34, 34, 34,  1,  1,  1,  1, 17, 17,  0,  1,  1,  0, 17, 17, 17, 34, 33, 17, 17, 17,  1, &
  0, 32, 32, 32, 32,  1,  1,  1, 16, 17, 17, 17, 17, 17,  1,  1, 17, 18, 34, 34, 17, 17, 17,  1, &
  1,  0,  0,  0,  0, 16, 16, 16, 16, 17, 17,  0,  1, 17,  1,  1, 17, 34, 34, 34, 33, 17, 17,  1, &
  1, 16, 34, 34,  1, 16, 17,  1,  1,  1, 17, 16, 17, 16, 16, 17, 18, 34, 34, 34, 34, 17, 17,  1, &
  1, 16,  0,  0,  0, 17,  1,  1,  1,  1, 17, 16, 17, 16, 16, 17, 34, 34, 34, 34, 34, 65, 17,  1, &
  1, 16, 17, 16, 51,  1,  1,  1,  1,  1, 17, 16, 17, 16, 16, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  1, 16, 17,  0, 51,  0,  1,  1,  1,  0, 16,  0, 16,  0, 16, 20, 34, 34, 34, 34, 34, 36, 17,  1, &
  1,  3,  0,  0, 51,  0,  1,  1,  1,  0,  0,  1,  0,  0,  1, 66, 34, 34, 34, 34, 34, 34, 65,  1, &
  1,  3,  1, 16, 48, 17,  1,  1,  1,  1, 17, 17, 17, 16, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 16,  1, 16, 48,  0,  0,  0,  0,  1, 17, 17, 17, 16, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 17,  0,  3, 48, 17,  2, 34, 34,  0,  0,  0,  0,  0, 17, 34, 34, 34, 34, 34, 34, 34, 33,  1, &
  1, 17, 17,  3, 48, 17,  3, 51, 51, 34, 34, 34, 34, 32, 17, 34, 34, 34, 33, 34, 34, 34, 33,  1, &
  1, 17, 17, 16,  1, 16,  3, 48, 51, 51, 51, 51, 51, 51,  1, 18, 34, 34, 65, 66, 34, 34, 17,  1, &
  1, 17, 17, 17, 17,  3,  0, 48,  0, 48,  0, 48,  0, 48,  0, 17, 34, 36, 17, 20, 34, 33, 17,  1, &
  1, 17, 17, 17, 16, 51, 51, 51,  3, 51,  3, 51,  3, 51,  3,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )
  
 integer(1), parameter, dimension(PictSizeX2,PictSizeY) :: Pks = reshape( (/ &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, &
  1, 17, 17, 17, 17, 17, 17, 17,  3, 51, 51, 51, 51, 51, 51, 51, 51, 51, 48, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 16, 17, 17, 17, 16, 51, 51, 51, 48, 48, 48, 48, 48, 48,  1, 17, 17, 17, 17,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 17,  0, 48, 48, 48, 32, 32, 32, 32, 32, 17, 17, 17, 17, 17,  1, &
  1, 17, 17, 16,  0,  0, 17, 17, 17, 32, 32, 32, 34, 34, 34, 34, 34,  1, 17, 17, 17, 17, 16,  1, &
  1, 17, 17,  0,  0,  0,  1, 17, 17, 16, 34, 34, 32,  0,  0,  0,  0,  1, 17, 17, 17, 17,  0,  1, &
  1, 17, 16,  0,  0,  0,  0, 17, 17, 16,  0,  0,  0, 17, 17, 17, 17,  0, 17, 17, 17, 16, 48,  1, &
  1, 17,  0,  0,  0,  0,  0,  1, 17, 16, 16, 16, 16, 17, 17, 17, 17,  1,  1, 17, 17, 16, 48,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 16, 16, 16, 16,  0,  0, 16,  0,  1,  1, 17, 17, 16, 48,  1, &
  1, 16,  0,  0,  0,  0,  0,  0, 17, 16, 16, 16, 16,  1,  0,  1,  0,  0, 17, 17, 17, 16, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1, 17, 17, 16, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1, 17, 17, 16, 48,  1, &
  1,  0,  0,  0,  0,  0,  0,  0,  1, 16, 16, 16, 16, 17, 17,  1, 17,  1,  1, 17, 17, 16, 48,  1, &
  1,  0,  0,  0, 16, 16,  0,  0,  1, 16, 16, 16, 16, 17, 16,  0, 17,  1,  1, 17, 17, 16, 48,  1, &
  1, 16,  0,  1, 16, 17,  0,  0, 17, 16, 16, 16, 16, 16,  1, 17,  1,  1,  1, 17, 17, 16, 48,  1, &
  1, 17, 17, 17,  0,  1, 17, 17, 16,  0, 16, 16, 16, 16, 16,  0, 17,  0, 16,  1, 17, 16, 48,  1, &
  1, 17, 17, 16,  0,  0, 17, 17,  1, 16, 16, 16, 16, 17, 17, 17, 17,  1,  1, 16, 17, 16, 48,  1, &
  1, 17, 17, 17, 17, 17, 17, 16, 16, 16, 16, 16, 16, 17,  1, 17,  1,  1,  1, 16, 17, 16, 48,  1, &
  1, 17, 17, 17, 17, 17, 16,  0, 16,  1, 16, 16, 16,  1,  1,  1,  1,  1, 16,  3,  0, 16, 48,  1, &
  1, 17, 17, 17, 17, 16,  0, 48,  1, 17,  1, 16, 16, 17,  1,  0,  1, 16,  0, 48,  1,  0, 48,  1, &
  1, 17, 17, 17, 16,  3,  2, 48,  0,  0, 16,  1,  1, 16, 16, 16, 48, 32,  3,  0, 16, 32, 48,  1, &
  1, 17, 17, 16,  0,  3,  0,  3,  0, 48,  0,  0,  0,  1,  1,  0, 48,  0, 48,  1,  0,  0, 48,  1, &
  1, 17, 16,  3,  3,  0, 48,  3,  2,  0, 48, 48, 48,  0,  0,  3,  2,  3,  0, 16, 34, 32, 48,  1, &
  1, 16,  2, 32,  0,  0, 48, 32, 48, 34,  0,  0,  0, 48, 48,  3,  0,  3,  0, 16, 32,  0, 48,  1, &
  1,  2, 51,  0, 48,  0, 48,  0, 48, 34, 35, 51, 48,  0,  0, 48, 32, 48,  1,  2, 34, 32, 48,  1, &
  0, 50, 32,  0,  0, 48,  3,  0, 48, 34, 35,  0,  3, 50, 32, 48,  0, 48,  1,  2,  0,  0, 48,  1, &
  2, 51, 32,  3,  3,  3,  3,  2,  3,  2, 35, 51, 51, 50, 32, 48,  3,  0, 16, 34, 34, 32, 48,  1, &
  2, 35,  0,  0,  3,  0,  3,  0,  3,  0,  0,  0,  0,  0,  3,  2,  3,  0, 16, 32,  0,  0,  0,  1, &
  3, 34,  0, 48, 48, 51, 48, 48,  3,  0, 17, 16, 17, 16,  3,  0,  3,  0, 16, 32, 51, 51,  0,  1, &
  3, 50,  0,  0, 48, 48,  0, 48,  3,  1, 16, 17, 16, 17,  3,  0, 48,  1,  2,  3,  0,  3, 51,  1, &
  2, 51,  0, 48, 48, 51, 48, 48, 32, 48, 16, 16, 16, 17,  3,  0, 48,  1,  2,  3, 48, 16, 51,  1, &
  2, 35,  0,  0, 48, 48,  0, 48,  0, 48, 17, 16, 17, 16, 48, 32, 48,  0,  2, 32,  0,  1,  0,  1, &
  3, 34,  0, 48, 48, 51, 51,  3,  0, 48, 16, 17, 16, 16, 48,  0, 48, 48,  0, 34, 34, 32,  0,  1, &
  2, 50,  0,  0, 48, 48,  0,  3,  0, 48,  0,  0,  0,  0, 48,  3,  0,  0,  0,  0,  0,  1, 17,  1, &
  2, 35,  0, 48,  3,  3, 51,  3,  0, 48, 51, 51, 51, 51, 48,  3,  0,  0,  0,  0, 17, 16,  0,  1, &
  0,  2,  0,  0,  3,  3,  0,  3,  0, 48,  0,  0,  0,  0, 48,  3,  3, 48,  0, 48,  1,  1, 17,  1, &
  1, 16, 16,  3,  0, 48, 51,  3,  2, 48,  0, 34, 32,  0, 48,  3,  0,  3, 48,  0,  1, 16,  0,  1, &
  0,  0, 17,  0,  0, 51,  0,  3,  0, 48,  0, 34, 32,  0, 50,  3,  3, 48, 48,  3,  0, 16, 17,  1, &
  1, 17,  1,  0, 48,  0, 51,  3,  0, 48,  0,  0,  0,  0, 48,  3,  0,  3,  3,  0,  0,  2,  0,  1, &
  0,  0, 17, 16,  0,  0,  0,  3,  2, 48, 51, 51, 51, 51, 48,  3,  3, 51,  3,  0, 48,  3, 34,  1, &
  1, 17,  0,  0,  0, 48,  0,  3,  0, 48,  0,  0,  0,  0, 50,  3,  0,  0, 48, 48,  0,  2, 50,  1, &
  0,  0, 34, 34, 32,  0, 48, 48,  0, 48, 16, 17, 16, 16, 48,  3,  3, 51, 48, 48, 48,  2, 35,  1, &
  0,  1,  0,  0, 34,  0,  0, 48, 32, 48, 17, 16, 17, 16, 48,  0, 48,  0, 48, 48,  0,  3, 34,  1, &
  3, 48, 16, 51,  2,  1,  0, 48,  3,  1, 16, 16, 16, 16, 48, 32, 48, 51, 48, 48, 48,  3, 50,  1, &
  3, 51,  0,  3,  2,  1,  0, 48,  3,  1, 16, 17, 16, 17,  3,  0, 48,  0, 48, 48,  0,  2, 51,  1, &
  0,  3, 51, 48, 32, 16,  3,  0,  3,  0, 17, 16, 17, 16,  3,  0, 48, 51, 48, 48, 48,  2, 35,  1, &
  0,  0,  0,  0, 32, 16,  3,  2,  3,  0,  0,  0,  0,  0,  3,  0,  3,  0,  3,  0,  0,  3, 34,  1, &
  0, 48, 34, 34, 32, 16,  3,  0, 48, 34, 51, 51, 51, 34,  3,  2,  3,  3,  3,  3,  0, 35, 50,  1, &
  0, 48,  0,  2,  1,  0, 48,  0, 48, 34, 51,  0,  3, 34, 32, 48,  3,  0, 48,  0,  0, 34, 48,  1, &
  0, 48, 34, 34,  1,  0, 48, 32, 48,  0,  0, 51, 51, 34, 32, 48,  0, 48,  0, 48,  3, 50,  1,  1, &
  0, 48,  0, 32, 16,  3,  0,  3,  0, 48, 48,  0,  0,  2, 32, 48, 32, 48,  0,  0, 34,  0, 17,  1, &
  0, 48, 34, 32, 16,  3,  2,  3,  0,  0,  0, 48, 48, 48,  2,  3,  0, 48,  3,  3,  0, 17, 17,  1, &
  0, 48,  0,  1,  0, 48,  0, 48,  1,  0,  0,  0,  0,  0, 48,  3,  0,  3,  0,  0, 17, 17, 17,  1, &
  0, 48, 32, 16,  3,  0, 32, 48, 16, 16, 16,  1,  0, 16,  0,  0, 50,  3,  0, 17, 17, 17, 17,  1, &
  0, 48,  1,  0, 48,  0, 17,  0,  1,  1, 16, 16, 17,  1, 17,  0, 48,  0, 17, 17, 17, 17, 17,  1, &
  0, 48, 16,  3,  0, 17,  1,  1,  1,  1,  0, 16, 16, 17,  0, 16,  0, 17, 17, 17, 17, 17, 17,  1, &
  0, 48, 17, 16, 17,  1,  1,  1, 17,  1, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17,  1, &
  0, 48, 17, 16, 17,  1,  1, 17, 17, 17, 16, 16, 16, 16, 17,  1, 17, 16,  0,  0, 17, 17, 17,  1, &
  0, 48, 17, 17,  0, 16,  1, 16,  0, 16, 16, 16, 16, 16,  0, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  0, 48, 17, 17, 17,  1,  1,  1, 17,  0, 16, 16, 16, 16, 17, 16,  0,  1, 16, 17,  0,  0, 17,  1, &
  0, 48, 17, 17, 17,  1,  1, 16,  0, 17, 16, 16, 16, 16, 17,  0,  0,  0, 16, 16,  0,  0,  1,  1, &
  0, 48, 17, 17, 17,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 48, 17, 17, 17,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 48, 17, 17, 17,  1,  1, 17,  1, 17, 16, 16, 16, 16, 17,  0,  0,  0,  0,  0,  0,  0,  1,  1, &
  0, 48, 17, 17, 17, 16,  0,  1,  0,  1,  0, 16, 16, 16, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  0, 48, 17, 17, 17,  1,  0,  0, 16,  0,  0, 16, 16, 16, 17, 16,  0,  0,  0,  0,  0,  0, 17,  1, &
  0, 48, 17, 17, 17,  1,  1, 17, 16, 17, 16, 16, 16, 16, 17, 17,  0,  0,  0,  0,  0,  1, 17,  1, &
  0, 48, 17, 17, 17, 16,  1, 17, 17, 17, 16,  0,  0,  0, 17, 17, 16,  0,  0,  0,  0, 17, 17,  1, &
  0,  1, 17, 17, 17, 17,  0,  0,  0,  0,  0, 34, 34, 32, 17, 17, 17,  0,  0,  0,  1, 17, 17,  1, &
  0, 17, 17, 17, 17, 17,  2, 34, 34, 34, 34, 32, 32, 32, 33, 17, 17, 16,  0,  0, 17, 17, 17,  1, &
  1, 17, 17, 17, 17, 16, 32, 32, 32, 32, 32, 48, 48, 48,  1, 17, 17, 17,  0,  1, 17, 17, 17,  1, &
  1, 17, 17, 17, 17,  0, 48, 48, 48, 48, 48, 51, 51, 51, 48, 17, 17, 17, 16, 17, 17, 17, 17,  1, &
  1, 17, 17, 17, 16, 51, 51, 51, 51, 51, 51, 51, 51, 51, 51,  1, 17, 17, 17, 17, 17, 17, 17,  1, &
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1 /), (/PictSizeX2,PictSizeY/) )

integer(1), parameter, dimension(PictSizeX2,PictSizeY,3,4) :: CardPict = &
	reshape( (/ PJC, PQC, PKC, &
		    PJS, PQS, PKS, &
		    PJH, PQH, PKH, &
		    PJD, PQD, PKD /), (/PictSizeX2,PictSizeY,3,4/) )

logical, parameter :: X = .true.
logical, parameter :: o = .false.

integer, parameter :: SizeNumX = 9
integer, parameter :: SizeNumY = 14

logical*1, dimension(SizeNumX,SizeNumY), parameter :: num2 = &
		  reshape( (/ o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,o,o,o,o,o,X,X,o, &
			      o,o,o,o,o,X,X,X,o, &
			      o,o,o,o,X,X,X,o,o, &
			      o,o,o,X,X,X,o,o,o, &
			      o,o,X,X,X,o,o,o,o, &
			      o,X,X,X,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num3 = &
		  reshape( (/ o,X,X,X,X,X,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,X,X,o,o, &
			      o,o,o,o,X,X,o,o,o, &
			      o,o,o,X,X,o,o,o,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,o,X,X,X,X,X,X,o, &
			      o,o,o,o,o,o,X,X,o, &
			      o,o,o,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &	
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num4 = &
		  reshape( (/ o,o,o,o,o,X,X,o,o, &
			      o,o,o,o,X,X,X,o,o, &
			      o,o,o,X,X,X,X,o,o, &
 			      o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,o,X,X,o,o, &
			      X,X,X,o,o,X,X,o,o, &
			      X,X,X,X,X,X,X,X,X, &
			      X,X,X,X,X,X,X,X,X, &
			      o,o,o,o,o,X,X,o,o, &
			      o,o,o,o,o,X,X,o,o, &
			      o,o,o,o,X,X,X,X,o, &
			      o,o,o,o,X,X,X,X,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num5 = &
		  reshape( (/ o,X,X,X,X,X,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,o,o,o, &
			      o,X,X,o,o,o,o,o,o, &
			      o,X,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,o,o,o,o,X,X,o, &
			      o,o,o,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num6 = &
		  reshape( (/ o,o,o,X,X,X,X,o,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,o,o,o,o,o, &
			      o,X,X,o,o,o,o,o,o, &
			      o,X,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num7 = &
		  reshape( (/ o,X,X,X,X,X,X,X,o, &
   			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,o,o,o,o,X,X,o,o, &
			      o,o,o,o,o,X,X,o,o, &
			      o,o,o,o,X,X,o,o,o, &
			      o,o,o,o,X,X,o,o,o, &
			      o,o,o,o,X,X,o,o,o, &
			      o,o,o,X,X,o,o,o,o, &
			      o,o,o,X,X,o,o,o,o, &
			      o,o,o,X,X,o,o,o,o, &
			      o,o,o,X,X,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num8 = &
	          reshape( (/ o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num9 = &
		  reshape( (/ o,o,X,X,X,X,X,o,o, &
 		  	      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
  		              o,X,X,o,o,o,X,X,o, &
		              o,X,X,o,o,o,X,X,o, &
		              o,X,X,o,o,o,X,X,o, &
		              o,X,X,X,X,X,X,X,o, &
		              o,o,X,X,X,X,X,X,o, &
		              o,o,o,o,o,o,X,X,o, &
		              o,o,o,o,o,X,X,X,o, &
		              o,o,X,X,X,X,X,o,o, &
		              o,o,X,X,X,X,o,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: num10 = &
		  reshape( (/ X,X,o,o,X,X,X,X,o, &
			      X,X,o,X,X,X,X,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,o,o,X,X, &
			      X,X,o,X,X,X,X,X,X, &
			      X,X,o,o,X,X,X,X,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: numJ = &
		  reshape( (/ o,o,o,o,X,X,X,X,o, &
			      o,o,o,o,X,X,X,X,o, &
      			      o,o,o,o,o,X,X,o,o, &
      			      o,o,o,o,o,X,X,o,o, &
      			      o,o,o,o,o,X,X,o,o, &
      			      o,o,o,o,o,X,X,o,o, &
      			      o,o,o,o,o,X,X,o,o, &
      			      o,o,o,o,o,X,X,o,o, &
			      o,X,X,o,o,X,X,o,o, &
			      o,X,X,o,o,X,X,o,o, &
			      o,X,X,X,X,X,X,o,o, &
			      o,o,X,X,X,X,o,o,o, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: numQ = &
	          reshape( (/ o,o,X,X,X,X,X,o,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      o,o,X,X,X,X,X,o,o, &
			      o,o,o,o,o,X,X,X,o, &
			      o,o,o,o,o,o,X,X,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: numK = &
		  reshape( (/ X,X,X,X,o,X,X,X,X, &
			      X,X,X,X,o,X,X,X,X, &
			      o,X,X,o,o,X,X,o,o, &
			      o,X,X,o,X,X,o,o,o, &
			      o,X,X,X,X,o,o,o,o, &
			      o,X,X,X,o,o,o,o,o, &
			      o,X,X,X,X,o,o,o,o, &
			      o,X,X,o,X,X,o,o,o, &
			      o,X,X,o,o,X,X,o,o, &
			      o,X,X,o,o,o,X,X,o, &
			      X,X,X,X,o,X,X,X,X, &
			      X,X,X,X,o,X,X,X,X, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
logical*1, dimension(SizeNumX,SizeNumY), parameter :: numA = &
		  reshape( (/ o,o,o,o,X,o,o,o,o, &
			      o,o,o,o,X,o,o,o,o, &
			      o,o,o,X,X,X,o,o,o, &
			      o,o,o,X,X,X,o,o,o, &
			      o,o,X,X,o,X,X,o,o, &
			      o,o,X,X,o,X,X,o,o, &
			      o,X,X,o,o,o,X,X,o, &
			      o,X,X,X,X,X,X,X,o, &
			      X,X,X,X,X,X,X,X,X, &
			      X,X,o,o,o,o,o,X,X, &
			      X,X,o,o,o,o,o,X,X, &
			      X,X,o,o,o,o,o,X,X, &
			      o,o,o,o,o,o,o,o,o, &
			      o,o,o,o,o,o,o,o,o /), (/SizeNumX,SizeNumY/) )
			      		
logical*1, dimension(SizeNumX,SizeNumY,13), parameter :: CardNum = &
	reshape( (/ numA, num2, num3, num4, num5, &
		    num6, num7, num8, num9, num10, &
		    numJ, numQ, numK /), (/SizeNumX,SizeNumY,13/) )

integer, parameter :: SmallSymX = 11
integer, parameter :: SmallSymY = 11

logical*1, dimension(SmallSymX,SmallSymY), parameter :: SmallClub = &
	reshape( (/ o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,X,o,X,X,X,o,X,o,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,o,X,X,o,X,o,X,X,o,o, &
		    o,o,o,o,o,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,o,o,o,o,o,o,o,o /), (/SmallSymX,SmallSymY/) )

logical*1, dimension(SmallSymX,SmallSymY), parameter :: SmallHeart = &
	reshape( (/ o,o,o,o,o,o,o,o,o,o,o, &
		    o,o,X,X,X,o,X,X,X,o,o, &
		    o,X,X,X,X,o,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,o,X,X,X,X,X,X,X,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,o,o,X,o,o,o,o,o, &
		    o,o,o,o,o,o,o,o,o,o,o /), (/SmallSymX,SmallSymY/) )

logical*1, dimension(SmallSymX,SmallSymY), parameter :: SmallDiamond = &
	reshape( (/ o,o,o,o,o,o,o,o,o,o,o, &
		    o,o,o,o,o,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,X,X,X,X,X,X,X,o,o, &
		    o,o,X,X,X,X,X,X,X,o,o, &
		    o,o,X,X,X,X,X,X,X,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,o,o,X,o,o,o,o,o, &
		    o,o,o,o,o,o,o,o,o,o,o /), (/SmallSymX,SmallSymY/) )

logical*1, dimension(SmallSymX,SmallSymY), parameter :: SmallSpade = &
	reshape( (/ o,o,o,o,o,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,X,X,X,X,X,o,o,o, &
		    o,o,X,X,X,X,X,X,X,o,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,o, &
		    o,o,X,X,o,X,o,X,X,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,o,X,X,X,o,o,o,o, &
		    o,o,o,o,o,o,o,o,o,o,o /), (/SmallSymX,SmallSymY/) )

logical*1, dimension(SmallSymX,SmallSymY,4), parameter :: SmallSymbol = &
	reshape( (/SmallClub, SmallSpade, SmallHeart, SmallDiamond/), &
		 (/SmallSymX,SmallSymY,4/) )
		    
integer, parameter :: LargeSymX = 15
integer, parameter :: LargeSymY = 15

logical*1, dimension(LargeSymX,LargeSymY), parameter :: LargeClub = &
	reshape( (/ o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,X,X,X,o,X,X,X,o,X,X,X,o,o, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    o,X,X,X,X,X,o,X,o,X,X,X,X,X,o, &
		    o,o,X,X,X,o,o,X,o,o,X,X,X,o,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o /), (/LargeSymX,LargeSymY/) )

logical*1, dimension(LargeSymX,LargeSymY), parameter :: LargeHeart = &
	reshape( (/ o,o,X,X,X,X,o,o,o,X,X,X,X,o,o, &
		    o,X,X,X,X,X,X,o,X,X,X,X,X,X,o, &
		    X,X,X,X,X,X,X,o,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    o,o,X,X,X,X,X,X,X,X,X,X,X,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,o,o,X,o,o,o,o,o,o,o /), (/LargeSymX,LargeSymY/) )
		    
logical*1, dimension(LargeSymX,LargeSymY), parameter :: LargeDiamond = & 
	reshape( (/ o,o,o,o,o,o,o,X,o,o,o,o,o,o,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,X,X,X,X,X,X,X,X,X,X,X,o,o, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    o,o,X,X,X,X,X,X,X,X,X,X,X,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, & 
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,o,o,X,o,o,o,o,o,o,o /), (/LargeSymX,LargeSymY/) )

logical*1, dimension(LargeSymX,LargeSymY), parameter :: LargeSpade = &
	reshape( (/ o,o,o,o,o,o,o,X,o,o,o,o,o,o,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o, &
		    o,o,o,o,X,X,X,X,X,X,X,o,o,o,o, &
		    o,o,o,X,X,X,X,X,X,X,X,X,o,o,o, &
		    o,o,X,X,X,X,X,X,X,X,X,X,X,o,o, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    o,X,X,X,X,X,X,X,X,X,X,X,X,X,o, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,X,X,X,X,X,X,X,X,X, &
		    X,X,X,X,X,X,o,X,o,X,X,X,X,X,X, &
		    o,X,X,X,X,o,o,X,o,o,X,X,X,X,o, &
		    o,o,o,o,o,o,X,X,X,o,o,o,o,o,o, &
		    o,o,o,o,o,X,X,X,X,X,o,o,o,o,o /), (/LargeSymX,LargeSymY/) )

logical*1, dimension(LargeSymX,LargeSymY,4), parameter :: LargeSymbol = &
	reshape( (/LargeClub, LargeSpade, LargeHeart, LargeDiamond/), &
		 (/LargeSymX,LargeSymY,4/) )

integer, parameter :: CardSuitColor( 4 ) = &
   (/ ColorIndexBlack, ColorIndexBlack, ColorIndexRed, ColorIndexRed /)

integer, public :: CardBackColor = ColorIndexGreen

type CL
	integer*1 x, y
end type

integer i_init

type(CL), parameter, dimension(10,10) :: CardLoc = reshape( (/ &
  CL(29,42), (CL(0,0),i_init=1,9), &
  CL(29,11), CL(29,72), (CL(0,0),i_init=1,8), &
  CL(29,11), CL(29,42), CL(29,72), (CL(0,0),i_init=1,7), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), (CL(0,0),i_init=1,6), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(29,42), (CL(0,0),i_init=1,5), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(16,42), CL(42,42), &
	(CL(0,0),i_init=1,4), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(16,42), CL(42,42), &
	CL(29,26), (CL(0,0),i_init=1,3), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(16,32), CL(42,32), &
	CL(16,51), CL(42,51), (CL(0,0),i_init=1,2), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(16,32), CL(42,32), &
	CL(16,51), CL(42,51), CL(29,42), CL(0,0), &
  CL(16,11), CL(42,11), CL(16,72), CL(42,72), CL(16,32), CL(42,32), &
	CL(16,51), CL(42,51), CL(29,21), CL(29,62) /), (/ 10, 10 /) )

integer(1), parameter, dimension(4) :: CardLoc9 = (/ -2, -2, 3, 0 /)

type(card), parameter :: CardBlank = CARD(0_1, 0_1)

type(bitmap), pointer :: CardBitmap
type(bitmap), pointer :: BlankCardBitmap

contains

subroutine CardDrawInit()
integer ix, iy

CardBitmap => BitmapAllocate( CardSizeX, CardSizeY )
call BitmapSetPalette( CardBitmap, &
	 		(/  BitmapRGB(   0,   0,   0 ), &   ! 0 = black
			    BitmapRGB( #FF, #FF, #FF ), &   ! 1 = white
	                    BitmapRGB( #FF,   0,   0 ), &   ! 2 = red
 	                    BitmapRGB( #FF, #FF,   0 ), &   
	                    BitmapRGB( #FF,   0, #FF ), &   ! 4 = 
	                    BitmapRGB(   0, #FF,   0 ), &   ! 
	                    BitmapRGB(   0, #80,   0 ), &   ! 6 = green
	                    BitmapRGB(   0,   0, #80 ) /) ) ! 7 = blue

BlankCardBitmap => BitmapAllocate( CardSizeX, CardSizeY )
call BitmapSetPalette( BlankCardBitmap, &
		        (/ BitmapRGB(   0,   0,   0 ), &
			   BitmapRGB( #FF, #FF, #FF ), &
			   BitmapRGB(   0,   0, #80 ) /) )

! Border
BlankCardBitmap%Pixel(1:CardSizeX,1) = 0
BlankCardBitmap%Pixel(1:CardSizeX,CardSizeY) = 0
BlankCardBitmap%Pixel(1,1:CardSizeY) = 0
BlankCardBitmap%Pixel(CardSizeX,1:CardSizeY) = 0

    do iy = 2,CardSizeY-1
	do ix = 2,CardSizeX-1
 		if( iand(ix+iy,3) < 2 ) then
			BlankCardBitmap%Pixel(ix,iy) = 2
		else
			BlankCardBitmap%Pixel(ix,iy) = 1
		endif
	end do
    end do	

end subroutine CardDrawInit

subroutine CardDraw( CardToDraw, X, Y )
type(card) CardToDraw
integer X, Y
integer ix, iy, LocX, LocY, i

if( CardToDraw%Number == CardBlank%Number ) then
    call BitmapPut( BlankCardBitmap, X, Y )
    return
end if

! Card Background
CardBitmap%Pixel = ColorIndexWhite

if( CardToDraw%Number > 10 ) then
	LocX = 2
else
	LocX = 3
endif

! Border
CardBitmap%Pixel(1:CardSizeX,1) = ColorIndexBlack
CardBitmap%Pixel(1:CardSizeX,CardSizeY) = ColorIndexBlack
CardBitmap%Pixel(1,1:CardSizeY) = ColorIndexBlack
CardBitmap%Pixel(CardSizeX,1:CardSizeY) = ColorIndexBlack

! Numbers: Left Upper and Right Lower
where( CardNum( :,:,CardToDraw%Number ) ) 
CardBitmap%Pixel(LocX+1:LocX+SizeNumX,5:5+SizeNumY-1) = CardSuitColor( CardToDraw%Suit ) 
CardBitmap%Pixel(CardSizeX-LocX:CardSizeX-SizeNumX-LocX+1:-1, &
	   CardSizeY-4:CardSizeY-SizeNumY-4+1:-1) &
				          = CardSuitColor( CardToDraw%Suit )
endwhere

! Small Symbols: Left Upper and Right Lower
where( SmallSymbol( :,:,CardToDraw%Suit ) )
CardBitmap%Pixel(LocX:LocX+SmallSymX-1,19:19+SmallSymY-1) &
					  = CardSuitColor( CardToDraw%Suit ) 
CardBitmap%Pixel(CardSizeX-LocX+1:CardSizeX-SmallSymX-LocX+2:-1, &
	   CardSizeY-18:CardSizeY-SmallSymY-18+1:-1) &
					  = CardSuitColor( CardToDraw%Suit ) 
endwhere

! Large Symbols: Middle of card
if( CardToDraw%Number > 10 ) then
  do iy = 1, PictSizeY
    do ix = 1, PictSizeX2
   	CardBitmap%Pixel( 12+(ix*2-1),11+iy ) = &
		ISHL(CardPict(ix,iy,CardToDraw%Number-10,CardToDraw%Suit),-4)
	CardBitmap%Pixel( 12+(ix*2),11+iy ) = &
		IAND(CardPict(ix,iy,CardToDraw%Number-10,CardToDraw%Suit),15)
    end do
  end do
else
  do i = 1, CardToDraw%Number 
    LocY = CardLoc(i,CardToDraw%Number)%y
    LocX = CardLoc(i,CardToDraw%Number)%x
    if( CardToDraw%Number == 9 .and. i == 9 ) then
	LocY = LocY + CardLoc9( CardToDraw%Suit )
    endif
    if( LocY <= CardSizeY/2 ) then
      where( LargeSymbol( :,:,CardToDraw%Suit ) ) &
         CardBitmap%Pixel(LocX:LocX+LargeSymX-1,LocY:LocY+LargeSymY-1) = &
		CardSuitColor( CardToDraw%Suit )
    else
      where( LargeSymbol( :,:,CardToDraw%Suit ) ) &
         CardBitmap%Pixel(LocX:LocX+LargeSymX-1,LocY+LargeSymY-1:LocY:-1) = &
		CardSuitColor( CardToDraw%Suit )
    endif
  end do
endif

call BitmapPut( CardBitmap, X, Y )

end subroutine CardDraw
		
end module

module cards
use cardbasic
use carddeck
use cardhand
use carddrawmod
contains
subroutine CardInit()
call CardDrawInit()
end subroutine
end module cards
