/****************************************************************************
*
*  APPLE //E EMULATOR FOR WINDOWS                    
*
*  Copyright (C) 1994-96, Michael O'Brien.  All rights reserved.
*
***/

#include "stdhdr.h"
#pragma  hdrstop
#include "applewin.h"

#define  VIEWPORTX   5
#define  VIEWPORTY   5
#define  VIEWPORTCX  560
#define  VIEWPORTCY  384
#define  BUTTONX     (VIEWPORTCX+(VIEWPORTX<<1))
#define  BUTTONY     0
#define  BUTTONCX    45
#define  BUTTONCY    45
#define  FSVIEWPORTX (640-BUTTONCX-5-VIEWPORTCX)
#define  FSVIEWPORTY (480-VIEWPORTCY>>1)
#define  FSBUTTONX   (640-BUTTONCX)
#define  FSBUTTONY   ((480-VIEWPORTCY>>1)-1)
#define  BUTTONS     8

#define  BTN_HELP    0
#define  BTN_RUN     1
#define  BTN_DRIVE1  2
#define  BTN_DRIVE2  3
#define  BTN_FULLSCR 4
#define  BTN_COLOR   5
#define  BTN_DEBUG   6
#define  BTN_SETUP   7

TCHAR   computerchoices[] = TEXT("Apple ][+\0")
                            TEXT("Apple //e\0");
TCHAR   joystickchoices[] = TEXT("Disabled\0")
                            TEXT("PC Joystick\0")
                            TEXT("Keyboard (standard)\0")
                            TEXT("Keyboard (centering)\0")
                            TEXT("Mouse\0");
TCHAR   serialchoices[]   = TEXT("None\0")
                            TEXT("COM1\0")
                            TEXT("COM2\0")
                            TEXT("COM3\0")
                            TEXT("COM4\0");
TCHAR   soundchoices[]    = TEXT("Disabled\0")
                            TEXT("PC Speaker (direct)\0")
                            TEXT("PC Speaker (translated)\0")
                            TEXT("Sound Card\0");
TCHAR   videochoices[]    = TEXT("Monochrome\0")
                            TEXT("Color (standard)\0")
                            TEXT("Color (optimized)\0");
TCHAR   discchoices[]     = TEXT("Authentic Speed\0")
                            TEXT("Enhanced Speed\0");

HBITMAP capsbitmap[2];
HBITMAP diskbitmap[3];
HBITMAP buttonbitmap[BUTTONS];

BOOL    active          = 0;
HBRUSH  btnfacebrush    = (HBRUSH)0;
HPEN    btnfacepen      = (HPEN)0;
HPEN    btnhighlightpen = (HPEN)0;
HPEN    btnshadowpen    = (HPEN)0;
int     buttonactive    = -1;
int     buttondown      = -1;
int     buttonover      = -1;
int     buttonx         = BUTTONX;
int     buttony         = BUTTONY;
HRGN    clipregion      = (HRGN)0;
HDC     framedc         = (HDC)0;
RECT    framerect       = {0,0,0,0};
HWND    framewindow     = (HWND)0;
BOOL    fullscreen      = 0;
BOOL    helpquit        = 0;
BOOL    painting        = 0;
HFONT   smallfont       = (HFONT)0;
HWND    tooltipwindow   = (HWND)0;
BOOL    usingcursor     = 0;
int     viewportx       = VIEWPORTX;
int     viewporty       = VIEWPORTY;

LPDIRECTDRAW        directdraw = (LPDIRECTDRAW)0;
LPDIRECTDRAWSURFACE surface    = (LPDIRECTDRAWSURFACE)0;

void    DrawStatusArea (HDC passdc, BOOL drawflags);
void    EnableTrackbar (HWND window, BOOL enable);
void    FillComboBox (HWND window, int controlid, LPCTSTR choices, int currentchoice);
void    ProcessButtonClick (int button);
void    RelayEvent (UINT message, WPARAM wparam, LPARAM lparam);
void    ResetMachineState ();
void    SetFullScreenMode ();
void    SetNormalMode ();
void    SetUsingCursor (BOOL);

//===========================================================================
BOOL CALLBACK ConfigDlgProc (HWND   window,
                             UINT   message,
                             WPARAM wparam,
                             LPARAM lparam) {
  static BOOL afterclose = 0;
  switch (message) {

    case WM_COMMAND:
      switch (LOWORD(wparam)) {

        case IDOK: {
          BOOL  newcomptype   = (BOOL) SendDlgItemMessage(window,101,CB_GETCURSEL,0,0);
          DWORD newvidtype    = (DWORD)SendDlgItemMessage(window,105,CB_GETCURSEL,0,0);
          DWORD newjoytype    = (DWORD)SendDlgItemMessage(window,102,CB_GETCURSEL,0,0);
          DWORD newsoundtype  = (DWORD)SendDlgItemMessage(window,103,CB_GETCURSEL,0,0);
          DWORD newserialport = (DWORD)SendDlgItemMessage(window,104,CB_GETCURSEL,0,0);
          BOOL  newdisktype   = (BOOL) SendDlgItemMessage(window,109,CB_GETCURSEL,0,0);
          if (newcomptype != apple2e)
            if (MessageBox(window,
                           TEXT("You have changed the emulated computer ")
                           TEXT("type.  This change will not take effect ")
                           TEXT("until the next time you restart the ")
                           TEXT("emulator.\n\n")
                           TEXT("Would you like to restart the emulator now?"),
                           TEXT("Configuration"),
                           MB_ICONQUESTION | MB_YESNO | MB_SETFOREGROUND) == IDYES)
              afterclose = 2;
          if (newdisktype != enhancedisk)
            if (MessageBox(window,
                           TEXT("You have changed the disk speed setting.  ")
                           TEXT("This change will not take effect ")
                           TEXT("until the next time you restart the ")
                           TEXT("emulator.\n\n")
                           TEXT("Would you like to restart the emulator now?"),
                           TEXT("Configuration"),
                           MB_ICONQUESTION | MB_YESNO | MB_SETFOREGROUND) == IDYES)
              afterclose = 2;
          if (!JoySetEmulationType(window,newjoytype)) {
            afterclose = 0;
            return 0;
          }
          if (!SpkrSetEmulationType(window,newsoundtype)) {
            afterclose = 0;
            return 0;
          }
          if (videotype != newvidtype) {
            videotype = newvidtype;
            VideoReinitialize();
            if ((mode != MODE_LOGO) && (mode != MODE_DEBUG))
              VideoRedrawScreen();
          }
          CommSetSerialPort(window,newserialport);
          if (IsDlgButtonChecked(window,106))
            speed = SPEED_NORMAL;
          else
            speed = SendDlgItemMessage(window,108,TBM_GETPOS,0,0);
#define SAVE(a,b) RegSaveValue(TEXT("Configuration"),a,1,b);
          SAVE(TEXT("Computer Emulation"),newcomptype);
          SAVE(TEXT("Joystick Emulation"),joytype);
          SAVE(TEXT("Sound Emulation")   ,soundtype);
          SAVE(TEXT("Serial Port")       ,serialport);
          SAVE(TEXT("Custom Speed")      ,IsDlgButtonChecked(window,107));
          SAVE(TEXT("Emulation Speed")   ,speed);
          SAVE(TEXT("Video Emulation")   ,videotype);
          SAVE(TEXT("Enhance Disk Speed"),newdisktype);
#undef SAVE
          EndDialog(window,1);
          if (afterclose)
            PostMessage(framewindow,WM_USER+afterclose,0,0);
          break;
        }

        case IDCANCEL:
          EndDialog(window,0);
          break;

        case 106:
          SendDlgItemMessage(window,108,TBM_SETPOS,1,SPEED_NORMAL);
          EnableTrackbar(window,0);
          break;

        case 107:
          SetFocus(GetDlgItem(window,108));
          EnableTrackbar(window,1);
          break;

        case 108:
          CheckRadioButton(window,106,107,107);
          EnableTrackbar(window,1);
          break;

        case 111:
          afterclose = 1;
          PostMessage(window,WM_COMMAND,IDOK,(LPARAM)GetDlgItem(window,IDOK));
          break;

        case 112:
          RegSaveValue(TEXT(""),TEXT("RunningOnOS"),0,0);
          if (MessageBox(window,
                         TEXT("The emulator has been set to recalibrate ")
                         TEXT("itself the next time it is started.\n\n")
                         TEXT("Would you like to restart the emulator now?"),
                         TEXT("Configuration"),
                         MB_ICONQUESTION | MB_YESNO | MB_SETFOREGROUND) == IDYES) {
            afterclose = 2;
            PostMessage(window,WM_COMMAND,IDOK,(LPARAM)GetDlgItem(window,IDOK));
          }
          break;

      }
      break;

    case WM_HSCROLL:
      CheckRadioButton(window,106,107,107);
      break;

    case WM_INITDIALOG:
      FillComboBox(window,101,computerchoices,apple2e);
      FillComboBox(window,105,videochoices,videotype);
      FillComboBox(window,102,joystickchoices,joytype);
      FillComboBox(window,103,soundchoices,soundtype);
      FillComboBox(window,104,serialchoices,serialport);
      FillComboBox(window,109,discchoices,enhancedisk);
      SendDlgItemMessage(window,108,TBM_SETRANGE,1,MAKELONG(0,40));
      SendDlgItemMessage(window,108,TBM_SETPAGESIZE,0,5);
      SendDlgItemMessage(window,108,TBM_SETTICFREQ,10,0);
      SendDlgItemMessage(window,108,TBM_SETPOS,1,speed);
      {
        BOOL custom = 1;
        if (speed == 10) {
          custom = 0;
          RegLoadValue(TEXT("Configuration"),TEXT("Custom Speed"),
                       1,(DWORD *)&custom);
        }
        CheckRadioButton(window,106,107,106+custom);
        SetFocus(GetDlgItem(window,custom ? 108 : 106));
        EnableTrackbar(window,custom);
      }
      afterclose = 0;
      break;

    case WM_LBUTTONDOWN: {
      POINT pt = {LOWORD(lparam),
                  HIWORD(lparam)};
      ClientToScreen(window,&pt);
      RECT rect;
      GetWindowRect(GetDlgItem(window,108),&rect);
      if ((pt.x >= rect.left) && (pt.x <= rect.right) &&
          (pt.y >= rect.top) && (pt.y <= rect.bottom)) {
        CheckRadioButton(window,106,107,107);
        EnableTrackbar(window,1);
        SetFocus(GetDlgItem(window,108));
        ScreenToClient(GetDlgItem(window,108),&pt);
        PostMessage(GetDlgItem(window,108),WM_LBUTTONDOWN,
                    wparam,MAKELONG(pt.x,pt.y));
      }
      break;
    }

    case WM_SYSCOLORCHANGE:
      SendDlgItemMessage(window,108,WM_SYSCOLORCHANGE,0,0);
      break;

  }
  return 0;
}

//===========================================================================
void CreateGdiObjects () {
  ZeroMemory(buttonbitmap,BUTTONS*sizeof(HBITMAP));
#define LOADBUTTONBITMAP(bitmapname)  LoadImage(instance,bitmapname,   \
                                                IMAGE_BITMAP,0,0,      \
                                                LR_CREATEDIBSECTION |  \
                                                LR_LOADMAP3DCOLORS |   \
                                                LR_LOADTRANSPARENT);
  buttonbitmap[BTN_HELP   ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("HELP_BUTTON"));
  buttonbitmap[BTN_RUN    ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("RUN_BUTTON"));
  buttonbitmap[BTN_DRIVE1 ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DRIVE1_BUTTON"));
  buttonbitmap[BTN_DRIVE2 ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DRIVE2_BUTTON"));
  buttonbitmap[BTN_FULLSCR] = (HBITMAP)LOADBUTTONBITMAP(TEXT("FULLSCR_BUTTON"));
  buttonbitmap[BTN_COLOR  ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("COLOR_BUTTON"));
  buttonbitmap[BTN_DEBUG  ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DEBUG_BUTTON"));
  buttonbitmap[BTN_SETUP  ] = (HBITMAP)LOADBUTTONBITMAP(TEXT("SETUP_BUTTON"));
  capsbitmap[0] = (HBITMAP)LOADBUTTONBITMAP(TEXT("CAPSOFF_BITMAP"));
  capsbitmap[1] = (HBITMAP)LOADBUTTONBITMAP(TEXT("CAPSON_BITMAP"));
  diskbitmap[0] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DISKOFF_BITMAP"));
  diskbitmap[1] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DISKREAD_BITMAP"));
  diskbitmap[2] = (HBITMAP)LOADBUTTONBITMAP(TEXT("DISKWRITE_BITMAP"));
  btnfacebrush    = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
  btnfacepen      = CreatePen(PS_SOLID,1,GetSysColor(COLOR_BTNFACE));
  btnhighlightpen = CreatePen(PS_SOLID,1,GetSysColor(COLOR_BTNHIGHLIGHT));
  btnshadowpen    = CreatePen(PS_SOLID,1,GetSysColor(COLOR_BTNSHADOW));
  smallfont = CreateFont(11,6,0,0,FW_NORMAL,0,0,0,ANSI_CHARSET,
                         OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,
                         DEFAULT_QUALITY,VARIABLE_PITCH | FF_SWISS,
                         TEXT("Small Fonts"));
}

//===========================================================================
void DeleteGdiObjects () {
  int loop;
  for (loop = 0; loop < BUTTONS; loop++)
    DeleteObject(buttonbitmap[loop]);
  for (loop = 0; loop < 2; loop++)
    DeleteObject(capsbitmap[loop]);
  for (loop = 0; loop < 3; loop++)
    DeleteObject(diskbitmap[loop]);
  DeleteObject(btnfacebrush);
  DeleteObject(btnfacepen);
  DeleteObject(btnhighlightpen);
  DeleteObject(btnshadowpen);
  DeleteObject(smallfont);
}

//===========================================================================
void Draw3dRect (HDC dc, int x1, int y1, int x2, int y2, BOOL out) {
  SelectObject(dc,GetStockObject(NULL_BRUSH));
  SelectObject(dc,out ? btnshadowpen : btnhighlightpen);
  POINT pt[3];
  pt[0].x = x1;    pt[0].y = y2-1;
  pt[1].x = x2-1;  pt[1].y = y2-1;
  pt[2].x = x2-1;  pt[2].y = y1; 
  Polyline(dc,(LPPOINT)&pt,3);
  SelectObject(dc,(out == 1) ? btnhighlightpen : btnshadowpen);
  pt[1].x = x1;    pt[1].y = y1;
  pt[2].x = x2;    pt[2].y = y1;
  Polyline(dc,(LPPOINT)&pt,3);
}

//===========================================================================
void DrawBitmapRect (HDC dc, int x, int y, LPRECT rect, HBITMAP bitmap) {
  HDC memdc = CreateCompatibleDC(dc);
  SelectObject(memdc,bitmap);
  BitBlt(dc,x,y,
         rect->right  + 1 - rect->left,
         rect->bottom + 1 - rect->top,
         memdc,
         rect->left,
         rect->top,
         SRCCOPY);
  DeleteDC(memdc);
}

//===========================================================================
void DrawButton (HDC passdc, int number) {
  FrameReleaseDC();
  HDC dc = (passdc ? passdc : GetDC(framewindow));
  int x  = buttonx;
  int y  = buttony+number*BUTTONCY;
  if (number == buttondown) {
    int loop = 0;
    while (loop++ < 3)
      Draw3dRect(dc,x+loop,y+loop,x+BUTTONCX,y+BUTTONCY,0);
    RECT rect = {0,0,39,39};
    DrawBitmapRect(dc,x+4,y+4,&rect,buttonbitmap[number]);
  }
  else {
    Draw3dRect(dc,x+1,y+1,x+BUTTONCX,y+BUTTONCY,1);
    Draw3dRect(dc,x+2,y+2,x+BUTTONCX-1,y+BUTTONCY-1,1);
    RECT rect = {1,1,40,40};
    DrawBitmapRect(dc,x+3,y+3,&rect,buttonbitmap[number]);
  }
  if ((number == BTN_DRIVE1) || (number == BTN_DRIVE2)) {
    int  offset = (number == buttondown) << 1;
    RECT rect = {x+offset+3,
                 y+offset+31,
                 x+offset+42,
                 y+offset+42};
    SelectObject(dc,smallfont);
    SetTextColor(dc,RGB(0,0,0));
    SetTextAlign(dc,TA_CENTER | TA_TOP);
    SetBkMode(dc,TRANSPARENT);
    ExtTextOut(dc,x+offset+22,rect.top,ETO_CLIPPED,&rect,
               DiskGetName(number-BTN_DRIVE1),
               MIN(8,_tcslen(DiskGetName(number-BTN_DRIVE1))),
               NULL);
  }
  if (!passdc)
    ReleaseDC(framewindow,dc);
}

//===========================================================================
void DrawCrosshairs (int x, int y) {
  static int lastx = 0;
  static int lasty = 0;
  FrameReleaseDC();
  HDC dc = GetDC(framewindow);
#define LINE(x1,y1,x2,y2) MoveToEx(dc,x1,y1,NULL); LineTo(dc,x2,y2);

  // ERASE THE OLD CROSSHAIRS
  if (lastx && lasty)
    if (fullscreen) {
      int loop = 4;
      while (loop--) {
        RECT rect = {0,0,5,5};
        switch (loop) {
          case 0: OffsetRect(&rect,lastx-2,FSVIEWPORTY-5);           break;
          case 1: OffsetRect(&rect,lastx-2,FSVIEWPORTY+VIEWPORTCY);  break;
          case 2: OffsetRect(&rect,FSVIEWPORTX-5,         lasty-2);  break;
          case 3: OffsetRect(&rect,FSVIEWPORTX+VIEWPORTCX,lasty-2);  break;
        }
        FillRect(dc,&rect,(HBRUSH)GetStockObject(BLACK_BRUSH));
      }
    }
    else {
      int loop = 5;
      while (loop--) {
        switch (loop) {
          case 0: SelectObject(dc,GetStockObject(BLACK_PEN));  break;
          case 1: // fall through
          case 2: SelectObject(dc,btnshadowpen);               break;
          case 3: // fall through
          case 4: SelectObject(dc,btnfacepen);                 break;
        }
        LINE(lastx-2,VIEWPORTY-loop-1,
             lastx+3,VIEWPORTY-loop-1);
        LINE(VIEWPORTX-loop-1,lasty-2,
             VIEWPORTX-loop-1,lasty+3);
        if ((loop == 1) || (loop == 2))
          SelectObject(dc,btnhighlightpen);
        LINE(lastx-2,VIEWPORTY+VIEWPORTCY+loop,
             lastx+3,VIEWPORTY+VIEWPORTCY+loop);
        LINE(VIEWPORTX+VIEWPORTCX+loop,lasty-2,
             VIEWPORTX+VIEWPORTCX+loop,lasty+3);
      }
    }

  // DRAW THE NEW CROSSHAIRS
  if (x && y) {
    int loop = 4;
    while (loop--) {
      if ((loop == 1) || (loop == 2))
        SelectObject(dc,GetStockObject(WHITE_PEN));
      else
        SelectObject(dc,GetStockObject(BLACK_PEN));
      LINE(x+loop-2,viewporty-5,
           x+loop-2,viewporty);
      LINE(x+loop-2,viewporty+VIEWPORTCY+4,
           x+loop-2,viewporty+VIEWPORTCY-1);
      LINE(viewportx-5,           y+loop-2,
           viewportx,             y+loop-2);
      LINE(viewportx+VIEWPORTCX+4,y+loop-2,
           viewportx+VIEWPORTCX-1,y+loop-2);
    }
  }
#undef LINE
  lastx = x;
  lasty = y;
  ReleaseDC(framewindow,dc);
}

//===========================================================================
void DrawFrameWindow () {
  FrameReleaseDC();
  PAINTSTRUCT ps;
  HDC         dc = (painting ? BeginPaint(framewindow,&ps)
                             : GetDC(framewindow));
  VideoRealizePalette(dc);

  if (!fullscreen) {

    // DRAW THE 3D BORDER AROUND THE EMULATED SCREEN
    Draw3dRect(dc,
               VIEWPORTX-2,VIEWPORTY-2,
               VIEWPORTX+VIEWPORTCX+2,VIEWPORTY+VIEWPORTCY+2,
               0);
    Draw3dRect(dc,
               VIEWPORTX-3,VIEWPORTY-3,
               VIEWPORTX+VIEWPORTCX+3,VIEWPORTY+VIEWPORTCY+3,
               0);
    SelectObject(dc,btnfacepen);
    Rectangle(dc,
              VIEWPORTX-4,VIEWPORTY-4,
              VIEWPORTX+VIEWPORTCX+4,VIEWPORTY+VIEWPORTCY+4);
    Rectangle(dc,
              VIEWPORTX-5,VIEWPORTY-5,
              VIEWPORTX+VIEWPORTCX+5,VIEWPORTY+VIEWPORTCY+5);

    // DRAW THE TOOLBAR BUTTONS
    int loop = BUTTONS;
    while (loop--)
      DrawButton(dc,loop);
  }

  // DRAW THE STATUS AREA
  DrawStatusArea(dc,DRAW_BACKGROUND | DRAW_LEDS);
  if (painting)
    EndPaint(framewindow,&ps);
  else
    ReleaseDC(framewindow,dc);

  // DRAW THE CONTENTS OF THE EMULATED SCREEN
  if (mode == MODE_LOGO)
    VideoDisplayLogo();
  else if (mode == MODE_DEBUG)
    DebugDisplay(1);
  else
    VideoRedrawScreen();
}

//===========================================================================
void DrawStatusArea (HDC passdc, int drawflags) {
  FrameReleaseDC();
  HDC  dc     = (passdc ? passdc : GetDC(framewindow));
  int  x      = buttonx;
  int  y      = buttony+BUTTONS*BUTTONCY+1;
  int  drive1 = 0;
  int  drive2 = 0;
  BOOL caps   = 0;
  DiskGetLightStatus(&drive1,&drive2);
  KeybGetCapsStatus(&caps);
  if (fullscreen) {
    SelectObject(dc,smallfont);
    SetBkMode(dc,OPAQUE);
    SetBkColor(dc,RGB(0,0,0));
    SetTextAlign(dc,TA_LEFT | TA_TOP);
    SetTextColor(dc,RGB((drive1==2 ? 255 : 0),(drive1==1 ? 255 : 0),0));
    TextOut(dc,x+ 3,y+2,TEXT("1"),1);
    SetTextColor(dc,RGB((drive2==2 ? 255 : 0),(drive2==1 ? 255 : 0),0));
    TextOut(dc,x+13,y+2,TEXT("2"),1);
    if (apple2e) {
      SetTextAlign(dc,TA_RIGHT | TA_TOP);
      SetTextColor(dc,(caps ? RGB(128,128,128) :
                              RGB(  0,  0,  0)));
      TextOut(dc,x+BUTTONCX,y+2,TEXT("Caps"),4);
    }
    SetTextAlign(dc,TA_CENTER | TA_TOP);
    SetTextColor(dc,(mode == MODE_PAUSED ||
                     mode == MODE_STEPPING ? RGB(255,255,255) :
                                             RGB(  0,  0,  0)));
    TextOut(dc,x+BUTTONCX/2,y+13,(mode == MODE_PAUSED ? TEXT(" Paused ") :
                                                        TEXT("Stepping")),8);
  }
  else {
    if (drawflags & DRAW_BACKGROUND) {
      SelectObject(dc,GetStockObject(NULL_PEN));
      SelectObject(dc,btnfacebrush);
      Rectangle(dc,x,y,x+BUTTONCX+2,y+35);
      Draw3dRect(dc,x+1,y+3,x+BUTTONCX,y+31,0);
      SelectObject(dc,smallfont);
      SetTextAlign(dc,TA_CENTER | TA_TOP);
      SetTextColor(dc,RGB(0,0,0));
      SetBkMode(dc,TRANSPARENT);
      TextOut(dc,x+ 7,y+7,TEXT("1"),1);
      TextOut(dc,x+25,y+7,TEXT("2"),1);
    }
    if (drawflags & DRAW_LEDS) {
      RECT rect = {0,0,8,8};
      DrawBitmapRect(dc,x+12,y+8,&rect,diskbitmap[drive1]);
      DrawBitmapRect(dc,x+30,y+8,&rect,diskbitmap[drive2]);
      if (apple2e) {
        RECT rect = {0,0,30,8};
        DrawBitmapRect(dc,x+7,y+19,&rect,capsbitmap[caps != 0]);
      }
    }
    if (drawflags & DRAW_TITLE) {
      TCHAR title[40];
      _tcscpy(title,apple2e ? TITLE : TEXT("Apple ][+ Emulator"));
      switch (mode) {
        case MODE_PAUSED:   _tcscat(title,TEXT(" [Paused]"));    break;
        case MODE_STEPPING: _tcscat(title,TEXT(" [Stepping]"));  break;
      }
      SendMessage(framewindow,WM_SETTEXT,0,(LPARAM)title);
    }
  }
  if (!passdc)
    ReleaseDC(framewindow,dc);
}

//===========================================================================
void EnableTrackbar (HWND window, BOOL enable) {
  EnableWindow(GetDlgItem(window,108),enable);
  int loop = 120;
  while (loop++ < 124)
    EnableWindow(GetDlgItem(window,loop),enable);
}

//===========================================================================
void EraseButton (int number) {
  RECT rect;
  rect.left   = buttonx;
  rect.right  = rect.left+BUTTONCX;
  rect.top    = buttony+number*BUTTONCY;
  rect.bottom = rect.top+BUTTONCY;
  InvalidateRect(framewindow,&rect,1);
}

//===========================================================================
void FillComboBox (HWND window, int controlid, LPCTSTR choices, int currentchoice) {
  HWND combowindow = GetDlgItem(window,controlid);
  SendMessage(combowindow,CB_RESETCONTENT,0,0);
  while (*choices) {
    SendMessage(combowindow,CB_ADDSTRING,0,(LPARAM)(LPCTSTR)choices);
    choices += _tcslen(choices)+1;
  }
  SendMessage(combowindow,CB_SETCURSEL,currentchoice,0);
}

//===========================================================================
LRESULT CALLBACK FrameWndProc (HWND   window,
                               UINT   message,
                               WPARAM wparam,
                               LPARAM lparam) {
  switch (message) {

    case WM_ACTIVATE:
      JoyReset();
      SetUsingCursor(0);
      break;

    case WM_ACTIVATEAPP:
      active = wparam;
      break;

    case WM_CLOSE:
      if (fullscreen)
        SetNormalMode();
      if (!IsIconic(window))
        GetWindowRect(window,&framerect);
      RegSaveValue(TEXT("Preferences"),TEXT("Window X-Position"),1,framerect.left);
      RegSaveValue(TEXT("Preferences"),TEXT("Window Y-Position"),1,framerect.top);
      FrameReleaseDC();
      SetUsingCursor(0);
      if (helpquit) {
        helpquit = 0;
        HtmlHelp(NULL,NULL,HH_CLOSE_ALL,0);
      }
      break;

    case WM_CHAR:
      if ((mode == MODE_RUNNING) || (mode == MODE_LOGO) ||
          ((mode == MODE_STEPPING) && (wparam != TEXT('\x1B'))))
        KeybQueueKeypress((int)wparam,1);
      else if ((mode == MODE_DEBUG) || (mode == MODE_STEPPING))
        DebugProcessChar((TCHAR)wparam);
      break;

    case WM_CREATE:
      framewindow = window;
      CreateGdiObjects();
      SpkrInitialize();
      DragAcceptFiles(window,1);
      break;

    case WM_DDE_INITIATE: {
      ATOM application = GlobalAddAtom(TEXT("applewin"));
      ATOM topic       = GlobalAddAtom(TEXT("system"));
      if(LOWORD(lparam) == application && HIWORD(lparam) == topic)
        SendMessage((HWND)wparam,WM_DDE_ACK,(WPARAM)window,MAKELPARAM(application,topic));
      GlobalDeleteAtom(application);
      GlobalDeleteAtom(topic);
      break;
    }

    case WM_DDE_EXECUTE: {
      LPTSTR filename = (LPTSTR)GlobalLock((HGLOBAL)lparam);
      int error = DiskInsert(0,filename,0,0);
      if (!error) {
        if (!fullscreen)
          DrawButton((HDC)0,BTN_DRIVE1);
        SetForegroundWindow(window);
        ProcessButtonClick(BTN_RUN);
      }
      else
        DiskNotifyInvalidImage(filename,error);
      GlobalUnlock((HGLOBAL)lparam);
      break;
    }

    case WM_DESTROY:
      DragAcceptFiles(window,0);
      DebugDestroy();
      if (!restart) {
        DiskDestroy();
        ImageDestroy();
      }
      CommDestroy();
      CpuDestroy();
      MemDestroy();
      SpkrDestroy();
      VideoDestroy();
      DeleteGdiObjects();
      PostQuitMessage(0);
      break;

    case WM_DISPLAYCHANGE:
      VideoReinitialize();
      break;

    case WM_DROPFILES: {
      TCHAR filename[MAX_PATH];
      DragQueryFile((HDROP)wparam,0,filename,sizeof(filename));
      POINT point;
      DragQueryPoint((HDROP)wparam,&point);
      RECT rect;
      rect.left   = buttonx;
      rect.right  = rect.left+BUTTONCX+1;
      rect.top    = buttony+BTN_DRIVE2*BUTTONCY+1;
      rect.bottom = rect.top+BUTTONCY;
      int error = DiskInsert(PtInRect(&rect,point) ? 1 : 0,filename,0,0);
      if (!error) {
        if (!fullscreen)
          DrawButton((HDC)0,PtInRect(&rect,point) ? BTN_DRIVE2 : BTN_DRIVE1);
        rect.top = buttony+BTN_DRIVE1*BUTTONCY+1;
        if (!PtInRect(&rect,point)) {
          SetForegroundWindow(window);
          ProcessButtonClick(BTN_RUN);
        }
      }
      else
        DiskNotifyInvalidImage(filename,error);
      DragFinish((HDROP)wparam);
      break;
    }

    case WM_KEYDOWN:
      if ((wparam >= VK_F1) && (wparam <= VK_F8) && (buttondown == -1)) {
        SetUsingCursor(0);
        buttondown = wparam-VK_F1;
        if (fullscreen && (buttonover != -1)) {
          if (buttonover != buttondown)
            EraseButton(buttonover);
          buttonover = -1;
        }
        DrawButton((HDC)0,buttondown);
      }
      else if (wparam == VK_F9) {
        videotype++;
        if (videotype > 2)
          videotype = 0;
        VideoReinitialize();
        if ((mode != MODE_LOGO) && (mode != MODE_DEBUG))
          VideoRedrawScreen();
        RegSaveValue(TEXT("Configuration"),TEXT("Video Emulation"),1,videotype);
      }
      else if (wparam == VK_CAPITAL)
        KeybToggleCapsLock();
      else if (wparam == VK_PAUSE) {
        SetUsingCursor(0);
        switch (mode) {
          case MODE_RUNNING:  mode = MODE_PAUSED;              break;
          case MODE_PAUSED:   mode = MODE_RUNNING;             break;
          case MODE_STEPPING: DebugProcessChar(TEXT('\x1B'));  break;
        }
        DrawStatusArea((HDC)0,DRAW_TITLE);
        if ((mode != MODE_LOGO) && (mode != MODE_DEBUG))
          VideoRedrawScreen();
        resettiming = 1;
      }
      else if ((mode == MODE_RUNNING) || (mode == MODE_LOGO) || (mode == MODE_STEPPING)) {
        BOOL autorep  = ((lparam & 0x40000000) != 0);
        BOOL extended = ((lparam & 0x01000000) != 0);
        if ((!JoyProcessKey((int)wparam,extended,1,autorep)) &&
            (mode != MODE_LOGO))
          KeybQueueKeypress((int)wparam,0);
      }
      else if (mode == MODE_DEBUG)
        DebugProcessCommand(wparam);
      if (wparam == VK_F10) {
        SetUsingCursor(0);
        return 0;
      }
      break;

    case WM_KEYUP:
      if ((wparam >= VK_F1) && (wparam <= VK_F8) && (buttondown == (int)wparam-VK_F1)) {
        buttondown = -1;
        if (fullscreen)
          EraseButton(wparam-VK_F1);
        else
          DrawButton((HDC)0,wparam-VK_F1);
        ProcessButtonClick(wparam-VK_F1);
      }
      else
        JoyProcessKey((int)wparam,((lparam & 0x01000000) != 0),0,0);
      break;

    case WM_LBUTTONDOWN:
      if (buttondown == -1) {
        int x = LOWORD(lparam);
        int y = HIWORD(lparam);
        if ((x >= buttonx) &&
            (y >= buttony) &&
            (y <= buttony+BUTTONS*BUTTONCY)) {
          buttonactive = buttondown = (y-buttony-1)/BUTTONCY;
          DrawButton((HDC)0,buttonactive);
          SetCapture(window);
        }
        else if (usingcursor)
          if (wparam & (MK_CONTROL | MK_SHIFT))
            SetUsingCursor(0);
          else
            JoySetButton(0,1);
        else if ((x < buttonx) && JoyUsingMouse() &&
                 ((mode == MODE_RUNNING) || (mode == MODE_STEPPING)))
          SetUsingCursor(1);
      }
      RelayEvent(WM_LBUTTONDOWN,wparam,lparam);
      break;

    case WM_LBUTTONUP:
      if (buttonactive != -1) {
        ReleaseCapture();
        if (buttondown == buttonactive) {
          buttondown = -1;
          if (fullscreen)
            EraseButton(buttonactive);
          else
            DrawButton((HDC)0,buttonactive);
          ProcessButtonClick(buttonactive);
        }
        buttonactive = -1;
      }
      else if (usingcursor)
        JoySetButton(0,0);
      RelayEvent(WM_LBUTTONUP,wparam,lparam);
      break;

    case WM_MOUSEMOVE: {
      int x = LOWORD(lparam);
      int y = HIWORD(lparam);
      int newover = (((x >= buttonx) &&
                      (x <= buttonx+BUTTONCX) &&
                      (y >= buttony) &&
                      (y <= buttony+BUTTONS*BUTTONCY))
                     ? (y-buttony-1)/BUTTONCY : -1);
      if (buttonactive != -1) {
        int newdown = (newover == buttonactive) ? buttonactive : -1;
        if (newdown != buttondown) {
          buttondown = newdown;
          DrawButton((HDC)0,buttonactive);
        }
      }
      else if (fullscreen && (newover != buttonover) && (buttondown == -1)) {
        if (buttonover != -1)
          EraseButton(buttonover);
        buttonover = newover;
        if (buttonover != -1)
          DrawButton((HDC)0,buttonover);
      }
      else if (usingcursor) {
        DrawCrosshairs(x,y);
        JoySetPosition(x-viewportx-2,VIEWPORTCX-4,
                       y-viewporty-2,VIEWPORTCY-4);
      }
      RelayEvent(WM_MOUSEMOVE,wparam,lparam);
      break;
    }

    case WM_NOTIFY:
      if(((LPNMTTDISPINFO)lparam)->hdr.hwndFrom == tooltipwindow &&
         ((LPNMTTDISPINFO)lparam)->hdr.code == TTN_GETDISPINFO)
        ((LPNMTTDISPINFO)lparam)->lpszText =
          (LPTSTR)DiskGetFullName(((LPNMTTDISPINFO)lparam)->hdr.idFrom);
      break;

    case WM_PAINT:
      if (GetUpdateRect(window,NULL,0)) {
        painting = 1;
        DrawFrameWindow();
        painting = 0;
      }
      break;

    case WM_PALETTECHANGED:
      if ((HWND)wparam == window)
        break;
      // fall through

    case WM_QUERYNEWPALETTE:
      DrawFrameWindow();
      break;

    case WM_RBUTTONDOWN:
    case WM_RBUTTONUP:
      if (usingcursor)
        JoySetButton(1,(message == WM_RBUTTONDOWN));
      RelayEvent(message,wparam,lparam);
      break;

    case WM_SYSCOLORCHANGE:
      DeleteGdiObjects();
      CreateGdiObjects();
      break;

    case WM_SYSCOMMAND:
      switch (wparam & 0xFFF0) {
        case SC_KEYMENU:
          if (fullscreen && active)
            return 0;
          break;
        case SC_MINIMIZE:
          GetWindowRect(window,&framerect);
          break;
      }
      break;

    case WM_SYSKEYDOWN:
      PostMessage(window,WM_KEYDOWN,wparam,lparam);
      if ((wparam == VK_F10) || (wparam == VK_MENU))
        return 0;
      break;

    case WM_SYSKEYUP:
      PostMessage(window,WM_KEYUP,wparam,lparam);
      break;

    case WM_USER+1: {
      if (mode != MODE_LOGO)
        if (MessageBox(framewindow,
                       TEXT("Running the benchmarks will reset the state of ")
                       TEXT("the emulated machine, causing you to lose any ")
                       TEXT("unsaved work.\n\n")
                       TEXT("Are you sure you want to do this?"),
                       TEXT("Benchmarks"),
                       MB_ICONQUESTION | MB_YESNO | MB_SETFOREGROUND) == IDNO)
          break;
      UpdateWindow(window);
      ResetMachineState();
      mode = MODE_LOGO;
      DrawStatusArea((HDC)0,DRAW_TITLE);
      HCURSOR oldcursor = SetCursor(LoadCursor(0,IDC_WAIT));
      VideoBenchmark();
      ResetMachineState();
      SetCursor(oldcursor);
      break;
    }

    case WM_USER+2:
      if (mode != MODE_LOGO)
        if (MessageBox(framewindow,
                       TEXT("Restarting the emulator will reset the state ")
                       TEXT("of the emulated machine, causing you to lose any ")
                       TEXT("unsaved work.\n\n")
                       TEXT("Are you sure you want to do this?"),
                       TEXT("Configuration"),
                       MB_ICONQUESTION | MB_YESNO | MB_SETFOREGROUND) == IDNO)
          break;
      restart = 1;
      PostMessage(window,WM_CLOSE,0,0);
      break;

  }
  return DefWindowProc(window,message,wparam,lparam);
}

//===========================================================================
void ProcessButtonClick (int button) {
  switch (button) {

    case BTN_HELP:
      {
        TCHAR filename[MAX_PATH];
        _tcscpy(filename,progdir);
        _tcscat(filename,TEXT("APPLEWIN.CHM"));
        HtmlHelp(framewindow,filename,HH_DISPLAY_TOC,0);
        helpquit = 1;
      }
      break;

    case BTN_RUN:
      if (mode == MODE_LOGO)
        DiskBoot();
      else if (mode == MODE_RUNNING)
        ResetMachineState();
      if ((mode == MODE_DEBUG) || (mode == MODE_STEPPING))
        DebugEnd();
      mode = MODE_RUNNING;
      DrawStatusArea((HDC)0,DRAW_TITLE);
      VideoRedrawScreen();
      resettiming = 1;
      break;

    case BTN_DRIVE1:
    case BTN_DRIVE2:
      DiskSelect(button-BTN_DRIVE1);
      if (!fullscreen)
        DrawButton((HDC)0,button);
      break;

    case BTN_FULLSCR:
      if (fullscreen)
        SetNormalMode();
      else
        SetFullScreenMode();
      break;

    case BTN_COLOR:
      VideoChooseColor();
      break;

    case BTN_DEBUG:
      if (mode == MODE_LOGO)
        ResetMachineState();
      if (mode == MODE_STEPPING)
        DebugProcessChar(TEXT('\x1B'));
      else if (mode == MODE_DEBUG)
        ProcessButtonClick(BTN_RUN);
      else {
        DebugBegin();
      }
      break;

    case BTN_SETUP:
      {
        DialogBox(instance,
                  TEXT("CONFIGURATION_DIALOG"),
                  framewindow,
                  (DLGPROC)ConfigDlgProc);
      }
      break;

  }
}

//===========================================================================
void RelayEvent (UINT message, WPARAM wparam, LPARAM lparam) {
  if (fullscreen)
    return;
  MSG msg;
  msg.hwnd    = framewindow;
  msg.message = message;
  msg.wParam  = wparam;
  msg.lParam  = lparam;
  SendMessage(tooltipwindow,TTM_RELAYEVENT,0,(LPARAM)&msg);
}

//===========================================================================
void ResetMachineState () {
  MemReset();
  DiskBoot();
  VideoResetState();
  CommReset();
  JoyReset();
}

//===========================================================================
void SetFullScreenMode () {
  fullscreen = 1;
  buttonover = -1;
  buttonx    = FSBUTTONX;
  buttony    = FSBUTTONY;
  viewportx  = FSVIEWPORTX;
  viewporty  = FSVIEWPORTY;
  GetWindowRect(framewindow,&framerect);
  SetWindowLong(framewindow,GWL_STYLE,WS_POPUP | WS_SYSMENU | WS_VISIBLE);
  DDSURFACEDESC ddsd;
  ddsd.dwSize = sizeof(ddsd);
  ddsd.dwFlags = DDSD_CAPS;
  ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
  if (DirectDrawCreate(NULL,&directdraw,NULL) != DD_OK ||
      directdraw->SetCooperativeLevel(framewindow,DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN) != DD_OK ||
      directdraw->SetDisplayMode(640,480,8) != DD_OK ||
      directdraw->CreateSurface(&ddsd,&surface,NULL) != DD_OK) {
    SetNormalMode();
    return;
  }
  InvalidateRect(framewindow,NULL,1);
}

//===========================================================================
void SetNormalMode () {
  fullscreen = 0;
  buttonx    = BUTTONX;
  buttony    = BUTTONY;
  viewportx  = VIEWPORTX;
  viewporty  = VIEWPORTY;
  directdraw->RestoreDisplayMode();
  directdraw->SetCooperativeLevel(NULL,DDSCL_NORMAL);
  SetWindowLong(framewindow,GWL_STYLE,
                WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX |
                WS_VISIBLE);
  SetWindowPos(framewindow,0,framerect.left,
                             framerect.top,
                             framerect.right - framerect.left,
                             framerect.bottom - framerect.top,
                             SWP_NOZORDER | SWP_FRAMECHANGED);
  if (surface) {
    surface->Release();
    surface = (LPDIRECTDRAWSURFACE)0;
  }
  directdraw->Release();
  directdraw = (LPDIRECTDRAW)0;
}

//===========================================================================
void SetUsingCursor (BOOL newvalue) {
  if (newvalue == usingcursor)
    return;
  usingcursor = newvalue;
  if (usingcursor) {
    SetCapture(framewindow);
    RECT rect = {viewportx+2,
                 viewporty+2,
                 viewportx+VIEWPORTCX-1,
                 viewporty+VIEWPORTCY-1};
    ClientToScreen(framewindow,(LPPOINT)&rect.left);
    ClientToScreen(framewindow,(LPPOINT)&rect.right);
    ClipCursor(&rect);
    ShowCursor(0);
    POINT pt;
    GetCursorPos(&pt);
    ScreenToClient(framewindow,&pt);
    DrawCrosshairs(pt.x,pt.y);
  }
  else {
    DrawCrosshairs(0,0);
    ShowCursor(1);
    ClipCursor(NULL);
    ReleaseCapture();
  }
}

//
// ----- ALL GLOBALLY ACCESSIBLE FUNCTIONS ARE BELOW THIS LINE -----
//

//===========================================================================
void FrameCreateWindow () {
  int width  = VIEWPORTCX + (VIEWPORTX<<1)
                          + BUTTONCX
                          + (GetSystemMetrics(SM_CXBORDER)<<1)
                          + 5;
  int height = VIEWPORTCY + (VIEWPORTY<<1)
                          + GetSystemMetrics(SM_CYBORDER)
                          + GetSystemMetrics(SM_CYCAPTION)
                          + 5;
  int xpos;
  if (!RegLoadValue(TEXT("Preferences"),TEXT("Window X-Position"),1,(DWORD *)&xpos))
    xpos = (GetSystemMetrics(SM_CXSCREEN)-width) >> 1;
  int ypos;
  if (!RegLoadValue(TEXT("Preferences"),TEXT("Window Y-Position"),1,(DWORD *)&ypos))
    ypos = (GetSystemMetrics(SM_CYSCREEN)-height) >> 1;
  framewindow = CreateWindow(TEXT("APPLE2FRAME"),apple2e ? TITLE
                                                         : TEXT("Apple ][+ Emulator"),
                             WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX |
                             WS_VISIBLE,
                             xpos,ypos,width,height,
                             HWND_DESKTOP,(HMENU)0,instance,NULL);
  InitCommonControls();
  tooltipwindow = CreateWindow(TOOLTIPS_CLASS,NULL,TTS_ALWAYSTIP, 
                               CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT, 
                               framewindow,(HMENU)0,instance,NULL); 
  TOOLINFO toolinfo;
  toolinfo.cbSize = sizeof(toolinfo);
  toolinfo.uFlags = TTF_CENTERTIP;
  toolinfo.hwnd = framewindow;
  toolinfo.hinst = instance;
  toolinfo.lpszText = LPSTR_TEXTCALLBACK;
  toolinfo.rect.left  = BUTTONX;
  toolinfo.rect.right = toolinfo.rect.left+BUTTONCX+1;
  toolinfo.uId = 0;
  toolinfo.rect.top    = BUTTONY+BTN_DRIVE1*BUTTONCY+1;
  toolinfo.rect.bottom = toolinfo.rect.top+BUTTONCY;
  SendMessage(tooltipwindow,TTM_ADDTOOL,0,(LPARAM)&toolinfo);
  toolinfo.uId = 1;
  toolinfo.rect.top    = BUTTONY+BTN_DRIVE2*BUTTONCY+1;
  toolinfo.rect.bottom = toolinfo.rect.top+BUTTONCY;
  SendMessage(tooltipwindow,TTM_ADDTOOL,0,(LPARAM)&toolinfo);
}

//===========================================================================
HDC FrameGetDC () {
  if (!framedc) {
    framedc = GetDC(framewindow);
    SetViewportOrgEx(framedc,viewportx,viewporty,NULL);
  }
  return framedc;
}

//===========================================================================
HDC FrameGetVideoDC (LPBYTE *addr, LONG *pitch) {
  if (fullscreen && active && !painting) {
    RECT rect = {FSVIEWPORTX,
                 FSVIEWPORTY,
                 FSVIEWPORTX+VIEWPORTCX,
                 FSVIEWPORTY+VIEWPORTCY};
    DDSURFACEDESC surfacedesc;
    surfacedesc.dwSize = sizeof(surfacedesc);
    if (surface->Lock(&rect,&surfacedesc,0,NULL) == DDERR_SURFACELOST) {
      surface->Restore();
      surface->Lock(&rect,&surfacedesc,0,NULL);
    }
    *addr  = (LPBYTE)surfacedesc.lpSurface+(VIEWPORTCY-1)*surfacedesc.lPitch;
    *pitch = -surfacedesc.lPitch;
    return (HDC)0;
  }
  else return FrameGetDC();
}

//===========================================================================
void FrameRefreshStatus (int drawflags) {
  DrawStatusArea((HDC)0,drawflags);
}

//===========================================================================
void FrameRegisterClass () {
  WNDCLASSEX wndclass;
  ZeroMemory(&wndclass,sizeof(WNDCLASSEX));
  wndclass.cbSize        = sizeof(WNDCLASSEX);
  wndclass.style         = CS_OWNDC | CS_BYTEALIGNCLIENT;
  wndclass.lpfnWndProc   = FrameWndProc;
  wndclass.hInstance     = instance;
  wndclass.hIcon         = LoadIcon(instance,TEXT("APPLEWIN_ICON"));
  wndclass.hCursor       = LoadCursor(0,IDC_ARROW);
  wndclass.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
  wndclass.lpszClassName = TEXT("APPLE2FRAME");
  wndclass.hIconSm       = (HICON)LoadImage(instance,TEXT("APPLEWIN_ICON"),
                                            IMAGE_ICON,16,16,LR_DEFAULTCOLOR);
  RegisterClassEx(&wndclass);
}

//===========================================================================
void FrameReleaseDC () {
  if (framedc) {
    SetViewportOrgEx(framedc,0,0,NULL);
    ReleaseDC(framewindow,framedc);
    framedc = (HDC)0;
  }
}

//===========================================================================
void FrameReleaseVideoDC () {
  if (fullscreen && active && !painting) {

    // THIS IS CORRECT ACCORDING TO THE DIRECTDRAW DOCS
    RECT rect = {FSVIEWPORTX,
                 FSVIEWPORTY,
                 FSVIEWPORTX+VIEWPORTCX,
                 FSVIEWPORTY+VIEWPORTCY};
    surface->Unlock(&rect);

    // BUT THIS SEEMS TO BE WORKING
    surface->Unlock(NULL);
  }
}
