/****************************************************************************
*
*  APPLE //E EMULATOR FOR WINDOWS                    
*
*  Copyright (C) 1994-96, Michael O'Brien.  All rights reserved.
*
***/

#include "stdhdr.h"
#pragma  hdrstop
#include "applewin.h"

BOOL	apple2e	= 1;
BOOL	autoboot	= 0;
BOOL	behind	= 0;
BOOL	calibrating	= 0;
DWORD	clockgran	= 0;
DWORD	cumulativecycles	= 0;
DWORD	cyclegran	= 0;
DWORD	cyclenum	= 0;
DWORD	emulmsec	= 0;
DWORD	finegraindelay	= 1;
BOOL	fullspeed	= 0;
BOOL	i386	= 0;
HINSTANCE	instance	= (HINSTANCE)0;
DWORD	lastfastpaging	= 0;
DWORD	lasttrimimages	= 0;
int	mode	= MODE_LOGO;
DWORD	needsprecision	= 0;
BOOL	optenhancedisk	= 1;
BOOL	optmonochrome	= 0;
BOOL	optpopuplabels	= 1;
TCHAR	progdir[MAX_PATH]	= TEXT("");
BOOL	resettiming	= 0;
BOOL	restart	= 0;
DWORD	speed	= 100;
BOOL	win31	= 0;
BOOL	flash	= 0;
BOOL	autostarting	= 1;
BOOL	recalibrate	= 0;
char	*argv[5];	// command line parameters
int		argc	= 0;		// parameter count
DWORD	stime;	// time variable
DWORD	maxspeed;
#ifdef RAMWORKS
int		maxexpages=128;	// user requested ram pages
#endif

//===========================================================================
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
#define DEFAULT_NONFLIP_REFRESH_INTERVAL (50000) /* 20 fps */
#define MAX_NONFLIP_REFRESH_INTERVAL (250000) /* 4 fps */
#define MIN_NONFLIP_REFRESH_INTERVAL (cyclegran)
DWORD	nonflipRefreshInterval	=	DEFAULT_NONFLIP_REFRESH_INTERVAL;	
DWORD	minNonflipRefreshInterval = DEFAULT_NONFLIP_REFRESH_INTERVAL;	/* for debugging */
DWORD	maxNonflipRefreshInterval = DEFAULT_NONFLIP_REFRESH_INTERVAL;	/* for debugging */
BOOL	dodynamic=1;	// if we want to adjust the refresh rate
#endif

//===========================================================================

#ifdef SHOW_FPS
DWORD	showFps;
DWORD	totalFrames; 
DWORD	startTime; 
DWORD	endTime;
BOOL	showframes=0;

void InitFps(){
	showFps = 1;
	totalFrames = 0;
	startTime = GetTickCount();
}

void FpsDialog() {
	if (!showframes) return;
	if (showFps) {
		endTime = GetTickCount(); //pete - frame rate
		DWORD elapsedTime = endTime -startTime; //pete - frame rate
		int fp100s = totalFrames * 100000 / elapsedTime;
		static char fpsMessage[128];
		wsprintf(fpsMessage,TEXT("%d.%02d frames DRAWN per second since emulator was reset.\n")
			                TEXT("FPS counter reset."),
							fp100s/100,fp100s%100);
		MessageBox(NULL,fpsMessage,"Frame Rate",MB_OK|MB_ICONINFORMATION);
		InitFps();
	}
};

#endif

//===========================================================================
void CheckFastPaging () {
	if ((pages >= 10) && CpuSupportsFastPaging()) {
		lastfastpaging = cumulativecycles;
		if (cpuemtype == CPU_COMPILING) {
			lasttrimimages = cumulativecycles;
			//MemSetFastPaging(1);
		}
	}
}

//===========================================================================
void CheckCpuType () {
	SYSTEM_INFO sysinfo;
	GetSystemInfo(&sysinfo);
	i386 = (sysinfo.dwProcessorType == PROCESSOR_INTEL_386);
}

//===========================================================================
void CheckWindowsVersion () {
	win31 = ((LOBYTE(LOWORD(GetVersion())) == 3) &&
		(HIBYTE(LOWORD(GetVersion())) <= 0x20));
}

//===========================================================================
int ContinueExecution () {
	static BOOL finegrainlast   = 1;
	static BOOL finegraintiming = 1;
	static BOOL normaldelays    = 0;
	static BOOL pageflipping    = 0;
	static BOOL flashlast = 0;
	static DWORD lastnormaldelays = 1;
	static DWORD lastcycles       = 0;
	static DWORD lastdelay        = 1;	
	// RUN THE CPU, DISK, AND JOYSTICK TIMERS FOR ONE CLOCK TICK'S
	// WORTH OF CYCLES
	int waiting = 0;
	BOOL skippedfinegrain = 0;
	BOOL ranfinegrain     = 0;
	{
		int   loop        = 1+(cyclegran >= 20000);
		DWORD cyclestorun = cyclegran >> (DWORD)(cyclegran >= 20000);
		while (loop--) {
			cyclenum = 0;
			if (((cumulativecycles-needsprecision) > 1500000) && !finegraintiming) {
				do {
					DWORD executedcycles = CpuExecute(2000);
					cyclenum += executedcycles;
					CheckFastPaging();
					DiskUpdatePosition(executedcycles);
					JoyUpdatePosition(executedcycles);
					VideoUpdateVbl(executedcycles,
						(cyclestorun-cyclenum) <= 2000);
				} while (cyclenum < cyclestorun);
			} else {
				DWORD cyclesneeded = 0;
				do {
					DWORD startcycles = cyclenum;
					if (SpkrNeedsAccurateCycleCount()) {
						int loop2 = 3;
						while (loop2--) {
							cyclesneeded += 33;
							if (cyclenum < cyclesneeded)
								cyclenum += CpuExecute(cyclesneeded-cyclenum);
						}
					}
					else {
						cyclesneeded += 100;
						if (cyclenum < cyclesneeded)
							cyclenum += CpuExecute(cyclesneeded-cyclenum);
					}
					CheckFastPaging();
					DiskUpdatePosition(cyclenum-startcycles);
					JoyUpdatePosition(cyclenum-startcycles);
					VideoUpdateVbl(cyclenum-startcycles,
						(cyclestorun-cyclesneeded) < 1000);
					if (finegraintiming && finegrainlast)
						if ((!calibrating) && (SpkrCyclesSinceSound() > 50000))
							skippedfinegrain = 1;
						else {
							ranfinegrain = 1;
							DWORD loop = finegraindelay;
							while (loop--) ;
						}
				} while (cyclesneeded < cyclestorun);
			}
			cumulativecycles += cyclenum;
			SpkrUpdate(cyclenum);
			CommUpdate(cyclenum);
		}
	}
	emulmsec += clockgran;
	if (cpuemtype == CPU_FASTPAGING) {
		if ((!pages) && (cumulativecycles-lastfastpaging > 500000))
			MemSetFastPaging(0);
		else if (cumulativecycles-lasttrimimages > 500000) {
			MemTrimImages();
			lasttrimimages = cumulativecycles;
		}
	}
	pages = 0;
	
	// DETERMINE WHETHER THE SCREEN WAS UPDATED, THE DISK WAS SPINNING,
	// OR THE KEYBOARD I/O PORTS WERE BEING EXCESSIVELY QUERIED THIS
	// CLOCKTICK
	VideoCheckPage(0);
	BOOL diskspinning  = DiskIsSpinning();
	BOOL screenupdated = VideoHasRefreshed();
	BOOL systemidle    = (KeybGetNumQueries() > (clockgran << 2))
		&& (!calibrating) && (!ranfinegrain);
	fullspeed = ((speed == SPEED_MAX) || (GetKeyState(VK_SCROLL) < 0)) && !calibrating;
	if (screenupdated)
		pageflipping = 3;
	
	// IF A TWENTIETH OF A SECOND HAS ELAPSED AND THE SCREEN HAS NOT BEEN
	// UPDATED BUT IT APPEARS TO NEED UPDATING, THEN REFRESH IT
	// also refresh the screen if it's flash time
	static DWORD lasttime = 0;
	static DWORD lastflashtime = 0;
	DWORD currtime = GetTickCount();
	if (currtime-lastflashtime >= 125) {
		flash = flash?0:1;
		lastflashtime = currtime;
	}
	if (mode != MODE_LOGO) {
		if (flash == flashlast) {
			static BOOL  anyupdates     = 0;
			static DWORD lastcycles     = 0;
			static BOOL  lastupdates[2] = {0,0};
			anyupdates |= screenupdated;
			if (((cumulativecycles-lastcycles) >= (DWORD) (
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
				nonflipRefreshInterval
#else
				50000 
#endif
				<< (DWORD)(behind || fullspeed || calibrating || ranfinegrain))) ) {
				lastcycles = cumulativecycles;
				if (((!anyupdates) && (!lastupdates[0]) && (!lastupdates[1]) &&
					VideoApparentlyDirty()) || (flash != flashlast)) {
					VideoCheckPage(1);
					if ((!fullspeed) || 
						(currtime-lasttime >= (DWORD)((graphicsmode || !systemidle) ? 32 : 16))) {
						VideoRefreshScreen();
						lasttime = currtime;
						flashlast = flash;
					}
					screenupdated = 1;
				}
				lastupdates[1] = lastupdates[0];
				lastupdates[0] = anyupdates;
				anyupdates     = 0;
				if (pageflipping)
					pageflipping--;
			}
		} else {
			VideoRefreshScreen();
			lasttime = currtime;
			flashlast = flash;
		}
	}
	
	// IF WE ARE CURRENTLY CALIBRATING THE SYSTEM THEN ADJUST FINE
	// GRAIN TIMING
	if (calibrating) {
		if (cumulativecycles-lastcycles >= 100000) {
			lastcycles = cumulativecycles;
			if (finegraintiming && finegrainlast)
				if (normaldelays+lastnormaldelays <= (DWORD)(clockgran <= 25))
					if (calibrating < 3) {
						calibrating++;
						finegraindelay = lastdelay;
						normaldelays   = 1;
						resettiming    = 1;
					}
					else
						calibrating = 0;
				else {
					lastdelay = finegraindelay;
					switch (calibrating) {
					case 1: finegraindelay += 100;  break;
					case 2: finegraindelay += 10;   break;
					case 3: ++finegraindelay;       break;
				}
			}
			lastnormaldelays = normaldelays;
			normaldelays     = 0;
		}
	}
	
	// IF WE ARE NOT CALIBRATING THE SYSTEM, TURN FINE GRAIN TIMING
	// ON OR OFF BASED ON WHETHER THE DISK IS SPINNING, THE SCREEN IS
	// BEING UPDATED, OR THE SYSTEM IS IDLE
	finegrainlast   = finegraintiming;
	finegraintiming = calibrating ||
		((!systemidle) && (!diskspinning) && (!fullspeed) &&
		(!pageflipping) && (!screenupdated) && 
		SpkrNeedsFineGrainTiming() && !VideoApparentlyDirty());
	
	// COMPARE THE EMULATOR'S CLOCK TO THE REAL TIME CLOCK
	{
		static DWORD milliseconds = 0;
		static DWORD microseconds = 0;
		DWORD currtime = GetTickCount();
		if ((!fullspeed) &&
			((!optenhancedisk) || (!diskspinning)) &&
			(!resettiming)) {
			if ((speed == SPEED_NORMAL) || calibrating)
				milliseconds += clockgran;
			else {
				DWORD delta = (DWORD)((clockgran*2000.0)/pow(2.0,speed/100.0));
				milliseconds += (delta/1000);
				microseconds += (delta % 1000);
				if (microseconds >= 1000) {
					microseconds -= 1000;
					++milliseconds;
				}
			}
			
			// DETERMINE WHETHER WE ARE AHEAD OF OR BEHIND REAL TIME
			if (currtime > milliseconds+1000) {
				behind       = 0;
				milliseconds = currtime;
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
				if (!calibrating)
					nonflipRefreshInterval = DEFAULT_NONFLIP_REFRESH_INTERVAL;	
#endif
			}
			else if ((currtime >= milliseconds+100) && !calibrating) {
				behind = 1;
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
				if (!calibrating) {
					DWORD tweak = (currtime - milliseconds)*1000;
					nonflipRefreshInterval += tweak;
					nonflipRefreshInterval = MIN(nonflipRefreshInterval,MAX_NONFLIP_REFRESH_INTERVAL); 
					maxNonflipRefreshInterval = MAX(maxNonflipRefreshInterval,nonflipRefreshInterval);
				}
#endif
			}
			else if (milliseconds > currtime) {
				static DWORD ahead     = 0;
				static DWORD lastreset = 0;
				ahead += milliseconds-currtime;
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
				if (!calibrating && dodynamic) {
					DWORD tweak = (ahead*1000)/10;
					if (nonflipRefreshInterval > tweak)
						nonflipRefreshInterval -= tweak;
					nonflipRefreshInterval = MAX(nonflipRefreshInterval,MIN_NONFLIP_REFRESH_INTERVAL);
					minNonflipRefreshInterval = MIN(minNonflipRefreshInterval,nonflipRefreshInterval);
				}
#endif
				if (currtime-lastreset > 500) {
					if (ahead >= 200)
						behind = 0;
					ahead     = 0;
					lastreset = currtime;
				}
			}
			
			// IF WE ARE AHEAD OF REAL TIME, WAIT FOR REAL TIME TO CATCH UP
			if ((milliseconds-currtime > 0) &&
				(milliseconds-currtime < 200)) {
				++normaldelays;
				if ((skippedfinegrain || !finegraintiming) && !behind)
					Sleep(1);
				do {
					if (systemidle && !behind) {
						Sleep(1);
						waiting++;
					}
					currtime = GetTickCount();
				} while ((milliseconds-currtime > 0) &&
					(milliseconds-currtime < 200));
			}
			else if (currtime-milliseconds > 250)
				milliseconds += 100;
			
		}
		else {
			behind       = fullspeed;
			milliseconds = currtime;
			resettiming  = 0;
		}
	}
	return ((calibrating-2)*50+(normaldelays+lastnormaldelays <= (DWORD)(clockgran <= 25))*25);
}

//===========================================================================
void DetermineClockGranularity () {
	clockgran = 50;
	DWORD oldticks = GetTickCount();
	int   loop     = 40;
	while (loop-- && (clockgran >= 20)) {
		DWORD newticks;
		do
		newticks = GetTickCount();
		while (oldticks >= newticks);
		if (newticks-oldticks >= 10)
			clockgran = MIN(clockgran,newticks-oldticks);
		oldticks = newticks;
	}
	cyclegran = clockgran*1000;
}

//===========================================================================
// Calibration dialog procedure
LRESULT CALLBACK DlgProc (HWND   window,
                          UINT   message,
                          WPARAM wparam,
                          LPARAM lparam) {
	if (message == WM_CREATE) {
		RECT rect;
		GetWindowRect(window,&rect);
		SIZE size;
		size.cx = rect.right-rect.left;
		size.cy = rect.bottom-rect.top;
		rect.left   = (GetSystemMetrics(SM_CXSCREEN)-size.cx) >> 1;
		rect.top    = (GetSystemMetrics(SM_CYSCREEN)-size.cy) >> 1;
		rect.right  = rect.left+size.cx;
		rect.bottom = rect.top +size.cy;
		MoveWindow(window,
			rect.left,
			rect.top,
			rect.right-rect.left,
			rect.bottom-rect.top,
			0);
#ifdef CTL3D_GO
		Ctl3dSubclassDlgEx(window,0xFFFF);  // comment out to use default dialog styles.
#endif
		ShowWindow(window,SW_SHOW);
	}
	if (win31 && (message == WM_CTLCOLORSTATIC)) {
		SetBkColor((HDC)wparam,0xC0C0C0);
		return (LRESULT)GetStockObject(LTGRAY_BRUSH);
	}
	return DefWindowProc(window,message,wparam,lparam);
}

//===========================================================================
void EnterMessageLoop () {
	MSG message;
	while (GetMessage(&message,0,0,0)) {
		TranslateMessage(&message);
		DispatchMessage(&message);
		while ((mode == MODE_RUNNING) || (mode == MODE_STEPPING) || calibrating)
			if (PeekMessage(&message,0,0,0,PM_REMOVE)) {
				if (message.message == WM_QUIT)
					return;
				TranslateMessage(&message);
				DispatchMessage(&message);
			}
			else if (mode == MODE_STEPPING)
				DebugContinueStepping();
			else {
				ContinueExecution();
				if (fullspeed)
					ContinueExecution();
			}
	}
	while (PeekMessage(&message,0,0,0,PM_REMOVE)) ;
}

//===========================================================================
void GetProgramDirectory () {
	GetModuleFileName((HINSTANCE)0,progdir,MAX_PATH);
	progdir[MAX_PATH-1] = 0;
	int loop = _tcslen(progdir);
	while (loop--)
		if ((progdir[loop] == TEXT('\\')) ||
			(progdir[loop] == TEXT(':'))) {
			progdir[loop+1] = 0;
			break;
		}
}

//===========================================================================
BOOL LoadCalibrationData () {
	if (recalibrate) return 0;
#define LOAD(a,b,c) if (!RegLoadValue(a,b,0,c)) return 0;
	DWORD buildnumber = 0;
	DWORD runningonos = 0;
	LOAD(TEXT(""),TEXT("CurrentBuildNumber"),&buildnumber);
	LOAD(TEXT(""),TEXT("RunningOnOS"),&runningonos);
	if (buildnumber != BUILDNUMBER)
		return 0;
	if (runningonos != GetVersion())
		return 0;
	LOAD(TEXT("Calibration"),TEXT("Clock Granularity"),&clockgran);
	LOAD(TEXT("Calibration"),TEXT("Cycle Granularity"),&cyclegran);
	LOAD(TEXT("Calibration"),TEXT("Precision Timing") ,&finegraindelay);
	LOAD(TEXT("Calibration"),TEXT("MaxSpeed") ,&maxspeed);
#undef LOAD
	return (clockgran && cyclegran && finegraindelay);
}

//===========================================================================
void LoadConfiguration () {
#define LOAD(a,b) RegLoadValue(TEXT("Configuration"),a,1,b);
	LOAD(TEXT("Computer Emulation")  ,(DWORD *)&apple2e);
	LOAD(TEXT("Joystick Emulation")  ,&joytype);
	LOAD(TEXT("Sound Emulation")     ,&soundtype);
	LOAD(TEXT("Serial Port")         ,&serialport);
	LOAD(TEXT("Emulation Speed")     ,&speed);
	LOAD(TEXT("Enhance Disk Speed")  ,(DWORD *)&optenhancedisk);
#ifdef MOUSE_MOVEMENT_JOYSTICK
	LOAD(TEXT("Centering Mouse Range")  ,&mouseRange);
#endif
	LOAD(TEXT("Monochrome Video")    ,(DWORD *)&optmonochrome);
#undef LOAD
}

//===========================================================================
void PerformCalibration () {
	// REGISTER THE WINDOW CLASS OF THE CALIBRATION DIALOG BOX
	WNDCLASS wndclass;
	ZeroMemory(&wndclass,sizeof(WNDCLASS));
	wndclass.lpfnWndProc   = DlgProc;
	wndclass.cbWndExtra    = DLGWINDOWEXTRA;
	wndclass.hInstance     = instance;
	wndclass.hIcon         = LoadIcon(instance,MAKEINTRESOURCE(APPLEWIN_ICON));
	wndclass.hCursor       = LoadCursor(0,IDC_WAIT);
	wndclass.hbrBackground = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
	wndclass.lpszClassName = TEXT("APPLE2CALIBRATION");
	RegisterClass(&wndclass);
	
	// CREATE THE CALIBRATION DIALOG BOX
	HWND dlgwindow = CreateDialog(instance,
		MAKEINTRESOURCE(CALIBRATION_DIALOG),
		(HWND)0,
		NULL);
	
	// PROCESS MESSAGES UNTIL THE DIALOG BOX IS FULLY DRAWN
	MSG message;
	while (PeekMessage(&message,0,0,0,PM_REMOVE)) {
		TranslateMessage(&message);
		DispatchMessage(&message);
	}
	
	// WAIT ONE SECOND FOR DISK ACTIVITY TO STOP
	Sleep(1000);
	
	// SET UP MEMORY FOR THE CALIBRATION
	FillMemory(mem,0x102,0x3E);
	*(mem+0x102) = 0x4C;
	*(mem+0x103) = 0x00;
	*(mem+0x104) = 0x00;
	regs.pc = 0;

			SetDlgItemInt(dlgwindow,CALIBRATION_PROGRESS,50+ContinueExecution(),FALSE);

	stime=GetTickCount();	// how many miliseconds since windows started
	CpuExecute(1023000);	// Run 1023000 ccles (1 second of Apple ][)
	stime = GetTickCount() - stime;	// now, how long did that take in milliseconds?
	maxspeed = 100000.0/stime;	// highest running speed on this computer * 100

	// PERFORM THE CALIBRATION
	calibrating    = 1;
	finegraindelay = maxspeed;
	while (calibrating) {
		SetDlgItemInt(dlgwindow,CALIBRATION_PROGRESS,50+ContinueExecution(),FALSE);
	}
	
	// CLOSE THE DIALOG BOX
	PostMessage(dlgwindow,WM_CLOSE,0,0);
	
}

//===========================================================================
void RegisterExtensions () {
	TCHAR command[MAX_PATH];
	GetModuleFileName((HMODULE)0,command,MAX_PATH);
	command[MAX_PATH-1] = 0;
	TCHAR icon[MAX_PATH];
	wsprintf(icon,TEXT("%s,1"),(LPCTSTR)command);
	_tcscat(command,TEXT(" %1"));
	RegSetValue(HKEY_CLASSES_ROOT,".bin",REG_SZ,"DiskImage",10);
	RegSetValue(HKEY_CLASSES_ROOT,".do" ,REG_SZ,"DiskImage",10);
	RegSetValue(HKEY_CLASSES_ROOT,".dsk",REG_SZ,"DiskImage",10);
	RegSetValue(HKEY_CLASSES_ROOT,".nib",REG_SZ,"DiskImage",10);
	RegSetValue(HKEY_CLASSES_ROOT,".po" ,REG_SZ,"DiskImage",10);
	RegSetValue(HKEY_CLASSES_ROOT,
		"DiskImage",
		REG_SZ,"Disk Image",21);
	RegSetValue(HKEY_CLASSES_ROOT,
		"DiskImage\\DefaultIcon",
		REG_SZ,icon,_tcslen(icon)+1);
	RegSetValue(HKEY_CLASSES_ROOT,
		"DiskImage\\shell\\open\\command",
		REG_SZ,command,_tcslen(command)+1);
}

//===========================================================================
void SaveCalibrationData () {
	RegSaveValue(TEXT(""),TEXT("CurrentBuildNumber"),0,BUILDNUMBER);
	RegSaveValue(TEXT(""),TEXT("CurrentBuildNumber"),1,BUILDNUMBER);
	RegSaveValue(TEXT(""),TEXT("RunningOnOS")       ,0,GetVersion());
	RegSaveValue(TEXT(""),TEXT("RunningOnOS")       ,1,GetVersion());
	RegSaveValue(TEXT("Calibration"),TEXT("Clock Granularity"),0,clockgran);
	RegSaveValue(TEXT("Calibration"),TEXT("Cycle Granularity"),0,cyclegran);
	RegSaveValue(TEXT("Calibration"),TEXT("Precision Timing") ,0,finegraindelay);
	RegSaveValue(TEXT("Calibration"),TEXT("MaxSpeed") ,0,maxspeed);
}

//===========================================================================
void ProcessCommandLine(LPSTR instr) {
	int pos=0, cpos=0;
	BOOL inquote=FALSE;
	static char *arguments=(char *)malloc(strlen(instr) *(sizeof(char)));
	argc = 0;
	argv[0]=NULL;
	while (instr[pos]) {
		argv[argc]=arguments+cpos;	// pointer to pos char in instr
		argc++;
		while ((instr[pos] != '\0') && (inquote || (instr[pos]!=' ')) ){
			if (instr[pos]=='\"') {
				inquote = TRUE-inquote;
			} else {
				arguments[cpos]=instr[pos];
				cpos++;
			}
			pos++;
		}
		arguments[cpos]='\0';
		cpos++;
		if (instr[pos]) pos++;	// let the while end on a single NULL
		if (argv[argc-1][0]=='/') { // it's a command line switch, so check if it's valid
			argc--;	// it's not a file name - don't keep it.
			switch (argv[argc][1]) {
#ifdef SHOW_FPS
			case 'F':	// Frame Rate display
				showframes = 1;
				break;
#endif
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
			case 'D':	// Dynamic Nonflip off
				dodynamic = 0;
				break;
#endif
			case 'S':	// Autostart off
				autostarting = 0;
				break;
			case 'C':	// Recalibrate
				recalibrate = 1;
				break;
#ifdef RAMWORKS
			case 'R':	// Ram size
				maxexpages = atoi(argv[argc]+2);
				if (maxexpages > 127) maxexpages=128;
				else if (maxexpages <1) maxexpages=1;
				break;
#endif
			default:
				{
					char linefmtmsg[512];
				wsprintf(linefmtmsg,TEXT("Invalid Command Line\n")
					TEXT("%s received\n")
					TEXT("Invalid Flag: %s\n\n")
					TEXT("Usage:\nAppleWin [/S] [/C]")
#ifdef RAMWORKS
					TEXT(" [/R{pages}]")
#endif
#ifdef DYNAMIC_NONFLIP_REFRESH_INTERVAL
					TEXT(" [/D]")
#endif
#ifdef SHOW_FPS
					TEXT(" [/F]")
#endif
					TEXT(" [{disk1} [{disk2}]]\n"),
					instr,argv[argc]);
				MessageBox(NULL,linefmtmsg,"Frame Rate",MB_OK|MB_ICONINFORMATION);
				exit(2);
				}
			}
		}
	}
}

//===========================================================================
int APIENTRY WinMain (HINSTANCE passinstance, HINSTANCE, LPSTR incmdline, int) {
	ProcessCommandLine(incmdline);
	
	// DO ONE-TIME INITIALIZATION
	instance = passinstance;
	CheckCpuType();
	CheckWindowsVersion();
	GdiSetBatchLimit(512);
	GetProgramDirectory();
	RegisterExtensions();
#ifdef CTL3D_GO
	Ctl3dRegister(instance);	// comment out for default dialogs
#endif
	FrameRegisterClass();
	ImageInitialize();
	if (!DiskInitialize())
		return 1;
	
	do {
		
		// DO INITIALIZATION THAT MUST BE REPEATED FOR A RESTART
		restart = 0;
		mode    = MODE_LOGO;
		LoadConfiguration();
		DebugInitialize();
		JoyInitialize();
		MemInitialize();
		VideoInitialize();
		if (!LoadCalibrationData()) {
			DetermineClockGranularity();
			PerformCalibration();
			SaveCalibrationData();
			MemDestroy();
			MemInitialize();
		}
		FrameCreateWindow();
		
		// ENTER THE MAIN MESSAGE LOOP
#ifdef SHOW_FPS
		showFps = 0;
#endif
		EnterMessageLoop();
#ifdef SHOW_FPS
		FpsDialog();
#endif
		
	} while (restart);
	return 0;
}
